//
// File:        Method.java
// Package:     gov.llnl.babel.symbols
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Method.java,v 1.18 2003/01/08 20:27:08 dahlgren Exp $
// Description: SIDL method (modifiers, return type, name, args, and throws)
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolUtilities;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * The <code>Method</code> class defines a SIDL method, including modifiers,
 * return type, name, arguments, and exceptions.  Methods may have one of
 * three definition modifiers: final, static, or abstract.  They may also
 * have one of two communication modifiers: local or oneway.  The return type
 * is either a type or null (which indicates void).  The method name is a
 * simple identifier string.  The optional arguments are an ordered collection
 * or arguments.  The method may throw any number of exceptions.
 */
public class Method implements Cloneable {
  public static final int NORMAL   = 0; // communication/definition modifier
  public static final int ABSTRACT = 1; // definition modifier
  public static final int FINAL    = 2; // definition modifier
  public static final int STATIC   = 3; // definition modifier
  public static final int LOCAL    = 1; // communication modifier
  public static final int ONEWAY   = 2; // communication modifier

  public static final String[] s_def_mod = {
    "", "abstract", "final", "static",
  };

  public static final String[] s_comm_mod = {
    "", "local", "oneway",
  };

  private ArrayList d_arguments;
  private Comment   d_comment;
  private int       d_communication_modifier;
  private int       d_definition_modifier;
  private String    d_short_name;	// for langs that support overloading
  private String    d_extension;       // short + extension = long name
  private String    d_long_name;       // keep for convenience
  private HashSet   d_references;
  private HashSet   d_basicarrays;     // basic array types references
  private boolean   d_return_copy;
  private Type      d_return_type;
  private HashSet   d_throws;

  /**
   * Create an empty <code>Method</code> object that will be built by
   * calls to other member functions.
   */
  public Method() {
    d_arguments              = new ArrayList();
    d_comment                = null;
    d_communication_modifier = NORMAL;
    d_definition_modifier    = NORMAL;
    d_short_name             = null;
    d_extension              = null;
    d_long_name              = null;
    d_references             = new HashSet();
    d_basicarrays            = new HashSet();
    d_return_copy            = false;
    d_return_type            = null;
    d_throws                 = new HashSet();
  }

  /**
   * Return a shallow copy of this method object.  This method should
   * never fail since this method is cloneable.  Send in the clones.
   */
  public Method cloneMethod() {
    Method m = null;
    try {
      m = (Method) super.clone();
    } catch (CloneNotSupportedException ex) {
      throw new java.lang.InternalError(ex.getMessage());
    }
    return m;
  }

  /**
   * Add another argument to the end of the list of method arguments.
   */
  public void addArgument(Argument arg) {
    d_arguments.add(arg);
    addTypeReferences(arg.getType());
  }

  /**
   * Return the array of arguments in an <code>ArrayList</code> container.
   */
  public ArrayList getArgumentList() {
    return d_arguments;
  }

  /**
   * Set the comment for the method.  This may be null if there is no
   * comment.
   */
  public void setComment(Comment comment) {
    d_comment = comment;
  }

  /**
   * Get the comment for the method.  This may be null if there is no
   * comment.
   */
  public Comment getComment() {
    return d_comment;
  }

  /**
   * Set the communication modifier for the method.  Valid modifiers are
   * NORMAL, LOCAL, or ONEWAY.
   */
  public void setCommunicationModifier(int modifier) {
    d_communication_modifier = modifier;
  }

  /**
   * Get the communication modifier for this method.  Valid values are
   * NORMAL, LOCAL, or ONEWAY.
   */
  public int getCommunicationModifier() {
    return d_communication_modifier;
  }

  /**
   * Return the communication modifier string for this method.  
   */
  public String getCommunicationModifierString() {
    return s_comm_mod[d_communication_modifier];
  }

  /**
   * Set the definition modifier for the method.  Valid modifiers are
   * NORMAL, ABSTRACT, FINAL, or STATIC.
   */
  public void setDefinitionModifier(int modifier) {
    d_definition_modifier = modifier;
  }

  /**
   * Return the definition modifier for the method.  Valid modifiers are
   * NORMAL, ABSTRACT, FINAL, or STATIC.
   */
  public int getDefinitionModifier() {
    return d_definition_modifier;
  }

  /**
   * Return the explicit definition modifier string for the method based
   * on the type of extendable in which it belongs.  At this time, ABSTRACT 
   * is implied for an Interface and all of the others (except NORMAL) 
   * are invalid.
   *
   * @param is_interface  the boolean indication of the type of 
   *                      extendable to which the method belongs.
   * @return the string associated with the explicit definition modifier
   */
  public String getDefinitionModifier(boolean is_interface) {
    String mod = null;
    if (  (!is_interface)
       || (d_definition_modifier == NORMAL) ) {
      mod = s_def_mod[d_definition_modifier];
    }
    return mod;
  }

  /**
   * Return whether the method is abstract.
   */
  public boolean isAbstract() {
    return d_definition_modifier == ABSTRACT;
  }

  /**
   * Return whether the methos is final.
   */
  public boolean isFinal() {
    return d_definition_modifier == FINAL;
  }

  /**
   * Return whether the method is static.
   */
  public boolean isStatic() {
    return d_definition_modifier == STATIC;
  }

  /**
   * Return <code>true</code> if and only if at least one argument of
   * this method is an array with an ordering specification.  Calling
   * this on methods without array arguments will return <code>false</code>.
   * Calling this method on something like <code>void doIt(in array&lt;int,
   * 2, row-major&gt; x);</code> will return <code>true</code>.
   */
  public boolean hasArrayOrderSpec() {
    if (d_return_type.hasArrayOrderSpec()) return true;
    for(Iterator i = d_arguments.iterator(); i.hasNext(); ){
      if (((Argument)i.next()).hasArrayOrderSpec()) return true;
    }
    return false;
  }

  /**
   * Set the names of the method (a standard SIDL identifier).
   */
  public void setMethodName(String shortName, String extension) {
    d_short_name = shortName;
    d_extension  = extension;
    d_long_name  = shortName + extension;
  }

  /**
   * Set the names of the method (a standard SIDL identifier).  Use of
   * this method will set the long and short name to be identical.
   */
  public void setMethodName(String shortName) {
    setMethodName(shortName, "");
  }

  /**
   * Return the short name of the method (a standard SIDL identifier).
   */
  public String getShortMethodName() {
    return d_short_name;
  }

  /**
   * Return the name extension of the method (a standard SIDL identifier).
   */
  public String getNameExtension() {
    return d_extension;
  }

  /**
   * Return the long name of the method (a standard SIDL identifier).
   */
  public String getLongMethodName() {
    return d_long_name;
  }

  /**
   * Set the copy mode for the return type.  If true, then the return
   * interface or class is copied to the caller.
   */
  public void setReturnCopy(boolean copy) {
    d_return_copy = copy;
  }

  /**
   * Get the copy mode for the return type.  If true, then the return
   * interface or class is copied to the caller.
   */
  public boolean isReturnCopy() {
    return d_return_copy;
  }

  /**
   * Set the return type for the method.  A void return type
   * must be represented by a <code>Type</code> object with value
   * <code>Type.VOID</code>, not a null reference.
   */
  public void setReturnType(Type type) {
    d_return_type = type;
    addTypeReferences(type);
  }

  /**
   * Get the return type for the method.  A void return type is
   * represented by a <code>Type</code> object with a value of
   * <code>Type.VOID</code>.
   */
  public Type getReturnType() {
    return(d_return_type);
  }

  /**
   * Add a symbol identifier to the list of supported exceptions for this
   * method.  No error checking is performed to ensure that the specified
   * symbol is valid in the throws clause; such checking must be performed
   * by the parser.
   */
  public void addThrows(SymbolID id) {
    d_throws.add(id);
    d_references.add(id);
  }

  /**
   * Retrieve the set of exceptions that may be thrown by this method.
   * The exceptions are returned in a <code>Set</code> object that contains
   * the <code>SymbolID</code> symbol identifiers for the exception types.
   */
  public Set getThrows() {
    return d_throws;
  }

  /**
   * Return the set of symbols referred to by this method.  Each element
   * of the set is a <code>SymbolID</code>.
   */
  public Set getSymbolReferences() {
    return d_references;
  }

  /**
   * Return the set of basic array references <code>SymbolID</code>s.
   * This set includes arrays of the fundamental types such as double,
   * int, etc. as <code>SymbolID</code> objects.
   */
  public Set getBasicArrays() {
    return d_basicarrays;
  }


  /**
   * Return the string corresponding to the return type for this method.
   *
   * @param parent_pkg the string containing the parent package.  When not
   *                   null, it is used to strip the package from the return
   *                   string if it is in the specified package.
   * @return           the string containing the possibly abbreviated return
   *                   type.
   */
  public String getReturnType(String parent_pkg) {
    StringBuffer result = new StringBuffer();
    String rtype = d_return_type.getTypeString();
    if (parent_pkg != null) {
      result.append(SymbolUtilities.getSymbolName(rtype, parent_pkg));
    } else {
      result.append(rtype);
    }
    return result.toString();
  }


  /**
   * Return the concatenation of the explicit definition modifier, copy,
   * return type, name and extension.
   *
   * @param is_interface True if interface and want the implicit definition
   *                     modifier excluded from the result; otherwise, false
   * @param parent_pkg   the string containing the parent package.  When not
   *                     null, it is used to strip the package from the return
   *                     string if it is in the specified package.
   * @return             the string containing the preface
   */
  public String getSignaturePreface(boolean is_interface, String parent_pkg) {
    StringBuffer preface = new StringBuffer();

    String dmod = getDefinitionModifier(is_interface);
    if (dmod != null) {
      preface.append(dmod + " ");
    }
    if (d_return_copy) {
      preface.append("copy ");
    }
    preface.append(getReturnType(parent_pkg));
    preface.append(" ");
    preface.append(d_short_name);
    if ((d_extension != null) && (d_extension != "")) {
      preface.append(" [");
      preface.append(d_extension);
      preface.append("] ");
    }
    return preface.toString();
  }


  /**
   * Return the signature of the method, including the definition modifier
   * based on the extendable type.  Also, optionally abbreviate type if in 
   * specified package.
   *
   * @param is_interface True if interface and want the implicit definition
   *                     modifier excluded from the result; otherwise, false
   * @param parent_pkg   the string containing the parent package.  When not
   *                     null, it is used to strip the package from the return
   *                     string if it is in the specified package.
   * @return             the string containing the full signature
   */
  public String getSignature(boolean is_interface, String parent_pkg) {
    StringBuffer signature = new StringBuffer();

    signature.append(getSignaturePreface(is_interface, parent_pkg));
    signature.append("(");
    for (Iterator a = d_arguments.iterator(); a.hasNext(); ) {
      signature.append(((Argument) a.next()).getArgumentString());
      if (a.hasNext()) {
        signature.append(", ");
      }
    }
    signature.append(") " + getCommunicationModifierString());

    if (!d_throws.isEmpty()) {
      signature.append(" throws ");
      for (Iterator t = d_throws.iterator(); t.hasNext(); ) {
        SymbolID sid = (SymbolID) t.next();
        String fname = sid.getFullName();
        if (parent_pkg != null) {
          signature.append(SymbolUtilities.getSymbolName(fname, parent_pkg));
        } else {
          signature.append(fname);
        }
        if (t.hasNext()) {
          signature.append(", ");
        }
      }
    }

    return signature.toString();
  }

  /**
   * Return the signature of the method.  The signature does not include
   * the attributes abstract, final, or static.  It also does not abbreviate
   * package names.
   */
  public String getSignature() {
    return getSignature(false, null);
  }

  /**
   * Compare the signatures of two methods.  For signatures to match, the
   * methods must have the same return types and mode, the same names, the
   * same arguments, and the same throws clauses.  They must also have the
   * same communication modifiers.  The signature does not include modifiers
   * static, abstract, or final.
   */
  public boolean sameSignature(Method m) {
    /*
     * Check modifiers, method names, and return types
     */
    if (d_communication_modifier != m.d_communication_modifier) {
      return false;
    }
    if (!d_short_name.equals(m.d_short_name)) {
      return false;
    }
    if (!d_extension.equals(m.d_extension)) {
      return false;
    }
    if (d_return_copy != m.d_return_copy) {
      return false;
    }
    if (d_return_type == null) {
      if (m.d_return_type != null) {
        return false;
      }
    } else if (!d_return_type.equals(m.d_return_type)) {
      return false;
    }

    /*
     * Check that the two throws lists are the same
     */
    if (d_throws.size() != m.d_throws.size()) {
      return false;
    }
    for (Iterator i = d_throws.iterator(); i.hasNext(); ) {
      if (!m.d_throws.contains(i.next())) {
        return false;
      }
    }

    return sameArguments(m);
  }

  /**
   * Compare the base signatures of the two methods.  For them to match, the
   * methods must have the same short and the same arguments.
   */
  public boolean sameBaseSignature(Method m) {
    /*
     * Check the names
     */
    if (!d_short_name.equals(m.d_short_name)) {
      return false;
    }

    return sameArguments(m);
  }

  /**
   * Compare the arguments of the two methods.  Note that ordering counts!
   * That is the arguments are the same if the same ones appear in the
   * same order.
   */
  private boolean sameArguments(Method m) {
    /*
     * Check that the arguments are the same
     */
    int size = d_arguments.size();
    if (size != m.d_arguments.size()) {
      return false;
    } else {
      for (int a = 0; a < size; a++) {
        if (!d_arguments.get(a).equals(m.d_arguments.get(a))) {
          return false;
        }
      }
    }

    return true;
  }

  /**
   * If the specified type is an enumerated type, an interface, a class,
   * or an array of these objects, then add the symbol identifier of the
   * symbol to the symbol reference set.
   */
  private void addTypeReferences(Type type) {
    if (type != null) {
      if (type.getType() == Type.SYMBOL) {
        d_references.add(type.getSymbolID());
      } else if (type.getType() == Type.ARRAY) {
        type = type.getArrayType();
        addTypeReferences(type);
        if ((type.getType() >= Type.BOOLEAN) &&
            (type.getType() <= Type.STRING)) {
          Version v = new Version();
          v.appendVersionNumber(0);
          d_basicarrays.add(new SymbolID("SIDL." + type.getTypeString(), v));
        }
      }
    }
  }
}
