//
// File:        Symbol.java
// Package:     gov.llnl.babel.symbols
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Symbol.java,v 1.14 2003/10/05 17:49:26 norris Exp $
// Description: abstract base class for all symbols in a symbol table
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Metadata;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Date;
import java.util.Set;

/**
 * The <code>Symbol</code> abstract base class describes the basic
 * characteristics shared by all symbols in the symbol table.  All
 * SIDL symbols have a symbol identifier, a symbol type, a metadata
 * description, and an associated comment.  Four different types of
 * symbols are supported: package, enum, class, and interface.
 */
public abstract class Symbol implements Comparable {
  // these have to mesh with the constants in Type.java
  public final static int ENUM      = 11;// must be numerically least
  public final static int CLASS     = 12;
  public final static int INTERFACE = 13;
  public final static int PACKAGE   = 14;

  public final static String s_type[] = {
   "class", "enum", "interface", "package"
  };

  public final static String SCOPE = ".";

  private Comment  d_comment;
  private Metadata d_metadata;
  private SymbolID d_symbol_id;
  private int      d_symbol_type;
  private boolean  d_user_specified;

  /**
   * Initialize the <code>Symbol</code> abstact base class by providing
   * a symbol identifier and its type (one of CLASS, ENUM, INTERFACE, or
   * PACKAGE).  The metadata is constructed using the current date and time.
   */
  public Symbol(SymbolID id, int type, Comment comment) {
    d_comment     = comment;
    d_metadata    = new Metadata(new Date());
    d_symbol_id   = id;
    d_symbol_type = type;
    d_user_specified = false;
  }

  /**
   * Initialize the <code>Symbol</code> abstract base class by providing
   * a symbol identifier, a symbol type, a comment, and a metadata object.
   */
  public Symbol(SymbolID id, int type, Comment comment, Metadata metadata) {
    d_comment     = comment;
    d_metadata    = metadata;
    d_symbol_id   = id;
    d_symbol_type = type;
    d_user_specified = false;
  }

  /**
   * Return the comment associated with this symbol.  The return argument
   * may be null if no comment was defined.
   */
  public Comment getComment() {
    return d_comment;
  }

  /**
   * Alter the comment.
   * @param comment the new comment.
   */
  public void setComment(Comment comment) {
    d_comment = comment;
  }
   
  /** 
   * Indicate whether this is a user-specified symbol (or one resulting 
   * from dependency resolution)
   */
  public boolean getUserSpecified() {
      return d_user_specified;
  } 

  /** 
   * Specify whether this symbol was given by the user on the Babel command line
   */
  public void setUserSpecified(boolean val) {
    d_user_specified = val;
  }

  /**
   * Return the metadata associated with this symbol.
   */
  public Metadata getMetadata() {
    return d_metadata;
  }

  /**
   * Add a (keyword,value) pair to the metadata for this symbol.
   */
  public void addMetadata(String keyword, String value) {
    d_metadata.addMetadata(keyword, value);
  }

  /**
   * Return the identifier associated with this symbol.
   */
  public SymbolID getSymbolID() {
    return d_symbol_id;
  }

  /**
   * Return the fully qualified name associated with this symbol.
   */
  public String getFullName() {
    return d_symbol_id.getFullName();
  }

  /**
   * Return the type of this symbol, one of CLASS, ENUM, INTERFACE, or
   * PACKAGE.
   */
  public int getSymbolType() {
    return d_symbol_type;
  }

  /**
   * Return the type string associated with this symbol.  The return argument
   * may be null if no type was defined.
   */
  public String getSymbolTypeString() {
    String s = null;
    if ( (CLASS <= d_symbol_type) && (d_symbol_type <= PACKAGE) ) {
      s = s_type[d_symbol_type];
    }
    return s;
  }

  /**
   * Return the set of symbols (in the form of <code>SymbolID</code>)
   * referenced by this particular symbol.  The return argument may be
   * null if this symbol contains no external references.
   */
  public abstract Set getSymbolReferences();

  /**
   * Return the symbol references for the whole type hierarchy rooted 
   * with this symbol. For packages, this recurses down the package
   * hierarchy.
   */
  public abstract Set getAllSymbolReferences();

  /**
   * Return the set of all references to arrays of fundamental types.
   */
  public abstract Set getBasicArrayRefs();

  /**
   * Compare this <code>Symbol</code> with the argument passed in.
   */
  public int compareTo(Object o)
  {
    Symbol sym = (Symbol)o;
    return d_symbol_id.compareTo(sym.d_symbol_id);
  }
}
