/*

 ============================================================================
                   The Apache Software License, Version 1.1
 ============================================================================

 Copyright (C) 1999-2003 The Apache Software Foundation. All rights reserved.

 Redistribution and use in source and binary forms, with or without modifica-
 tion, are permitted provided that the following conditions are met:

 1. Redistributions of  source code must  retain the above copyright  notice,
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

 3. The end-user documentation included with the redistribution, if any, must
    include  the following  acknowledgment:  "This product includes  software
    developed  by the  Apache Software Foundation  (http://www.apache.org/)."
    Alternately, this  acknowledgment may  appear in the software itself,  if
    and wherever such third-party acknowledgments normally appear.

 4. The names "Batik" and  "Apache Software Foundation" must  not  be
    used to  endorse or promote  products derived from  this software without
    prior written permission. For written permission, please contact
    apache@apache.org.

 5. Products  derived from this software may not  be called "Apache", nor may
    "Apache" appear  in their name,  without prior written permission  of the
    Apache Software Foundation.

 THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 FITNESS  FOR A PARTICULAR  PURPOSE ARE  DISCLAIMED.  IN NO  EVENT SHALL  THE
 APACHE SOFTWARE  FOUNDATION  OR ITS CONTRIBUTORS  BE LIABLE FOR  ANY DIRECT,
 INDIRECT, INCIDENTAL, SPECIAL,  EXEMPLARY, OR CONSEQUENTIAL  DAMAGES (INCLU-
 DING, BUT NOT LIMITED TO, PROCUREMENT  OF SUBSTITUTE GOODS OR SERVICES; LOSS
 OF USE, DATA, OR  PROFITS; OR BUSINESS  INTERRUPTION)  HOWEVER CAUSED AND ON
 ANY  THEORY OF LIABILITY,  WHETHER  IN CONTRACT,  STRICT LIABILITY,  OR TORT
 (INCLUDING  NEGLIGENCE OR  OTHERWISE) ARISING IN  ANY WAY OUT OF THE  USE OF
 THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 This software  consists of voluntary contributions made  by many individuals
 on  behalf of the Apache Software  Foundation. For more  information on the
 Apache Software Foundation, please see <http://www.apache.org/>.

*/

package org.apache.batik.dom.svg;

import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.svg.SVGAnimatedBoolean;

/**
 * This class implements the {@link SVGAnimatedBoolean} interface.
 *
 * @author <a href="mailto:stephane@hillion.org">Stephane Hillion</a>
 * @version $Id: SVGOMAnimatedBoolean.java,v 1.4 2003/08/08 11:38:59 vhardy Exp $
 */
public class SVGOMAnimatedBoolean
    implements SVGAnimatedBoolean,
               LiveAttributeValue {

    /**
     * The associated element.
     */
    protected AbstractElement element;

    /**
     * The attribute's namespace URI.
     */
    protected String namespaceURI;

    /**
     * The attribute's local name.
     */
    protected String localName;

    /**
     * The actual boolean value.
     */
    protected boolean baseVal;

    /**
     * The default's attribute value.
     */
    protected String defaultValue;

    /**
     * Whether the mutation comes from this object.
     */
    protected boolean mutate;

    /**
     * Creates a new SVGOMAnimatedBoolean.
     * @param elt The associated element.
     * @param ns The attribute's namespace URI.
     * @param ln The attribute's local name.
     * @param attr The attribute node, if any.
     * @param val The default attribute value, if missing.
     */
    public SVGOMAnimatedBoolean(AbstractElement elt,
                                String ns,
                                String ln,
                                Attr attr,
                                String val) {
        element = elt;
        namespaceURI = ns;
        localName = ln;
        if (attr != null) {
            String s = attr.getValue();
            baseVal = "true".equals(s);
        }
        defaultValue = val;
    }

    /**
     * <b>DOM</b>: Implements {@link SVGAnimatedBoolean#getBaseVal()}.
     */
    public boolean getBaseVal() {
        return baseVal;
    }

    /**
     * <b>DOM</b>: Implements {@link SVGAnimatedBoolean#setBaseVal(boolean)}.
     */
    public void setBaseVal(boolean baseVal) throws DOMException {
        if (this.baseVal != baseVal) {
            mutate = true;
            this.baseVal = baseVal;
            element.setAttributeNS(namespaceURI, localName,
                                   (baseVal) ? "true" : "false");
            mutate = false;
        }
    }

    /**
     * <b>DOM</b>: Implements {@link SVGAnimatedBoolean#getAnimVal()}.
     */
    public boolean getAnimVal() {
        throw new RuntimeException("!!! TODO: getAnimVal()");
    }

    /**
     * Called when an Attr node has been added.
     */
    public void attrAdded(Attr node, String newv) {
        if (!mutate) {
            baseVal = "true".equals(newv);
        }
    }

    /**
     * Called when an Attr node has been modified.
     */
    public void attrModified(Attr node, String oldv, String newv) {
        if (!mutate) {
            baseVal = "true".equals(newv);
        }
    }

    /**
     * Called when an Attr node has been removed.
     */
    public void attrRemoved(Attr node, String oldv) {
        if (!mutate) {
            baseVal = "true".equals(defaultValue);
        }
    }
}
