(* PDF Bookmarks *)
open Utility

type target = int (* Just page number for now *)

type bookmark =
  {level : int;
   text : string;
   target : target;
   isopen : bool}

let remove_bookmarks pdf =
  match Pdf.lookup_direct pdf "/Root" pdf.Pdf.trailerdict with
  | None -> raise (Pdf.PDFError "remove_boomarks: Bad PDF: no root")
  | Some catalog ->
      let catalog' = Pdf.remove_dict_entry catalog "/Outlines" in
        let newcatalognum = Pdf.addobj pdf catalog' in
          {pdf with
            Pdf.root = newcatalognum;
            Pdf.trailerdict =
              Pdf.add_dict_entry
                pdf.Pdf.trailerdict "/Root" (Pdf.Indirect newcatalognum)}


type ntree =
  Br of int * Pdf.pdfobject * ntree list

let rec print_ntree (Br (i, _, l)) =
  Printf.printf "%i (" i;
  iter print_ntree l;
  flprint ")"

let fresh source pdf =
  incr source; Pdf.maxobjnum pdf + !source

(* Flatten a tree and produce a root object for it. Return a list of
(num, pdfobject) pairs with the root first. *)
let flatten_tree source pdf = function
  | [] ->
      let n = fresh source pdf in
        [(n, Pdf.Dictionary [])], n
  | tree ->
      let root_objnum = fresh source pdf in
      (* Add /Parent links to root *)
      let tree =
        let add_root_parent (Br (i, dict, children)) =
          Br
            (i,
             Pdf.add_dict_entry dict "/Parent" (Pdf.Indirect root_objnum),
             children)
        in
          map add_root_parent tree
      in
        let rec really_flatten = function
          Br (i, pdfobject, children) ->
            (i, pdfobject) :: flatten (map really_flatten children)
        in
          let all_but_top = flatten (map really_flatten tree)
          and top, topnum =
            (* Make top level from objects at first level of tree *)
            match extremes tree with
              Br (first, _, _), Br (last, _, _) ->
                 (root_objnum, Pdf.Dictionary
                   [("/First", Pdf.Indirect first); ("/Last", Pdf.Indirect last)]),
                 root_objnum
          in
            top::all_but_top, topnum

(* Add /Count entries to an ntree *)
let add_counts tree = tree

(* Add /Parent entries to an ntree *)
let rec add_parent parent (Br (i, obj, children)) =
  let obj' =
    match parent with
    | None -> obj
    | Some parent_num ->
        Pdf.add_dict_entry obj "/Parent" (Pdf.Indirect parent_num)
  in
    Br (i, obj', map (add_parent (Some i)) children)

(* Add /First and /Last entries to an ntree *)
let rec add_firstlast (Br (i, obj, children)) =
  match children with
  | [] -> (Br (i, obj, children))
  | c ->
      match extremes c with
        Br (i', _, _), Br (i'', _, _) ->
          let obj = Pdf.add_dict_entry obj "/First" (Pdf.Indirect i') in
            let obj = Pdf.add_dict_entry obj "/Last" (Pdf.Indirect i'') in
              (Br (i, obj, map add_firstlast children))
       
(* Add /Next and /Prev entries to an ntree *)
let rec add_next (Br (i, obj, children)) =
  match children with
  | [] -> Br (i, obj, children)
  | [_] -> Br (i, obj, map add_next children)
  | c::cs ->
      let numbers = map (fun (Br (i, _, _)) -> i) cs in
        let children' =
          (map2
             (fun (Br (i, obj, children)) nextnum ->
                Br (i,
                    Pdf.add_dict_entry obj "/Next" (Pdf.Indirect nextnum),
                    children))
             (all_but_last (c::cs))
             numbers)
          @ [last cs]
        in
          Br (i, obj, map add_next children')

let rec add_prev (Br (i, obj, children)) =
  match children with
  | [] -> Br (i, obj, children)
  | [_] -> Br (i, obj, map add_prev children)
  | c::cs ->
      let numbers = map (fun (Br (i, _, _)) -> i) (all_but_last (c::cs)) in
        let children' =
          c::
            map2
              (fun (Br (i, obj, children)) prevnum ->
                 Br (i,
                     Pdf.add_dict_entry obj "/Prev" (Pdf.Indirect prevnum),
                     children))
              cs
              numbers
        in
          Br (i, obj, map add_prev children')

(* Find a page indirect from the page tree of a document, given a page number. *)
let page_object_number pdf destpage =
  try
    Pdf.Indirect (select destpage (Pdf.page_reference_numbers pdf))
  with
    (* The page might not exist in the output *)
    Invalid_argument "select" -> dpr "3b"; Pdf.Null

(* Make a node from a given title, destination page number in a given PDF ond
open flag. *)
let node_of_line pdf title destpage isopen =
  if destpage > 0 then (* destpage = 0 means no destination. *)
    Pdf.Dictionary
      [("/Title", Pdf.String title);
       ("/Dest", Pdf.Array
         [page_object_number pdf destpage; Pdf.Name "/Fit"])]
  else
    Pdf.Dictionary [("/Title", Pdf.String title)]

(* Make an ntree list from a list of parsed bookmark lines. *)
let rec make_outline_ntree source pdf = function
  | [] -> []
  | (n, title, destpage, isopen)::t ->
      let lower, rest = cleavewhile (fun (n', _, _, _) -> n' > n) t in
        let node = node_of_line pdf title destpage isopen in
          Br (fresh source pdf, node, make_outline_ntree source pdf lower)
            ::make_outline_ntree source pdf rest

(* Temporary routine *)
let tuple_of_record r =
  r.level, r.text, r.target, r.isopen

(* Add bookmarks. *)
let add_bookmarks parsed pdf =
  let parsed = map tuple_of_record parsed in
  if parsed = [] then remove_bookmarks pdf else
  begin
    let source = ref 0 in
    let tree = make_outline_ntree source pdf parsed in
      (* Build the (object number, bookmark tree object) pairs. *)
      let pairs, tree_root_num =
        let tree =
          map add_firstlast tree
        in
          let tree =
            match add_next (add_prev (Br (0, Pdf.Null, tree))) with
              Br (_, _, children) -> children
          in
            flatten_tree source pdf (add_counts (map (add_parent None) tree))
      in
        (* Add the objects to the pdf *)
        iter
          (function x -> ignore (Pdf.addobj_given_num pdf x))
          pairs;
          (* Replace the /Outlines entry in the document catalog. *)
          match Pdf.lookup_direct pdf "/Root" pdf.Pdf.trailerdict with
          | None -> raise (Pdf.PDFError "Bad PDF: no root")
          | Some catalog ->
              let catalog' =
                Pdf.add_dict_entry catalog "/Outlines" (Pdf.Indirect tree_root_num)
              in
                let newcatalognum = Pdf.addobj pdf catalog' in
                  {pdf with
                    Pdf.root = newcatalognum;
                    Pdf.trailerdict =
                      Pdf.add_dict_entry
                        pdf.Pdf.trailerdict "/Root" (Pdf.Indirect newcatalognum)}
  end
