/*
 * $Header$
 * $Revision$
 * $Date$
 * ====================================================================
 *
 *  Copyright 2002-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;

/**
 *
 * Unit tests for {@link HttpConnection}.
 *
 * @author Sean C. Sullivan
 *
 * @version $Id$
 *
 */
public class TestHttpConnection extends TestLocalHostBase {
    
    // ------------------------------------------------------------ Constructor
    public TestHttpConnection(String testName) {
        super(testName);
    }

    // ------------------------------------------------------------------- Main
    public static void main(String args[]) {
        String[] testCaseName = { TestHttpConnection.class.getName() };
        junit.textui.TestRunner.main(testCaseName);
    }

    // ------------------------------------------------------- TestCase Methods

    public static Test suite() {
        return new TestSuite(TestHttpConnection.class);
    }


    // ----------------------------------------------------------- Test Methods

    public void testConstructThenClose() {
        HttpConnection conn = new HttpConnection(getHost(), getPort());
        conn.close();
        assertTrue( ! conn.isOpen() );
    }

    public void testConnTimeout() {

        // create a custom protocol that will delay for 500 milliseconds
        Protocol testProtocol = new Protocol(
            "timeout",
            new DelayedProtocolSocketFactory(
                500, 
                Protocol.getProtocol("http").getSocketFactory()
            ),
            getPort()
        );

        HttpConnection conn = new HttpConnection(getHost(), getPort(), testProtocol);
        // 1 ms is short enough to make this fail
        conn.setConnectionTimeout(1);
        try {
            conn.open();
            fail("Should have timed out");
        } catch(IOException e) {
            assertTrue(e instanceof HttpConnection.ConnectionTimeoutException);
            /* should fail */
        }
    }

    public void testForIllegalStateExceptions() {
        HttpConnection conn = new HttpConnection(getHost(), getPort());

        try {
            OutputStream out = conn.getRequestOutputStream();
            fail("getRequestOutputStream did not throw the expected exception");
        }
        catch (IllegalStateException expected) {
            // this exception is expected
        }
        catch (IOException ex) {
            fail("getRequestOutputStream did not throw the expected exception");
        }

        try {
            OutputStream out = new ChunkedOutputStream(conn.getRequestOutputStream());
            fail("getRequestOutputStream(true) did not throw the expected exception");
        }
        catch (IllegalStateException expected) {
            // this exception is expected
        }
        catch (IOException ex) {
            fail("getRequestOutputStream(true) did not throw the expected exception");
        }

        try {
            InputStream in = conn.getResponseInputStream();
            fail("getResponseInputStream() did not throw the expected exception");
        }
        catch (IllegalStateException expected) {
            // this exception is expected
        }
        catch (IOException ex) {
            fail("getResponseInputStream() did not throw the expected exception");
        }

    }
    
    /**
     * A ProtocolSocketFactory that delays before creating a socket.
     */
    class DelayedProtocolSocketFactory implements ProtocolSocketFactory {
        
        private int delay;
        private ProtocolSocketFactory realFactory;
            
        public DelayedProtocolSocketFactory(int delay, ProtocolSocketFactory realFactory) {
            this.delay = delay;
            this.realFactory = realFactory;            
        }
                
        public Socket createSocket(
            String host,
            int port,
            InetAddress clientHost,
            int clientPort
        ) throws IOException, UnknownHostException {
            
            synchronized (this) {
                try {
                    this.wait(delay);
                } catch (InterruptedException e) {}
            }
            return realFactory.createSocket(host, port, clientHost, clientPort);
        }

        public Socket createSocket(String host, int port)
            throws IOException, UnknownHostException {
            synchronized (this) {
                try {
                    this.wait(delay);
                } catch (InterruptedException e) {}
            }
            return realFactory.createSocket(host, port);
        }

    }

}

