/////////////////////////////////////////////////////////////////////////////
// Name:        Cache.h
// Purpose:     Manage cache of transcoded files
// Author:      Alex Thuering
// Created:     29.08.2008
// RCS-ID:      $Id: Cache.h,v 1.2 2009/03/16 23:38:59 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#ifndef DVDSTYLER_CACHE_H
#define DVDSTYLER_CACHE_H

#include "DVD.h"
#include "mediatrc.h"
#include <wx/hashmap.h>

/**
 * Stores cache entry information
 */
class CacheEntry {
public:
	CacheEntry(Vob* vob, DVD* dvd, int idx);
	bool equals(const CacheEntry& entry);
	inline int GetIdx() { return m_idx; }
	inline bool IsUsed() { return m_used; }
	inline void SetUsed(bool used) { m_used = used; }
	
private:
	wxString m_videoFile;
	wxArrayString m_audioFiles;
	wxArrayString m_subtitleFiles;
	wxArrayInt m_formats;
    int m_videoBitrate;
    int m_audioBitrate;
    int m_idx;
    bool m_used;
};

WX_DECLARE_STRING_HASH_MAP(CacheEntry*, CacheMap);

/**
 * Manage cache of transcoded files
 */
class Cache {
public:
	/**
	 * Constructor
	 */
	Cache();
	
	/**
	 * Destructor
	 */
	~Cache();
	
	/**
	 * Sets the temporally directory for cache files
	 */
	bool SetTempDir(wxString tempDir);
	
	/**
	 * Returns temporally directory
	 */
	wxString GetTempDir();
	
	/**
	 * Return count of entries in cache
	 */
	int GetCount();
	
	/**
	 * Checks if there is a file for given VOB in the cache
	 * @return The file name in cache or empty string if no match entry in cache
	 */
	wxString Find(Vob* vob, DVD* dvd, bool removeOld = true);
	
	/**
	 * Adds an entry for given VOB to the cache
	 * @return The file name in cache
	 */
	wxString Add(Vob* vob, DVD* dvd);
	
	/**
	 * Removes all files from cache
	 */
	void Clear();
	
	/**
	 * Begins checking for unused entries
	 */
	void BeginClean();
	
	/**
	 * Removes all unused entries
	 */
	void EndClean();
private:
	wxString m_dir;
	CacheMap m_cacheMap;
	
	/**
	 * @return The file name of file in cache with given index
	 */
	wxString GetFileName(int index);
	
	/**
	 * @return The index for new cache entry
	 */
	int GenerateIndex();
	
	/**
	 * Loads list of cache entries
	 */
	bool Open();
	
	/**
	 * Saves list of cache entries
	 */
	void Save();
};

#endif //DVDSTYLER_CACHE_H
