/////////////////////////////////////////////////////////////////////////////
// Name:        PropDlg.h
// Purpose:     Base class for property dialogs
// Author:      Alex Thuering
// Created:     18.11.2003
// RCS-ID:      $Id: PropDlg.h,v 1.17 2010/02/22 20:19:08 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#ifndef PROP_DLG_H
#define PROP_DLG_H

#include <wx/wx.h>
#include <wx/grid.h>
#include <wx/dynarray.h>
#include <wx/filedlg.h>

/** Base class for property dialogs.
  * Allow you to simply create properties dialogs without any GUI designer.
  * Using this class, you can also create dynamic property dialog.
  */
class wxPropDlg: public wxDialog {
public:
	/**
	 * Constructor
	 */
    wxPropDlg(wxWindow *parent, wxString title = _("Properties"));
    /**
     * Destructor
     */
    virtual ~wxPropDlg();
	
protected:
	wxArrayInt m_types;
	wxArrayPtrVoid m_controls;
	wxArrayInt m_groupIds;
	int m_updateIndex; /** index of property by update (reset) */
	int m_currGroupId;
	void* m_currGroupChecker;
	
	/**
	 * Creates dialog with properties panel and button pane
	 */
	void Create(bool resetButton = false, bool dontShowCheckbox = false);
	
	/**
	 * Creates properties panel.
	 * if sizer != NULL adds properties, else restores default values (reset function).
	 */
	virtual void CreatePropPanel(wxSizer* sizer) = 0;
	
	/**
	 * Creates panel with buttons
	 */
	virtual void CreateButtonPane(wxSizer* sizer, bool resetButton = false, bool dontShowCheckbox = false);
	
	/**
	 * Restores default values in the properties dialog.
	 * Calls CreatePropPanel with sizer = NULL.
	 */
	virtual void Reset();
	
	/**
	 * Populate values on the model.
	 */
	virtual bool SetValues() = 0;

	// Add properties
	wxWindow* propWindow; // default this

	void AddText(wxSizer* sizer, wxString text, int flag = wxALIGN_CENTER_VERTICAL);

	void AddTextProp(wxSizer* sizer, const wxString& label, const wxString& value,
	  bool readonly = false, int width = -1);
	void AddStaticTextProp(wxSizer* sizer, const wxString& label, const wxString& value);
	/**
	 * Adds a static line with given caption to the window
	 */
	void AddStaticLine(wxSizer* sizer, wxString caption);
	/**
	 * Creates label and spin control.
	 * @returns subSizer if width > 0, null otherwise
	 */
	wxSizer* AddSpinProp(wxSizer* sizer, const wxString& label, int value, int min = 0,
	  int max = 100, int width = -1, const wxString& label2 = wxT(""), bool addSpacer = true);
	void AddCheckProp(wxSizer* sizer, const wxString& label, bool value,
	  bool readonly = false, int id = -1);
	void AddRadioProp(wxSizer* sizer, const wxString& label, bool value,
	  long style = wxRB_SINGLE, bool readonly = false, int id = -1);
	void AddRadioGroupProp(wxSizer* sizer, const wxArrayString& text,
	  int value, bool readonly = false);
	/**
	 * Creates label and combobox.
	 * @returns subSizer if width > 0, null otherwise
	 */
	wxSizer* AddComboProp(wxSizer* sizer, const wxString& label, const wxString& value,
	  const wxArrayString& choices, long style = 0, int width = -1, bool addSpacer = true,
	  int id = -1);
	void SetComboData(wxComboBox* ctrl, const wxString& value, const wxArrayString& choices);
	/**
	 * Creates label and choice control.
	 * @returns subSizer if width > 0, null otherwise
	 */
	wxSizer* AddChoiceProp(wxSizer* sizer, const wxString& label, const wxString& value,
	  const wxArrayString& choices, int width = -1, bool addSpacer = true, int id = -1);
	/**
	 * Creates label and choice control with item 'Custom'.
	 * @returns subSizer if width > 0, null otherwise
	 */
	wxSizer* AddChoiceCustomProp(wxSizer* sizer, const wxString& label, const wxString& value,
	  const wxArrayString& choices, int customItemIdx, int width = -1, bool addSpacer = true);
	/** Sets the last control as control to enter the custom value. */
	void SetLastControlCustom(int choiceCtrlIdx, bool enable);
	void SetChoiceData(wxChoice* ctrl, const wxString& value, const wxArrayString& choices);
	void AddGridProp(wxSizer* sizer, const wxArrayPtrVoid& data,
	  const wxString& rowTitle, bool editable);
	void SetGridData(wxGrid* grid, const wxArrayPtrVoid& data,
      const wxString& rowTitle, bool editable);
	void AddFontProp(wxSizer* sizer, const wxString& label,
	  wxFont font, wxString caption = _T("..."));
	void AddFontProp(wxSizer* sizer, const wxString& label,
	  wxFontData font, wxString caption = _T("..."));
	void AddColourProp(wxSizer* sizer, const wxString& label,
	  wxColour colour, int opacity = -1, wxString caption = _T("..."));
	void AddFileProp(wxSizer* sizer, const wxString& label, const wxString& value,
	  int dlgStyle = wxFD_OPEN, wxString caption = _T("..."), wxString wildcard = _T("*.*"));
	void AddDirectoryProp(wxSizer* sizer, const wxString& label,
	  const wxString& value, wxString caption = _T("..."));

	wxSizer* BeginGroup(wxSizer* sizer, wxString title = wxEmptyString,
	  wxString checkTitle = wxEmptyString, bool value = false,
	  bool readonly = false, bool vertical = true, int proportion = 0);
	void EndGroup();
	
	/**
	 * @return the pointer to the control with given index
	 */
	wxControl* GetControl(int index);
	/**
	 * @return the pointer to last added control
	 */
	wxControl* GetLastControl();
	/**
	 * @return index of the last added control
	 */
	int GetLastControlIndex();

	// Get values
	wxString GetString(int index);
	int GetInt(int index);
	bool GetBool(int index);
	void* GetClientData(int index);
	wxArrayPtrVoid GetGrid(int index);
	wxFontData GetFont(int index);
	wxColour GetColour(int index);
	
	void* m_currObject;
	int m_currObjectItem;
  	virtual void OnCancel(wxCommandEvent& event);
    virtual void OnOk(wxCommandEvent& event);
    virtual void OnReset(wxCommandEvent& event);
    virtual void OnSelectCustomChoice(wxCommandEvent& evt);
	virtual void OnSelectFont(wxCommandEvent& event);
	virtual void OnSelectColour(wxCommandEvent& event);
	virtual void OnSelectFile(wxCommandEvent& event);
	virtual void OnSelectDir(wxCommandEvent& event);
	virtual void OnGroupCheck(wxCommandEvent& event);
	virtual void OnCellLeftClick(wxGridEvent& event);
	virtual void OnCellRightClick(wxGridEvent& event);
	virtual void OnCellChange(wxGridEvent& event);
	virtual void OnRowDelete(wxCommandEvent& event);
	
  private:
    DECLARE_EVENT_TABLE()
};

#endif // PROP_DLG_H
