/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.target;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.team.core.Team;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.core.sync.IRemoteResource;
import org.eclipse.team.internal.core.streams.ProgressMonitorInputStream;
import org.eclipse.team.internal.core.target.IRemoteTargetResource;
import org.eclipse.team.internal.core.target.ITargetRunnable;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;
import org.eclipse.team.internal.ftp.client.Constants;
import org.eclipse.team.internal.ftp.client.FTPDirectoryEntry;
import org.eclipse.team.internal.ftp.client.FTPServerException;
import org.eclipse.team.internal.ftp.client.IFTPRunnable;

public class FTPRemoteTargetResource extends PlatformObject implements IRemoteTargetResource {

	FTPTargetProvider provider;
	private IPath providerRelativePath;
	private boolean isContainer; /* this is an immutable characteristic, once the handle is created */
	private FTPDirectoryEntry entry; /* chache of remote information */
	
	private static final FTPDirectoryEntry DOES_NOT_EXIST = new FTPDirectoryEntry(null, false, false, 0 , null);
	
	private static final int MAX_TRANSFER_SIZE = 32788;

	/**
	 * Constructor for FTPRemoteTargetResource.
	 */
	public FTPRemoteTargetResource(FTPTargetProvider provider, IPath urlRelativePath, boolean isContainer) {
		this(provider, urlRelativePath, isContainer, null);
	}

	/**
	 * Constructor for FTPRemoteTargetResource.
	 */
	public FTPRemoteTargetResource(FTPTargetProvider provider, IPath urlRelativePath, boolean isContainer, FTPDirectoryEntry entry) {
		this.provider = provider;
		this.providerRelativePath = urlRelativePath;
		this.isContainer = isContainer;
		this.entry = entry;
	}
	
	/**
	 * Delete the resource remotely. If the resource is a container,
	 * delete the memebers before deleting the resource irself.
	 */
	public void delete(IProgressMonitor progress) throws TeamException{
		provider.run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				if (isContainer()) {
					monitor.beginTask(null, 100);
					IRemoteResource[] children = members(Policy.subMonitorFor(monitor, 10));
					for (int i = 0; i < children.length; i++) {
						FTPRemoteTargetResource remoteResource = (FTPRemoteTargetResource)children[i];
						remoteResource.delete(Policy.subMonitorFor(monitor, 80 / children.length));
					}
					provider.deleteDirectory(getProviderRelativePath(), Policy.subMonitorFor(monitor, 10));
				} else {
					provider.deleteFile(getProviderRelativePath(), monitor);
				}
			}
		}, Policy.monitorFor(progress));
	}
	
	/**
	 * @see IRemoteTargetResource#exists()
	 */
	public boolean exists(IProgressMonitor progress) throws TeamException {
		entry = null;
		getEntry(progress);
		return entry != null;
	}

	/**
	 * @see IRemoteResource#getContents(IProgressMonitor)
	 */
	public InputStream getContents(IProgressMonitor progress) throws TeamException {
		final InputStream result[] = new InputStream[] { null };
		provider.run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				monitor.beginTask(null, 100);
				try {
					// get the size and only transfer if it is under the max transfer size
					int size = getSize(Policy.subMonitorFor(monitor, 10));
					if (size > MAX_TRANSFER_SIZE) {
						result[0] = new ByteArrayInputStream(Policy.bind("FTPRemoteTargetResource.remoteFileTooBig").getBytes()); //$NON-NLS-1$
						return;
					}
					final String title = Policy.bind("FTPClient.getFile", getProviderRelativePath().toString()); //$NON-NLS-1$
					monitor.subTask(title);
					InputStream in = provider.getContents(getProviderRelativePath(), isBinary(), Policy.subMonitorFor(monitor, 10));
					// setup progress monitoring
					IProgressMonitor subMonitor = Policy.subMonitorFor(monitor, 80);
					try {
						subMonitor.beginTask(null, size);
						subMonitor.subTask(Policy.bind("FTPClient.transferNoSize", title)); //$NON-NLS-1$
						in = new ProgressMonitorInputStream(in, 0, Constants.TRANSFER_PROGRESS_INCREMENT, subMonitor) {
							long previousBytesRead = 0;
							protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
								if (bytesRead == 0) return;
								long worked = bytesRead  - previousBytesRead;
								previousBytesRead = bytesRead;
								monitor.worked((int)worked);
								monitor.subTask(Policy.bind("FTPClient.transferUnknownSize", //$NON-NLS-1$
									new Object[] { title, Long.toString(bytesRead >> 10) }));
							}
						};
						
						// Transfer the contents
						byte[] buffer = new byte[1024];
						ByteArrayOutputStream out = new ByteArrayOutputStream();
						int read;
						try {
							try {
								while ((read = in.read(buffer)) >= 0) {
									out.write(buffer, 0, read);
								}
							} finally {
								out.close();
							}
						} catch (IOException e) {
							throw FTPPlugin.wrapException(e);
						} finally {
							try {
								in.close();
							} catch (IOException e) {
								FTPPlugin.log(FTPPlugin.wrapException(e));
							}
						}
						result[0] = new ByteArrayInputStream(out.toByteArray());
					} finally {
						subMonitor.done();
					}
				} finally {
					monitor.done();
				}
			}
		}, Policy.monitorFor(progress));
		return result[0];
	}

	/**
	 * Method isBinary.
	 * @return boolean
	 */
	private boolean isBinary() {
		return Team.getType(new IStorage() {
			public InputStream getContents() throws CoreException {
				// use empty contents when trying to determine the file type
				return new ByteArrayInputStream(new byte[0]);
			}
			public IPath getFullPath() {
				return new Path(getURL().getPath());
			}
			public String getName() {
				return FTPRemoteTargetResource.this.getName();
			}
			public boolean isReadOnly() {
				return true;
			}
			public Object getAdapter(Class adapter) {
				return null;
			}
		}) != Team.TEXT;
	}

	/**
	 * @see IRemoteTargetResource#getLastModified()
	 */
	public String getLastModified(IProgressMonitor progress) throws TeamException {
		FTPDirectoryEntry entry = getEntry(progress);
		if (entry == null) {
			throw new FTPException(Policy.bind("FTPRemoteTargetResource.doesNotExist", getURL().toString())); //$NON-NLS-1$
		}
		return entry.getModTime().toString();
	}

	/**
	 * Same as getLastModified except we force a trip to the server
	 */
	protected String getReleasedIdentifier(IProgressMonitor progress) throws TeamException {
		entry = null;
		return getReleasedIdentifierIfNecessary(progress);
	}
	
	protected String getReleasedIdentifierIfNecessary(IProgressMonitor progress) throws TeamException {
		// Append the size the the timestamp in an attempt to compensate for timestamp granularity of 1 minute
		progress = Policy.monitorFor(progress);
		progress.beginTask(null, 100);
		try {
			return getLastModified(Policy.subMonitorFor(progress, 50)) + getSize(Policy.subMonitorFor(progress, 50));
		} finally {
			progress.done();
		}
	}
	
	/**
	 * @see IRemoteResource#getName()
	 */
	public String getName() {
		IPath path = providerRelativePath;
		if (path.isEmpty() || path.isRoot()) {
			return getProviderURL().toExternalForm();
		}
		return path.lastSegment();
	}
	
	public FTPTargetProvider getProvider() {
		return provider;
	}
	
	public URL getProviderURL() {
		return provider.getURL();
	}
	
	public IPath getProviderRelativePath() {
		return providerRelativePath;
	}
	
	/**
	 * @see IRemoteTargetResource#getURL()
	 */
	public URL getURL() {
		try {
			return new URL(getProviderURL(), providerRelativePath.toString());
		} catch (MalformedURLException e) {
			FTPPlugin.log(new TeamException(Policy.bind("FTPRemoteTargetResource.invalidURL"), e)); //$NON-NLS-1$
			return null;
		}
	}

	/**
	 * @see IRemoteTargetResource#getSize()
	 */
	public int getSize(IProgressMonitor progress) throws TeamException {
		FTPDirectoryEntry entry = getEntry(progress);
		if (entry == null) {
			throw new FTPException(Policy.bind("FTPRemoteTargetResource.doesNotExist", getURL().toString())); //$NON-NLS-1$
		}
		return (int)entry.getSize();
	}
	
	/**
	 * @see IRemoteResource#isContainer()
	 */
	public boolean isContainer() {
		return isContainer;
	}
	
	/**
	 * @see IRemoteResource#members(IProgressMonitor)
	 */
	public IRemoteResource[] members(IProgressMonitor progress) throws TeamException {
		final FTPDirectoryEntry[] entries = provider.listFiles(getProviderRelativePath(), progress);
		IRemoteResource[] members = new IRemoteResource[entries.length];
		for (int i = 0; i < entries.length; i++) {
			FTPDirectoryEntry entry = entries[i];
			members[i] = new FTPRemoteTargetResource(provider, providerRelativePath.append(entry.getName()), 
				entry.hasDirectorySemantics(), entry);
		}
		return members;
	}
	
	/**
	 * Create the parents of the resource remotely. If the resource is a directory, create it as well
	 */
	public void mkdirs(IProgressMonitor progress) throws TeamException {
		try {
			if (isContainer()) {
				provider.createDirectory(getProviderRelativePath(), progress);
			} else {
				provider.createDirectory(getProviderRelativePath().removeLastSegments(1), progress);
			}
		} catch (FTPServerException e) {
			// Some servers use 550 (DOES_NOT_EXIST) when the directory already exists.
			// Therefore, don't report an error in either case. If the directory wasn't created
			// another operation will fail anyway so we don't need to report it here
			// P.S. The repercusions of this are that server that return 550 cannot create multiple
			// ancesters for a file (i.e. upload of f1/f2/file will fail if f1 doesn't exist.
			if (e.getStatus().getCode() == FTPException.DIRECTORY_EXIST || e.getStatus().getCode() == FTPException.DOES_NOT_EXIST) return;
			throw e;
		}
	}
	
	/**
	 * @see IRemoteTargetResource#getFile(String)
	 */
	public IRemoteTargetResource getFile(String name) {
		return new FTPRemoteTargetResource(provider, providerRelativePath.append(name), false /* is container */);
	}

	/**
	 * @see IRemoteTargetResource#getFolder(String)
	 */
	public IRemoteTargetResource getFolder(String name) {
		return new FTPRemoteTargetResource(provider, providerRelativePath.append(name), true /* is container */);
	}

	/**
	 * @see IRemoteTargetResource#getSite()
	 */
	public Site getSite() {
		return provider.getSite();
	}
	
	public String toString() {
		return getURL().toString();
	}

	private FTPDirectoryEntry getEntry(IProgressMonitor monitor) throws TeamException {
		if (entry == null) {
			if (isContainer()) {
				entry = provider.fetchEntry(getProviderRelativePath(), monitor);
			} else {
				entry = provider.fetchEntryForFile(getProviderRelativePath(), monitor);
			}
		}
		return entry;
	}
		
	public boolean equals(Object obj) {
		if(this == obj)
			return true;
		if(!(obj instanceof FTPRemoteTargetResource))
			return false;
		FTPRemoteTargetResource otherElement = ((FTPRemoteTargetResource)obj);
		return otherElement.getURL().equals(getURL());
	}
	
	public int hashCode() {
		return getURL().hashCode();
	}
	/**
	 * @see org.eclipse.team.internal.core.target.IRemoteTargetResource#canBeReached(IProgressMonitor)
	 */
	public boolean canBeReached(IProgressMonitor monitor) throws TeamException {
		provider.run(new ITargetRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
			}
		}, monitor);
		// We connected with no exception
		return true;
	}


}
