/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.target;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.core.sync.IRemoteResource;
import org.eclipse.team.internal.core.target.ResourceState;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;
import org.eclipse.team.internal.ftp.client.IFTPRunnable;

public class FTPResourceState extends ResourceState {
	
	protected FTPRemoteTargetResource remoteResource;

	/**
	 * Constructor for FTPResourceState.
	 * @param localResource
	 * @param rootUrl
	 */
	public FTPResourceState(FTPTargetProvider provider, IResource localResource) {
		super(localResource, provider.getURL());
		remoteResource = new FTPRemoteTargetResource(provider, localResource.getProjectRelativePath(), localResource.getType() != IResource.FILE);
	}

	public FTPResourceState(IResource localResource, FTPRemoteTargetResource remoteResource) {
		super(localResource, remoteResource.getProviderURL());
		this.remoteResource = remoteResource;
	}
	
	/**
	 * @see ResourceState#getReleasedIdentifier()
	 */
	public String getReleasedIdentifier(IProgressMonitor monitor) throws TeamException {
		return remoteResource.getReleasedIdentifier(monitor);
	}

	/**
	 * @see ResourceState#getReleasedIdentifier()
	 */
	public String getReleasedIdentifierIfNecessary(IProgressMonitor monitor) throws TeamException {
		return remoteResource.getReleasedIdentifierIfNecessary(monitor);
	}
	
	/**
	 * @see ResourceState#download(IProgressMonitor)
	 */
	public void download(IProgressMonitor progress) throws TeamException {
		// Download only applies to files as folders are created by super class
		if (localResource.getType() != IResource.FILE) return;
		
		// Download the file and set the local and remote base identifiers
		final IFile localFile = (IFile)localResource;
		remoteResource.getProvider().run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				monitor.beginTask(null, 100);
				try {
					// Get the remote base identifier first to ensure that we don't lose remote changes
					remoteBaseIdentifier = getReleasedIdentifierIfNecessary(Policy.subMonitorFor(monitor, 10));
					remoteResource.getProvider().getFile(localFile, Policy.subMonitorFor(monitor, 90));
					// Set the local base identifier (there's always a chance someone snuck in between lines)
					localBaseTimestamp = localFile.getModificationStamp();
				} finally {
					monitor.done();
				}
			}
		}, Policy.monitorFor(progress));
		
		// Remember the new base state
		storeState();
	}

	/**
	 * @see ResourceState#upload(IProgressMonitor)
	 */
	public void upload(IProgressMonitor progress) throws TeamException {
		// Upload only applies to files as folders are created using mkRemoteDirs()
		if (localResource.getType() != IResource.FILE) return;
		
		// Upload the file and set the local and remote base identifiers
		remoteResource.getProvider().run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				monitor.beginTask(null, 100);
				try {
					IFile localFile = (IFile)localResource;
					// Get the local base identifier first to ensure that we don't lose remote changes
					localBaseTimestamp = localFile.getModificationStamp();
					remoteResource.getProvider().putFile(localFile, Policy.subMonitorFor(monitor, 90));
					// Set the remote base identifier (there's always a chance someone snuck in between lines)
					if (FTPPlugin.getPlugin().isFetchRemoteTimestampImmediately()) {
						remoteBaseIdentifier = getReleasedIdentifier(Policy.subMonitorFor(monitor, 10));
					} else {
						// The fetching of the timestamp will be delayed.
						// This may result in the lose of remote changes in some circumstances.
						// However, the performance gain is large
						remoteResource.getProvider().needsRemoteIdentifier(getLocal());
					}
				} finally {
					monitor.done();
				}
			}
		}, Policy.monitorFor(progress));
		
		// Remember the new base state
		storeState();
	}

	/**
	 * @see ResourceState#delete(IProgressMonitor)
	 */
	public void delete(IProgressMonitor progress) throws TeamException {
		remoteResource.delete(progress);
		removeState();
	}

	/**
	 * @see ResourceState#hasRemote()
	 */
	public boolean hasRemote(IProgressMonitor progress) {
		try {
			return remoteResource.exists(progress);
		} catch (TeamException e) {
			return false;
		}
	}

	/**
	 * @see ResourceState#getRemoteType()
	 */
	public int getRemoteType() {
		if (remoteResource.isContainer())
			return IResource.FOLDER;
		return IResource.FILE;
	}

	/**
	 * @see ResourceState#getRemoteChildren()
	 */
	public ResourceState[] getRemoteChildren(IProgressMonitor monitor) throws TeamException {
		// Only containers have children.
		if (!remoteResource.isContainer())
			return new ResourceState[0];

		// Query the FTP server to find the children (i.e., members) of the receiver.
		IRemoteResource[] remoteChildren = remoteResource.members(monitor);

		// We will answer with an array of resource states describing the children.
		ResourceState[] result = new ResourceState[remoteChildren.length];
		
		// Have to contruct resource states from DAV resource handles.
		for (int i = 0; i < remoteChildren.length; i++) {
			FTPRemoteTargetResource remoteChild = (FTPRemoteTargetResource)remoteChildren[i];

			// Look-up the child, by name in the local resource, note that it may be missing.
			IContainer localContainer = (IContainer) localResource;
			String childName = remoteChild.getName();
			IResource localChild;
			if (remoteChild.isContainer())
				localChild = localContainer.getFolder(new Path(childName));
			else
				localChild = localContainer.getFile(new Path(childName));
				
			// Add the new FTP state to the result array.
			result[i] = new FTPResourceState(localChild, remoteChild);
		}

		// All children processed.
		return result;
	}
	
	/**
	 * @see ResourceState#mkRemoteDirs(IProgressMonitor)
	 */
	protected void mkRemoteDirs(IProgressMonitor monitor) throws TeamException {
		remoteResource.mkdirs(Policy.subMonitorFor(monitor, 90));
		storeState();
	}
	
	/*
	 * Answers true if the resource was out of date the last time it was fetched
	 */
	protected boolean wasOutOfDate(IProgressMonitor monitor) throws TeamException {
		if (!hasPhantom()) return false;
		if (remoteBaseIdentifier.equals(EMPTY_REMOTEBASEID)) return false;
		String releasedIdentifier = remoteResource.getReleasedIdentifierIfNecessary(monitor);
		return !remoteBaseIdentifier.equals(releasedIdentifier);
	}
	/**
	 * @see org.eclipse.team.internal.core.target.ResourceState#setRemoteBaseIdentifier(String)
	 */
	protected void recordReleasedIdentifier(IProgressMonitor monitor) throws TeamException {
		loadState();
		super.setRemoteBaseIdentifier(getReleasedIdentifierIfNecessary(monitor));
		storeState();
	}

}
