/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.ui;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.TargetManager;
import org.eclipse.team.internal.ftp.Policy;
import org.eclipse.team.internal.ftp.target.FTPSite;
import org.eclipse.team.internal.ui.target.TargetWizardPage;

/*
 * Wizard page for configuring a project with a FTP location.
 */
public class FTPMainPage extends TargetWizardPage {
	boolean isCheckout;
	
	// Widgets
	private Combo usernameCombo;
	private Combo hostnameUrlCombo;
	private Text password;
	//private Combo proxyCombo;
	private Combo timeoutCombo;
	private Button passiveButton;
	
	// Dialog store id constants
	private static final String STORE_USERNAME_ID =
		"FTPWizardMainPage.STORE_USERNAME_ID";//$NON-NLS-1$
	private static final String STORE_URL_ID =
		"FTPWizardMainPage.STORE_HOSTNAME_ID";//$NON-NLS-1$
//	private static final String STORE_PROXY_ID =
//		"FTPWizardMainPage.STORE_PATH_ID";//$NON-NLS-1$
	private static final String STORE_TIMEOUT_ID =
		"FTPWizardMainPage.STORE_MODULE_ID";//$NON-NLS-1$
	
	
	// In case the page was launched from a different wizard	
	private IDialogSettings settings;
		
	/*
	 * ConfigurationWizardMainPage constructor.
	 */
	public FTPMainPage(String pageName, String title, String description, ImageDescriptor titleImage) {
		super(pageName, title, titleImage);
		setDescription(description);
		setTitle(title);
	}	
	
	protected void setDialogSettings(IDialogSettings settings) {
		this.settings = settings;
	}
	
	protected IDialogSettings getDialogSettings() {
		return settings;
	}
	
	/*
	 * Creates the UI part of the page.
	 * 
	 * @param parent  the parent of the created widgets
	 */
	public void createControl(Composite parent) {
		Composite composite = createComposite(parent, 2);

		createLabel(composite, Policy.bind("FTPWizardMainPage.url")); //$NON-NLS-1$
		hostnameUrlCombo = createEditableCombo(composite);
		hostnameUrlCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				FTPMainPage.this.validateFields();
			}
		});
		
		Group loginDetailsTabGroup = new Group(composite, SWT.NONE);
		loginDetailsTabGroup.setText(Policy.bind("FTPWizardMainPage.loginFrame")); //$NON-NLS-1$
		GridData loginDetails = new GridData(GridData.FILL_HORIZONTAL);
		loginDetails.horizontalSpan = 2;
		loginDetailsTabGroup.setLayoutData(loginDetails);
		loginDetailsTabGroup.setLayout(new GridLayout(2,false));
		
		Button useAnon = new Button(loginDetailsTabGroup,SWT.RADIO);
		GridData data = new GridData(GridData.GRAB_HORIZONTAL);
		data.horizontalSpan = 2;
		useAnon.setLayoutData(data);
		useAnon.setText(Policy.bind("FTPWizardMainPage.anon")); //$NON-NLS-1$
		useAnon.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				if (((Button)event.widget).getSelection()) {
					usernameCombo.setEnabled(false);
					password.setEnabled(false);
					usernameCombo.setText(Policy.bind("FTPWizardMainPage.anonuser")); //$NON-NLS-1$
					password.setText(Policy.bind("FTPWizardMainPage.pwd")); //$NON-NLS-1$
				}
			}
		});
		Button useLoginDetails = new Button(loginDetailsTabGroup,SWT.RADIO);
		data = new GridData(GridData.GRAB_HORIZONTAL);
		data.horizontalSpan = 2;
		useLoginDetails.setLayoutData(data);
		useLoginDetails.setText(Policy.bind("FTPWizardMainPage.loginDetails")); //$NON-NLS-1$
		useLoginDetails.setSelection(true);
		useLoginDetails.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				if (((Button)event.widget).getSelection()) {
					usernameCombo.setText(""); //$NON-NLS-1$
					password.setText(""); //$NON-NLS-1$
					usernameCombo.setEnabled(true);
					password.setEnabled(true);
				}
			}
		});

		createLabel(loginDetailsTabGroup, Policy.bind("FTPWizardMainPage.userName")); //$NON-NLS-1$
		usernameCombo = createEditableCombo(loginDetailsTabGroup);
		usernameCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				FTPMainPage.this.validateFields();		
			}
		});
		
		createLabel(loginDetailsTabGroup, Policy.bind("FTPWizardMainPage.password")); //$NON-NLS-1$
		password = createTextField(loginDetailsTabGroup);
		password.setEchoChar('*'); //$NON-NLS-1$

//		createLabel(composite, Policy.bind("FTPWizardMainPage.proxy")); //$NON-NLS-1$
//		proxyCombo = createEditableCombo(composite);
//		proxyCombo.addListener(SWT.Modify, new Listener() {
//			public void handleEvent(Event e) {
//				FTPMainPage.this.validateFields();		
//			}
//		});

		createLabel(composite, Policy.bind("FTPWizardMainPage.timeout")); //$NON-NLS-1$
		timeoutCombo = createEditableCombo(composite);
		timeoutCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				FTPMainPage.this.validateFields();		
			}
		});
		
		Composite passiveButtonTabGroup = new Composite(composite, SWT.NONE);
		data = new GridData();
		data.horizontalSpan = 2;
		passiveButtonTabGroup.setLayoutData(data);
		passiveButtonTabGroup.setLayout(new FillLayout());

		passiveButton = new Button(passiveButtonTabGroup, SWT.CHECK);
		passiveButton.setText(Policy.bind("FTPWizardMainPage.passive")); //$NON-NLS-1$
			
		initializeValues();
		validateFields();
		hostnameUrlCombo.setFocus();
		setControl(composite);
	}
	
	public FTPSite getSite() {
		if(!isPageComplete()) {
			return null;
		}

		return new FTPSite(getCurrentValues());
	}

	private Properties getCurrentValues() {
		Properties result = new Properties();
		result.setProperty("location", getHostAsURL()); //$NON-NLS-1$
		result.setProperty("username", usernameCombo.getText()); //$NON-NLS-1$
		result.setProperty("password", password.getText()); //$NON-NLS-1$
//		result.setProperty("proxyURL", proxyCombo.getText()); //$NON-NLS-1$
		result.setProperty("connectionTimeout", timeoutCombo.getText()); //$NON-NLS-1$
		result.setProperty("passive", passiveButton.getSelection() ? "true" : "false"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		return result;
	}
			
	/*
	 * @see WizardPage#finish
	 */
	public boolean finish(IProgressMonitor monitor) {
		// Set the result to be the current values
		saveWidgetValues();
		return true;
	}
	
	/*
	 * Initializes states of the controls.
	 */
	private void initializeValues() {
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] hostNames = settings.getArray(STORE_URL_ID);
			if (hostNames != null) {
				for (int i = 0; i < hostNames.length; i++) {
					hostnameUrlCombo.add(hostNames[i]);
				}
			}
			String[] userNames = settings.getArray(STORE_USERNAME_ID);
			if (userNames != null) {
				for (int i = 0; i < userNames.length; i++) {
					usernameCombo.add(userNames[i]);
				}
			}
//			String[] proxies = settings.getArray(STORE_PROXY_ID);
//			if (proxies != null) {
//				for (int i = 0; i < proxies.length; i++) {
//					proxyCombo.add(proxies[i]);
//				}
//			}			
			String[] timeouts = settings.getArray(STORE_TIMEOUT_ID);
			if (timeouts != null) {
				for (int i = 0; i < timeouts.length; i++) {
					timeoutCombo.add(timeouts[i]);
				}
			}
			timeoutCombo.setText(String.valueOf(FTPSite.DEFAULT_TIMEOUT));
			passiveButton.setSelection(true);
		}
	}
	
	/*
	 * Saves the widget values
	 */
	private void saveWidgetValues() {
		// Update history
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] userNames = settings.getArray(STORE_USERNAME_ID);
			if (userNames == null) userNames = new String[0];
			userNames = addToHistory(userNames, usernameCombo.getText());
			settings.put(STORE_USERNAME_ID, userNames);
	
			String[] hostNames = settings.getArray(STORE_URL_ID);
			if (hostNames == null) hostNames = new String[0];
			hostNames = addToHistory(hostNames, hostnameUrlCombo.getText());
			settings.put(STORE_URL_ID, hostNames);
	
//			String[] paths = settings.getArray(STORE_PROXY_ID);
//			if (paths == null) paths = new String[0];
//			paths = addToHistory(paths, proxyCombo.getText());
//			settings.put(STORE_PROXY_ID, paths);

			String[] modules = settings.getArray(STORE_TIMEOUT_ID);
			if (modules == null) modules = new String[0];
			modules = addToHistory(modules, timeoutCombo.getText());
			settings.put(STORE_TIMEOUT_ID, modules);	
		}
	}

	private String getHostAsURL() {
		String host = hostnameUrlCombo.getText();
		if (host.length()!=0 && host.toLowerCase().indexOf("ftp://")!=0) host="ftp://"+host; //$NON-NLS-1$ //$NON-NLS-2$
		return host;
	}
	/*
	 * Validates the contents of the editable fields and set page completion 
	 * and error messages appropriately.
	 */
	private void validateFields() {
		// Validate the server URL
		String host=getHostAsURL();
		if (host.length() == 0) {
			setPageComplete(false);
			return;
		}  
		try {
			URL url = new URL(host);
			Site site = TargetManager.getSite(FTPSite.TYPE, url);
			if(site != null) {
				setMessage(Policy.bind("FTPWizardMainPage.locationExists"), WARNING); //$NON-NLS-1$
				setPageComplete(false);
				return;
			}
		} catch(MalformedURLException e) {
			setMessage(Policy.bind("FTPWizardMainPage.invalidServerURL"), WARNING); //$NON-NLS-1$
			setPageComplete(false);
			return;
		}

		String username = usernameCombo.getText();
		if(username.length() == 0) {
			setPageComplete(false);
			setMessage(Policy.bind("FTPWizardMainPage.usernameMustBeSpecified"), WARNING); //$NON-NLS-1$
			return;
		}

//		// Validate the proxy URL.
//		String proxy = proxyCombo.getText();
//		if (proxy.length() != 0) {
//			try {
//				new URL(proxy);
//			} catch(MalformedURLException e) {
//				setMessage(Policy.bind("FTPWizardMainPage.invalidProxyURL"), WARNING); //$NON-NLS-1$
//				setPageComplete(false);
//				return;
//			}
//		}
		
		// validate the timeout
		String timeout = timeoutCombo.getText();
		try {
			int timeoutInt = Integer.parseInt(timeout);
			if(timeoutInt <= 0) {
				setMessage(Policy.bind("FTPWizardMainPage.invalidTimeout"), WARNING); //$NON-NLS-1$
				setPageComplete(false);
				return;		
			}
		} catch(NumberFormatException e) {
			setMessage(Policy.bind("FTPWizardMainPage.invalidTimeout"), WARNING); //$NON-NLS-1$
			setPageComplete(false);
			return;		
		}
		
		setMessage(null);
		setPageComplete(true);
	}
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if (visible) {
			hostnameUrlCombo.setFocus();
		}
	}
}
