/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tomcat.internal;

import java.io.*;
import java.net.*;

import org.apache.catalina.*;
import org.apache.catalina.connector.http.*;
import org.apache.catalina.core.*;
import org.apache.catalina.logger.*;
import org.apache.catalina.realm.*;
import org.apache.catalina.startup.*;
import org.eclipse.core.runtime.*;
import org.eclipse.help.internal.appserver.*;

/**
 * Single engine, single host, single connector
 * Tomcat Application Server.
 */
public class TomcatAppServer implements IWebappServer {
	private String hostAddress;
	private int port;
	// false until an attempt to start Tomcat
	private boolean isStarted = false;
	// true after started without problems
	private boolean running = false;

	private Embedded embedded = null;
	private Engine engine = null;
	private Host host = null;

	/**
	 * Constructs this class, but does not instantiates
	 * or start Tomcat classes until webapp are added.
	 */
	public TomcatAppServer() {
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#start(int, java.lang.String)
	 */
	public void start(int port, String hostAddress) throws CoreException {
		this.hostAddress = hostAddress;
		this.port = port;

		if (isStarted) {
			return;
		}
		isStarted = true;
		try {
			FileLogger logger = new FileLogger();
			logger.setDirectory(
				TomcatPlugin.getDefault().getStateLocation().toOSString());
			embedded = new Embedded(logger, new MemoryRealm());
			embedded.setDebug(0);
			embedded.setLogger(logger);

			IPluginDescriptor descriptor =
				TomcatPlugin.getDefault().getDescriptor();
			URL installURL = descriptor.getInstallURL();
			String home = Platform.resolve(installURL).getFile();
			System.setProperty("catalina.home", home);
			String base = home;
			System.setProperty("catalina.base", base);

			// start now, and then add all the contexts..
			embedded.start();

			// Create a very basic container hierarchy
			engine = embedded.createEngine();
			engine.setDefaultHost("localhost");

			host = embedded.createHost("localhost", home + "/webapps");
			if (host instanceof StandardHost) {
				((StandardHost) host).setErrorReportValveClass(
					"org.eclipse.tomcat.internal.EclipseErrorReportValve");
			}
			engine.addChild(host);

			// Install the assembled container hierarchy
			PrintStream sysOut = System.out;
			// reassign standard output to prevent Tomcat from writing
			// its version message there.
			System.setOut(new PrintStream(new ByteArrayOutputStream()));
			try {
				embedded.addEngine(engine);
			} finally {
				System.setOut(sysOut);
			}

			// Root context
			Context root = embedded.createContext("", home + "/webapps/ROOT");
			// this line should be replaced once tomcat provides support
			// for setting the working directory
			if (root instanceof StandardContext) {
				((StandardContext) root).setWorkDir(
					getWorkingDirectory("ROOT"));
			}
			root.setLoader(
				embedded.createLoader(this.getClass().getClassLoader()));
			host.addChild(root);

			// Create Connector
			InetAddress iAddress = null;
			if (this.hostAddress != null) {
				try {
					iAddress = InetAddress.getByName(this.hostAddress);
				} catch (UnknownHostException uhe) {
					// will default to all interfaces
				}
			}
			Connector connector =
				embedded.createConnector(iAddress, this.port, false);
			PortTrackingServerSocketFactory sSocketFactory =
				new PortTrackingServerSocketFactory();
			connector.setFactory(sSocketFactory);
			// Override defaults on HttpConnector
			if (connector instanceof HttpConnector) {
				HttpConnector httpConnector = (HttpConnector) connector;
				Preferences pref =
					TomcatPlugin.getDefault().getPluginPreferences();
				int acceptCount = pref.getInt(TomcatPlugin.PREF_ACCEPT_COUNT);
				if (acceptCount > 0) {
					httpConnector.setAcceptCount(acceptCount);
				}
				int maxProcessors =
					pref.getInt(TomcatPlugin.PREF_MAX_PROCESSORS);
				if (maxProcessors > 0) {
					httpConnector.setMaxProcessors(maxProcessors);
				}
				int minProcessors =
					pref.getInt(TomcatPlugin.PREF_MIN_PROCESSORS);
				if (minProcessors > 0) {
					httpConnector.setMinProcessors(minProcessors);
				}
			}

			// add Connector to Tomcat
			embedded.addConnector(connector);

			this.port = sSocketFactory.getAssignedPort();
			if (this.port == 0) {
				// could not open the port
				if (port == 0) {
					// no port obtained
					throw new CoreException(
						new Status(
							IStatus.ERROR,
							TomcatPlugin.PLUGIN_ID,
							IStatus.OK,
							TomcatResources.getString(
								"TomcatAppServer.start.CannotOptainPort"),
							null));
				} else {
					// port busy
					throw new CoreException(
						new Status(
							IStatus.ERROR,
							TomcatPlugin.PLUGIN_ID,
							IStatus.OK,
							TomcatResources.getString(
								"TomcatAppServer.start.CannotOpenPort",
								"" + port),
							null));
				}
			}
			// if null passed for hostAddress, determine Address
			// that is available for local connections
			if (this.hostAddress == null) {
				this.hostAddress =
					LocalConnectionTest.getLocalInterface(this.port);
			}

			running = true;

		} catch (Exception exc) {
			TomcatPlugin.logError(
				TomcatResources.getString("TomcatAppServer.start"),
				exc);
			if (exc instanceof CoreException) {
				throw (CoreException) exc;
			} else {
				throw new CoreException(
					new Status(
						IStatus.ERROR,
						TomcatPlugin.PLUGIN_ID,
						IStatus.OK,
						TomcatResources.getString("TomcatAppServer.start"),
						exc));
			}
		}
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#start(java.lang.String, org.eclipse.core.runtime.IPath, java.lang.ClassLoader)
	 */
	public void start(String webappName, IPath path, ClassLoader customLoader)
		throws CoreException {

		if (!isStarted) {
			start(port, hostAddress);
		}
		if (!running) {
			throw new CoreException(
				new Status(
					IStatus.ERROR,
					TomcatPlugin.PLUGIN_ID,
					IStatus.OK,
					TomcatResources.getString(
						"TomcatAppServer.addingWebapp",
						webappName,
						path.toOSString()),
					null));
		}

		String contextPath = webappName;
		if (!contextPath.startsWith("/")) {
			contextPath = "/" + contextPath;
		}
		try {
			Context context =
				embedded.createContext(contextPath, path.toOSString());
			if (context instanceof StandardContext) {
				((StandardContext) context).setWorkDir(
					getWorkingDirectory(webappName));
			}

			WebAppClassLoader webappLoader =
				new WebAppClassLoader(customLoader);
			context.setLoader(embedded.createLoader(webappLoader));

			host.addChild(context);

		} catch (Exception exc) {
			throw new CoreException(
				new Status(
					IStatus.ERROR,
					TomcatPlugin.PLUGIN_ID,
					IStatus.OK,
					TomcatResources.getString(
						"TomcatAppServer.addingWebapp",
						webappName,
						path.toOSString()),
					exc));
		}
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#stop(java.lang.String)
	 */
	public void stop(String webappName) throws CoreException {
		if (!running) {
			return;
		}
		Context context = (Context) host.findChild("/" + webappName);
		if (context != null) {
			embedded.removeContext(context);
		}
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#getHost()
	 */
	public String getHost() {
		if (!running) {
			return null;
		}
		return hostAddress;
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#getPort()
	 */
	public int getPort() {
		if (!running) {
			return 0;
		}
		return port;
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#isRunning()
	 */
	public boolean isRunning() {
		return running;
	}

	/**
	 * @see org.eclipse.help.internal.appserver.IWebappServer#stop()
	 */
	public void stop() throws CoreException {
		if (!running) {
			return;
		}
		running = false;

		// Remove the engine (which should trigger removing the connector)
		try {
			embedded.removeEngine(engine);
		} catch (Exception exc) {
			throw new CoreException(
				new Status(
					IStatus.ERROR,
					TomcatPlugin.PLUGIN_ID,
					IStatus.OK,
					TomcatResources.getString("TomcatAppServer.engineRemove"),
					exc));
		}
		// Shut down this tomcat server (should have nothing left to do)
		try {
			embedded.stop();
		} catch (LifecycleException e) {
			throw new CoreException(
				new Status(
					IStatus.ERROR,
					TomcatPlugin.PLUGIN_ID,
					IStatus.OK,
					TomcatResources.getString("TomcatAppServer.embeddedStop"),
					e));
		}
	}

	private String getWorkingDirectory(String webApp) {
		return TomcatPlugin
			.getDefault()
			.getStateLocation()
			.append(webApp)
			.toOSString();
	}
}
