/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.util.misc;

import java.util.List;
import java.util.ArrayList;
import java.util.AbstractList;
import java.lang.reflect.Array;

import com.sun.appserv.management.util.stringifier.ArrayStringifier;


/**
	A circular list implementation
 */
public final class CircularList<T> extends AbstractList<T>
{
	private final T[]       mObjects;
	private int					mNumItems;
	private int					mFirst;
	private OverflowHandler		mOverflowHandler;
	
		public
	CircularList( Class<T> theClass, final int size )
	{
		if ( size == 0 )
		{
			throw new IllegalArgumentException( "list must have at least one item" );
		}

		mObjects	= ArrayUtil.newArray( theClass, size );
		mFirst		= 0;
		mNumItems	= 0;
		
		mOverflowHandler	= null;
	}
	
	public interface OverflowHandler
	{
		void	handleBufferOverflow( Object o );
	}
	
		public void
	setOverflowHandler( final OverflowHandler handler )
	{
		mOverflowHandler	= handler;
	}
	
		public OverflowHandler
	getOverflowHandler( )
	{
		return( mOverflowHandler );
	}
	
		public final int
	size()
	{
		return( mNumItems );
	}
	
		public final int
	capacity()
	{
		return( mObjects.length );
	}
	
	
		public final void
	clear()
	{
		for( int i = 0; i < size(); ++i )
		{
			set( i, null );
		}
		mNumItems	= 0;
		++modCount;
	}
	
		private final int
	getPhysicalIndex( final int logicalIndex )
	{
		return( (mFirst + logicalIndex) % capacity() );
	}
	
		public final T
	get( final int i)
	{
		checkInBounds( i );
		
		return( mObjects[ getPhysicalIndex( i ) ] );
	}
	
	
		public final T
	set(
		final int	i,
		final T     item)
	{
		checkInBounds( i );
		
		final int	physicalIndex	= getPhysicalIndex( i );
		final T	oldItem	= mObjects[ physicalIndex ];
		mObjects[ physicalIndex ] = item;
		
		return( oldItem );
	}
	
	
		protected void
	discardedObject( final T o )
	{
		if ( mOverflowHandler != null )
		{
			mOverflowHandler.handleBufferOverflow( o );
		}
	}
	
		private final void
	store(
		final int	logicalIndex,
		final T		item )
	{
		mObjects[ getPhysicalIndex( logicalIndex ) ]	= item;
	}
	
		public final boolean
	add( final T item )
	{
		final int	capacity	= capacity();
		
		assert( mFirst < capacity );
		
		// if we're full before adding, then we'll be overwriting
		// the first item.
		if ( size() == capacity )
		{
			final T	overwrittenObject	= get( 0 );
			
			// first item will be overwritten; next one is first (oldest)
			mFirst	= (mFirst + 1) % capacity;
			store( capacity - 1, item );
			
			discardedObject( overwrittenObject );
			
		}
		else
		{
			store( mNumItems, item );
			++mNumItems;
		}
		
		++modCount;
		return( true );
	}
	
	/**
		May be added to the end only.
	 */
		public final void
	add(
		final int	index,
		final T 	item )
	{
		if ( index == mNumItems )
		{
			add( item );
		}
		else
		{
			throw new UnsupportedOperationException( "add not at end" );
		}
	}
	
		public final T
	remove( final int i)
	{
		T	result	= null;
		
		if ( i == 0 )
		{
			result	= removeFirst();
		}
		else if ( i == mNumItems - 1 )
		{
			result	= removeLast();
		}
		else
		{
			throw new UnsupportedOperationException();
		}
		
		++modCount;
		return( result );
	}
	
		private final void
	checkInBounds( final int i )
	{
		if ( i < 0 || i >= mNumItems )
		{
			throw new IndexOutOfBoundsException( "" + i );
		}
	}
	
		public final T
	removeFirst()
	{
		checkInBounds( 0 );
		
		final T	result	= get( 0 );
		--mNumItems;
		mFirst	= (mFirst + 1) % capacity();
		
		return( result );
	}
	
	
		public final T
	removeLast()
	{
		checkInBounds( 0 );
		
		final T	result	= get( mNumItems - 1 );
		--mNumItems;
		
		return( result );
	}
	
	
		public boolean
	equals( final Object rhsIn )
	{
		boolean	equal	= false;
		
		if ( rhsIn == this )
		{
			equal	= true;
		}
		else if ( ! (rhsIn instanceof CircularList) )
		{
			equal	= false;
		}
		else
		{
			final CircularList	rhs	= (CircularList)rhsIn;
			
			equal =  capacity() == rhs.capacity() &&
				size() == rhs.size();
			if ( equal )
			{
				final int	size	= size();
				for( int i = 0; i < size(); ++i )
				{
					if ( ! CompareUtil.objectsEqual( get( i ), rhs.get( i ) ) )
					{
						equal	= false;
						break;
					}
				}
			}
		}
		
		return( equal );
	}

		public String
	toString()
	{
		return( ArrayStringifier.stringify( toArray( ), ", " ) );
	}
}

