/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * RepositoryConfig.java
 *
 * Created on August 19, 2003, 1:59 PM
 */

package com.sun.enterprise.admin.servermgmt;

import java.util.HashMap;

import java.io.File;

import com.sun.enterprise.util.SystemPropertyConstants;

import com.sun.enterprise.util.io.FileUtils;

/**
 * This class represents a repository configuration. A repository can 
 * be either a domain, a node agent, or a server instance. Configuration
 * specific to each (DomainConfig, AgentConfig, InstanceConfig) is 
 * derived from this class. A repository config consists of the following
 * attributes:
 * 
 * 1)repositoryName -- domain or node agent name (e.g. domain1 or agent1)
 *
 * 2)repositoryRoot -- the parent directory of the repository (e.g.
 * $installDir/domains or $installDir/agents)
 *
 * 3)instanceName -- the optional server instance name (e.g. server1)
 *
 * 4)configurationName -- the optional configuration name of the server
 * instance (e.g. default-config). 
 *
 * Using (repositoryName, repositoryRoot, instanceName, configurationName)
 * syntax. Here are the following permutations:
 *
 * 1)For a domain: (domainRootDirectory, domainName, null, null) e.g.
 * ("/sun/appserver/domains", "domain1", null, null)
 *
 * 2)For a node agent: (agentRootDirectory, agentName, "agent", null) e.g
 * ("/sun/appserver/agents", "agent1", "agent", null). Note that the instance
 * name of a node agent is always the literal string "agent".
 * 
 * 3)For a server instance (agentRootDirectory, agentName, instanceName, 
 * configName) e.g.
 * ("/sun/appserver/agents", "agent1", "server1", "default-config")
 *
 * The RepositoryConfig class is an extensible HashMap that can contain
 * any attributes, but also relies on two system properties being
 * set:
 * 
 * 1)com.sun.aas.installRoot -- installation root directory stored under
 * the K_INSTALL_ROOT key.
 *
 * 2)com.sun.aas.configRoot -- configuration root (for locating asenv.conf) 
 * stored under the K_CONFIG_ROOT key.
 *
 * @author  kebbs
 */
public class RepositoryConfig extends HashMap {
    
    public static final String K_INSTALL_ROOT = "install.root";
    public static final String K_CONFIG_ROOT = "config.root";    
    
    //Name of the domain or node agent. Cannot be null.
    private String _repositoryName;
    
    //Root directory where the domain or node agent resides. Cannot be null
    private String _repositoryRoot;
    
    //Name of the server instance. May be null
    private String _instanceName;
    
    //Name of the configuration. May be null
    private String _configurationName;
    
    /** 
     * Creates a new instance of RepositoryConfig 
     * The K_INSTALL_ROOT and K_CONFIG_ROOT attributes are implicitly set     
     */
    public RepositoryConfig(String repositoryName, String repositoryRoot, String instanceName, 
        String configName) {
        _instanceName = instanceName;
        _repositoryName = repositoryName;        
        _repositoryRoot = repositoryRoot;
        _configurationName = configName;
        put(K_INSTALL_ROOT, getFilePath(SystemPropertyConstants.INSTALL_ROOT_PROPERTY));        
        put(K_CONFIG_ROOT, getFilePath(SystemPropertyConstants.CONFIG_ROOT_PROPERTY));         
    }

    public RepositoryConfig(String repositoryName, String repositoryRoot, String instanceName) {
        this(repositoryName, repositoryRoot, instanceName, null);
    }
    
    public RepositoryConfig(String repositoryName, String repositoryRoot) {
        this(repositoryName, repositoryRoot, null);
    }
    
    /** 
     * Creates a new instance of RepositoryConfig defined using the 
     * system property com.sun.aas.instanceRoot. It is assumed that this 
     * system property is a directory of the form:
     * <repositoryRootDirectory>/<repositoryName>/<instanceName>
     */
    public RepositoryConfig() {
        final File instanceRoot = new File(System.getProperty(SystemPropertyConstants.INSTANCE_ROOT_PROPERTY));        
        final File repositoryDir = instanceRoot.getParentFile();               
        _instanceName = instanceRoot.getName();
        _repositoryName = repositoryDir.getName();
        _repositoryRoot = FileUtils.makeForwardSlashes(repositoryDir.getParentFile().getAbsolutePath());
        _configurationName = null;
        put(K_INSTALL_ROOT, getFilePath(SystemPropertyConstants.INSTALL_ROOT_PROPERTY));        
        put(K_CONFIG_ROOT, getFilePath(SystemPropertyConstants.CONFIG_ROOT_PROPERTY));  
    }

    public String toString()
    {
        return ("repositoryRoot " + _repositoryRoot + " repositoryName " + _repositoryName +
            " instanceName " + _instanceName + " configurationName " + _configurationName);
    }
    
    protected String getFilePath(String propertyName) 
    {
        File f = new File(System.getProperty(propertyName));
        return FileUtils.makeForwardSlashes(f.getAbsolutePath());
    }

    public void setConfigurationName(String configurationName) {
        _configurationName = configurationName;
    }
    
    public String getConfigurationName() {
        return _configurationName;
    }
   
    public String getDisplayName() {
        return getRepositoryName();
    }
    
    public void setInstanceName(String instanceName) {
        _instanceName = instanceName;
    }
    
    public String getInstanceName() {
        return _instanceName;
    }
    
    public String getRepositoryName() {
        return _repositoryName;
    }
    
    protected void setRepositoryRoot(String repositoryRoot) {
        _repositoryRoot = repositoryRoot;
    }
    
    public String getRepositoryRoot()
    {
        return _repositoryRoot;
    }
    
    public String getInstallRoot()
    {
        return (String)get(K_INSTALL_ROOT);
    }
    
    public String getConfigRoot()
    {
        return (String)get(K_CONFIG_ROOT);
    }
}
