/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 package com.sun.enterprise.deployment;

import com.sun.enterprise.deployment.web.InitializationParameter;
import com.sun.enterprise.deployment.web.ContextParameter;
import com.sun.enterprise.deployment.web.EnvironmentEntry;
import com.sun.enterprise.deployment.web.WebDescriptor;
import com.sun.enterprise.util.LocalStringManagerImpl;
import com.sun.enterprise.util.RelativePathResolver;
import com.sun.enterprise.util.TypeUtil;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

    /** 
    ** The EnvironmentProperty class hold the data about a single environment entry for J2EE components.
    ** @author Danny Coward 
    */
 
public class EnvironmentProperty extends Descriptor implements InitializationParameter, ContextParameter, WebDescriptor, EnvironmentEntry, InjectionCapable {
    private String value; 
    private String type;
    private Object valueObject;
    private boolean setValueCalled = false;

    // list of injection targes
    private Set<InjectionTarget> injectionTargets;

    private static Class[] allowedTypes = {
                                        int.class,
                                        boolean.class,
                                        double.class,
                                        float.class,
                                        long.class,
                                        short.class,
                                        byte.class,
                                        char.class,
					java.lang.String.class,
					java.lang.Boolean.class,
					java.lang.Integer.class,
					java.lang.Double.class,
					java.lang.Byte.class,
					java.lang.Short.class,
					java.lang.Long.class,
					java.lang.Float.class,
                                        java.lang.Character.class
					    };
   private static LocalStringManagerImpl localStrings =
	    new LocalStringManagerImpl(EnvironmentProperty.class);

    protected String mappedName;
						    
    /** 
    ** copy constructor.
    */

    public EnvironmentProperty(EnvironmentProperty other) {
	super(other);
	value = other.value;
	type = other.type;
	valueObject = other.valueObject;
    }  
				    
    /** 
    ** Construct an environment property if type String and empty string value and no description.
    */

    public EnvironmentProperty() {
    }  
    
     /** 
    ** Construct an environment property of given name value and description. 
    */
    
    public EnvironmentProperty(String name, String value, String description) {
	this(name, value, description, null);
    }  
    
    /** 
    ** Construct an environment property of given name value and description and type.
    ** Throws an IllegalArgumentException if bounds checking is true and the value cannot be
    ** reconciled with the given type. 
    */ 
    
    public EnvironmentProperty(String name, String value, String description, String type) {
	super(name, description);
	this.value = value;
	checkType(type);
	this.type = type;
    } 
    
    /** 
    ** Returns the String value of this environment property 
    */
    
    public String getValue() {
	if (this.value == null) {
	    this.value = "";
	}
	return value;
    }
    
    /**
     * Returns a resolved value of this environment property
     */
    public String getResolvedValue() {
    	return RelativePathResolver.resolvePath(getValue());
    }
    
    /** 
     ** Returns the typed value object of this environment property. Throws an IllegalArgumentException if bounds checking is 
     ** true and the value cannot be
     ** reconciled with the given type. 
     */
     public Object getResolvedValueObject() {
 	if (this.valueObject == null) {
 	    this.valueObject = "";
 	}
 	return getObjectFromString(this.getResolvedValue(), this.getValueType()); 
     }
    
    /** 
    ** checks the given class type. throws an IllegalArgumentException if bounds checking
    ** if the type is not allowed.
    */
    
    private void checkType(String type) {
	if (type != null) {
	    Class typeClass = null;
	    // is it loadable ?
	    try {
		typeClass = Class.forName(type);
	    } catch (Throwable t) {
		if (this.isBoundsChecking()) {
		    throw new IllegalArgumentException(localStrings.getLocalString(
										   "enterprise.deployment..exceptiontypenotallowedpropertytype",
										   "{0} is not an allowed property value type", new Object[] {type}));
		} else {
		    return;
		}
	    }
	    boolean allowedType = false;
	    for (int i = 0; i < allowedTypes.length; i++) {
		if (allowedTypes[i].equals(typeClass)) {
		    allowedType = true;
		    break;
		}
	    }
	    if (this.isBoundsChecking() && !allowedType) {
		throw new IllegalArgumentException(localStrings.getLocalString(
										   "enterprise.deployment.exceptiontypenotallowedprprtytype",
										   "{0} is not an allowed property value type", new Object[] {type}));
	    }
	}
    }
    
    /** 
    ** Returns the typed value object of this environment property. Throws an IllegalArgumentException if bounds checking is 
    ** true and the value cannot be
    ** reconciled with the given type. 
    */
    public Object getValueObject() {
	if (this.valueObject == null) {
	    this.valueObject = "";
	}
	return getObjectFromString(this.getValue(), this.getValueType()); 
    }
    
    /** 
    ** Returns value type of this environment property. 
    */
    
    public Class getValueType() {
	if (this.type == null) {
	    return String.class;
	} else {
	    try {
		return Class.forName(this.type);
	    } catch (Throwable t) {
		return null;
	    }
	}
    }
    
     /** 
    ** Returns value type of this environment property. Throws Illegal argument exception if this is not an
    ** allowed type and bounds checking.
    */
    
    public void setType(String type) {
	checkType(type);
	this.type = type;
    }
    
     /** 
    ** Returns value type of this environment property as a classname. 
    */
    
    public String getType() {
	if (type == null && this.isBoundsChecking()) {
            return String.class.getName();
	} else {
            return type;
        }
    }

    public void setMappedName(String mName) {
        mappedName = mName;
    }

    public String getMappedName() {
        return (mappedName != null)? mappedName : "";
    }

    
     /** 
    ** Sets the value of the environment property to the given string.
    */
    
    public void setValue(String value) {
	this.value = value;
        this.setValueCalled = true;
	super.changed();
    }

    public boolean hasAValue() {
        return setValueCalled;
    }
    
     /** 
    ** Returns true if the argument is an environment property of the same name, false else.
    */
    
    public boolean equals(Object other) {
	if (other instanceof EnvironmentProperty &&
	    this.getName().equals( ((EnvironmentProperty) other).getName() )) {
		return true;
	}
	return false;
    }
    
    /** 
    ** The hashCode of an environment property is the same as that of the name String.
    */
    public int hashCode() {
	return this.getName().hashCode();
    }
    
    /** 
    ** Returns a String representation of this environment property.
    */
    public void print(StringBuffer toStringBuffer) {
	toStringBuffer.append("Env-Prop: ").append(super.getName()).append("@");
        printInjectableResourceInfo(toStringBuffer);
        toStringBuffer.append("@").append(this.getType()).append("@").append(this.getValue()).append("@").append("@").append(super.getDescription());
    }
    
    private Object getObjectFromString(String string, Class type) {
        if (type == null && !this.isBoundsChecking()) {
            Object obj = getValueObjectUsingAllowedTypes(string);
            if (obj != null) return obj;
        }
	if (string == null || ("".equals(string) && !type.equals(String.class))) {
	    return null;
	}
	try {
            if (String.class.equals(type)) {
		return string;
            } else if (Boolean.class.equals(type)) {
		return new Boolean(string);
	    } else if (Integer.class.equals(type)) {
		return new Integer(string);
	    } else if (Double.class.equals(type)) {
		return new Double(string);
	    } else if (Float.class.equals(type)) {
		return new Float(string);
	    } else if (Short.class.equals(type)) {
		return new Short(string);
	    } else if (Byte.class.equals(type)) {
		return new Byte(string);
	    } else if (Long.class.equals(type)) {
		return new Long(string);
	    } else if (Character.class.equals(type)) {
                if (string.length() != 1) {
                    throw new IllegalArgumentException();
                } else {
                    return new Character(string.charAt(0));
                }
            }
	} catch (Throwable t) {
	    throw new IllegalArgumentException(localStrings.getLocalString(
									   "enterprise.deployment.exceptioncouldnotcreateinstancetype",
									   "Could not create instance of {0} from {1}\n reason: {2}" + t, new Object[] {type, string, t}));
	}
	throw new IllegalArgumentException(localStrings.getLocalString(
								       "enterprise.deployment.exceptionillegaltypeenvproperty",
								       "Illegal type for environment properties: {0}", new Object[] {type}));
    }
	

    private Object getValueObjectUsingAllowedTypes(String string) 
                                  throws IllegalArgumentException {
        if (this.type.equals(int.class.getName())) {
            return new Integer(string);
        } else if (this.type.equals(long.class.getName())) {
            return new Long(string);
        } else if (this.type.equals(short.class.getName())) {
            return new Short(string);
        } else if (this.type.equals(boolean.class.getName())) {
            return new Boolean(string);
        } else if (this.type.equals(float.class.getName())) {
            return new Float(string);
        } else if (this.type.equals(double.class.getName())) {
            return new Double(string);
        } else if (this.type.equals(byte.class.getName())) {
            return new Byte(string);
        } else if (this.type.equals(char.class.getName())) {
            if (string.length() != 1) {
                throw new IllegalArgumentException();
            } else {
                return new Character(string.charAt(0));
            }
        } 
        return null;
    }

    //
    // InjectableResource implementation
    //
    public void addInjectionTarget(InjectionTarget target) {
        if (injectionTargets==null) {
            injectionTargets = new HashSet<InjectionTarget>();
        }
        boolean found = false;
        for (InjectionTarget injTarget : injectionTargets) {
            if (injTarget.equals(target)) {
                found = true;
                break;
            }
        }
        if (!found) {
            injectionTargets.add(target);
        }
    }
    
    public Set<InjectionTarget> getInjectionTargets() {
        return injectionTargets;
    }

    public boolean isInjectable() {
        return (injectionTargets!=null && injectionTargets.size()>0);
        //return (getInjectTargetName() != null);
    }

    public String getComponentEnvName() {
        return getName();
    }

    public String getInjectResourceType() {
        return type;
    }

    public void setInjectResourceType(String resourceType) {
        type = resourceType;
    }
    

    public StringBuffer printInjectableResourceInfo
        (StringBuffer toStringBuffer) {
        
        if( isInjectable() ) {
            for (InjectionTarget target : getInjectionTargets()) {
                if( target.isFieldInjectable() ) {
                    toStringBuffer.append("Field-Injectable Resource. Class name = ").
                            append(target.getClassName()).append(" Field name=").
                            append(target.getFieldName());
                } else {
                    toStringBuffer.append("Method-Injectable Resource. Class name =").
                            append(target.getClassName()).append(" Method =").                            
                            append(target.getMethodName());
                }
            }
        } else {
            toStringBuffer.append("Non-Injectable Resource");
        }

        return toStringBuffer;
    }
    
    // 
    // End InjectableResource implementation
    //
    
    
}
