/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.annotation.handlers;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.interceptor.AroundInvoke;
import javax.interceptor.Interceptors;
import javax.ejb.PostActivate;
import javax.ejb.PrePassivate;

import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.EjbBundleDescriptor;
import com.sun.enterprise.deployment.InterceptorBindingDescriptor;
import com.sun.enterprise.deployment.LifecycleCallbackDescriptor;
import com.sun.enterprise.deployment.EjbInterceptor;
import static com.sun.enterprise.deployment.LifecycleCallbackDescriptor.CallbackType;
import com.sun.enterprise.deployment.EjbSessionDescriptor;
import com.sun.enterprise.deployment.MethodDescriptor;
import com.sun.enterprise.deployment.annotation.AnnotationInfo;
import com.sun.enterprise.deployment.annotation.AnnotationProcessorException;
import com.sun.enterprise.deployment.annotation.HandlerProcessingResult;
import com.sun.enterprise.deployment.annotation.ProcessingContext;
import com.sun.enterprise.deployment.annotation.context.EjbContext;
import com.sun.enterprise.deployment.annotation.context.EjbInterceptorContext;
import com.sun.enterprise.deployment.annotation.impl.ComponentDefinition;

/**
 * This handler is responsible for handling javax.ejb.Interceptors
 *
 */
public class InterceptorsHandler extends AbstractAttributeHandler {
    
    public InterceptorsHandler() {
    }
    
    /**
     * @return the annoation type this annotation handler is handling
     */
    public Class<? extends Annotation> getAnnotationType() {
        return Interceptors.class;
    }    
        
    protected HandlerProcessingResult processAnnotation(AnnotationInfo ainfo,
            EjbContext[] ejbContexts) throws AnnotationProcessorException {

        Interceptors interceptors = (Interceptors) ainfo.getAnnotation();

        
        EjbBundleDescriptor ejbBundle = 
            ((EjbDescriptor)ejbContexts[0].getDescriptor()).
                getEjbBundleDescriptor();
        
        // Process each of the interceptor classes.
        for(Class interceptor : interceptors.value()) {
            processInterceptorClass(interceptor, ejbBundle, ainfo);
        }

        for(EjbContext next : ejbContexts) {

            EjbDescriptor ejbDescriptor = (EjbDescriptor) next.getDescriptor();

            // Create binding information.  
            InterceptorBindingDescriptor binding = 
                new InterceptorBindingDescriptor();

            binding.setEjbName(ejbDescriptor.getName());

            for(Class interceptor : interceptors.value()) {
                binding.appendInterceptorClass(interceptor.getName());
            }
            
            if(ElementType.METHOD.equals(ainfo.getElementType())) {
                Method m = (Method) ainfo.getAnnotatedElement();
                MethodDescriptor md = 
                    new MethodDescriptor(m, MethodDescriptor.EJB_BEAN);
                binding.setBusinessMethod(md);
            }

            // All binding information processed from annotations should go
            // before the binding information processed from the descriptors.
            // Since descriptors are processed first, always place the binding
            // info at the front.  The binding information from the descriptor
            // is ordered, but there is no prescribed order in which the 
            // annotations are processed, so all that matters is that it's
            // before the descriptor bindings and that the descriptor binding
            // order is preserved.
            ejbBundle.prependInterceptorBinding(binding);
        }

        return getDefaultProcessedResult();
    }

    private void processInterceptorClass(Class interceptorClass,
            EjbBundleDescriptor ejbBundle, AnnotationInfo ainfo) 
        throws AnnotationProcessorException {
                
        Set<LifecycleCallbackDescriptor> aroundInvokeDescriptors =
            new HashSet<LifecycleCallbackDescriptor>();
        Set<LifecycleCallbackDescriptor> postActivateDescriptors =
            new HashSet<LifecycleCallbackDescriptor>();
        Set<LifecycleCallbackDescriptor> prePassivateDescriptors =
            new HashSet<LifecycleCallbackDescriptor>();
        
        ComponentDefinition cdef = new ComponentDefinition(interceptorClass);
        for(Method m : cdef.getMethods()) {
            if( m.getAnnotation(AroundInvoke.class) != null ) {
                aroundInvokeDescriptors.add(getLifecycleCallbackDescriptor(m));
            }
            if( m.getAnnotation(PostActivate.class) != null ) {
                postActivateDescriptors.add(getLifecycleCallbackDescriptor(m));
            }
            if( m.getAnnotation(PrePassivate.class) != null ) {
                prePassivateDescriptors.add(getLifecycleCallbackDescriptor(m));
            }
        }
        
        EjbInterceptor interceptor =
            ejbBundle.getInterceptorByClassName(interceptorClass.getName());
        if (interceptor == null) {
            interceptor = new EjbInterceptor();
            interceptor.setInterceptorClassName(interceptorClass.getName());
            // Add interceptor to the set of all interceptors in the ejb-jar
            ejbBundle.addInterceptor(interceptor);
        }
        
        if (aroundInvokeDescriptors.size() > 0) {
            interceptor.addAroundInvokeDescriptors(aroundInvokeDescriptors);
        }
        
        if (postActivateDescriptors.size() > 0) {
            interceptor.addCallbackDescriptors(CallbackType.POST_ACTIVATE,
                postActivateDescriptors);
        }
        
        if (prePassivateDescriptors.size() > 0) {
            interceptor.addCallbackDescriptors(CallbackType.PRE_PASSIVATE,
                prePassivateDescriptors);
        }

        // process resource related annotations
        EjbInterceptorContext ejbInterceptorContext =
            new EjbInterceptorContext(interceptor);
        ProcessingContext procContext = ainfo.getProcessingContext();
        procContext.pushHandler(ejbInterceptorContext);
        procContext.getProcessor().process(
            procContext, new Class[] { interceptorClass });
        return;
    }

    /**
     * @return an array of annotation types this annotation handler would 
     * require to be processed (if present) before it processes it's own 
     * annotation type.
     */
    public Class<? extends Annotation>[] getTypeDependencies() {
        return getEjbAnnotationTypes();
    }

    private LifecycleCallbackDescriptor getLifecycleCallbackDescriptor(Method m) {
        LifecycleCallbackDescriptor lccDesc = new LifecycleCallbackDescriptor();
        lccDesc.setLifecycleCallbackClass(m.getDeclaringClass().getName());
        lccDesc.setLifecycleCallbackMethod(m.getName());
        return lccDesc;
    }
}
