/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.node;

import com.sun.enterprise.deployment.util.DOLUtils;
import java.io.InputStream;
import java.util.logging.Level;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 *Provides access to schemas and DTDs to Java Web Start-launched app clients 
 *that do not have an app server product installation at hand.
 *<p>
 *The DTDs and schemas are assumed to be in the classpath so that
 *schemas are at /schemas/<schema-name> and DTDs at /dtds/<dtd-name>
 *
 * @author tjquinn
 */
public class SaxParserHandlerBundled extends SaxParserHandler {

    /** prefixes for the paths to use for looking up schemas and dtds as resources */
    private static final String BUNDLED_SCHEMA_ROOT = "/schemas";
    private static final String BUNDLED_DTD_ROOT = "/dtds";
    
    /** Creates a new instance of SaxParserHandlerBundled */
    public SaxParserHandlerBundled() {
    }
    
    /**
     *Returns an InputSource for the requested DTD or schema.
     *<p>
     *This implementation returns an InputSource that wraps the result 
     *of getResourceAsStream, having
     *located the requested schema in the classpath.
     *@param the public ID of the requested entity
     *@param the system ID of the requested entity
     *@return InputSource for the requested entity; null if not available
     *@throws SAXException in case of errors resolving the entity
     */
    public InputSource resolveEntity(String publicID, String systemID) throws SAXException {
        InputSource result = null;
        
        /*
         *This logic was patterned after that in the superclass.
         */
        try {
            if(DOLUtils.getDefaultLogger().isLoggable(Level.FINE)) {
                DOLUtils.getDefaultLogger().fine("Asked to resolve  " + publicID + " system id = " + systemID);
            }
            if (publicID==null) {
                // unspecified schema
                if (systemID==null || systemID.lastIndexOf('/')==systemID.length()) {
                    return null;
                }
                
                /*
                 *Attempt to open a stream to the requested resource as a schema.
                 */
                InputStream is = openSchemaStream(systemID);
                
                if (is != null) {
                    /*
                     *We found the entity, so wrap an InputSource around the stream.
                     */
                    result = new InputSource(is);
                } else {
                    /*
                     *The entity was not found, so wrap an InputSource around the system ID string instead.
                     */
                    result = new InputSource(systemID);
                }
            } else {
                /*
                 *Try to find a DTD for the entity.
                 */
                if (mapping.containsKey(publicID)) {
                    this.publicID = publicID;
                    InputStream is = openDTDStream(publicID);
                    if (is != null) {
                        result = new InputSource(is);
                    } 
                } else if (systemID != null) {
                    /*
                     *The DTD lookup failed but a system ID was also specified.  Try
                     *looking up the DTD in the schemas path, because some reside
                     *there and were not registered in the DTD map by SaxParserHandler.
                     */
                    InputStream is = openSchemaStream(systemID);
                    if (is != null) {
                        result = new InputSource(is);
                    }
                }
            }
        } catch (Exception exc) {
            exc.printStackTrace();
            throw new SAXException(exc);
        }
        return result;
    }
    
    /**
     *Returns an InputStream for the schema with the requested system ID.
     *@param systemID of the schema
     *@return an InputStream to the selected schema; null if the schema is not available
     */
    private InputStream openSchemaStream(String systemID) {
        String targetID = BUNDLED_SCHEMA_ROOT + "/" + systemID.substring(systemID.lastIndexOf("/") + 1 );
        InputStream result = this.getClass().getResourceAsStream(targetID);
        return result;
    }
    
    /**
     *Returns an InputStream for the DTD with the requested public ID.
     *@param the publicID of the DTD requested
     *@return an InputStream to the selected DTD; null if the DTD is not available
     */
    private InputStream openDTDStream(String publicID) {
        String targetID = BUNDLED_DTD_ROOT + "/" + mapping.get(publicID);
        InputStream result = this.getClass().getResourceAsStream(targetID);
        return result;
    }
}
