/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

//
// Copyright 2001 iPlanet/ Sun Microsystems, Inc. All Rights Reserved.
//
// Author : darpan
// Module : Utils
//
package com.sun.enterprise.util.scheduler;

import com.sun.enterprise.util.ApproximateClock;
//Bug 4677074 begin
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
//Bug 4677074 end

/**
* PeriodicEventScheduler manages tasks of the type PeriodicallyServicable. Such 
* objects can be added to the manager and the time interval in these objects
* defines at what frequency the service() method of these tasks gets invoked by 
* the manager.<br>
* It is critical that the service() operation be as brisk as possible for the 
* manager to work effectively.<br>
* PeriodicEventScheduler is internally set as daemon.
* <BR> <I>$Source: /cvs/glassfish/appserv-commons/src/java/com/sun/enterprise/util/scheduler/PeriodicEventScheduler.java,v $</I>
* @author     $Author: tcfujii $
* @version    1.0 $Revision: 1.3 $ $Date: 2005/12/25 04:12:29 $
* @see PeriodicallyServicable
*/
public class PeriodicEventScheduler
    implements Runnable
{
//Bug 4677074 begin
    static Logger _logger=LogDomains.getLogger(LogDomains.UTIL_LOGGER);
//Bug 4677074 end
    // TODO: What the scheduler can also do, is to put these tasks at the top of 
    // the worker thread pool task queue, instead of executing the task on itself.
    
    /** Internal Thread holder */
    protected Thread _thread;    
    /** Scheduler instance holder */
    private static PeriodicEventScheduler _instance = null;
    /** Instance lock */
    private static Object instanceLock = new Object();
    /** Sorted task list */
    protected TimedTaskList sortedList;
    
    /** Thread runs while bRun is true (unless shutdown) */
    protected boolean bRun;
    /** Debug flag */
    protected boolean bDebug=false;
    /** Current time holder */
    protected transient long counter=0;
    /** Delay for time approximation */
    protected long delay_time_approx = 100;
    /** Approximate clock holder */
    protected ApproximateClock clock = null;

    static
    {
        _instance = new PeriodicEventScheduler();
    }
    
    /**
    * Method to access the PeriodicEventScheduler singleton. 
    * @return singleton object.
    */
    public static PeriodicEventScheduler getInstance()
    {
        return _instance;   
    }

    
    /**
    * Constructor invoked once for the singleton.
    */
    private PeriodicEventScheduler()
    {
        super();
        bRun = true;
        sortedList = new TimedTaskList();
        clock = new ApproximateClock (delay_time_approx);
        counter = getTime();
        _thread = new Thread (this, "PeriodicEventScheduler");
		_thread.setDaemon(true);
		_thread.start();
    }
    
    /**
    * Get approximate or actual time.
    * @return long current time.
    */
    private long getTime()
    {
        return clock.getActualTime(); // TODO : change actual time to perhaps approx time
    }

    /**
    * Add a PeriodicallyServicable object to the 'timed task execution queue'.
    * @param seconds time in seconds after which the task will be invoked for 
    * the first time. <I>Do not confuse with the frequency period which is given
    * by the getTimeIntervalForService() method on this object.</I>
    * @param obj object that has to be serviced in timed interval fashion.
    * @param startingTime start calling service() method in startingTime + frequency time.
    * @return boolean true on success, false otherwise
    */
    public synchronized boolean addTimeRepeatableTask(PeriodicallyServicable obj, int startingTime)
    {
        if(startingTime < 0 || obj.getFrequency() < 1)
        {
//Bug 4677074            if(bDebug) System.out.println("PeriodicEventScheduler::addTimeRepeatableTask() rejected task" + obj.toString());
//Bug 4677074 begin
	    if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"PeriodicEventScheduler::addTimeRepeatableTask() rejected task" + obj.toString());
//Bug 4677074 end

            return false;
        }
        boolean bool = sortedList.addTask(obj, startingTime, counter);
        synchronized(instanceLock)
        {
            instanceLock.notify();
        }
        return bool;
    }
    
    /**
    * Remove the servicable object from the task list.
    * @param obj PeriodicallyServicable object to be removed from the task list.
    * @return boolean true on success, false otherwise
    */
    public synchronized boolean removeTimeRepeatableTask(PeriodicallyServicable obj)
    {
        if(executingTask.equals(obj))
        {
            removeExecutingTask=true;
            return true;
        }
        else
            return sortedList.removeTask(obj);            
    }
    
    /**
    * Insert the task object into the task list.
    * @param taskObj task object to insert.
    * @return boolean true on success, false otherwise
    */
    protected synchronized boolean insertSorted(TaskData taskObj)
    {
        return sortedList.insertTask(taskObj);
    }
    
    /** Maintained by the run() method */
    private PeriodicallyServicable executingTask=null;
    /** On remove, a check is made if the executingTask is the one to be removed. This is the flag set. */
    private boolean removeExecutingTask=false;
    
    /**
    * Start running the thread.
    */
    public void run()
    {
        TaskData task=null;

        while(bRun)
        {
            try
            {
                //if(bDebug) System.out.println("---run()" + sortedList.toString() + "---");
//Bug 4677074 begin
		//if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"---run()" + sortedList.toString() + "---");
//Bug 4677074 end
                task = sortedList.getFirstTask();
                //if(bDebug) System.out.println("---run()" + sortedList.toString() + "+++");
//Bug 4677074 begin
                //if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"---run()" + sortedList.toString() + "+++");
//Bug 4677074 end
                if(null==task)
                {
                    synchronized(instanceLock)
                    {
                        instanceLock.wait();
                        continue; // fetches a task
                    }
                }
                
                executingTask=task.obj;
                
                // got the first task
                counter = getTime();
                long sleepingTime = task.abs_execute_time - counter;
                if(sleepingTime > 0L)
                {
                    try
                    {
//Bug 4677074                        if(bDebug) System.out.println("Current time=" + (int)(counter/1000) + ", Sleeping for " + sleepingTime + " msec.");
//Bug 4677074 begin
			if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"Current time=" + (int)(counter/1000) + ", Sleeping for " + sleepingTime + " msec.");
//Bug 4677074 end
                        Thread.sleep( sleepingTime );
                    }
                    catch(InterruptedException ieInner)
                    {
//Bug 4677074                        if(bDebug) System.out.println("PeriodicEventScheduler::run() > " + ieInner);
//Bug 4677074 begin
			if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"PeriodicEventScheduler::run() > " + ieInner);
//Bug 4677074 end
                    }
                }else
                {
                    // a decision has to be made immediately if we want to execute 
                    // the task even if we missed the right time to execute it
                    if (!task.obj.getExecutionTolerance(Math.abs(sleepingTime)) )
                    {
//Bug 4677074                         if(bDebug) System.out.println("Missed scheduling for " + task.obj.toString());
//Bug 4677074 begin
			if(com.sun.enterprise.util.logging.Debug.enabled)  _logger.log(Level.FINE,"Missed scheduling for " + task.obj.toString());
//Bug 4677074 end
                        continue;
                    } else
                    {
//Bug 4677074                        if(bDebug) System.out.println("Executing after missing scheduling for " + task.obj.toString());
//Bug 4677074 begin
			if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"Executing after missing scheduling for " + task.obj.toString());
//Bug 4677074 end
                    }
                }
              
                // now we can execute this task in multiple ways, one is on this thread,
                // other is on other task queues (by putting this task on the front of the Q)
                task.obj.service();
                
            }catch(InterruptedException ieOuter)
            {
//Bug 4677074                if(bDebug) System.out.println("PeriodicEventScheduler::run() > " + ieOuter);
//Bug 4677074 begin
			if(com.sun.enterprise.util.logging.Debug.enabled)  _logger.log(Level.FINE,"PeriodicEventScheduler::run() > " + ieOuter);
//Bug 4677074 end
            }
            catch(Exception e)
            {
                System.out.println("PeriodicEventScheduler::run() > " + e);
//Bug 4677074 begin
		_logger.log(Level.WARNING,"iplanet_util.generic_exception",e);
//Bug 4677074 end
            }
            finally
            {
                if (null!=task)
                {
                    counter = getTime();
                    // now put this task back into the Q
                    task.abs_execute_time = counter;
                    //if(bDebug) System.out.println("Adding in list " + task.obj.toString());
//Bug 4677074 begin
		    //if(com.sun.enterprise.util.logging.Debug.enabled) _logger.log(Level.FINE,"Adding in list " + task.obj.toString());
//Bug 4677074 end
                    if(!removeExecutingTask)
                        insertSorted(task);
                    else
                        removeExecutingTask=false;
                    executingTask=null;
                }
            }
        } // while
    }
    
    public String toString()
    {
        return "[PeriodicEventScheduler: " + sortedList.toString() + "]";
    }
}

