/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.base.sfsb;

import com.sun.ejb.spi.sfsb.SFSBUUIDUtil;
import com.sun.enterprise.util.Utility;

/**
 * A utility class that generates stateful session keys using two longs
 * The session id generated by this class is guarenteed to be unique as 
 * long as the system clock is never reset to a previous value
 * 
 * The hashCode of the SessionKey generated by SimpleKeyGenerator 
 * also allows uniform distribution of keys when hashed in a HashMap
 *
 * @author  Mahesh Kannan
 */
public class SimpleKeyGenerator
    implements SFSBUUIDUtil
{
   
    private int idCounter;
    private long time;
    private long prefix;

    public SimpleKeyGenerator() {
        //Inital isCounter value
        this.idCounter = 0;

        //Set the default time value
        this.time = System.currentTimeMillis();

        //Default value for prefix
        this.prefix = 0;
    }

    public void setPrefix(int val) {
        this.prefix = val;
        this.prefix = this.prefix << 32;
    }

    /**
     * Create and return the sessionKey.
     * @return the sessionKey object
     */
    public Object createSessionKey() {
        int id = 0;
        synchronized (this) {
            id = ++idCounter;
            if (id < 0) {
                //idCounter wrapped around!!
                id = idCounter = 0;
                time = System.currentTimeMillis();
            }
        }

        return new SimpleSessionKey(time, prefix + id);
    }
    
    /**
     * Called from the Container before publishing an IOR. 
     * The method must convert the sessionKey into a byte[]
     * @return A byte[] representation of the key. The byte[] 
     * could be created using serialization.
     */
    public byte[] keyToByteArray(Object sessionKey) {
        SimpleSessionKey key = (SimpleSessionKey) sessionKey;
        byte[] array = new byte[16];

        Utility.longToBytes(key.time, array, 0);
        Utility.longToBytes(key.id, array, 8);

        return array;
    }
    
     /**
      * Return the sessionKey that represents the sessionKey. 
      * This has to be super efficient as the container calls this 
      * method on every invocation. Two objects obtained from identical 
      * byte[] must satisfy both o1.equals(o2) and 
      * o1.hashCode() == o2.hashCode()
      * @return the sessionKey object
      */
    public Object byteArrayToKey(byte[] array, int startIndex, int len) {
        long myTime = Utility.bytesToLong(array, startIndex);
        long myId = Utility.bytesToLong(array, startIndex+8);

        return new SimpleSessionKey(myTime, myId);
    }

}

class SimpleSessionKey
    implements java.io.Serializable
{

    long time;
    long id;

    public SimpleSessionKey(long time, long id) {
        this.time = time;
        this.id = id;
    }
    
    public int hashCode() {
        return (int) id;
    }

    public boolean equals(Object otherObj) {
        if (otherObj instanceof SimpleSessionKey) {
            SimpleSessionKey other = (SimpleSessionKey) otherObj;
            return ((id == other.id) && (time == other.time));
        }

        return false;
    }

    public String toString() {
        return "" + time + "-" + id;
    }

}
    
