/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.base.stats;

import java.lang.reflect.Method;

import java.util.ArrayList;

import com.sun.enterprise.admin.monitor.stats.TimeStatisticImpl;
import com.sun.enterprise.admin.monitor.stats.MutableTimeStatisticImpl;

/**
 * A Class for providing stats for an EJB method
 *  All concrete S1AS containers instantiate one instance of
 *  this class for each EJB method. 
 *
 * @author Mahesh Kannan
 */

public class MethodMonitor {

    private String		methodName; 
    private boolean		monitorOn = true;

    private static ThreadLocal	execThreadLocal = new ThreadLocal();
    private Object		lock = new Object();
    private int			successCount = 0;
    private int			errorCount = 0;
    private int			invocationCount = 0;
    private long		totalExecutionTime = 0;

    private MutableTimeStatisticImpl  methodStat;

    public MethodMonitor(Method method) {
	this.methodName = constructMethodName(method);
	this.monitorOn = true;
    }

    void setMutableTimeStatisticImpl(MutableTimeStatisticImpl methodStat) {
	this.methodStat = methodStat;
    }

    public void preInvoke() {
	if (monitorOn) {
	    ArrayList list = (ArrayList) execThreadLocal.get();
	    if (list == null) {
		list = new ArrayList(5);
		execThreadLocal.set(list);
	    }
	    list.add(new Long(System.currentTimeMillis()));
	    synchronized (lock) {
		invocationCount++;
	    }
	}
    }

    public void postInvoke(Throwable th) {
	if (monitorOn) {
	    ArrayList list = (ArrayList) execThreadLocal.get();
	    if ( (list != null) && (list.size() > 0) ) {
		int index = list.size();
		Long startTime = (Long) list.remove(index-1);
		synchronized(lock) {
		    if (th == null) {
			successCount++;
		    } else {
			errorCount++;
		    }
		    if (startTime != null) {
			long diff = System.currentTimeMillis()
			    - startTime.longValue();
			totalExecutionTime = diff;

			methodStat.incrementCount(diff);
		    }
		}
	    }
	}
    }

    public void resetAllStats(boolean monitorOn) {
	successCount = 0;
	errorCount = 0;
	invocationCount = 0;
	totalExecutionTime = 0;
	this.monitorOn = monitorOn;
    }

    public String getMethodName() {
	return this.methodName;
    }

    public int getTotalInvocations() {
	return invocationCount;
    }

    public long getExecutionTime() {
	return totalExecutionTime;
    }

    public int getTotalNumErrors() {
	return errorCount;
    }

    public int getTotalNumSuccess() {
	return successCount;
    }

    public void appendStats(StringBuffer sbuf) {
	sbuf.append("\n\t[Method ")
	    .append("name=").append(methodName).append("; ")
	    .append("invCount=").append(invocationCount).append("; ")
	    .append("success=").append(successCount).append("; ")
	    .append("errors=").append(errorCount).append("; ")
	    .append("totalTime=").append(totalExecutionTime).append("]");
    }


    private String constructMethodName(Method method) {
	StringBuffer sbuf = new StringBuffer();
	sbuf.append(method.getName());
	Class[] paramTypes = method.getParameterTypes();
	int sz = paramTypes.length;
	if (sz > 0) {
	    String dash = "-";
	    for (int i=0; i<sz; i++) {
		sbuf.append(dash)
		    .append(paramTypes[i].getName());
	    }
	}
	return sbuf.toString();
    }

}
