/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.ejb.containers;

import javax.ejb.EJBException;
import javax.ejb.AccessLocalException;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Proxy;
import java.lang.reflect.Method;

import java.util.Map;
import com.sun.ejb.Invocation;
import com.sun.enterprise.Switch;
import com.sun.ejb.ComponentContext;
import com.sun.ejb.InvocationInfo;
import com.sun.enterprise.InvocationManager;
import com.sun.ejb.Container;
import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.WebServiceEndpoint;

import com.sun.ejb.containers.util.InvocationHandlerUtil;

/** 
 * This is a proxy invocation handler for web service ejb invocations.
 * A single instance of this invocation handler is used for all
 * web service invocations to a particular ejb endpoint, so it must support
 * concurrent use.
 *
 * @author Kenneth Saks
 */    

public final class WebServiceInvocationHandler extends EJBLocalRemoteObject 
    implements InvocationHandler {

    private WebServiceEndpoint endpoint_;
    private Class ejbClass_;
    private Class serviceEndpointIntfClass_;
    private InvocationManager invManager_;
    private boolean hasHandlers_;
    private Map invocationInfoMap_;

    public WebServiceInvocationHandler(Class ejbClass, 
                                       WebServiceEndpoint endpoint,
                                       Class serviceEndpointIntfClass,
                                       Map invocationInfoMap) {
        ejbClass_ = ejbClass;
        serviceEndpointIntfClass_ = serviceEndpointIntfClass;
        endpoint_ = endpoint;
        hasHandlers_ = endpoint.hasHandlers();
        Switch theSwitch = Switch.getSwitch();
        invManager_ = theSwitch.getInvocationManager();
        invocationInfoMap_ = invocationInfoMap;
    }

    public Object invoke(Object proxy, Method method, Object[] args) 
        throws Throwable {
        try {
        container.onEnteringContainer();
        // NOTE : be careful with "args" parameter.  It is null
        //        if method signature has 0 arguments.

        Class methodClass = method.getDeclaringClass();
        if( methodClass == java.lang.Object.class )  {
            return InvocationHandlerUtil.
                invokeJavaObjectMethod(this, method, args);    
        }

        Object returnValue = null;
        Invocation inv = null;

        try {
            // Invocation was created earlier in the web service dispatching
            inv = (Invocation) invManager_.getCurrentInvocation();

            inv.ejbObject = this;
            
            // things can become hairy here. This handler may have been created 
            // with a dummy SEI to satisfy the EJB container. In such cases, we must
            // find the right method object on the SIB.
            if (endpoint_.getServiceEndpointInterface().equals(ejbClass_.getName())) {
                // we need to substiture the method object
                method = ejbClass_.getMethod(method.getName(), method.getParameterTypes());
            }
            inv.method = method;
            inv.clientInterface = serviceEndpointIntfClass_;

            inv.invocationInfo = (InvocationInfo)
                invocationInfoMap_.get(inv.method);

            if( inv.invocationInfo == null ) {
                throw new EJBException
                    ("Web service Invocation Info lookup failed for " +
                     "method " + inv.method);
            }

            inv.transactionAttribute = inv.invocationInfo.txAttr;

            if( hasHandlers_ ) {
                // Handler performed method authorization already
            } else {
                StatelessSessionContainer container = 
                    (StatelessSessionContainer) getContainer();                

                boolean authorized = container.authorize(inv);
                if( !authorized ) {
                    throw new AccessLocalException("Client not authorized " +
                                                   "to access " + inv.method);
                }
            }

            ComponentContext ctx = container.getContext(inv);
            inv.context  = ctx;
            inv.ejb      = ctx.getEJB();
            inv.instance = inv.ejb;

            container.preInvokeTx(inv);

            // Enterprise Bean class doesn't necessarily implement
            // web service endpoint interface, so we can't directly
            // dispatch through the given method object.
            Method beanClassMethod = ejbClass_.getMethod
                (method.getName(), method.getParameterTypes());
            inv.beanMethod = beanClassMethod;
            inv.methodParams = args;
            returnValue = container.intercept(inv);
        } catch(NoSuchMethodException nsme) {
            inv.exception = nsme;
        } catch(InvocationTargetException ite) {
            inv.exception = ite.getCause();
        } catch(Throwable c) {
            inv.exception = c;
        } finally {
            // ejb web service dispatcher handles postInvoke
        }
        if (inv.exception != null) {
            if(inv.exception instanceof java.lang.RuntimeException) {
                throw (java.lang.RuntimeException)inv.exception; 
            } else if (inv.exception instanceof Exception) {
                throw inv.exception;
            } else {
                EJBException ejbEx = new EJBException();
                ejbEx.initCause(inv.exception);
                throw ejbEx;
            }
        }
        return returnValue;
        } finally {
            container.onLeavingContainer();
        }
    }
    
}
