/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.common;

//JDK imports

//JMX imports
import javax.management.MBeanServer;

// i18n import
import com.sun.enterprise.admin.util.SOMLocalStringsManager;

/**
    A class to create instances of MBeanServer. Note that this is not analogous
    to classic Factory Pattern, since this factory needs to be initialized
    according to context. Server Object Model requires MBeanServer reference 
    only when it runs in the context of admin server (meaning it is called from
    the admin html GUI). In case of clients that do not run in the JVM of the
    admin server, i.e. CLI and Plugin, Server Object Model does not need the
    reference to MBeanServer as it goes via the HTTP connector.
    Hence this class provides a method to initialize the factory. The factory
    should be initialized at admin server startup. This is currently done
    in AdminService class.

    @version 1.0
    @author Kedar Mhaswade
*/

public class MBeanServerFactory
{
    public static final String kDefaultInitializerClassName =
        "com.sun.enterprise.admin.server.core.AdminService";

    private static MBeanServer sMBeanServerInstance = null;

	// i18n SOMLocalStringsManager
	private static SOMLocalStringsManager localizedStrMgr =
		SOMLocalStringsManager.getManager( MBeanServerFactory.class );

    /**
        Initializes this factory. Also creates the singleton MBeanServer instance
        in the admin server's JVM. It does not allow any class to initialize the 
        factory. Currently only the default initializer is able to create the 
        instance. If the initializer is not the default one then IllegalArgumentException
        will be thrown. The initializer has to initialize the MBeanServer instance.
        The parameters may not be null. 
        @param initializer instance of Object with proper class.
        @param mbs instance of MBeanServer's implementation that acts as singleton.
        @throws IllegalArgumentException if the contract is not satisfied by caller. 
        @throws IllegalStateException if the initialize has been called already.
    */
    public static void initialize(Object initializer, MBeanServer mbs)
    {
        if (sMBeanServerInstance != null)
        {
			String msg = localizedStrMgr.getString( "admin.common.already_initialized" );
            throw new IllegalStateException( msg );
        }
        if (initializer == null || mbs == null)
        { 
			String msg = localizedStrMgr.getString( "admin.common.null_arg" );
            throw new IllegalArgumentException( msg );
        }
        if (!initializer.getClass().getName().equals(kDefaultInitializerClassName))
        {
			String msg = localizedStrMgr.getString( "admin.common.invalid_initializer" );
            throw new IllegalArgumentException( msg );
        }
        sMBeanServerInstance = mbs;
    }
    
    /**
        Returns the instance of singleton (for Admin Server's JVM) for MBeanServer.
        Returns null if the factory is not initiazed earlier. Thus a null will be
        returned when this method is called in a client's context. In case of
        Server context, since the initiazation of factory is expected to provide
        the implementation, this should return a non null value (a valid instance).
        @return instance of MBeanServer, null if Factory is not initialized.
    */
    public static MBeanServer getMBeanServer()
    {
        return ( sMBeanServerInstance );
    }
}
