/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.common.domains.registry;

import java.util.Iterator;
import java.util.HashMap;
import java.io.Serializable;
import java.io.File;
import java.lang.UnsupportedOperationException;
import java.util.NoSuchElementException;
import java.io.IOException;
import java.io.EOFException;

/**
   This class provides a registry for domains, abstracting the
   persistent storage and coordination activities required to permit
   uniqueness invariants to be maintained between domains no matter
   which JVM they are being referred from.
   <p>
   The directory in which the persistent store is kept is denoted by
   the <code>System</code> property
   <code>com.sun.aas.admin.configRoot</code>. The name of the
   file is <code>domains.ser</code>
   <p>
   The principal invariants this registry maintains is that no two
   domains shall have the same name, and no two domains shall have the
   same location.
   <p>
   The registry can be considered to be a table whose keys are domain
   name and domain location, and whose values are the remote contact
   information by which the adminstration server within that domain
   can be contacted remotely.
   <p>
   A row in this table is abstracted into the {@link DomainEntry} class. 
   *
   * @author <a href="mailto:toby.h.ferguson@sun.com">Toby H Ferguson</a>
   * @version $Revision: 1.3 $
   */
public class DomainRegistry implements DomainRegistryI
{

		/**
		 * Return an instance of the DomainRegistry
		 * @return a <code>DomainRegistry</code> value
		 @throws DomainRegistryException if there was a problem
		 instantiating the class
		 */
  synchronized static public DomainRegistry newInstance() throws DomainRegistryException {
	if (instance == null){
	  instance = new DomainRegistry();
	  instance.init();
	}
	return instance;
  }

	public synchronized void registerDomain(DomainEntry de) throws DomainRegistryException {
		prepareForUpdate();
		try {
		  registry.registerDomain(de);
		}
		catch (DomainRegistryException e){
		  store.unlock();
		  throw e;
		}
		saveRegistry();
	}

	public void unregisterDomain(String domain_name) throws DomainRegistryException {
	  prepareForUpdate();
	  try{
		registry.unregisterDomain(domain_name);
	  }
	  catch (DomainRegistryException e){
		store.unlock();
		throw e;
	  }
	  saveRegistry();
	  
	}

	public void unregisterDomain(DomainEntry de) throws DomainRegistryException {
	  this.unregisterDomain(de.getName());
	}

	public void reregisterDomain(DomainEntry de) throws DomainRegistryException {
	  prepareForUpdate();
	  try {
		registry.reregisterDomain(de);
	  }
	  catch (DomainRegistryException e){
		store.unlock();
		throw e;
	  }
	  saveRegistry();

	}
  
	public boolean containsDomain(DomainEntry de) throws DomainRegistryException {
	  refreshRegistry();
	  return registry.containsDomain(de);
	}

	public DomainEntry getDomain(String name) throws DomainRegistryException {
	  refreshRegistry();
	  return registry.getDomain(name);
	}

  public Iterator iterator() throws DomainRegistryException {
	refreshRegistry();
	return registry.iterator();
  }
  
	public int size() throws DomainRegistryException {
	  refreshRegistry();
	  return registry.size();
	}
	

	void reset() throws IOException {
	  store.unlock();
	  registry = null;
	  store = null;
	  instance = null;
	}

  private void refreshRegistry() throws DomainRegistryException {
	if (lastModified < store.lastModified()){
	  try {
		registry = getRegistryFromStore();
	  }
	  catch (Exception te){
		throw new DomainRegistryException("problem reading from store", te);
	  }
	}
  }

  private Registry getRegistryFromStore() throws IOException, TimeoutException, ClassNotFoundException {
	Registry br;
	br = (Registry) store.readObject();
	return (br != null ? br : new Registry());
  }
  

  private void prepareForUpdate() throws DomainRegistryException {
	try {
	  store.lock();
	}
	catch (Exception e){
	  throw new DomainRegistryException("problem locking store ", e);
	}
	refreshRegistry();
  }

  private void saveRegistry() throws DomainRegistryException {
	try {
	  store.writeObject(registry);
	  store.unlock();
	  lastModified = store.lastModified();
	}
	catch (Exception e){
	  e.printStackTrace();
	  throw new DomainRegistryException("couldn't save registry", e);
	}
  }
  
  
  private void init()  throws DomainRegistryException {
	try {
	  store = LockingStoreFactory.getInstance();
	  registry = getRegistryFromStore();
	}
	catch (Exception e){
	  throw new DomainRegistryException("couldn't initialize registry. Error message: "+ e.getMessage());
	}
  }

/*
  PersistentStore getPS(){
	return (PersistentStore) store;
  }
 */
 
  private Registry registry;
  private LockingStore store;
  private static DomainRegistry instance = null;
  private long lastModified = 0;
  
  private DomainRegistry(){}

}
