/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.diagnostics.util;

import java.io.File;

import java.io.FilenameFilter;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.text.ParseException;

import com.sun.enterprise.diagnostics.Constants;
import com.sun.enterprise.diagnostics.DiagnosticException;
/**
 * Log file name is assumed to be in server.log_YYYY-MM-DDTHH-MM-SS format
 * eg.server.log_2005-03-04T12-21-53 format. Files are sorted in ascending order.
 * @author Manisha Umbarje
 */
public class LogNameFilter implements FilenameFilter {

    private Date startDate;
    private Date endDate;
    private String fileNamePrefix;
    
    private static final SimpleDateFormat dateFormat = 
    new SimpleDateFormat(Constants.DATE_PATTERN);

    public LogNameFilter (String fileName, Date startDate, Date endDate) {
	    this.startDate = startDate;
	    this.endDate = endDate;
	    fileNamePrefix = fileName;
    }

    public boolean accept (File aDir, String fileName) {
	if (aDir == null || fileName == null)
	    return false;
	if (fileName.indexOf(fileNamePrefix)<0)
	    return false;
	int datePatternIndex = fileName.indexOf(Constants.FILENAME_DATE_SEPARATOR);

	if (datePatternIndex > 0) {
	    try {
		// fileDate indicates date of last entry in the file
		Date fileDate = dateFormat.parse
		(fileName.substring(datePatternIndex+1,datePatternIndex+11));

		if (fileDate.compareTo(startDate) >=0 && 
		    fileDate.compareTo(endDate) <=0)
		{
		    return true;
		}
		else
		    return false;
	    } catch(Exception e) {
		return false;
	    }
	} else {
	    try
	    {
		// server.log, check if date of first entry is before
		// the endDate, assuption here is every new log file start
		// with new logEntry and not partial entry
		return endDate.after(getDateOfFirstLogEntry(aDir,fileName));
	    } catch (DiagnosticException de) {
		    return false;
	    }
	}
    }

    private Date getDateOfFirstLogEntry(File aDir, String fileName)
			    throws DiagnosticException {
	if (aDir == null || fileName == null)
	    return null;

	try {
	    File file = new File(aDir, fileName);
	    BufferedReader reader = new BufferedReader(new FileReader(file));
	    String firstEntry = reader.readLine();

	    return dateFormat.parse(firstEntry.substring
	    (Constants.ENTRY_DATE_BEGIN_INDEX,Constants.ENTRY_DATE_LENGTH));
	} catch (FileNotFoundException fnf) {
	    throw new DiagnosticException(fnf.getMessage());
	} catch (IOException io) {
	    throw new DiagnosticException(io.getMessage());
	} catch (ParseException pe) {
	    throw new DiagnosticException(pe.getMessage());
	}
    }
}
