/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.audit;

import java.util.Properties;

import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.AuditModule;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.config.serverbeans.SecurityService;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.AuditModuleEvent;
import com.sun.enterprise.admin.event.AuditModuleEventListener;

/**
 * Listener interface to handle audit module events.
 * So that audit module can be dynamically created/update/deleted.
 * @author Shing Wai Chan
 */
public class AuditModuleEventListenerImpl implements AuditModuleEventListener {

    /**
     * New audit module created.
     * It is called whenever a AuditModuleEvent with action of
     * AuditModuleEvent.ACTION_CREATE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void auditModuleCreated(AuditModuleEvent event)
             throws AdminEventListenerException {
        try {
            String moduleName = event.getModuleName();
            AuditModule am = getAuditModule(moduleName,
                    event.getConfigContext());
            String classname = am.getClassname();
            Properties props = getAuditModuleProperties(am);      

            AuditManager manager =
                    AuditManagerFactory.getInstance().getAuditManagerInstance();
            manager.addAuditModule(moduleName, classname, props);
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }

    /**
     * Audit module deleted.
     * It is called whenever a AuditModuleEvent with action of
     * AuditModuleEvent.ACTION_DELETE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void auditModuleDeleted(AuditModuleEvent event)
             throws AdminEventListenerException {
        try {
            String moduleName = event.getModuleName();
            AuditManager manager =
                    AuditManagerFactory.getInstance().getAuditManagerInstance();
            manager.removeAuditModule(moduleName);
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }

    /**
     * Audit module updated (attributes change).
     * It is called whenever a AuditModuleEvent with action of
     * AuditModuleEvent.ACTION_UPDATE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void auditModuleUpdated(AuditModuleEvent event)
             throws AdminEventListenerException {
        try {
            String moduleName = event.getModuleName();

            AuditModule am = getAuditModule(moduleName,
                    event.getConfigContext());
            String classname = am.getClassname();
            Properties props = getAuditModuleProperties(am);      

            AuditModule oldAm = getAuditModule(moduleName,
                    event.getOldConfigContext());
            String oldClassname = oldAm.getClassname();
            Properties oldProps = getAuditModuleProperties(oldAm);      

            AuditManager manager =
                    AuditManagerFactory.getInstance().getAuditManagerInstance();
            if (!classname.equals(oldClassname)) {
                manager.addAuditModule(moduleName, classname, props);
            } else if (!props.equals(oldProps)) {
                com.sun.appserv.security.AuditModule auditModule =
                        manager.getAuditModule(moduleName);
                auditModule.init(props);
            }
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }

    private AuditModule getAuditModule(String moduleName,
            ConfigContext configContext) throws ConfigException {
        SecurityService security = 
            ServerBeansFactory.getSecurityServiceBean(configContext);
        return security.getAuditModuleByName(moduleName);

    }

    private Properties getAuditModuleProperties(AuditModule am) {
        ElementProperty[] elementProps = am.getElementProperty();
        int size = (elementProps != null) ? elementProps.length : 0;
        Properties props = new Properties();
        //XXX should we set this?
        props.setProperty(AuditManager.NAME, am.getName());
        props.setProperty(AuditManager.CLASSNAME, am.getClassname());
        for (int i = 0; i < size; i++) {
            props.setProperty(elementProps[i].getName(),
                    elementProps[i].getValue());
        }
        return props;
    }
}
