/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.server;

import java.io.File;
import java.util.ArrayList;
import java.io.IOException;

import com.sun.enterprise.Switch;
import com.sun.enterprise.server.J2EEServer;
import com.sun.enterprise.security.audit.AuditModuleEventListenerImpl;
import com.sun.enterprise.security.audit.SecurityServiceEventListenerImpl;
import com.sun.enterprise.security.auth.realm.AuthRealmEventListenerImpl;
import com.sun.enterprise.security.auth.realm.UserMgmtEventListenerImpl;

import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.config.serverbeans.TransactionService;
import com.sun.enterprise.config.serverbeans.Applications;
import com.sun.enterprise.config.serverbeans.ServerXPathHelper;
import com.sun.enterprise.config.ConfigBeansFactory;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;

import com.sun.enterprise.instance.InstanceFactory;
import com.sun.enterprise.instance.InstanceEnvironment;
import com.sun.enterprise.instance.AppsManager;
import com.sun.enterprise.instance.EjbModulesManager;
import com.sun.enterprise.instance.ConnectorModulesManager;
import com.sun.enterprise.loader.ClassLoaderUtils;
import com.sun.enterprise.util.logging.Debug;

import com.sun.enterprise.util.ConnectorClassLoader;
import com.sun.enterprise.resource.ResourceInstaller;

import com.sun.appserv.server.ServerLifecycle;
import com.sun.appserv.server.ServerLifecycleException;

import com.sun.enterprise.server.pluggable.InternalServicesList;

import com.sun.enterprise.admin.event.AdminEventListenerRegistry;
import com.sun.enterprise.admin.event.jms.JmsServiceEvent;
import com.sun.enterprise.admin.event.jms.JmsHostEvent;
import com.sun.enterprise.admin.alert.AlertConfigurator;
//import com.sun.enterprise.admin.event.EventTester;

//IASRI 4717059 BEGIN
//import com.sun.ejb.containers.ReadOnlyBeanNotifierImpl;
//IASRI 4717059 END

import java.util.logging.Level;
import java.util.logging.Logger;
import com.sun.logging.LogDomains;
import com.sun.enterprise.server.logging.ServerLogManager;

/**
 * ApplicationServer is the main entry point for iAS 7.0 J2EE server instance.
 * This code implements the server lifecycle state transitions and it relies on
 * various iAS 7.0 subsystems such as Web/EJB containers throught its lifecycle.
 * com.sun.enterprise.J2EEServer is the RI counerpart of this code; the evolving
 * iAS server implementation replaces functionality of RI J2EEServer.
 */
public class ApplicationServer {

    /** server logger */
    static Logger _logger = LogDomains.getLogger(LogDomains.CORE_LOGGER);

    public static java.io.PrintStream ostream = System.out;
    public static java.io.PrintStream estream = System.err;

    private static final String USER_TX = "java:comp/UserTransaction";
    private static final String CODEBASE_PROP = "java.rmi.server.codebase";

    /** 
     * Default services implementing lifecycle interface for orderly startup/
     * shutdown and hot deployment.
     * Each service entry is of the form {name, className}.
     * Note: Make sure LifecycleModuleService is the last subsystem in this list.
     */
    static String[][]servicesByName;


    /** ServerContext -- server-wide runtime environment context **/
    private static ServerContext context = null;

    /** services -- standard runtime service objects **/
    private ArrayList services = new ArrayList();

    /** common class loader for the server */
    private ClassLoader commonClassLoader;

    /** share class loader for the server */
    private ClassLoader connectorClassLoader;

    /** 
     * Default constructor.
     */
    public ApplicationServer() {}

    /**
     * Server is initializing subsystems and setting up the runtime environment.
     * Prepare for the beginning of active use of the public methods of this
     * subsystem. This method is called before any of the public methods of 
     * this subsystem are utilized.  
     *
     * @param context ServerContext the server runtime context.
     *
     * @exception IllegalStateException if this subsystem has already been
     *  started
     * @exception ServerLifecycleException if this subsystem detects a fatal 
     *  error that prevents this subsystem from being used
     */
    public void onInitialization(ServerContext context) 
                throws ServerLifecycleException
    {
        // sets the server context
        this.context = context;

        // initialize all server loggers that were not initialized
        // Note: reInitializeServerLoggers should be called after 
        //       setting the server context and before performing any
        //       other server startup activities that needs a logger

        ServerLogManager.reInitializeServerLoggers();
        
        // print jvm info
        printStartupInfo();

        // system class loader 
        ClassLoader parentOfCommonCL = this.getClass().getClassLoader();
        //Shared CL
        //In the new CL hierarchy the parent of common classloader
        //is the shared chain.
        if(Boolean.getBoolean(com.sun.enterprise.server.PELaunch.USE_NEW_CLASSLOADER_PROPERTY))
            parentOfCommonCL = PELaunch.getSharedChain();

        InstanceEnvironment env = this.context.getInstanceEnvironment();

        try {

            String dir     =  env.getLibClassesPath();
            String jarDir  = env.getLibPath();

            // constructs the common class loader. System class loader 
            // is the parent of common class loader. Common class loader 
            // includes <instance>/lib/classes dir, <instance>/lib/*.jar 
            // and <instance>/lib/*.zip
            commonClassLoader = 
                ClassLoaderUtils.getClassLoader(new File[] {new File(dir)}, 
                                                new File[] {new File(jarDir)},
                                                parentOfCommonCL
                                                );

            // ignore if null
            if (commonClassLoader == null) {
                commonClassLoader = parentOfCommonCL;
            }
        } catch (IOException ioe) {
            _logger.log(Level.WARNING, "server.ioexception", ioe);
            commonClassLoader = parentOfCommonCL;
        } catch (Throwable th) {
            _logger.log(Level.WARNING, "server.exception", th);
            commonClassLoader = parentOfCommonCL;
        }

        // Create the connector class loader. This will be the parent class
        // loader for all j2ee components!! Common class loader is the 
        // parent of connector class loader.
        connectorClassLoader =
            ConnectorClassLoader.getInstance(commonClassLoader);
        
        // sets the class loaders in the server context
        if (this.context instanceof ServerContextImpl) {
            ServerContextImpl contextImpl = (ServerContextImpl) this.context;

            // set the common classloader in the server context
            contextImpl.setCommonClassLoader(commonClassLoader);

            // sets the shared class loader
            contextImpl.setSharedClassLoader(connectorClassLoader);

            // sets the life cycle parent class loader
            contextImpl.setLifecycleParentClassLoader(connectorClassLoader);
        }

        // final common class loader for the anonymous inner class
        final ClassLoader commonCL = commonClassLoader;

        // set the common class loader as the thread context class loader
            java.security.AccessController.doPrivileged(
                    new java.security.PrivilegedAction() {
                public Object run() {
                    Thread.currentThread().setContextClassLoader(commonCL);
                    return null;
                }
            }
            );

	//added the pluggable interface way of getting the list of services
	InternalServicesList servicesList = 
		context.getPluggableFeatureFactory().getInternalServicesList();

	servicesByName = servicesList.getServicesByName();
	if (servicesByName == null) {
	      _logger.log(Level.SEVERE, "services.null");
	      throw new ServerLifecycleException();
	}

        // security manager is set inside SecurityLifecycle

	// Instantiate the built-in runtime services
	services = instantiateRuntimeServices();
	        
        // Initialize the built-in runtime services
        for (int i = 0; i < services.size(); i++) {
            Object service = services.get(i);
            if (service instanceof ServerLifecycle) {
	        try {
                    ((ServerLifecycle) service).onInitialization(context);
		} catch (ServerLifecycleException e) {		   
		    _logger.log(Level.SEVERE, "service.notinit", 
				new Object[] {service, e.toString()});
		    throw e;
		}
            }
        }

        //add listeners for security dynamic reconfiguration
        AdminEventListenerRegistry.addAuditModuleEventListener(
                new AuditModuleEventListenerImpl());
        AdminEventListenerRegistry.addAuthRealmEventListener(
                new AuthRealmEventListenerImpl());
        AdminEventListenerRegistry.addSecurityServiceEventListener(
                new SecurityServiceEventListenerImpl());
        AdminEventListenerRegistry.addUserMgmtEventListener(
                new UserMgmtEventListenerImpl());

        // Call RI J2EEServer initialization code. Note that all resources
        // specified in server.xml will be loaded by the common class loader.
	try {
	    J2EEServer.main(context);
	} catch (Exception e) {
	    throw new ServerLifecycleException(e);
	}

        // final connector class loader for the anonymous inner class
        final ClassLoader connCL = connectorClassLoader;

        // set the connector class loader as the thread context class loader
        java.security.AccessController.doPrivileged(
            new java.security.PrivilegedAction() {
                public Object run() {
                    Thread.currentThread().
                        setContextClassLoader(connCL);
                        return null;
                    }
                }
        );

    }

    /** 
     * Go through the list of built-in runtime services and instantiate them.
     */
    protected ArrayList instantiateRuntimeServices() throws ServerLifecycleException{

        ArrayList serviceList = new ArrayList();

        // Instantiate service objects 
        for (int i=0; i < servicesByName.length; i++) {
            try {
                String[] def = servicesByName[i];
                Class c  = Class.forName(def[1]);
                Object o = c.newInstance();

                serviceList.add(o);
            } catch (Exception ex) {
	        _logger.log(Level.SEVERE, "server.exception", ex);
		throw new ServerLifecycleException(ex.getMessage());
	    }

        }
        return serviceList;
    }

    /**
     * Server is starting up applications
     *
     * @param context ServerContext the server runtime context.
     *
     * @exception ServerLifecycleException if this subsystem detects a fatal 
     *  error that prevents this subsystem from being used
     *
     * Note: Currently, with the J2EEServer code handles initialization and
     * applications startup in one place <code> J2EEServer.run() </code>.
     *
     * XXX: This separation needs to be worked on to make it cleaner.
     */
    public void onStartup() throws ServerLifecycleException
    {
        // startup the built-in runtime services
        for (int i = 0; i < services.size(); i++) {
            Object service = services.get(i);

            if (service instanceof ServerLifecycle) {
	        try {
                    ((ServerLifecycle) service).onStartup(context);
		} catch (ServerLifecycleException e) {		  
		    _logger.log(Level.SEVERE, "service.notstarted", 
				new Object[] {service, e.toString()});
		    throw e;
		}
            }
        }
        // _REVISIT_: Mover AlertConfigurator to configure after onReady()
        AlertConfigurator.getAlertConfigurator( ).configure( );
    }

    /**
     * Server has complted loading the applications and is ready to 
     * serve requests.
     *
     * @exception ServerLifecycleException if this subsystem detects a fatal 
     *  error that prevents this subsystem from being used
     */
    public void onReady() throws ServerLifecycleException {

        try {
            // registers the resource manager as a listener for the 
            // resource deploy events
            AdminEventListenerRegistry.addResourceDeployEventListener(
                                              new ResourceManager(context));
            // registers listeners for JmsService and JmsHost changes.
            AdminEventListenerRegistry.addEventListener(JmsServiceEvent.eventType,
                                       new JmsServiceEventListener());
            AdminEventListenerRegistry.addEventListener(JmsHostEvent.eventType,
                                       new JmsHostEventListener());

            // Since onReady is called by a new thread, we have to again set
            // the common class loader as the thread context class loader
            java.security.AccessController.doPrivileged(
                new java.security.PrivilegedAction() {
                    public Object run() {
                        Thread.currentThread().setContextClassLoader
                            (commonClassLoader);
                        return null;
                    }
                }
            );            
            
            /*
             * com.sun.enterprise.server.startupHook is an internal
             * undocumented property that can be used by iplanet developers
             * to invoke some custom code just after server startup happens.
             * Currently it is used by the ORB to easily benchmark and profile
             * the local invocation path of remote interfaces in the EJB
             * without having to deal with servlets or client/server processes
             * 
             * If the property com.sun.enterprise.server.startupHook is
             * defined then "main" in that class is executed
             * Note: This should always be the last piece of code executed
             *       in this method
             */
            String startupHook = null;

            startupHook = System.getProperty(
                              "com.sun.enterprise.server.startupHook");
            if (startupHook != null) {
                try {
                    // Call the custom startup hook specified
                    Class hookClass = Class.forName(startupHook);
                    java.lang.reflect.Method hookMethod =
                        hookClass.getMethod("main",
                                    new Class[] { ServerContext.class });
                    hookMethod.invoke(null, new Object[] { context });
                }
                catch (Exception ex) {
                    if (Debug.enabled) {
                          _logger.log(Level.FINE, "server.exception", ex);
                    }
                }
            }

        } catch (Exception ee) {
            throw new ServerLifecycleException(ee);
        }


        // notify the built-in runtime services that we are ready
        for (int i = 0; i < services.size(); i++) {
            Object service = services.get(i);

            if (service instanceof ServerLifecycle) {
	        try {
                    ((ServerLifecycle) service).onReady(context);
		} catch (ServerLifecycleException e) {
		  _logger.log(Level.SEVERE, "service.notready", 
			      new Object[] {service, e.toString()});
		}
            }
        }
    }

    /**
     * Server is shutting down applications
     *
     * @exception ServerLifecycleException if this subsystem detects a fatal 
     *  error that prevents this subsystem from being used
     */
    public void onShutdown() throws ServerLifecycleException 
    {
	
        // shutdown the built-in runtime services (in the reverse order).
        for (int i = services.size(); i > 0; i--) {
            Object service = services.get(i-1);

	    _logger.log(Level.FINE,"service.shutdown",
					services.get(i-1));
	    try {
                if (service instanceof ServerLifecycle) {
                    ((ServerLifecycle) service).onShutdown();
		}
	    } catch (Exception e) {
		//do nothing since we want to continue shutting down other LFC
		//modules
                _logger.log(Level.WARNING, "server.exception", e);	     
	    }            
        }
    }

    /**
     * Server is terminating the subsystems and the runtime environment.
     * Gracefully terminate the active use of the public methods of this
     * subsystem.  This method should be the last one called on a given
     * instance of this subsystem.
     *
     * @exception ServerLifecycleException if this subsystem detects a fatal 
     *  error that prevents this subsystem from being used
     */
    public void onTermination() throws ServerLifecycleException 
    {
        //Kill all connections in the appserver's connection pools
	if (_logger.isLoggable( Level.FINE ) ) {
	    _logger.log(Level.FINE, "Killing all pools in the appserver");
	}
	try {
	    Switch.getSwitch().getPoolManager().killAllPools();
	} catch( Throwable t ) {
	    if (_logger.isLoggable( Level.FINE ) ) {
	        _logger.log(Level.FINE, "exception : " +t);
	    }
	}
 // terminate the built-in runtime services (in the reverse order)
        for (int i = services.size(); i > 0; i--) {
            Object service = services.get(i-1);

	    _logger.log(Level.FINE,"service.shutdown",
					services.get(i-1));
	    try {
                if (service instanceof ServerLifecycle) {
                    ((ServerLifecycle) service).onTermination();
                }
	    } catch (Exception e) {
		//do nothing since we want to continue shutting down other LFC
		//modules
                _logger.log(Level.WARNING, "server.exception", e);	
	    }
        }

        services = null;
    }

    /**
     * Get the server runtime context; returns a valid context only 
     * after initialization; a null otherwise.
     *
     * @return    a server runtime context or null if not initialized
     */
    public static ServerContext getServerContext() {
        return context;
    }

    // Called by PE Main
    protected void setServerContext(ServerContext context) {
        this.context = context;
    }

    private void printStartupInfo() {
        _logger.log(Level.INFO, "j2eerunner.printstartinfo", 	
                        new Object[] {
                           System.getProperty("java.vm.name"), 
                           System.getProperty("java.version"), 
                           System.getProperty("java.vm.vendor") });
    }
}
