/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.webservice;

import java.io.IOException;

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;

import javax.xml.soap.SOAPMessage;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.ws.handler.MessageContext.Scope;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import com.sun.xml.ws.spi.runtime.RuntimeEndpointInfo;
import com.sun.xml.ws.server.Tie;
import com.sun.xml.ws.spi.runtime.WSRtObjectFactory;
import com.sun.xml.ws.spi.runtime.WSConnection;
import com.sun.xml.ws.spi.runtime.WebServiceContext;

import com.sun.enterprise.webservice.monitoring.WebServiceEngineImpl;
import com.sun.enterprise.webservice.monitoring.EndpointImpl;
import com.sun.enterprise.webservice.monitoring.HttpRequestInfoImpl;
import com.sun.enterprise.webservice.monitoring.HttpResponseInfoImpl;
import com.sun.enterprise.webservice.monitoring.ThreadLocalInfo;

/**
 * Implementation of the Ejb Message Dispatcher for EJB3 endpoints.
 *
 * @author Jerome Dochez
 */
public class Ejb3MessageDispatcher implements EjbMessageDispatcher {
    
    private static Logger logger = 
        LogDomains.getLogger(LogDomains.EJB_LOGGER);    
    
    private static WsUtil wsUtil = new WsUtil();
    
    // all jaxrpc invocations of all endpoints go through the same tie
    private static Tie tie = new Tie();
    private static WSRtObjectFactory rpcFactory = WSRtObjectFactory.newInstance();
    private WebServiceEngineImpl wsEngine;
    
    /** Creates a new instance of Ejb3MessageDispatcher */
    public Ejb3MessageDispatcher() {
        wsEngine = WebServiceEngineImpl.getInstance();
    }
    
    public void invoke(HttpServletRequest req, 
                       HttpServletResponse resp,
                       EjbRuntimeEndpointInfo endpointInfo) {
        
        
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "JAXWS WebServiceDispatcher " + req.getMethod() + 
                   " entering for " + req.getRequestURI() + " and query string " + req.getQueryString());
        }
       
        String method = req.getMethod();
        
        try {
            if( method.equals("POST") ) {
                
                handlePost(req, resp, endpointInfo);
                
            } else if( method.equals("GET") ) {
                
                handleGet(req, resp, endpointInfo);
                
            } else {
                
                String errorMessage =  "Unsupported method request = [" 
                    + method + "] for endpoint " + 
                    endpointInfo.getEndpoint().getEndpointName() + " at " + 
                    endpointInfo.getEndpointAddressUri();
                logger.warning(errorMessage);
                wsUtil.writeInvalidMethodType(resp, errorMessage);
            }
        } catch(Exception e) {
            logger.log(Level.WARNING, "ejb endpoint exception", e);
        }
    } 
    
    private void handlePost(HttpServletRequest req,
                            HttpServletResponse resp,
                            EjbRuntimeEndpointInfo endpointInfo)
        throws IOException    {
        
        EndpointImpl endpoint = null;               
        String messageID = null;
        
        try {
            
            try {
                
                RuntimeEndpointInfo jaxwsInfo = endpointInfo.prepareInvocation(true);
                if (jaxwsInfo != null) {
                    
                    // get the endpoint info
                    endpoint = (EndpointImpl) endpointInfo.getEndpoint().getExtraAttribute(EndpointImpl.NAME);
                    
                    if (endpoint!=null) {
                        // first global notification
                        if (wsEngine.hasGlobalMessageListener()) {
                            messageID = wsEngine.preProcessRequest(endpoint);
                        }
                    } else {
                        logger.fine("Missing internal monitoring info to trace " + req.getRequestURI());
                    }
                    

                    
                    WSConnection connection = rpcFactory.createWSConnection(req, resp);                    
                    WebServiceContext wsCtxt = jaxwsInfo.getWebServiceContext();
                    MessageContext msgCtxt = wsCtxt.getMessageContext();
                    msgCtxt.put(MessageContext.SERVLET_REQUEST, req);
                    msgCtxt.setScope(MessageContext.SERVLET_REQUEST, Scope.APPLICATION);
                    msgCtxt.put(MessageContext.SERVLET_RESPONSE, resp);
                    msgCtxt.setScope(MessageContext.SERVLET_RESPONSE, Scope.APPLICATION);
                    msgCtxt.put(MessageContext.HTTP_REQUEST_METHOD, req.getMethod());
                    msgCtxt.setScope(MessageContext.HTTP_REQUEST_METHOD, Scope.APPLICATION);
                    if (req.getQueryString() != null) {
                        msgCtxt.put(MessageContext.QUERY_STRING, req.getQueryString());
                        msgCtxt.setScope(MessageContext.QUERY_STRING, Scope.APPLICATION);
                    }
                    if (req.getPathInfo() != null) {
                        msgCtxt.put(MessageContext.PATH_INFO, req.getPathInfo());
                        msgCtxt.setScope(MessageContext.PATH_INFO, Scope.APPLICATION);
                    }
                    msgCtxt.put(MessageContext.HTTP_REQUEST_HEADERS, connection.getHeaders());
                    msgCtxt.setScope(MessageContext.HTTP_REQUEST_HEADERS, Scope.APPLICATION);

                    // our container related information
                    msgCtxt.put(EndpointImpl.MESSAGE_ID, messageID);
                    tie.handle(connection, jaxwsInfo);
                    
                }
                                
            } finally {
                
                // Always call release, even if an error happened
                // during getImplementor(), since some of the
                // preInvoke steps might have occurred.  It's ok
                // if implementor is null.
                endpointInfo.releaseImplementor();
                
            }    
             
        } catch (Throwable e) {
            
            String errorMessage = "invocation error on ejb endpoint " +
            endpointInfo.getEndpoint().getEndpointName() + " at " +
            endpointInfo.getEndpointAddressUri() + " : " + e.getMessage();
            logger.log(Level.WARNING, errorMessage, e);

            String binding = endpointInfo.getEndpoint().getProtocolBinding();
            WsUtil.raiseException(resp, binding, errorMessage);
        }
        
        // final tracing notification
        if (messageID!=null) {
            HttpResponseInfoImpl response = new HttpResponseInfoImpl(resp);
            wsEngine.postProcessResponse(messageID, response);
        }        
    }    
    
    private void handleGet(HttpServletRequest req, 
                           HttpServletResponse resp,
                           EjbRuntimeEndpointInfo endpointInfo)
        throws IOException
    {
       
        wsUtil.handleGet(req, resp, endpointInfo.getEndpoint());           

    }      
    
}
