/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.webservice.spi;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Level;

import javax.xml.namespace.QName;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.bind.JAXBContext;

import com.sun.enterprise.deployment.ServiceReferenceDescriptor;
import com.sun.enterprise.webservice.ServiceRefDescUtil;
import com.sun.enterprise.webservice.WsUtil;

public class ServiceDelegateImpl extends javax.xml.ws.spi.ServiceDelegate {

   private javax.xml.ws.spi.ServiceDelegate delegate;
   private ArrayList<InvInterfaceCreationListener> listeners = new ArrayList();
   private ServiceReferenceDescriptor serviceRef;
   
   public ServiceDelegateImpl(javax.xml.ws.spi.ServiceDelegate delegate) {
       this.delegate = delegate;
       serviceRef = ServiceRefDescUtil.getReference();
   }
   
  /** The getPort method returns a stub. A service client
   *  uses this stub to invoke operations on the target 
   *  service endpoint. The <code>serviceEndpointInterface</code> 
   *  specifies the service endpoint interface that is supported by
   *  the created dynamic proxy or stub instance. 
   *
   *  @param portName  Qualified name of the service endpoint in 
   *                   the WSDL service description
   *  @param serviceEndpointInterface Service endpoint interface 
   *                   supported by the dynamic proxy or stub
   *                   instance
   *  @return Object Proxy instance that 
   *                 supports the specified service endpoint 
   *                 interface
   *  @throws WebServiceException This exception is thrown in the
   *                   following cases:
   *                   <UL>
   *                   <LI>If there is an error in creation of 
   *                       the proxy
   *                   <LI>If there is any missing WSDL metadata
   *                       as required by this method
   *                   <LI>Optionally, if an illegal 
   *                       <code>serviceEndpointInterface</code>
   *                       or <code>portName</code> is specified
   *                   </UL>  
   *  @see java.lang.reflect.Proxy
   *  @see java.lang.reflect.InvocationHandler
  **/			
  @Override  
  public <T> T getPort(QName portName,
	               Class<T> serviceEndpointInterface) {
      T port = delegate.getPort(portName, serviceEndpointInterface);
      portCreated(port, serviceEndpointInterface);
      return port;
  }

  /** The getPort method returns a stub. The parameter 
   *  <code>serviceEndpointInterface</code> specifies the service 
   *  endpoint interface that is supported by the returned proxy.
   *  In the implementation of this method, the JAX-WS 
   *  runtime system takes the responsibility of selecting a protocol
   *  binding (and a port) and configuring the proxy accordingly. 
   *  The returned proxy should not be reconfigured by the client.
   *
   *  @param serviceEndpointInterface Service endpoint interface
   *  @return Object instance that supports the 
   *                   specified service endpoint interface
   *  @throws WebServiceException
   *                   <UL>
   *                   <LI>If there is an error during creation
   *                       of the proxy
   *                   <LI>If there is any missing WSDL metadata
   *                       as required by this method
   *                   <LI>Optionally, if an illegal 
   *                       <code>serviceEndpointInterface</code>
   *                       is specified
   *                   </UL>  
  **/
  public <T> T getPort(Class<T> serviceEndpointInterface) {
    T port = delegate.getPort(serviceEndpointInterface);
    portCreated(port, serviceEndpointInterface);
    return port;
  }

  /** Creates a new port for the service. Ports created in this way contain
   *  no WSDL port type information and can only be used for creating 
   *  <code>Dispatch</code>instances.
   *
   *  @param portName  Qualified name for the target service endpoint
   *  @param bindingId A URI identifier of a binding.
   *  @param endpointAddress Address of the target service endpoint as a URI
   *  @throws WebServiceException If any error in the creation of
   *  the port
   *
   *  @see javax.xml.ws.soap.SOAPBinding#SOAP11HTTP_BINDING
   *  @see javax.xml.ws.soap.SOAPBinding#SOAP12HTTP_BINDING
   *  @see javax.xml.ws.soap.SOAPBinding#SOAP11HTTP_MTOM_BINDING
   *  @see javax.xml.ws.soap.SOAPBinding#SOAP12HTTP_MTOM_BINDING
   *  @see javax.xml.ws.http.HTTPBinding#HTTP_BINDING
   **/
  public void addPort(QName portName, String bindingId,
      String endpointAddress) {
      delegate.addPort(portName, bindingId, endpointAddress);
  }

  /** Creates a <code>Dispatch</code> instance for use with objects of
   *  the users choosing.
   *
   *  @param portName  Qualified name for the target service endpoint
   *  @param type The class of object used to messages or message
   *  payloads. Implementations are required to support
   *  javax.xml.transform.Source and javax.xml.soap.SOAPMessage.
   *  @param mode Controls whether the created dispatch instance is message
   *  or payload oriented, i.e. whether the user will work with complete
   *  protocol messages or message payloads. E.g. when using the SOAP
   *  protocol, this parameter controls whether the user will work with
   *  SOAP messages or the contents of a SOAP body. Mode must be MESSAGE
   *  when type is SOAPMessage.
   *
   *  @return Dispatch instance
   *  @throws WebServiceException If any error in the creation of
   *                   the <code>Dispatch</code> object
   *  @see javax.xml.transform.Source
   *  @see javax.xml.soap.SOAPMessage
   **/
  public <T> Dispatch<T> createDispatch(QName portName, Class<T> type, Service.Mode mode) {
      Dispatch<T> dispatch = delegate.createDispatch(portName, type, mode);
      dispatchCreated(dispatch, type);
      return dispatch;
  }

  /** Creates a <code>Dispatch</code> instance for use with JAXB
   *  generated objects.
   *
   *  @param portName  Qualified name for the target service endpoint
   *  @param context The JAXB context used to marshall and unmarshall
   *  messages or message payloads.
   *  @param mode Controls whether the created dispatch instance is message
   *  or payload oriented, i.e. whether the user will work with complete
   *  protocol messages or message payloads. E.g. when using the SOAP
   *  protocol, this parameter controls whether the user will work with
   *  SOAP messages or the contents of a SOAP body.
   *
   *  @return Dispatch instance
   *  @throws ServiceException If any error in the creation of
   *                   the <code>Dispatch</code> object
   *
   *  @see javax.xml.bind.JAXBContext
   **/
  public Dispatch<Object> createDispatch(QName portName, 
              JAXBContext context, Service.Mode mode) {
      
      Dispatch<Object> dispatch = delegate.createDispatch(portName, context, mode);
      dispatchCreated(dispatch, Object.class);
      return dispatch;
      
  }


  /** Gets the name of this service.
   *  @return Qualified name of this service
  **/
  public QName getServiceName() {
      return delegate.getServiceName();
  }

  /** Returns an <code>Iterator</code> for the list of 
   *  <code>QName</code>s of service endpoints grouped by this
   *  service
   *
   *  @return Returns <code>java.util.Iterator</code> with elements
   *          of type <code>javax.xml.namespace.QName</code>
   *  @throws WebServiceException If this Service class does not
   *          have access to the required WSDL metadata  
  **/		
  public Iterator<javax.xml.namespace.QName> getPorts() {
      return delegate.getPorts();
  }

  /** Gets the location of the WSDL document for this Service.
   *
   *  @return URL for the location of the WSDL document for 
   *          this service
  **/
  public java.net.URL getWSDLDocumentLocation() {
      return delegate.getWSDLDocumentLocation();
  }

  /** 
   * Returns the configured handler resolver.
   *
   *  @return HandlerResolver The <code>HandlerResolver</code> being
   *          used by this <code>Service</code> instance, or <code>null</code>
   *          if there isn't one.
  **/
  public HandlerResolver getHandlerResolver() {
      return delegate.getHandlerResolver();
  }

  /**
   *  Sets the <code>HandlerResolver</code> for this <code>Service</code>
   *  instance.
   *  <p>
   *  The handler resolver, if present, will be called once for each
   *  proxy or dispatch instance that is created, and the handler chain
   *  returned by the resolver will be set on the instance.
   *
   *  @param handlerResolver The <code>HandlerResolver</code> to use
   *         for all subsequently created proxy/dispatch objects.
   *
   *  @see javax.xml.ws.handler.HandlerResolver
  **/
  public void setHandlerResolver(HandlerResolver handlerResolver) {
      delegate.setHandlerResolver(handlerResolver);
  }

  /**
   * Returns the executor for this <code>Service</code>instance.
   *
   * The executor is used for all asynchronous invocations that
   * require callbacks.
   *
   * @return The <code>java.util.concurrent.Executor</code> to be
   *         used to invoke a callback.
   * 
   * @see java.util.concurrent.Executor
   **/
   public java.util.concurrent.Executor getExecutor() {
       return delegate.getExecutor();
   }
  
  /**
   * Sets the executor for this <code>Service</code> instance.
   *
   * The executor is used for all asynchronous invocations that
   * require callbacks.
   *
   * @param executor The <code>java.util.concurrent.Executor</code>
   *        to be used to invoke a callback.
   *
   * @throws SecurityException If the instance does not support
   *         setting an executor for security reasons (e.g. the
   *         necessary permissions are missing).
   * 
   * @see java.util.concurrent.Executor
   **/
   public void setExecutor(java.util.concurrent.Executor executor) {
       delegate.setExecutor(executor);
   }
       
  
 
  /**
   * notifies our listener of a port creation 
   */
  private <T> void portCreated(T port, Class<T> serviceEndpointInterface) {
      for (InvInterfaceCreationListener listener : listeners) {
          try {
            listener.portCreated(port, serviceEndpointInterface);
          } catch(Throwable t) {
                 WsUtil.getDefaultLogger().severe("Exception " 
                         + t.getMessage() + " in portCreationListener : " + listener);
                 WsUtil.getDefaultLogger().log(Level.FINE, t.getMessage(), t);              
          }
      }
  }
  
  /**
   * Add a new listener for port creation
   * @param the listener
   */
  public void addListener(InvInterfaceCreationListener listener) {
    listeners.add(listener);
  }
  
  /**
   * notifies our listeners of a Dispatch creation
   */
  private <T> void dispatchCreated(Dispatch<T> dispatch, Class<T> serviceEndpointInterface) {
      for (InvInterfaceCreationListener listener : listeners) {
          try {
            listener.dispatchCreated(dispatch, serviceEndpointInterface);
          } catch(Throwable t) {
                 WsUtil.getDefaultLogger().severe("Exception " 
                         + t.getMessage() + " in dispatchCreationListener : " + listener);
                 WsUtil.getDefaultLogger().log(Level.FINE, t.getMessage(), t);              
          }
      }
  } 
  
  /**
   * @return the service reference descriptor used to describe this 
   * Service instance
   */
  public ServiceReferenceDescriptor getServiceReference() {
      return serviceRef;
  }
            
}
        