/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
/**
 * This interface defines the methods an instance of a <code>ReadTask</code> 
 * will invoke. The <code>ReadTask</code> will use an implementation of this 
 * interface tp predict if the NIO code>ByteBuffer</code> has been fully 
 * read and can be processed.
 *
 * <code>ReadTask</code> will invoke the method in the following order:
 *
 * (a) allocate(...)
 * (b) preParse(...)
 * (c) parse(...)
 * (d) contentLength() AND headerLength();
 * (d) postParse(...)
 * 
 * The algorithm will stop once (c) return <code>true</code>
 * 
 * @author Jean-Francois Arcand
 */
public interface StreamAlgorithm{
    
    
    /**
     * Return the stream content-length. If the content-length wasn't parsed,
     * return -1.
     */
    public int contentLength();
    
    
    /**
     * Return the stream header length. The header length is the length between
     * the start of the stream and the first occurance of character '\r\n' .
     */
    public int headerLength();
    
    
    /**
     * Allocate a <code>ByteBuffer</code>
     * @param useDirect allocate a direct <code>ByteBuffer</code>.
     * @param useView allocate a view <code>ByteBuffer</code>.
     * @return a new <code>ByteBuffer</code>
     */
    public ByteBuffer allocate(boolean useDirect, boolean useView);
    
    
    /**
     * Before parsing the bytes, initialize and prepare the algorithm.
     * @param byteBuffer the <code>ByteBuffer</code> used by this algorithm
     * @return <code>ByteBuffer</code> used by this algorithm
     */
    public ByteBuffer preParse(ByteBuffer byteBuffer);
    
    
    /**
     * Parse the <code>ByteBuffer</code> and try to determine if the bytes
     * stream has been fully read from the <code>SocketChannel</code>.
     * @paran byteBuffer the bytes read.
     * @return true if the algorithm determines the end of the stream.
     */
    public boolean parse(ByteBuffer byteBuffer);
    
    
    /**
     * After parsing the bytes, post process the <code>ByteBuffer</code> 
     * @param byteBuffer the <code>ByteBuffer</code> used by this algorithm
     * @return <code>ByteBuffer</code> used by this algorithm
     */
    public ByteBuffer postParse(ByteBuffer byteBuffer);  
    
    
    /**
     * Recycle the algorithm.
     */
    public void recycle();

    
    /**
     * The <code>Handler</code> associated with this algorithm.
     */
    public Handler getHandler();

    
    /**
     * Set the <code>SocketChannel</code> used by this algorithm
     */
    public void setSocketChannel(SocketChannel socketChannel);
    
    
    /**
     * Set the <code>port</code> this algorithm is used.
     */
    public void setPort(int port);
    
    
    /**
     * Return the port
     */
    public int getPort();
}

