/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly.async;

import com.sun.enterprise.web.connector.grizzly.AsyncExecutor;
import com.sun.enterprise.web.connector.grizzly.AsyncFilter;
import com.sun.enterprise.web.connector.grizzly.AsyncHandler;
import com.sun.enterprise.web.connector.grizzly.ProcessorTask;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.logging.Level;

/**
 * Default implementation of the <code>AsyncExecutor</code>. This class will
 * execute a <code>ProcessorTask</code> asynchronously, by interrupting the 
 * process based on the logic defined in its associated <code>AsyncFilter</code>
 * If no <code>AsyncFilter</code> are defined, the <code>ProcessorTask</code>
 * will not be interrupted and executed synchronously.
 *
 * @author Jeanfrancois Arcand
 */
public class DefaultAsyncExecutor implements AsyncExecutor{

    private final static String ASYNC_FILTER = 
            "com.sun.enterprise.web.connector.grizzly.asyncFilters";

    
    /**
     * The <code>AsyncProcessorTask</code> used to wrap the 
     * <code>ProcessorTask</code>
     */
    private AsyncProcessorTask asyncProcessorTask;
       
    
    /**
     * The associated <code>ProcessorTask</code>
     */
    private ProcessorTask processorTask;
    
    
    /**
     * The <code>AsyncFilter</code> to execute asynchronous operations on 
     * a <code>ProcessorTask</code>.
     */
    private static AsyncFilter[] sharedAsyncFilters = null;;
    
    
    /**
     * The <code>AsyncFilter</code> to execute asynchronous operations on 
     * a <code>ProcessorTask</code>.
     */
    private ArrayList<AsyncFilter> asyncFilters = 
            new ArrayList<AsyncFilter>();
    
    
    /**
     * Do we need to invoke filters?
     */
    private boolean invokeFilter = true;

    
    /**
     * Loads filters implementation.
     */
    static {
        loadFilters();
    }
    
    
    public DefaultAsyncExecutor(){
        init();
    }
    
    
    private void init(){
        if (sharedAsyncFilters != null){
            for (AsyncFilter o : sharedAsyncFilters){
                asyncFilters.add(o);
            }
        }
        
    }
    
    // ------------------------------------------------Asynchrounous Execution --/
    
    /**
     * Pre-execute a <code>ProcessorTask</code> by parsing the request 
     * line.
     */
    public boolean preExecute() throws Exception{
        processorTask = asyncProcessorTask.getProcessorTask();        
        processorTask.preProcess();
        processorTask.parseRequest();
        return true;
    }
    
    
    /**
     * Interrupt the <code>ProcessorTask</code> if <code>AsyncFilter</code>
     * has been defined.
     * @return true if the execution can continue, false if delayed.
     */
    public boolean interrupt() throws Exception{
        if ( asyncFilters == null || asyncFilters.size() == 0 ) {
            processorTask.invokeAdapter();
            return true;
        } else {
            processorTask.getAsyncHandler()
                .addToInterruptedQueue(asyncProcessorTask);
                                         
            return invokeFilters(); 
        }
    }

    
    /**
     * Invoke the <code>AsyncFilter</code>
     */
    private boolean invokeFilters(){
        boolean continueExec = true;
        for (AsyncFilter asf: asyncFilters){
            continueExec = asf.doFilter(this);
            if ( !continueExec ){
                break;
            }
        }
        return continueExec;
    }
    
    
    /**
     * Post-execute the <code>ProcessorTask</code> by preparing the response,
     * flushing the response and then close or keep-alive the connection.
     */
    public boolean postExecute() throws Exception{
        processorTask.postResponse();
        processorTask.postProcess();        
        processorTask.terminateProcess();
        
        // De-reference so under stress we don't have a simili leak.
        processorTask = null;
        return false;
    }

      
    /**
     * Set the <code>AsyncProcessorTask</code>.
     */
    public void setAsyncProcessorTask(AsyncProcessorTask asyncProcessorTask){
        this.asyncProcessorTask = asyncProcessorTask;
    }
    
    
    /**
     * Return <code>AsyncProcessorTask</code>.
     */
    public AsyncProcessorTask getAsyncProcessorTask(){
        return asyncProcessorTask;
    }
    
    
    // --------------------------------------------------------- Util --------//  
    
    
    /**
     * Load the list of <code>AsynchFilter</code>.
     */
    protected static void loadFilters(){      
        if ( System.getProperty(ASYNC_FILTER) != null){
            StringTokenizer st = new StringTokenizer(
                    System.getProperty(ASYNC_FILTER),",");
            
            sharedAsyncFilters = new AsyncFilter[st.countTokens()];    
            int i = 0;
            while (st.hasMoreTokens()){
                AsyncFilter asf = loadInstance(st.nextToken());
                if ( asf != null )
                    sharedAsyncFilters[i++] = asf;                
            } 
        }   
    }    
    
    
    /**
     * Instanciate a class based on a property.
     */
    private static AsyncFilter loadInstance(String property){        
        Class className = null;                               
        try{                              
            className = Class.forName(property);
            return (AsyncFilter)className.newInstance();
        } catch (ClassNotFoundException ex){
            SelectorThread.logger().log(Level.WARNING,ex.getMessage(),ex);
        } catch (InstantiationException ex){
            SelectorThread.logger().log(Level.WARNING,ex.getMessage(),ex);            
        } catch (IllegalAccessException ex){
            SelectorThread.logger().log(Level.WARNING,ex.getMessage(),ex);            
        }
        return null;
    }   

    
    /**
     * Add an <code>AsyncFilter</code>
     */
    public void addAsyncFilter(AsyncFilter asyncFilter) {
        asyncFilters.add(asyncFilter);
    }

    
    /**
     * Remove an <code>AsyncFilter</code>
     */
    public boolean removeAsyncFilter(AsyncFilter asyncFilter) {
        return asyncFilters.remove(asyncFilter);
    }
}
