/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.tools.verifier.appclient;

import java.io.File;
import java.io.FileInputStream;
import java.util.Iterator;
import java.util.Set;

import com.sun.enterprise.deployment.ApplicationClientDescriptor;
import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.ServiceReferenceDescriptor;
import com.sun.enterprise.deployment.io.AppClientDeploymentDescriptorFile;
import com.sun.enterprise.tools.verifier.CheckMgr;
import com.sun.enterprise.tools.verifier.FrameworkContext;
import com.sun.enterprise.tools.verifier.JarCheck;
import com.sun.enterprise.tools.verifier.Result;
import com.sun.enterprise.tools.verifier.tests.ComponentNameConstructor;
import com.sun.enterprise.tools.verifier.tests.dd.ParseDD;
import com.sun.enterprise.tools.verifier.wsclient.WebServiceClientCheckMgrImpl;

/**
 * Application Client harness
 */
public class AppClientCheckMgrImpl extends CheckMgr implements JarCheck {

    /**
     * name of the file containing the list of tests for the application client
     * architecture
     */
    private static final String testsListFileName = "TestNamesAppClient.xml"; // NOI18N
    private static final String sunONETestsListFileName = getSunPrefix()
            .concat(testsListFileName);

    public AppClientCheckMgrImpl(FrameworkContext frameworkContext) {
        this.frameworkContext = frameworkContext;
    }

    /**
     * Check method introduced for WebServices integration
     *
     * @param descriptor appclient descriptor
     */
    public void check(Descriptor descriptor) throws Exception {
        // run persistence tests first.
        checkPersistenceUnits(ApplicationClientDescriptor.class.cast(descriptor));
        //An ApplicationClient can have WebService References
        checkWebServicesClient(descriptor);

        if (frameworkContext.isPartition() &&
                !frameworkContext.isAppClient()) 
            return;
        // run the ParseDD test
        if (getSchemaVersion(descriptor).compareTo("1.4") < 0) { // NOI18N
            AppClientDeploymentDescriptorFile ddf = new AppClientDeploymentDescriptorFile();
            File file = new File(getAbstractArchiveUri(descriptor),
                    ddf.getDeploymentDescriptorPath());
            FileInputStream is = new FileInputStream(file);
            try {
                if (is != null) {
                    Result result = new ParseDD().validateAppClientDescriptor(is);
                    result.setComponentName(getArchiveUri(descriptor));
                    setModuleName(result);
                    frameworkContext.getResultManager().add(result);
                }
            } finally {
                try {
                    if(is!=null)
                        is.close();
                } catch(Exception e) {}
            }
        }

        super.check(descriptor);
    }

    /**
     * return the configuration file name for the list of tests pertinent to the
     * connector architecture
     *
     * @return <code>String</code> filename containing the list of tests
     */
    protected String getTestsListFileName() {
        return testsListFileName;
    }

    /**
     * return the configuration file name for the list of tests pertinent to the
     * application client architecture
     *
     * @return <code>String</code> filename containing the list of tests
     */
    protected String getSunONETestsListFileName() {
        return sunONETestsListFileName;
    }

    protected String getSchemaVersion(Descriptor descriptor) {
        return ((RootDeploymentDescriptor) descriptor).getSpecVersion();
    }

    protected void setModuleName(Result r) {
        r.setModuleName(Result.APPCLIENT);
    }

    protected void checkWebServicesClient(Descriptor descriptor)
            throws Exception {
        if (frameworkContext.isPartition() &&
                !frameworkContext.isWebServicesClient()) 
            return;
        WebServiceClientCheckMgrImpl webServiceClientCheckMgr = 
                                new WebServiceClientCheckMgrImpl(frameworkContext);
        ApplicationClientDescriptor desc = (ApplicationClientDescriptor) descriptor;
        if (desc.hasWebServiceClients()) {
            Set serviceRefDescriptors = desc.getServiceReferenceDescriptors();
            Iterator it = serviceRefDescriptors.iterator();
            while (it.hasNext()) {
                webServiceClientCheckMgr.setVerifierContext(context);
                webServiceClientCheckMgr.check(
                        (ServiceReferenceDescriptor) it.next());
            }
        } else // set not applicable for all tests in WebServices for this Appclient Bundle 
            webServiceClientCheckMgr.setVerifierContext(context);
    }

    protected ComponentNameConstructor getComponentNameConstructor(
            Descriptor descriptor) {
        return new ComponentNameConstructor((ApplicationClientDescriptor)descriptor);
    }

}
