/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.jdo.api.persistence.enhancer.classfile;

import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

/**
 * A list of attributes within a class file.
 * These lists occur in several places within a class file
 *    - at class level
 *    - at method level
 *    - at field level
 *    - at attribute level
 */

public class AttributeVector {

  /* Vector of ClassAttribute */
  private ClassAttribute attributes[] = null;

  /**
   * Returns the i'th attribute in the array
   */
  private ClassAttribute attrAt(int i) {
    return attributes[i];
  }

  /**
   * Construct an empty AttributeVector
   */
  public AttributeVector() { }

  /**
   * Add an element to the vector
   */
  public void addElement(ClassAttribute attr) {
    if (attributes == null)
      attributes = new ClassAttribute[1];
    else {
      ClassAttribute newAttributes[] = new ClassAttribute[attributes.length+1];
      System.arraycopy(attributes, 0, newAttributes, 0, attributes.length);
      attributes = newAttributes;
    }
    attributes[attributes.length-1] = attr;
  }

  public Enumeration elements() {
    class AttributeVectorEnumeration implements Enumeration {
      private ClassAttribute[] attributes;
      private int current = 0;

      AttributeVectorEnumeration(ClassAttribute attrs[]) {
	attributes = attrs;
      }

      public boolean hasMoreElements() {
	return attributes != null && current < attributes.length;
      }
      public Object nextElement() {
	if (!hasMoreElements())
	  throw new NoSuchElementException();
	return attributes[current++];
      }
    }

    return new AttributeVectorEnumeration(attributes);
  }

  /**
   * Look for an attribute of a specific name
   */
  public ClassAttribute findAttribute(String attrName) {
    Enumeration e = elements();
    while (e.hasMoreElements()) {
      ClassAttribute attr = (ClassAttribute) e.nextElement();
      if (attr.attrName().asString().equals(attrName))
	return attr;
    }
    return null;
  }

  /**
   * General attribute reader
   */
  static AttributeVector readAttributes(
	DataInputStream data, ConstantPool constantPool)
	throws IOException {
    AttributeVector attribs = new AttributeVector();
    int n_attrs = data.readUnsignedShort();
    while (n_attrs-- > 0) {
      attribs.addElement(ClassAttribute.read(data, constantPool));
    }
    return attribs;
  }

  /**
   * ClassMethod attribute reader
   */
  static AttributeVector readAttributes(
	DataInputStream data, CodeEnv codeEnv)
	throws IOException {
    AttributeVector attribs = new AttributeVector();
    int n_attrs = data.readUnsignedShort();
    while (n_attrs-- > 0) {
      attribs.addElement(ClassAttribute.read(data, codeEnv));
    }
    return attribs;
  }

  /**
   * Write the attributes to the output stream
   */
  void write(DataOutputStream out) throws IOException {
    if (attributes == null) {
      out.writeShort(0);
    } else {
      out.writeShort(attributes.length);
      for (int i=0; i<attributes.length; i++)
	attributes[i].write(out);
    }
  }

  /**
   * Print a description of the attributes
   */
  void print(PrintStream out, int indent) {
    if (attributes != null) {
      for (int i=0; i<attributes.length; i++)
	attributes[i].print(out, indent);
    }
  }

  /**
   * Print a brief summary of the attributes
   */
  void summarize() {
    System.out.println((attributes == null ? 0 : attributes.length) +
		       " attributes");//NOI18N
  }

}

