/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * Timer.java
 *
 * Create on March 3, 2000
 */

package com.sun.jdo.spi.persistence.support.sqlstore.utility;

/**
 * This class represents an asynchronous interval timer.
 * The timer waits for a specified interval, then posts a
 * tick event, called a <code>tick</code>, to a 
 * <code>TickEventListener</code> object that you create. 
 * Once the timer is started, it continues
 * producing tick events at each interval until it is stopped.
 * <p>
 * To use a timer, you define a class that implements the
 * <code>TickEventListener</code> interface. In the <code>tick</code>
 * method, you write the code that
 * that performs an action when the timer interval expires. 
 * <p>
 * You then instantiate an object of your class
 * and use it to start the timer and set the interval. For example:
 * <pre>
 * public class TimerUser implements TickEventListener
 * {
 *    public Timer myTimer;
 * 
 *    public void startTimer(int interval, int active)
 *    {
 *      myTimer = 
 *       new Timer(this, interval, (active == 1) ? true : false);
 *    }
 * 
 *    public void stopTimer()
 *    {
 *      if (myTimer != null)
 *         myTimer.setState(false);
 *    }
 * 
 *    public void tick(TickEvent evt)
 *    {
 *       System.out.println(
 *            "Tick count is " + evt.timer.getTickCount());
 *    }
 * }
 * </pre>
 * @see com.forte.util.TickEventListener
 * @see com.forte.util.TickEvent
 */
public class Timer extends Object implements Runnable
{
	/**
	* The thread running this object.
	*/
	private Thread					runner;
	
	/**
	* True when the alarm is armed.
	*/
	private boolean						active;

	/**
	* True when used for single tick
	*/
	private boolean						tickOnce;

	/**
	* The timer interval in milliseconds. 
	*/
	private long						interval;

	/**
	* The number timer intervals expired.
	*/
	private long						ticks;

	/**
	* The owning object that will get the callback. 
	*/
	private TickEventListener			owner;

	/**
	* True when the timer is terminating.
	*/
	private boolean						isTerminated;

    /**
     * @ForteInternal
     * The number of times ForteThreads have suspended to wait for an
	 * installed timer to time out. This will be used as value for
	 * TimeOutSuspends instrument of the TaskMgr agent.
     */
    public static int					suspends			= 0;
	
    /**
     * @ForteInternal
     * The number of times ForteThreads have resumed after waiting for an
	 * installed timer to time out. This will be used as value for
	 * TimeOutResumes instrument of the TaskMgr agent.
     */
    public static int					resumes				= 0;


	/**
	 * Constructs a timer object that is ready to be run.
	 * .
	 * @param active causes the timer to become active (<code>true</code>)
	 * or inactive (<code>false</code>). 
	 * @param interval the timeout in milliseconds (1000 = 1 second). 
	 * @param obj a listener for a TickEvent.
	 * @param tickOnce indicates that the TickEvent will only be
	 * delivered once (the default is repeated delivery on each tick).
	 */
	public Timer(TickEventListener obj, long interval, boolean active)
	{
		this(obj, interval, active, false, null);
	}

	public Timer(TickEventListener obj, long interval, boolean active,
				 boolean tickOnce, String name)
	{
		Thread	thd;
		this.owner = obj;
		this.interval = interval;
		this.active = active;
		this.tickOnce = tickOnce;
		this.isTerminated = false;
		thd = new Thread(this);
		thd.setDaemon(true);
		if (name != null)
		{
			thd.setName(name);
		}
		else
		{
			thd.setName("TimerThread" + System.identityHashCode(this)); // NOI18N
		}
		thd.start();
	}

	/**
	 * Sets the interval in milliseconds between ticks of the timer.
	 * 
	 * @param interval the new interval in milliseconds. 
	 */
	public void setInterval(long interval)
	{
		synchronized(this)
		{
			if (this.interval == interval)
				return;

			this.interval = interval;

			if (this.active)
				this.runner.interrupt();
		}
	}

	/**
	 * Gets the current interval between ticks (in milliseconds). 
	 * 
	 * @return The current interval between ticks (in milliseconds). 
	 */
	//
	// CHANGES
	//     17-jun-1997
	//         Created (jak)
	//
	public long getInterval()
	{
		return this.interval;
	}

    /** 
     * Gets the number of ticks that have accumulated.
    * @return The number of accumulated ticks. 
     */
    //
    // CHANGES
    //     17-jun-1997
    //         Created (jak)
    //
    public long getTickCount()
    {
		return this.ticks;
    }

	/**
	 * Resets the number of ticks that have occurred to
	 * the specified number. 
	 * 
	 * @param count the new tick count. 
	 */
	//
	// CHANGES
	//     17-jun-1997
	//         Created (jak)
	//
	public void setTickCount(long count)
	{
		this.ticks = count;	
	}

	/**
	 * Turns the timer on or off, enabling or disabling the 
	 * delivery of tick events to the tick event listener.
	 * 
	 * @param state causes the timer to become active (<code>true</code>)
	 * or inactive (<code>false</code>). 
	 */
	//
	// CHANGES
	//     17-jun-1997
	//         Created (jak)
	//
	public void setState(boolean state)
	{
		synchronized(this)
		{	
			if (this.active == state)
				return;

			if (this.active)
				this.runner.interrupt();
			else
				this.notify();

			this.active = state;	
		}
	}

	/**
	 * Terminate the timer (disable it).  
	 */
	//
	// CHANGES
	//     31-aug-1999
	//         Created (ncg)
	//
	public void terminate()
	{
		synchronized(this)
		{	
			this.isTerminated = true;
			if (this.runner != null)
				this.setState(false);
		}
	}

	/**
	 * Run the thread and make sure cleanup happens.
	 */
	public void run()
	{
		try
		{
			this.runMe();
		}
		finally
		{
			this.cleanForGC();
		}
	}

    /**
	 * Runs the timer thread. 
	 */
	void runMe()
	{
		/*		Thread	runThread;

		// Volatile check to see if terminated before we actually start.
		if (this.isTerminated)
		{
			return;
		}

		this.runner = runThread = Thread.currentThread();

		while (true)
		{
			synchronized(this)
			{
				// Safe check to see if terminated before we actually run.
				if (this.isTerminated)
				{
					part.modifyDaemonCount(-1);
					return;
				}

				while (!this.active)
				{
					try
					{
						// Wait to become active.
						this.wait();
					}
					catch (InterruptedException ie)
					{
						if (this.runner == null || this.isTerminated)
						{
							part.modifyDaemonCount(-1);
							return;
						}
						if (runThread.isKilledOn())
						{
							part.modifyDaemonCount(-1);
							runThread.isKilled();	
							return;
						}
					}
				}
			}

			try
			{
				// Wait for out timeout.
				Timer.suspends++;
				runThread.sleep(this.interval);
			}
			catch(InterruptedException ie)
			{
				if (this.runner == null || this.isTerminated)
				{
					part.modifyDaemonCount(-1);
					return;
				}
				if (runThread.isKilledOn())
				{
					part.modifyDaemonCount(-1);
					runThread.isKilled();	
					return;
				}

				// Some internal state changed, skip the tick.
				continue;
			}

			this.ticks++;
			Timer.resumes++;
			TickEventListener  theOwner; // local copy to avoid races
            if ((theOwner = this.owner) != null) 
				 theOwner.tick(new TickEvent(this)); 

			if (this.tickOnce || this.isTerminated)
			{
				part.modifyDaemonCount(-1);
				return;
			}
		}

		*/
	}

	/**
	 * Clen up.
	 */
	private void cleanForGC()
	{
		this.owner = null;	
		this.runner = null;	
	}

	/**
	 * Kills the thread running the timer and destroys the timer object.
	 */
	public void destroy()
	{
		try
		{
			this.cleanForGC();
			Thread	runThread;
			if ((runThread = this.runner) != null)
			{
				//	runThread.stopThread();
			}
		}
		catch (Exception e)
		{
		}
	}

	/**
	 * Destroys the timer. 
	 * 
	 */
	public void finalize()
	{
		this.destroy();
	}
}
