/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.jauth.callback;

import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import javax.security.auth.callback.Callback;
import javax.security.auth.x500.X500Principal;

/**
 * Callback for private key and corresponding certificate chain.
 *
 * @version %I%, %G%
 */
public class PrivateKeyCallback implements Callback {

    private Request request;
    private PrivateKey key;
    private Certificate[] chain;

    /**
     * Marker interface for private key request types.
     */
    public static interface Request { };

    /**
     * Request type for private keys that are identified via an alias.
     */
    public static class AliasRequest implements Request {
	private String alias;

	/**
	 * Construct an AliasRequest with an alias.
	 *
	 * <p> The alias is used to directly identify the private key
	 * to be returned.  The corresponding certificate chain for the
	 * private key is also returned.
	 *
	 * <p> If the alias is null,
	 * the handler of the callback relies on its own default.
	 *
	 * @param alias name identifier for the private key, or null.
	 */
	public AliasRequest(String alias) {
	    this.alias = alias;
	}

	/**
	 * Get the alias.
	 *
	 * @return the alias, or null.
	 */
	public String getAlias() {
	    return alias;
	}
    }

    /**
     * Request type for private keys that are identified via a SubjectKeyID
     */
    public static class SubjectKeyIDRequest implements Request {
	private byte[] id;

	/**
	 * Construct a SubjectKeyIDRequest with an subjectKeyID.
	 *
	 * <p> The subjectKeyID is used to directly identify the private key
	 * to be returned.  The corresponding certificate chain for the
	 * private key is also returned.
	 *
	 * <p> If the subjectKeyID is null,
	 * the handler of the callback relies on its own default.
	 *
	 * @param subjectKeyID identifier for the private key, or null.
	 */
	public SubjectKeyIDRequest(byte[] subjectKeyID) {
	    if (subjectKeyID != null) {
		this.id = (byte[])subjectKeyID.clone();
	    }
	}

	/**
	 * Get the subjectKeyID.
	 *
	 * @return the subjectKeyID, or null.
	 */
	public byte[] getSubjectKeyID() {
	    return id;
	}
    }

    /**
     * Request type for private keys that are identified via an
     * issuer/serial number.
     */
    public static class IssuerSerialNumRequest implements Request {
	private X500Principal issuer;
	private BigInteger serialNum;

	/**
	 * Constructs a IssuerSerialNumRequest with an issuer/serial number.
	 *
	 * <p> The issuer/serial number are used to identify a
	 * public key certificate.  The corresponding private key
	 * is returned in the callback.  The corresponding certificate chain
	 * for the private key is also returned.
	 *
	 * If the issuer/serialNumber parameters are null,
	 * the handler of the callback relies on its own defaults.
	 *
	 * @param issuer the X500Principal name of the certificate issuer,
	 *		or null.
	 *
	 * @param serialNumber the serial number of the certificate,
	 *		or null.
	 */
	public IssuerSerialNumRequest(X500Principal issuer,
					BigInteger serialNumber) {
	    this.issuer = issuer;
	    this.serialNum = serialNumber;
	}

	/**
	 * Get the issuer.
	 *
	 * @return the issuer, or null.
	 */
	public X500Principal getIssuer() {
	    return issuer;
	}

	/**
	 * Get the serial number.
	 *
	 * @return the issuer, or null.
	 */
	public BigInteger getSerialNum() {
	    return serialNum;
	}
    }

    /**
     * Constructs this PrivateKeyCallback with a private key Request object.
     *
     * <p> The <i>request</i> object identifies the private key
     * to be returned.  The corresponding certificate chain for the
     * private key is also returned.
     *
     * <p> If the <i>request</i> object is null,
     * the handler of the callback relies on its own default.
     *
     * @param request identifier for the private key, or null.
     */
    public PrivateKeyCallback(Request request) {
	this.request = request;
    }

    /**
     * Get the Request object which identifies the private key to be returned.
     *
     * @return the Request object which identifies the private key
     *		to be returned, or null.  If null, the handler of the callback
     *		relies on its own default.
     */
    public Request getRequest() {
	return request;
    }

    /**
     * Set the requested private key.
     *
     * <p> If the requested private key or chain could not be found,
     * then both values must be set to null.
     *
     * @param key the private key, or null.
     * @param chain the corresponding certificate chain, or null.
     */
    public void setKey(PrivateKey key, Certificate[] chain) {
	this.key = key;
	this.chain = chain;
    }

    /**
     * Get the requested private key.
     *
     * @return the private key, or null if the key could not be found.
     */
    public PrivateKey getKey() {
	return key;
    }

    /**
     * Get the requested certificate chain.
     *
     * @return the certificate chain, or null if the chain could not be found.
     */
    public Certificate[] getChain() {
	return chain;
    }
}
