/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.queryframework;

import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.util.Comparator;
import java.lang.reflect.Constructor;
import oracle.toplink.essentials.exceptions.*;
import oracle.toplink.essentials.internal.security.PrivilegedAccessHelper;
import oracle.toplink.essentials.internal.security.PrivilegedGetConstructorFor;
import oracle.toplink.essentials.internal.security.PrivilegedInvokeConstructor;
import oracle.toplink.essentials.internal.security.PrivilegedNewInstanceFromClass;

/**
 * <p><b>Purpose</b>: A SortedCollectionContainerPolicy is ContainerPolicy whose
 * container class implements the SortedInterface interface.
 * Added for BUG # 3233263
 * <p>
 * <p><b>Responsibilities</b>:
 * Provide the functionality to operate on an instance of a SortedSet.
 *
 * @see ContainerPolicy
 * @see MapContainerPolicy
 */
public class SortedCollectionContainerPolicy extends CollectionContainerPolicy {
    protected Comparator m_comparator = null;

    /**
     * INTERNAL:
     * Construct a new policy.
     */
    public SortedCollectionContainerPolicy() {
        super();
    }

    /**
     * INTERNAL:
     * Construct a new policy for the specified class.
     */
    public SortedCollectionContainerPolicy(Class containerClass) {
        super(containerClass);
    }

    /**
     * INTERNAL:
     * Sets a comparator object for this policy to use when instantiating
     * a new SortedSet object.
     */
    public void setComparator(Comparator comparator) {
        m_comparator = comparator;
    }

    /**
     * INTERNAL:
     * Return the stored comparator
     */
    public Comparator getComparator() {
        return m_comparator;
    }

    /**
     * INTERNAL
     * Override from ContainerPolicy. Need to maintain the comparator in the
     * new instance
     */
    public Object containerInstance() {
        try {
            if (m_comparator != null) {
                Object[] arguments = new Object[] { m_comparator };
                Class[] constructClass = new Class[] { Comparator.class };
                Constructor constructor = null;
                if (PrivilegedAccessHelper.shouldUsePrivilegedAccess()){
                    try {
                        constructor = (Constructor)AccessController.doPrivileged(new PrivilegedGetConstructorFor(getContainerClass(), constructClass, false));
                        return AccessController.doPrivileged(new PrivilegedInvokeConstructor(constructor, arguments));
                    } catch (PrivilegedActionException exception) {
                        throw QueryException.couldNotInstantiateContainerClass(getContainerClass(), exception.getException());
                    }
                } else {
                    constructor = PrivilegedAccessHelper.getConstructorFor(getContainerClass(), constructClass, false);
                    return PrivilegedAccessHelper.invokeConstructor(constructor, arguments);
                }
            } else {
                if (PrivilegedAccessHelper.shouldUsePrivilegedAccess()){
                    try {
                        return AccessController.doPrivileged(new PrivilegedNewInstanceFromClass(getContainerClass()));
                    } catch (PrivilegedActionException exception) {
                        throw QueryException.couldNotInstantiateContainerClass(getContainerClass(), exception.getException());
                    }
                } else {
                    return PrivilegedAccessHelper.newInstanceFromClass(getContainerClass());
                }
            }
        } catch (Exception ex) {
            throw QueryException.couldNotInstantiateContainerClass(getContainerClass(), ex);
        }
    }
}
