/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.sessions;

import oracle.toplink.essentials.exceptions.*;
import oracle.toplink.essentials.threetier.*;
import oracle.toplink.essentials.internal.identitymaps.IdentityMapManager;
import oracle.toplink.essentials.queryframework.*;
import oracle.toplink.essentials.internal.sessions.AbstractSession;
import oracle.toplink.essentials.internal.sessions.AbstractRecord;

/**
 * Provides isolation support by allowing a client session
 * to have a local cache of the subset of the classes.
 * This can be used to avoid caching frequently changing data,
 * or for security or VPD purposes.
 */
public class IsolatedClientSession extends oracle.toplink.essentials.threetier.ClientSession {
    public IsolatedClientSession(ServerSession parent, ConnectionPolicy connectionPolicy) {
        super(parent, connectionPolicy);
    }

    /**
    * INTERNAL:
    * Set up the IdentityMapManager.  This method allows subclasses of Session to override
    * the default IdentityMapManager functionality.
    */
    public void initializeIdentityMapAccessor() {
        this.identityMapAccessor = new IsolatedClientSessionIdentityMapAccessor(this, new IdentityMapManager(this));
    }

    /**
    * INTERNAL:
    * Helper method to calculate whether to execute this query locally or send
    * it to the server session.
    */
    protected boolean shouldExecuteLocally(DatabaseQuery query) {
        if (isIsolatedQuery(query)) {
            return true;
        }
        return isInTransaction();
    }

    /**
    * INTERNAL: Answers if this query is an isolated query and must be
    * executed locally.
    */
    protected boolean isIsolatedQuery(DatabaseQuery query) {
        query.checkDescriptor(this);
        if (query.isDataModifyQuery() || ((query.getDescriptor() != null) && query.getDescriptor().isIsolated())) {
            // For CR#4334 if in transaction stay on client session.
            // That way client's write accessor will be used for all queries.
            // This is to preserve transaction isolation levels.
            // also if this is an isolated class and we are in an isolated session
            //load locally. 
            return true;
        }
        return false;

    }

    /**
    * INTERNAL:
    * Gets the next link in the chain of sessions followed by a query's check
    * early return, the chain of sessions with identity maps all the way up to
    * the root session.
    * <p>
    * Used for session broker which delegates to registered sessions, or UnitOfWork
    * which checks parent identity map also.
    * @param canReturnSelf true when method calls itself.  If the path
    * starting at <code>this</code> is acceptable.  Sometimes true if want to
    * move to the first valid session, i.e. executing on ClientSession when really
    * should be on ServerSession.
    * @param terminalOnly return the session we will execute the call on, not
    * the next step towards it.
    * @return this if there is no next link in the chain
    */
    public AbstractSession getParentIdentityMapSession(DatabaseQuery query, boolean canReturnSelf, boolean terminalOnly) {
        if ((query != null) && isIsolatedQuery(query)) {
            return this;
        } else {
            return getParent().getParentIdentityMapSession(query, canReturnSelf, terminalOnly);
        }
    }

    /**
    * INTERNAL:
    * Gets the session which this query will be executed on.
    * Generally will be called immediately before the call is translated,
    * which is immediately before session.executeCall.
    * <p>
    * Since the execution session also knows the correct datasource platform
    * to execute on, it is often used in the mappings where the platform is
    * needed for type conversion, or where calls are translated.
    * <p>
    * Is also the session with the accessor.  Will return a ClientSession if
    * it is in transaction and has a write connection.
    * @return a session with a live accessor
    * @param query may store session name or reference class for brokers case
    */
    public AbstractSession getExecutionSession(DatabaseQuery query) {
        if (shouldExecuteLocally(query)) {
            return this;
        } else {
            return getParent().getExecutionSession(query);
        }
    }

    /**
    * INTERNAL:
    * Isolated sessions must forward call execution to its parent, unless in a transaction.
    * This is required as isolated sessions are always the execution session for isolated classes.
    */
    public Object executeCall(Call call, AbstractRecord translationRow, DatabaseQuery query) throws DatabaseException {
        if (isInTransaction()) {
            return super.executeCall(call, translationRow, query);
        }
        return getParent().executeCall(call, translationRow, query);
    }
}
