/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * Copyright 2004-2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */
//----------------------------------------------------------------------------
//
// Module:      Log.java
//
// Description: Transaction state logger.
//
// Product:     com.sun.jts.CosTransactions
//
// Author:      Simon Holdsworth
//
// Date:        March, 1997
//
// Copyright (c):   1995-1997 IBM Corp.
//
//   The source code for this program is not published or otherwise divested
//   of its trade secrets, irrespective of what has been deposited with the
//   U.S. Copyright Office.
//
//   This software contains confidential and proprietary information of
//   IBM Corp.
//----------------------------------------------------------------------------

package com.sun.jts.CosTransactions;

// Import required classes.

import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import com.sun.jts.utils.LogFormatter;


/**The Log class provides operations that control the physical log
 * as an entity versus the individual LogFiles that form the log. It supports
 * the initialisation, opening and termination of the log. Different physical
 * logs can be placed on the system with only minor changes to the methods
 * contained in this class.
 *
 * @version 0.01
 *
 * @author Simon Holdsworth, IBM Corporation
 *
 * @see
*/
//----------------------------------------------------------------------------
// CHANGE HISTORY
//
// Version By     Change Description
//   0.01  SAJH   Initial implementation.
//-----------------------------------------------------------------------------

class Log {

    /**A reference to the LogControl object.
     * @seecom.ibm.jts.logger.LogControl
     */
    private LogControl logControl = null;

    /**The log path.
     */
    // private static String logPath = null;
    private String logPath = null;
	/*
		Logger to log transaction messages
	*/
	static Logger _logger = LogDomains.getLogger(LogDomains.TRANSACTION_LOGGER);

    /**Default Log constructor.
     *
     * @param
     *
     * @return
     *
     * @see
     */
    Log() {
        // We need to ensure that messaging is initialised as this may be called
        // prior to SOMTR_Init.

        // Initialise the instance variables.

        logControl = null;

        // Get the log path.  If it is not set, or blank, then set it to the current
        // directory.

        if( logPath == null ) {
            int[] result = new int[1];
            logPath = Configuration.getDirectory(Configuration.LOG_DIRECTORY,
                                                 Configuration.JTS_SUBDIRECTORY,
                                                 result);

            // If a default was used, display a message.

            if( result[0] == Configuration.DEFAULT_USED ||
                result[0] == Configuration.DEFAULT_INVALID ) {

                // In the case where the SOMBASE default is used, only display a message
                // if an invalid value was specified in the environment value.

                if( logPath.length() > 0 ) {
						_logger.log(Level.WARNING,"jts.invalid_log_path",logPath);
                }

                // In the case where the SOMBASE default is invalid, the value returned is
                // the invalid default. We then default to the current directory.

                if( result[0] == Configuration.DEFAULT_INVALID ) {
						_logger.log(Level.WARNING,"jts.invalid_default_log_path");
                    logPath = "."/*#Frozen*/;
                }
            }
        }

    }


    Log(String logPath) {
        logControl = null;
        this.logPath = logPath;
    }

    /**Default Log destructor.
     *
     * @param
     *
     * @return
     *
     * @see
     */
    public void finalize() {

        logControl = null;
        logPath = null;

    }

    /**Initialises the log.
     *
     * @param
     *
     * @return
     *
     * @see
     */
    boolean initialise() {
        boolean result = true;

        // Call the initialize operation for the log

        logControl = new LogControl();
        logControl.initLog(false,false,logPath);

        return result;
    }


    /**Opens the log for the given server.
     * <p>
     * The given LogSOS object, if any, will be called in the event of the log
     * going short-on-storage. A LogFile object reference is returned that is used
     * for operations on the specific portion of the log.
     *
     * @param serverName  The name of the server whose log file is being opened.
     * @param upcall      The object which will handle upcalls from the log.
     *
     * @return  The object representing the physical log file.
     *
     * @see
     */
    LogFile open( String          serverName,
                  LogUpcallTarget upcall ) {

        LogFile logFile = null;

        boolean[] newLog = new boolean[1];  newLog[0] = true;

        // Open the log using the server name.

        try {
            LogHandle handle = logControl.openFile(serverName,upcall,null,newLog);

            // Create a new LogFile object with the handle to represent the open log.

            logFile = new LogFile(handle);
        }

        // If the log open failed, report the error.

        catch( LogException le ) {
			_logger.log(Level.SEVERE,"jts.log_error",le.toString());
			 String msg = LogFormatter.getLocalizedMessage(_logger,"jts.log_error",
			 							new java.lang.Object[] {le.toString()});
			  throw  new org.omg.CORBA.INTERNAL(msg);
        }

        return logFile;
    }

    /**Terminates the log.
     *
     * @param
     *
     * @return
     *
     * @see
     */
    boolean terminate() {

        boolean result = true;

        // No special action needed after the close for the logger

        return result;
    }

    /**Determines whether a log file exists for the given server.
     * <p>
     * This method may be used without initialising the Log object to determine
     * whether recovery should be performed, without initialising the log or the OTS.
     *
     * @param String
     *
     * @return
     *
     * @see
     */
    static boolean checkFileExists( String serverName ) {
        boolean exists = false;

        // Check whether the file exists.

        if( serverName != null ) {
            String logPath = null;
            int[] result = new int[1];
            logPath = Configuration.getDirectory(Configuration.LOG_DIRECTORY,
                                                 Configuration.JTS_SUBDIRECTORY,
                                                 result);

            // If a default was used, display a message.

            if( result[0] == Configuration.DEFAULT_USED ||
                result[0] == Configuration.DEFAULT_INVALID ) {

                // In the case where the SOMBASE default is used, only display a message
                // if an invalid value was specified in the environment value.

                if( logPath.length() > 0 ) {
                     _logger.log(Level.WARNING,"jts.invalid_log_path",logPath);
                }

                // In the case where the SOMBASE default is invalid, the value returned is
                // the invalid default. We then default to the current directory.

                if( result[0] == Configuration.DEFAULT_INVALID ) {
                                                _logger.log(Level.WARNING,"jts.invalid_default_log_path");
                    logPath = "."/*#Frozen*/;
                }
            }
            exists = LogControl.checkFileExists(serverName,logPath);
        }


        return exists;
    }

    /**Dumps the state of the object.
     *
     * @param
     *
     * @return
     *
     * @see
     */
    void dump() {
        //! somtrDUMP_OBJECT_HEADER;

        // Dump all of the instance variables in the LogFile object, without going
        // any further down object references.

        logControl.dump();
    }
}
