/*
 * $Id: buffer.c,v 1.4 2002/01/06 23:03:58 binnema Exp $
 */

/*
** Copyright (C) 2001 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#include <string.h>
#include "buffer.h"

Buffer*
buffer_new ()
{
	Buffer *new_buffer;

	new_buffer = (Buffer*) malloc (sizeof(Buffer));
	if (!new_buffer)
		return NULL;

	new_buffer->_data  = NULL;
	new_buffer->_len   = 0;

	return new_buffer;
}


Buffer* 
buffer_destroy (Buffer *buffer)
{
	assert (buffer);

	free (buffer->_data);
	free (buffer);

	return NULL;
}



void 
buffer_set_data (Buffer* buffer, const char* data, size_t len)
{
	assert (buffer);
	assert (data || len == 0);
	
	buffer->_data = (char*) realloc (buffer->_data, len);
	if (!buffer->_data)
		return;
		
	buffer->_data = memcpy (buffer->_data, data, len);
	buffer->_len  = len;
}




void
buffer_manage_data (Buffer* buffer, char* data, size_t len)
{
	assert (buffer);
	assert (data || len == 0);

	buffer->_data = data;
	buffer->_len  = len;
}



char        
buffer_at (const Buffer* buffer, BufferIterator pos)
{       
	assert (buffer);
	assert (pos < buffer_end(buffer));
	
	return buffer->_data[pos];
}
                   

	
const char* 
buffer_data (const Buffer* buffer)
{
	assert (buffer);
	return buffer->_data;
}
	


size_t 
buffer_length (const Buffer *buffer)
{
	assert (buffer);
	return buffer->_len;
}


	
void
buffer_append (Buffer* buffer, const char* data, size_t len)
{
	assert (buffer);
	assert (data || len == 0);
	
	buffer->_data = (char*)realloc (buffer->_data, buffer->_len + len);
	memcpy (buffer->_data + buffer->_len, data, len);

	buffer->_len += len;
}

void 
buffer_prepend (Buffer *buffer, const char *data, size_t len)
{
	assert (buffer);
	assert (data || len == 0);
	
	buffer_insert (buffer, data, len, 0);
}


void
buffer_insert (Buffer *buffer, const char *data, size_t len, size_t pos)
{
	assert (buffer);
	assert (data || len == 0);
	assert (buffer->_len > pos);
	
	buffer->_data = (char*)realloc (buffer->_data, buffer->_len + len);
		
	memmove (buffer->_data + pos + len, 
		buffer->_data + pos, 
		buffer->_len - pos);
	memmove (buffer->_data + pos, data, len);

	buffer->_len += len;
}


void
buffer_erase (Buffer *buffer, size_t pos, size_t len)
{
	assert (buffer);
	assert (buffer->_len >= pos + len);
	
	if (buffer->_len >= pos + len)
		memcpy (buffer->_data + pos,
			buffer->_data + pos + len,
			buffer->_len  - pos - len);

	buffer->_data = realloc (buffer->_data, buffer->_len - len);
	buffer->_len  -= len;
}




BufferIterator 
buffer_find (const Buffer *buffer, const char* pattern, size_t pattern_len)
/* brute force --> todo: kmp search */	
{	
	int i, j, match;
	
	assert (buffer);
	assert (pattern || pattern_len == 0);
	
	for (i = 0; i <= buffer->_len; ++i) {
		for (j = 0; j < pattern_len; ++j) {
			match = (buffer->_data[i+j] == pattern[j]);
			if (!match) 
				break;
		}
		if (match && (j == pattern_len)) 
			break;
	}

	return i;
}



void
buffer_clear (Buffer *buffer)
{
	assert (buffer);

	free (buffer->_data);

	buffer->_data = NULL;
	buffer->_len  = 0;
}
