/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/intern/m4_scenegraph_dev.h>

#include <math.h>


static Float Interpolate(Float keyValue1, Float keyValue2, Float fraction)
{
	return (keyValue2 - keyValue1) * fraction + keyValue1;
}

static Float GetInterpolateFraction(Float key1, Float key2, Float fraction)
{
	Float keyDiff = key2 - key1;
	assert((fraction >= key1) && (fraction <= key2));
	if (fabs(keyDiff) < M4_EPSILON_FLOAT) return 0;
	return ((fraction - key1) / keyDiff);
}

#ifdef M4_DEF_CoordinateInterpolator2D

void CI2D_SetFraction(SFNode *n)
{
	Float frac;
	u32 numElemPerKey, i, j;
	B_CoordinateInterpolator2D *_this = (B_CoordinateInterpolator2D *) n;

	if (! _this->key.count) return;
	if (_this->keyValue.count % _this->key.count) return;
	
	numElemPerKey = _this->keyValue.count / _this->key.count;
	//set size
	if (_this->value_changed.count != numElemPerKey)
		MFField_Alloc(&_this->value_changed, FT_MFVec2f, numElemPerKey);


	if (_this->set_fraction < _this->key.vals[0]) {
		for (i=0; i<numElemPerKey; i++)
			_this->value_changed.vals[i] = _this->keyValue.vals[i];
	} else if (_this->set_fraction > _this->key.vals[_this->key.count - 1]) {
		for (i=0; i<numElemPerKey; i++)
			_this->value_changed.vals[i] = _this->keyValue.vals[(_this->keyValue.count) - numElemPerKey + i];
	} else {
		for (j = 1; j < _this->key.count; j++) {
			// Find the key values the fraction lies between
			if ( _this->set_fraction < _this->key.vals[j-1]) continue;
			if (_this->set_fraction >= _this->key.vals[j]) continue;

			frac = GetInterpolateFraction(_this->key.vals[j-1], _this->key.vals[j], _this->set_fraction);
			for (i=0; i<numElemPerKey; i++) {
				_this->value_changed.vals[i].x = Interpolate(_this->keyValue.vals[(j-1)*numElemPerKey + i].x,
															_this->keyValue.vals[(j)*numElemPerKey + i].x, 
															frac);
				_this->value_changed.vals[i].y = Interpolate(_this->keyValue.vals[(j-1)*numElemPerKey + i].y,
															_this->keyValue.vals[(j)*numElemPerKey + i].y,
															frac);
			}
			break;
		}
	}
	//invalidate
	Node_OnEventOutSTR(n, "value_changed");
}
#endif


Bool CI_SetFraction(Float fraction, MFVec3f *vals, MFFloat *key, MFVec3f *keyValue)
{
	Float frac;
	u32 numElemPerKey, i, j;

	if (! key->count) return 0;
	if (keyValue->count % key->count) return 0;
	
	numElemPerKey = keyValue->count / key->count;

	if (vals->count != numElemPerKey) MFField_Alloc(vals, FT_MFVec3f, numElemPerKey);

	if (fraction < key->vals[0]) {
		for (i=0; i<numElemPerKey; i++)
			vals->vals[i] = keyValue->vals[i];
	} else if (fraction > key->vals[key->count - 1]) {
		for (i=0; i<numElemPerKey; i++)
			vals->vals[i] = keyValue->vals[(keyValue->count) - numElemPerKey + i];
	} else {
		for (j = 1; j < key->count; j++) {
			// Find the key values the fraction lies between
			if (fraction < key->vals[j-1]) continue;
			if (fraction >= key->vals[j]) continue;

			frac = GetInterpolateFraction(key->vals[j-1], key->vals[j], fraction);
			for (i=0; i<numElemPerKey; i++) {
				vals->vals[i].x = Interpolate(keyValue->vals[(j-1)*numElemPerKey + i].x,
															keyValue->vals[(j)*numElemPerKey + i].x,
															frac);
				vals->vals[i].y = Interpolate(keyValue->vals[(j-1)*numElemPerKey + i].y,
															keyValue->vals[(j)*numElemPerKey + i].y,
															frac);
				vals->vals[i].z = Interpolate(keyValue->vals[(j-1)*numElemPerKey + i].z,
															keyValue->vals[(j)*numElemPerKey + i].z, 
															frac);
			}
			break;
		}
	}
	return 1;
}


#ifdef M4_DEF_CoordinateInterpolator

void CoordInt_SetFraction(SFNode *n)
{
	B_CoordinateInterpolator *_this = (B_CoordinateInterpolator *) n;

	if (CI_SetFraction(_this->set_fraction, &_this->value_changed, &_this->key, &_this->keyValue)) 
		Node_OnEventOutSTR(n, "value_changed");
}
#endif



#ifdef M4_DEF_NormalInterpolator

void NormInt_SetFraction(SFNode *n)
{
	u32 i;
	B_NormalInterpolator *_this = (B_NormalInterpolator *) n;

	if (!CI_SetFraction(_this->set_fraction, &_this->value_changed, &_this->key, &_this->keyValue)) return;
	/*renorm*/
	for (i=0; i<_this->value_changed.count; i++) {
		SFVec3f *v = &_this->value_changed.vals[i];
		Float res = (Float) sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
		v->x /= res;
		v->y /= res;
		v->z /= res;
	}
	Node_OnEventOutSTR(n, "value_changed");
}
#endif


#ifdef M4_DEF_ColorInterpolator

void ColorInt_SetFraction(SFNode *node)
{
	u32 i;
	Float frac;
	B_ColorInterpolator *_this = (B_ColorInterpolator *)node;


	if (! _this->key.count) return;
	if (_this->keyValue.count != _this->key.count) return;
	
	// The given fraction is less than the specified range
	if (_this->set_fraction < _this->key.vals[0]) {
		_this->value_changed = _this->keyValue.vals[0];
	} else if (_this->set_fraction >= _this->key.vals[_this->key.count-1]) {
		_this->value_changed = _this->keyValue.vals[_this->keyValue.count-1];
	} else {
		for (i=1; i<_this->key.count; i++) {
			// Find the key values the fraction lies between
			if (_this->set_fraction < _this->key.vals[i-1]) continue;
			if (_this->set_fraction >= _this->key.vals[i]) continue;

			frac = GetInterpolateFraction(_this->key.vals[i-1], _this->key.vals[i], _this->set_fraction);
			_this->value_changed.red = Interpolate(_this->keyValue.vals[i-1].red, 
													_this->keyValue.vals[i].red, 
													frac);
			_this->value_changed.green = Interpolate(_this->keyValue.vals[i-1].green,
													_this->keyValue.vals[i].green, 
													frac);
			_this->value_changed.blue = Interpolate(_this->keyValue.vals[i-1].blue,
													_this->keyValue.vals[i].blue,
													frac);
			break;
		}
	}
	Node_OnEventOutSTR(node, "value_changed");
}
#endif

#ifdef M4_DEF_PositionInterpolator2D

void PosInt2D_SetFraction(SFNode *node)
{
	B_PositionInterpolator2D *_this = (B_PositionInterpolator2D *)node;
	u32 i;
	Float frac;

	if (! _this->key.count) return;
	if (_this->keyValue.count != _this->key.count) return;
	
	// The given fraction is less than the specified range
	if (_this->set_fraction < _this->key.vals[0]) {
		_this->value_changed = _this->keyValue.vals[0];
	} else if (_this->set_fraction >= _this->key.vals[_this->key.count-1]) {
		_this->value_changed = _this->keyValue.vals[_this->keyValue.count-1];
	} else {
		for (i=1; i<_this->key.count; i++) {
			// Find the key values the fraction lies between
			if (_this->set_fraction < _this->key.vals[i-1]) continue;
			if (_this->set_fraction >= _this->key.vals[i]) continue;

			frac = GetInterpolateFraction(_this->key.vals[i-1], _this->key.vals[i], _this->set_fraction);
			_this->value_changed.x = Interpolate(_this->keyValue.vals[i-1].x, _this->keyValue.vals[i].x, frac);
			_this->value_changed.y = Interpolate(_this->keyValue.vals[i-1].y, _this->keyValue.vals[i].y, frac);
			break;
		}
	}
	Node_OnEventOutSTR(node, "value_changed");
}
#endif

#ifdef M4_DEF_PositionInterpolator

void PosInt_SetFraction(SFNode *node)
{
	u32 i;
	Float frac;
	B_PositionInterpolator *_this = (B_PositionInterpolator *)node;

	if (! _this->key.count) return;
	if (_this->keyValue.count != _this->key.count) return;
	
	// The given fraction is less than the specified range
	if (_this->set_fraction < _this->key.vals[0]) {
		_this->value_changed = _this->keyValue.vals[0];
	} else if (_this->set_fraction >= _this->key.vals[_this->key.count-1]) {
		_this->value_changed = _this->keyValue.vals[_this->keyValue.count-1];
	} else {
		for (i=1; i<_this->key.count; i++) {
			// Find the key values the fraction lies between
			if (_this->set_fraction < _this->key.vals[i-1]) continue;
			if (_this->set_fraction >= _this->key.vals[i]) continue;

			frac = GetInterpolateFraction(_this->key.vals[i-1], _this->key.vals[i], _this->set_fraction);
			_this->value_changed.x = Interpolate(_this->keyValue.vals[i-1].x, _this->keyValue.vals[i].x, frac);
			_this->value_changed.y = Interpolate(_this->keyValue.vals[i-1].y, _this->keyValue.vals[i].y, frac);
			_this->value_changed.z = Interpolate(_this->keyValue.vals[i-1].z, _this->keyValue.vals[i].z, frac);
			break;
		}
	}
	Node_OnEventOutSTR(node, "value_changed");
}
#endif

#ifdef M4_DEF_ScalarInterpolator

void ScalarInt_SetFraction(SFNode *node)
{
	B_ScalarInterpolator *_this = (B_ScalarInterpolator *)node;
	u32 i;
	Float frac;

	if (! _this->key.count) return;
	if (_this->keyValue.count != _this->key.count) return;
	
	// The given fraction is less than the specified range
	if (_this->set_fraction < _this->key.vals[0]) {
		_this->value_changed = _this->keyValue.vals[0];
	} else if (_this->set_fraction >= _this->key.vals[_this->key.count-1]) {
		_this->value_changed = _this->keyValue.vals[_this->keyValue.count-1];
	} else {
		for (i=1; i<_this->key.count; i++) {
			// Find the key values the fraction lies between
			if (_this->set_fraction < _this->key.vals[i-1]) continue;
			if (_this->set_fraction >= _this->key.vals[i]) continue;

			frac = GetInterpolateFraction(_this->key.vals[i-1], _this->key.vals[i], _this->set_fraction);
			_this->value_changed = Interpolate(_this->keyValue.vals[i-1], _this->keyValue.vals[i], frac);
			break;
		}
	}
	Node_OnEventOutSTR(node, "value_changed");
}

#endif

#ifdef M4_DEF_OrientationInterpolator

#ifndef M_PI
#define M_PI					3.14159265358979323846f
#endif

/*taken from freeWRL*/
SFRotation Rotation_Interpolate(SFRotation kv1, SFRotation kv2, Float fraction)
{
	SFRotation res;
	Float newa, olda;
	Bool stzero = ( (Float) fabs(kv1.angle) < M4_EPSILON_FLOAT) ? 1 : 0;
	Bool endzero = ( (Float) fabs(kv2.angle) < M4_EPSILON_FLOAT) ? 1 : 0;
	Float testa = kv1.xAxis*kv2.xAxis + kv1.yAxis*kv2.yAxis + kv1.yAxis*kv2.yAxis;

	if (testa>= 0.0) {
		res.xAxis = kv1.xAxis + fraction*(kv2.xAxis-kv1.xAxis);
		res.yAxis = kv1.yAxis + fraction*(kv2.yAxis-kv1.yAxis);
		res.zAxis = kv1.zAxis + fraction*(kv2.zAxis-kv1.zAxis);
		newa = kv2.angle;
	} else {
		res.xAxis = kv1.xAxis + fraction*(-kv2.xAxis-kv1.xAxis);
		res.yAxis = kv1.yAxis + fraction*(-kv2.yAxis-kv1.yAxis);
		res.zAxis = kv1.zAxis + fraction*(-kv2.zAxis-kv1.zAxis);
		newa = -kv2.angle;
	}
	olda = kv1.angle;
	testa = newa - olda;
	/* make it so we smoothly transition */
	if (fabs(testa) > M_PI) {
		if (fabs(testa) > (M_PI*2)) {
			if (testa>0.0) {
				olda += M_PI*4;
			} else {
				newa += M_PI*4; 
			}
		} else {
			if (testa>0.0) {
				olda += M_PI*2;
			} else { 
				newa += M_PI*2; 
			}
		}
	}

	if (stzero || endzero) {
		res.xAxis = stzero ? kv2.xAxis : kv1.xAxis;
		res.yAxis = stzero ? kv2.yAxis : kv1.yAxis;
		res.zAxis = stzero ? kv2.zAxis : kv1.zAxis;
	}
	/* now that we have angles straight (hah!) bounds check result */
	res.angle = olda + fraction*(newa - olda);
	if (res.angle > M_PI*2) { 
		res.angle -= M_PI*2;
	} else if (res.angle <M_PI*2) {
		res.angle += M_PI*2;
	}
	return res;
}

void OrientInt_SetFraction(SFNode *node)
{
	u32 i;
	Float frac;
	B_OrientationInterpolator *_this = (B_OrientationInterpolator *)node;

	if (! _this->key.count) return;
	if (_this->keyValue.count != _this->key.count) return;
	
	// The given fraction is less than the specified range
	if (_this->set_fraction < _this->key.vals[0]) {
		_this->value_changed = _this->keyValue.vals[0];
	} else if (_this->set_fraction >= _this->key.vals[_this->key.count-1]) {
		_this->value_changed = _this->keyValue.vals[_this->keyValue.count-1];
	} else {
		for (i=1; i<_this->key.count; i++) {
			// Find the key values the fraction lies between
			if (_this->set_fraction < _this->key.vals[i-1]) continue;
			if (_this->set_fraction >= _this->key.vals[i]) continue;

			frac = GetInterpolateFraction(_this->key.vals[i-1], _this->key.vals[i], _this->set_fraction);
			_this->value_changed = Rotation_Interpolate(_this->keyValue.vals[i-1], _this->keyValue.vals[i], frac);
			break;
		}
	}
	Node_OnEventOutSTR(node, "value_changed");
}
#endif

