/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Authoring Tools sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

/*
		WARNING: this is a brute XMT parser, has nothing generic and only accepts 100% XMT-A files only
*/

#include <gpac/m4_author.h>
#include <gpac/intern/m4_scenegraph_dev.h>

#define XMT_LINE_SIZE	4000

typedef struct
{
	M4Err last_error;
	u32 line;

	char szMsg[512];

	FILE *input;
	Bool done;
	u32 unicode_type;

	/*extern node initializer*/
	void (*XMTNodeInit)(void *cbk, SFNode *n);
	/*extern error reporter*/
	void (*XMTOnError)(void *cbk, char *msg);
	/*callback for above functions*/
	void *xmt_cbck;

	/*routes are not created in the graph when parsing, so we need to track insert and delete/replace*/
	Chain *unresolved_routes, *inserted_routes;
	Chain *def_nodes;

	char line_buffer[XMT_LINE_SIZE], cur_buffer[500];
	s32 line_size, line_pos, line_start_pos;
	char *att_buffer;
	u32 att_buf_size;
	char *temp_att;

	/*result context*/
	M4SceneManager *ctx;
	/*current scene graph*/
	LPSCENEGRAPH scene_graph;
	/*set when parsing proto*/
	LPPROTO parsing_proto;

	/*current stream ID, AU time and RAP flag*/
	u32 stream_id;
	Double au_time;
	Bool au_is_rap;

	/*current BIFS stream & AU*/
	M4StreamContext *bifs_es;
	M4AUContext *bifs_au;
	u32 base_bifs_id;

	/*current OD stream & AU*/
	M4StreamContext *od_es;
	M4AUContext *od_au;
	u32 base_od_id;
	u32 bifs_w, bifs_h;
	Bool pixelMetrics;

	/*XMT is soooo ugly for descritpors...*/
	Chain *od_links;
	Chain *esd_links;
} XMTParser;


typedef struct
{
	char *desc_name;
	u32 ID;
	/*store nodes refering to this URL*/
	Chain *nodes;
	ObjectDescriptor *od;
} ODLink;

typedef struct
{
	char *desc_name;
	u32 ESID;
	ESDescriptor *esd;
} ESDLink;

void xmt_parse_command(XMTParser *parser, char *name, Chain *com_list);
void xmt_resolve_routes(XMTParser *parser);

#define XMT_REPORT_ERR(parser) \
	if (parser->XMTOnError) parser->XMTOnError(parser->xmt_cbck, parser->szMsg);	\
	else fprintf(stdout, parser->szMsg);	\

M4Err XMTLogErr1(XMTParser *parser, char *str)
{
	sprintf(parser->szMsg, "Error line %d: %s\n", parser->line, str);
	XMT_REPORT_ERR(parser);
	return parser->last_error = M4BadParam;
}

M4Err XMTLogErr2(XMTParser *parser, char *str, const char *field)
{
	sprintf(parser->szMsg, "Error line %d: \"%s\": %s\n", parser->line, field, str);
	XMT_REPORT_ERR(parser);
	return parser->last_error = M4BadParam;
}


SFNode *XMTNewNode(XMTParser *parser, u32 tag) 
{
	SFNode *n = SG_NewNode(parser->scene_graph, tag);
	if (n) Node_Init(n);
	return n;
}

void xmt_check_line(XMTParser *parser)
{
	u32 i;
	while (
		(parser->line_buffer[parser->line_pos]==' ') 
		|| (parser->line_buffer[parser->line_pos]=='\t') 
		) parser->line_pos++;

	if (parser->line_buffer[parser->line_pos]=='#') parser->line_size = parser->line_pos;
	else if ((parser->line_buffer[parser->line_pos]=='/') && (parser->line_buffer[parser->line_pos+1]=='/') ) parser->line_size = parser->line_pos;

	if (parser->line_size == parser->line_pos) {
		/*string based input - done*/
		if (!parser->input) return;

next_line:
		parser->line_buffer[0] = 0;
		i=0;
		parser->line_start_pos = ftell(parser->input);

		if (parser->unicode_type) {
			u8 c1, c2;
			unsigned short wchar;
			unsigned short l[XMT_LINE_SIZE];
			unsigned short *dst = l;
			Bool is_ret = 0;
			u32 go = XMT_LINE_SIZE - 1;
			u32 last_space_pos, last_space_pos_stream;
			last_space_pos = last_space_pos_stream = 0;
			while (go && !feof(parser->input) ) {
				c1 = fgetc(parser->input);
				c2 = fgetc(parser->input);
				/*Little-endian order*/
				if (parser->unicode_type==2) {
					if (c2) { wchar = c2; wchar <<=8; wchar |= c1; }
					else wchar = c1;
				} else {
					wchar = c1;
					if (c2) { wchar <<= 8; wchar |= c2;}
				}
				*dst = wchar;
				if (wchar=='\r') is_ret = 1;
				else if (wchar=='\n') {
					dst++;
					break;
				}
				else if (is_ret && wchar!='\n') {
					u32 fpos = ftell(parser->input);
					fseek(parser->input, fpos-2, SEEK_SET);
					is_ret = 1;
					break;
				}
				if (wchar==' ') {
					last_space_pos_stream = ftell(parser->input);
					last_space_pos = dst - l;
				}

				dst++;
				go--;
			}
			*dst = 0;

			if (!go) {
				u32 rew_pos = ftell(parser->input) - 2*(dst - &l[last_space_pos]);
				fseek(parser->input, rew_pos, SEEK_SET);
				l[last_space_pos+1] = 0;
			}

			/*check eof*/
			if (l[0]==0xFFFF) {
				parser->done = 1;
				return;
			}
			/*convert to mbc string*/
			dst = l;
			utf8_wcstombs(parser->line_buffer, XMT_LINE_SIZE, (const unsigned short **) &dst);

			if (!strlen(parser->line_buffer) && feof(parser->input)) {
				parser->done = 1;
				return;
			}
		} else {
			if ((fgets(parser->line_buffer, XMT_LINE_SIZE, parser->input) == NULL) 
				|| (!strlen(parser->line_buffer) && feof(parser->input))) {
				parser->done = 1;
				return;
			}
			/*watchout for long lines*/
			if (1 + strlen(parser->line_buffer) == XMT_LINE_SIZE) {
				u32 rew, pos;
				rew = 0;
				while (parser->line_buffer[strlen(parser->line_buffer)-1] != ' ') {
					parser->line_buffer[strlen(parser->line_buffer)-1] = 0;
					rew++;
				}
				pos = ftell(parser->input);
				fseek(parser->input, pos-rew, SEEK_SET);
			}
		}

		while (
			(parser->line_buffer[strlen(parser->line_buffer)-1]=='\n')
			|| (parser->line_buffer[strlen(parser->line_buffer)-1]=='\r')
			|| (parser->line_buffer[strlen(parser->line_buffer)-1]=='\t')
			)
			parser->line_buffer[strlen(parser->line_buffer)-1] = 0;

		parser->line_size = strlen(parser->line_buffer);
		parser->line_pos = 0;
		parser->line++;

		
		while ((parser->line_buffer[parser->line_pos]==' ') || (parser->line_buffer[parser->line_pos]=='\t'))
			parser->line_pos++;
		if ( (parser->line_buffer[parser->line_pos]=='#') 
			|| ( (parser->line_buffer[parser->line_pos]=='/') && (parser->line_buffer[parser->line_pos+1]=='/')) )
			goto next_line;
	}
	if (!parser->line_size) {
		if (!feof(parser->input)) xmt_check_line(parser);
		else parser->done = 1;
	}
	else if (!parser->done && (parser->line_size == parser->line_pos)) xmt_check_line(parser);
}
char *xmt_get_element(XMTParser *parser)
{
	s32 i;
	xmt_check_line(parser);

	if (parser->line_buffer[parser->line_pos] !='<') {
		/*not really clean...*/
		if (!strnicmp(parser->line_buffer + parser->line_pos, "NULL", 4)) {
			parser->line_pos += 4;
			return "NULL";
		}
		return NULL;
	}
	if (parser->line_buffer[parser->line_pos + 1] =='/') return NULL;
	else if ((parser->line_buffer[parser->line_pos + 1] =='!')
		&& (parser->line_buffer[parser->line_pos + 2] =='-')
		&& (parser->line_buffer[parser->line_pos + 3] =='-')) {
		char *end = strstr(parser->line_buffer, "-->");
		while (!end) {
			parser->line_pos = parser->line_size;
			xmt_check_line(parser);
			end = strstr(parser->line_buffer, "-->");
		}
		while ( (parser->line_buffer[parser->line_pos] != '-') 
			|| (parser->line_buffer[parser->line_pos+1] != '-')
			|| (parser->line_buffer[parser->line_pos+2] != '>') )
			parser->line_pos++;
		parser->line_pos += 3;

		return xmt_get_element(parser);
	}

	parser->line_pos++;
	xmt_check_line(parser);

	i=0;
	while (1) {
		if (!parser->line_buffer[parser->line_pos + i]) break;
		else if (parser->line_buffer[parser->line_pos + i] == '>') break;
		else if (parser->line_buffer[parser->line_pos + i] == ' ') break;
		else if ((parser->line_buffer[parser->line_pos + i] == '/') && (parser->line_buffer[parser->line_pos + i + 1] == '>')) break;
		else if (parser->line_buffer[parser->line_pos + i] == '\t') break;
		parser->cur_buffer[i] = parser->line_buffer[parser->line_pos + i];
		i++;
		if (parser->line_pos+i==parser->line_size) break;
	}
	parser->cur_buffer[i] = 0;
	parser->line_pos += i;
	return parser->cur_buffer;
}

void xmt_skip_attributes(XMTParser *parser)
{
	s32 i = 0;
	while (1) {
		if (!parser->line_buffer[parser->line_pos + i]) break;
		else if ( (parser->line_buffer[parser->line_pos + i] == '/')  && (parser->line_buffer[parser->line_pos + i + 1] == '>') ) break;
		else if (parser->line_buffer[parser->line_pos + i] == '>') {
			i++;
			break;
		}
		i++;
		if (parser->line_pos+i==parser->line_size) {
			parser->line_pos = parser->line_size;
			xmt_check_line(parser);
			i = 0;
		}
	}
	parser->cur_buffer[0] = 0;
	parser->line_pos += i;
}

Bool xmt_has_attributes(XMTParser *parser)
{
	while ((parser->line_buffer[parser->line_pos] == ' ') || (parser->line_buffer[parser->line_pos] == '\t') ) {
		parser->line_pos++;
		if (parser->line_pos==parser->line_size) xmt_check_line(parser);
	}
	if (parser->line_buffer[parser->line_pos] == '>') {
		parser->line_pos++;
		return 0;
	}
	if ((parser->line_buffer[parser->line_pos] == '/') && (parser->line_buffer[parser->line_pos+1] == '>')) {
		return 0;
	}
	return 1;
}

Bool xmt_element_done(XMTParser *parser, char *name)
{
	while ((parser->line_buffer[parser->line_pos] == ' ') || (parser->line_buffer[parser->line_pos] == '\t') ) {
		parser->line_pos++;
		if (parser->line_pos==parser->line_size) xmt_check_line(parser);
	}
	xmt_check_line(parser);
	if ((parser->line_buffer[parser->line_pos] == '/') && (parser->line_buffer[parser->line_pos+1] == '>')) {
		parser->line_pos += 2;
		return 1;
	}
	if (parser->line_buffer[parser->line_pos] != '<') return 0;
	if (parser->line_buffer[parser->line_pos + 1] != '/') return 0;
	if (strnicmp(&parser->line_buffer[parser->line_pos + 2], name, strlen(name))) return 0;
	xmt_skip_attributes(parser);
	return 1;
}

void xmt_skip_element(XMTParser *parser, char *name)
{
	char *str;
	char szElt[2048];
	if (!stricmp(name, "NULL")) return;

	strcpy(szElt, name);
	xmt_skip_attributes(parser);
	
	while (!xmt_element_done(parser, szElt) && !parser->last_error) {
		str = xmt_get_element(parser);
		xmt_skip_element(parser, str);
	}
}

char *xmt_get_attribute(XMTParser *parser)
{
	Bool str_start;
	s32 k = 0;
	s32 i = 0;
	while ((parser->line_buffer[parser->line_pos + i] == ' ') || (parser->line_buffer[parser->line_pos + i] == '\t') ) i++;
	while (1) {
		if (!parser->line_buffer[parser->line_pos + i] || (parser->line_pos+i==parser->line_size)) {
			xmt_check_line(parser);
			i = 0;
			continue;
		}
		else if (parser->line_buffer[parser->line_pos + i] == '=') break;
		parser->cur_buffer[k] = parser->line_buffer[parser->line_pos + i];
		i++;
		k++;
	}
	parser->cur_buffer[k] = 0;
	parser->line_pos += i + 1;
	str_start = 0;
	i=0;
	while ((parser->line_buffer[parser->line_pos + i] == ' ') || (parser->line_buffer[parser->line_pos + i] == '\t') ) i++;
	k = 0;
	if (!parser->att_buffer) {
		parser->att_buffer = malloc(sizeof(char) * 500);
		parser->att_buf_size = 500;
	}

	while (1) {
		if (!parser->line_buffer[parser->line_pos + i] || (parser->line_pos+i==parser->line_size)) {
			parser->line_pos = parser->line_size;
			xmt_check_line(parser);
			i = 0;
			continue;
		}
		else if (parser->line_buffer[parser->line_pos + i] == '\"') {
			if (str_start) break;
			str_start = 1;
		} else {
			if ((u32) k >= parser->att_buf_size) {
				parser->att_buf_size += 500;
				parser->att_buffer = realloc(parser->att_buffer, sizeof(char) * parser->att_buf_size);
			}
			parser->att_buffer[k] = parser->line_buffer[parser->line_pos + i];
			k++;
		}
		i++;
	}
	parser->att_buffer[k] = 0;
	parser->line_pos += i + 1;
	return parser->cur_buffer;
}

void xmt_new_od_link(XMTParser *parser, ObjectDescriptor *od, char *name)
{
	u32 i, ID, j;
	ODLink *odl;

	ID = 0;
	if (!strnicmp(name, "od", 2)) ID = atoi(name + 2);
	else if (!strnicmp(name, "iod", 3)) ID = atoi(name+ 3);
	/*be carefull, an url like "11-regression-test.mp4" will return 1 on sscanf :)*/
	else if (sscanf(name, "%d", &ID) == 1) {
		char szURL[20];
		sprintf(szURL, "%d", ID);
		if (strcmp(szURL, name)) {
			ID = 0;
		} else {
			name = NULL;
		}
	}
	else ID = 0;

	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		odl = ChainGetEntry(parser->od_links, i);
		if ( (ID && (odl->ID == ID))
			|| (odl->od == od)
			|| (odl->desc_name && name && !stricmp(odl->desc_name, name))
		) {
			if (!odl->od) odl->od = od;
			if (!odl->desc_name && name) odl->desc_name = strdup(name);
			if (!od->objectDescriptorID) {
				od->objectDescriptorID = ID;
			} else if (ID && (od->objectDescriptorID != ID)) {
				sprintf(parser->szMsg, "Error line %d: conflicting OD IDs %d vs %d\n", parser->line, ID, od->objectDescriptorID);
				XMT_REPORT_ERR(parser);
			}

			for (j=i+1; j<ChainGetCount(parser->od_links); j++) {
				ODLink *l2 = ChainGetEntry(parser->od_links, j);
				if (l2->od == od) {
					odl->ID = od->objectDescriptorID = odl->od->objectDescriptorID;
					ChainDeleteEntry(parser->od_links, j);
					if (l2->desc_name) free(l2->desc_name);
					free(l2);
					break;
				}
			}
			return;
		}
	}
	odl = malloc(sizeof(ODLink));
	memset(odl, 0, sizeof(ODLink));
	odl->nodes = NewChain();
	odl->od = od;
	if (ID) od->objectDescriptorID = ID;
	if (name) odl->desc_name = strdup(name);
	ChainAddEntry(parser->od_links, odl);
}
void xmt_new_od_link_from_node(XMTParser *parser, char *name, SFNode *in_node)
{
	u32 i, ID;
	ODLink *odl;

	ID = 0;
	if (!strnicmp(name, "od", 2)) ID = atoi(name + 2);
	else if (!strnicmp(name, "iod", 3)) ID = atoi(name + 3);
	/*be carefull, an url like "11-regression-test.mp4" will return 1 on sscanf :)*/
	else if (sscanf(name, "%d", &ID) == 1) {
		char szURL[20];
		sprintf(szURL, "%d", ID);
		if (strcmp(szURL, name)) {
			ID = 0;
		} else {
			name = NULL;
		}
	}
	else ID = 0;
	
	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		odl = ChainGetEntry(parser->od_links, i);
		if ( (name && odl->desc_name && !stricmp(odl->desc_name, name))
			|| (ID && odl->od && odl->od->objectDescriptorID==ID)
			|| (ID && (odl->ID==ID))
			) {
			if (in_node && (ChainFindEntry(odl->nodes, in_node)<0) ) ChainAddEntry(odl->nodes, in_node);
			return;
		}
	}
	odl = malloc(sizeof(ODLink));
	memset(odl, 0, sizeof(ODLink));
	odl->nodes = NewChain();
	if (in_node) ChainAddEntry(odl->nodes, in_node);
	if (ID) odl->ID = ID;
	else odl->desc_name = strdup(name);
	ChainAddEntry(parser->od_links, odl);
}

void xmt_new_esd_link(XMTParser *parser, ESDescriptor *esd, char *desc_name, u32 ID)
{
	u32 i;
	ESDLink *esdl;

	for (i=0; i<ChainGetCount(parser->esd_links); i++) {
		esdl = ChainGetEntry(parser->esd_links, i);
		if (esd!=esdl->esd) continue;
		if (ID) esdl->ESID = esdl->esd->ESID = ID;
		if (desc_name && !esdl->desc_name) esdl->desc_name = strdup(desc_name);
		return;
	}
	esdl = malloc(sizeof(ESDLink));
	memset(esdl, 0, sizeof(ESDLink));
	esdl->esd = esd;
	if (ID) esd->ESID = esdl->ESID = ID;
	if (desc_name) esdl->desc_name = strdup(desc_name);
	ChainAddEntry(parser->esd_links, esdl);
}

u32 xmt_get_node_id(XMTParser *parser)
{
	SFNode *n;
	u32 ID;
	if (sscanf(parser->att_buffer, "N%d", &ID) == 1) {
		ID ++;
		if (parser->bifs_es->next_node_id<ID) parser->bifs_es->next_node_id = ID + 1;
		n = SG_FindNode(parser->scene_graph, ID);
		if (n) {
			u32 nID = 1 + parser->bifs_es->next_node_id;
			parser->bifs_es->next_node_id ++;
			Node_SetDEF(n, nID, n->sgprivate->NodeName);
		}
	} else {
		ID = 1 + parser->bifs_es->next_node_id;
		parser->bifs_es->next_node_id ++;
	}
	return ID;
}

#define XMT_GET_VAL \
	char value[100];	\
	u32 i;			\
	char *str = parser->temp_att;	\
	if (!str) {		\
		XMTLogErr2(parser, "Number expected", name);	\
		return;		\
	}		\
	while (str[0] == ' ') str += 1;	\
	i = 0;	\
	while ((str[i] != ' ') && str[i]) {	\
		value[i] = str[i];			\
		i++;				\
	}					\
	value[i] = 0;	\
	while ((str[i] == ' ') && str[i]) i++;	\
	if (!str[i]) parser->temp_att = NULL;	\
	else parser->temp_att = &str[i];	

void xmt_parse_int(XMTParser *parser, const char *name, SFInt32 *val)
{
	XMT_GET_VAL
	*val = atoi(value);
}
void xmt_parse_float(XMTParser *parser, const char *name, SFFloat *val)
{
	XMT_GET_VAL
	*val = (Float) atof(value);
}
void xmt_parse_time(XMTParser *parser, const char *name, SFTime *val)
{
	XMT_GET_VAL
	*val = atof(value);
}
void xmt_parse_bool(XMTParser *parser, const char *name, SFBool *val)
{
	XMT_GET_VAL
	if (!stricmp(value, "1") || !stricmp(value, "true"))
		*val = 1;
	else
		*val = 0;
}


char *xmt_translate_string(char *str)
{
	char *value;
	u32 size, i, j;
	if (!str || !strlen(str)) return NULL;
	value = malloc(sizeof(char) * 500);
	size = 500;
	i = j = 0;
	while (str[i]) {
		if (j >= size) {
			size += 500;
			value = realloc(value, sizeof(char)*size);
		}
		if (str[i] == '&') {
			if (str[i+1]=='#') {
				char szChar[20], *end;
				u16 wchar[2];
				const unsigned short *srcp;
				strncpy(szChar, str+i, 10);
				end = strchr(szChar, ';');
				assert(end);
				end[1] = 0;
				i+=strlen(szChar);
				wchar[1] = 0;
				sscanf(szChar, "&#%hd;", &wchar[0]);
				srcp = wchar;
				j += utf8_wcstombs(&value[j], 20, &srcp);
			}
			else if (!strnicmp(&str[i], "&amp;", sizeof(char)*5)) {
				value[j] = '&';
				j++;
				i+= 5;
			}
			else if (!strnicmp(&str[i], "&lt;", sizeof(char)*4)) {
				value[j] = '<';
				j++;
				i+= 4;
			}
			else if (!strnicmp(&str[i], "&gt;", sizeof(char)*4)) {
				value[j] = '>';
				j++;
				i+= 4;
			}
			else if (!strnicmp(&str[i], "&apos;", sizeof(char)*6)) {
				value[j] = '\'';
				j++;
				i+= 6;
			}
			else if (!strnicmp(&str[i], "&quot;", sizeof(char)*6)) {
				value[j] = '\"';
				j++;
				i+= 6;
			}
		} else {
			value[j] = str[i];
			j++;
			i++;
		}
	}
	value[j] = 0;
	return value;
}

void xmt_parse_string(XMTParser *parser, const char *name, SFString *val)
{
	char value[5000];
	u32 i=0;
	u32 k=0;
	char *str = parser->temp_att;
	if (!str) {
		XMTLogErr2(parser, "String expected", name);
		return;
	}
	if (strchr(str, '\'')) {
		while (str[0] != '\'')  str += 1;
		str+=1;
		while (str[i] && (str[i] != '\'')) {
			/*handle UTF-8 - WARNING: if parser is in unicode string is already utf8 multibyte chars*/
			if (!parser->unicode_type && (str[i] & 0x80)) {
				value[k] = 0xc0 | ( (str[i] >> 6) & 0x3 );
				k++;
				str[i] &= 0xbf;
			}
			value[k] = str[i];
			i++;
			k++;
		}
		value[k] = 0;
		if (!str[i+1]) parser->temp_att = NULL;
		else parser->temp_att = &str[i+1];	
	} else {
		strcpy(value, str);
		parser->temp_att = NULL;
	}
	if (val->buffer) free(val->buffer);
	val->buffer = NULL;
	if (strlen(value)) val->buffer = xmt_translate_string(value);
}

void xmt_parse_url(XMTParser *parser, const char *name, SFURL *val, SFNode *owner)
{
	char value[5000], *tmp;
	u32 i=0;
	char *str = parser->temp_att;
	if (!str) {
		XMTLogErr2(parser, "String expected", name);
		return;
	}
	if (strchr(str, '\'')) {
		while (str[0] != '\'')  str += 1;
		str+=1;
		while ((str[i] != '\'') && str[i]) {
			value[i] = str[i];
			i++;
		}
		if (!str[i+1]) parser->temp_att = NULL;
		else parser->temp_att = &str[i+1];	
		value[i] = 0;
	} else {
		strcpy(value, str);
		parser->temp_att = NULL;
	}
	if (val->url) free(val->url);
	val->url = xmt_translate_string(value);
	val->OD_ID = 0;
	tmp = strstr(value, "#");
	if (tmp) tmp[0] = 0;
	if (!strnicmp(value, "od:", 3))
		xmt_new_od_link_from_node(parser, value+3, owner);
	else
		xmt_new_od_link_from_node(parser, value, owner);
}


void xmt_parse_script(XMTParser *parser, const char *name, SFScript *val)
{
	char *value;
	char *str = parser->temp_att;
	if (!str) {
		XMTLogErr2(parser, "String expected", name);
		return;
	}
	while ((str[0] == ' ') || (str[0] == '\t') ) str += 1;

	value = xmt_translate_string(str);
	if (val->script_text) free(val->script_text);
	val->script_text = value;
	parser->temp_att = NULL;
}

void xmt_sffield(XMTParser *parser, FieldInfo *info, SFNode *n)
{
	switch (info->fieldType) {
	case FT_SFInt32:
		xmt_parse_int(parser, info->name, (SFInt32 *)info->far_ptr); 
		break;
	case FT_SFBool:
		xmt_parse_bool(parser, info->name, (SFBool *)info->far_ptr); 
		break;
	case FT_SFFloat:
		xmt_parse_float(parser, info->name, (SFFloat *)info->far_ptr); 
		break;
	case FT_SFTime:
		xmt_parse_time(parser, info->name, (SFTime *)info->far_ptr); 
		break;
	case FT_SFColor:
		xmt_parse_float(parser, info->name, & ((SFColor *)info->far_ptr)->red); 
		xmt_parse_float(parser, info->name, & ((SFColor *)info->far_ptr)->green); 
		xmt_parse_float(parser, info->name, & ((SFColor *)info->far_ptr)->blue); 
		break;
	case FT_SFVec2f:
		xmt_parse_float(parser, info->name, & ((SFVec2f *)info->far_ptr)->x); 
		xmt_parse_float(parser, info->name, & ((SFVec2f *)info->far_ptr)->y); 
		break;
	case FT_SFVec3f:
		xmt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->x); 
		xmt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->y); 
		xmt_parse_float(parser, info->name, & ((SFVec3f *)info->far_ptr)->z); 
		break;
	case FT_SFRotation:
		xmt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->xAxis); 
		xmt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->yAxis); 
		xmt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->zAxis); 
		xmt_parse_float(parser, info->name, & ((SFRotation *)info->far_ptr)->angle); 
		break;
	case FT_SFString:
		xmt_parse_string(parser, info->name, (SFString*)info->far_ptr); 
		break;
	case FT_SFURL:
		xmt_parse_url(parser, info->name, (SFURL*)info->far_ptr, n); 
		break;
	case FT_SFCommandBuffer:
	{
		SFCommandBuffer *cb = (SFCommandBuffer *)info->far_ptr;
		xmt_skip_attributes(parser);
		while (!xmt_element_done(parser, "buffer") && !parser->last_error) {
			xmt_parse_command(parser, NULL, cb->commandList);
		}
	}
		break;
	case FT_SFScript:
		xmt_parse_script(parser, info->name, (SFScript *)info->far_ptr); 
		break;

	case FT_SFImage:
	{
		u32 k, size, v;
		SFImage *img = (SFImage *)info->far_ptr;
		xmt_parse_int(parser, "width", &img->width);
		if (parser->last_error) return;
		xmt_parse_int(parser, "height", &img->height);
		if (parser->last_error) return;
		xmt_parse_int(parser, "nbComp", &v);
		if (parser->last_error) return;
		img->numComponents = v;
		size = img->width * img->height * img->numComponents;
		if (img->pixels) free(img->pixels);
		img->pixels = malloc(sizeof(char) * size);
		for (k=0; k<size; k++) {
			char *name = "pixels";
			XMT_GET_VAL
			if (strstr(value, "0x")) sscanf(value, "%x", &v);
			else sscanf(value, "%d", &v);
			switch (img->numComponents) {
			case 1:
				img->pixels[k] = (char) v;
				break;
			case 2:
				img->pixels[k] = (char) (v>>8)&0xFF;
				img->pixels[k+1] = (char) (v)&0xFF;
				k++;
				break;
			case 3:
				img->pixels[k] = (char) (v>>16)&0xFF;
				img->pixels[k+1] = (char) (v>>8)&0xFF;
				img->pixels[k+2] = (char) (v)&0xFF;
				k+=2;
				break;
			case 4:
				img->pixels[k] = (char) (v>>24)&0xFF;
				img->pixels[k+1] = (char) (v>>16)&0xFF;
				img->pixels[k+2] = (char) (v>>8)&0xFF;
				img->pixels[k+3] = (char) (v)&0xFF;
				k+=3;
				break;
			}
		}
	}
		break;

	default:
		parser->last_error = M4NotSupported;
		break;

	}
}

void xmt_mffield(XMTParser *parser, FieldInfo *info, SFNode *n)
{
	FieldInfo sfInfo;

	sfInfo.fieldType = SG_GetSFType(info->fieldType);
	sfInfo.name = info->name;
	MFField_Reset(info->far_ptr, info->fieldType);

	parser->temp_att = parser->att_buffer;
	while (parser->temp_att && !parser->last_error) {
		MFField_Append(info->far_ptr, info->fieldType, &sfInfo.far_ptr);
		xmt_sffield(parser, &sfInfo, n);
	}
}


Bool XMTCheckNDT(XMTParser *parser, FieldInfo *info, SFNode *node, SFNode *parent)
{
	u32 i, tag;

#ifdef M4_DEF_Script
	if (parent->sgprivate->tag == TAG_Script) return 1;
#endif
	tag = node->sgprivate->tag;
	if (tag == TAG_ProtoNode) {
		tag = Proto_GetRenderingTag(((ProtoInstance *)node)->proto_interface);
		/*this happens with extern protos*/
		if (tag==TAG_UndefinedNode) return 1;
	}

	for (i=0;i<LAST_BIFS_VERSION; i++) {
		if (NDT_GetNodeType(info->NDTtype, tag, i+1)) return 1;
	}
	/*not defined yet, we assume NDT will be correct*/
	if (node->sgprivate->tag == TAG_UndefinedNode) return 1;
	/*not found*/
	sprintf(parser->szMsg, "Error line %d: node %s not valid in field %s\n", parser->line, Node_GetName(node), info->name);
	XMT_REPORT_ERR(parser);
	parser->last_error = M4InvalidNode;
	Node_Unregister(node, parent);
	return 0;
}

u32 GetXMTFieldTypeByName(const char *name)
{
	if (!stricmp(name, "Boolean")) return FT_SFBool;
	else if (!stricmp(name, "Integer")) return FT_SFInt32;
	else if (!stricmp(name, "Color")) return FT_SFColor;
	else if (!stricmp(name, "Vector2")) return FT_SFVec2f;
	else if (!stricmp(name, "Image")) return FT_SFImage;
	else if (!stricmp(name, "Time")) return FT_SFTime;
	else if (!stricmp(name, "Float")) return FT_SFFloat;
	else if (!stricmp(name, "Vector3")) return FT_SFVec3f;
	else if (!stricmp(name, "Rotation")) return FT_SFRotation;
	else if (!stricmp(name, "String")) return FT_SFString;
	else if (!stricmp(name, "Node")) return FT_SFNode;
	else if (!stricmp(name, "Booleans")) return FT_MFBool;
	else if (!stricmp(name, "Integers")) return FT_MFInt32;
	else if (!stricmp(name, "Colors")) return FT_MFColor;
	else if (!stricmp(name, "Vector2s")) return FT_MFVec2f;
	else if (!stricmp(name, "Images")) return FT_MFImage;
	else if (!stricmp(name, "Times")) return FT_MFTime;
	else if (!stricmp(name, "Floats")) return FT_MFFloat;
	else if (!stricmp(name, "Vector3s")) return FT_MFVec3f;
	else if (!stricmp(name, "Rotations")) return FT_MFRotation;
	else if (!stricmp(name, "Strings")) return FT_MFString;
	else if (!stricmp(name, "Nodes")) return FT_MFNode;
	else return FT_Unknown;
}
u32 GetXMTScriptEventTypeByName(const char *name)
{
	if (!stricmp(name, "eventIn")) return SFET_EventIn;
	else if (!stricmp(name, "eventOut")) return SFET_EventOut;
	if (!stricmp(name, "field")) return SFET_Field;
	else return ET_Unknown;
}
u32 GetXMTEventTypeByName(const char *name)
{
	if (!stricmp(name, "eventIn")) return ET_EventIn;
	else if (!stricmp(name, "eventOut")) return ET_EventOut;
	if (!stricmp(name, "field")) return ET_Field;
	if (!stricmp(name, "exposedField")) return ET_ExposedField;
	else return ET_Unknown;
}

SFNode *xmt_parse_node(XMTParser *parser, char *node_name, SFNode *parent);


void xmt_parse_field_node(XMTParser *parser, SFNode *node, FieldInfo *field)
{
	char *str;
	char szType[20];
	str = xmt_get_element(parser);
	strcpy(szType, str);

	if ((field->fieldType==FT_SFNode) && stricmp(str, "node") ) {
		XMTLogErr2(parser, "Invalid SFNode field declaration: expecting \"node\" parent element", str);
		return;
	}
	else if ((field->fieldType==FT_MFNode) && stricmp(str, "nodes")) {
		XMTLogErr2(parser, "Invalid MFNode field declaration: expecting \"nodes\" parent element", str);
		return;
	}
	xmt_skip_attributes(parser);
	while (!xmt_element_done(parser, szType)) {
		SFNode *n = xmt_parse_node(parser, NULL, node);
		if (n) {
			if (field->fieldType==FT_SFNode) * ((SFNode **)field->far_ptr) = n;
			else if (field->fieldType==FT_MFNode) ChainAddEntry(*(Chain **)field->far_ptr, n);
		}
	}
}

void xmt_parse_script_field(XMTParser *parser, SFNode *node)
{
	LPSCRIPTFIELD scfield;
	FieldInfo field;
	char *str, *val;
	u32 fieldType, eventType;
	char fieldName[1024];

	fieldType = eventType = 0;
	val = NULL;
	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		if (!stricmp(str, "name")) strcpy(fieldName, parser->att_buffer); 
		else if (!stricmp(str, "type")) fieldType = GetXMTFieldTypeByName(parser->att_buffer); 
		else if (!stricmp(str, "vrml97Hint")) eventType = GetXMTScriptEventTypeByName(parser->att_buffer); 
		else if (strstr(str, "value") || strstr(str, "Value")) val = strdup(parser->att_buffer);
	}
#ifdef M4_DEF_Script
	scfield = SG_NewScriptField(node, eventType, fieldType, fieldName);
#else
	scfield = NULL;
#endif
	if (!scfield) {
		XMTLogErr1(parser, "cannot create script field - check syntax");
		return;
	}
	if (val) {
		Node_GetFieldByName(node, fieldName, &field);
		str = parser->att_buffer;
		parser->att_buffer = parser->temp_att = val;
		if (SG_IsSFField(fieldType)) {
			xmt_sffield(parser, &field, node);
		} else {
			xmt_mffield(parser, &field, node);
		}
		parser->att_buffer = str;
		free(val);
	}
	/*for SF/MF Nodes*/
	if (!xmt_element_done(parser, "field")) {
		Node_GetFieldByName(node, fieldName, &field);
		xmt_parse_field_node(parser, node, &field);
		if (parser->last_error) return;
		xmt_element_done(parser, "field");
	}
}

void xmt_parse_ised(XMTParser *parser, SFNode *node)
{
	M4Err e;
	char *str;
	LPPROTOFIELD pf;
	FieldInfo pfield, nfield;
	char szNode[1024], szProto[1024];

	while (!xmt_element_done(parser, "IS")) {
		str = xmt_get_element(parser);
		if (!stricmp(str, "connect")) {
			while (xmt_has_attributes(parser)) {
				str = xmt_get_attribute(parser);
				if (!stricmp(str, "nodeField")) strcpy(szNode, parser->att_buffer);
				else if (!stricmp(str, "protoField")) strcpy(szProto, parser->att_buffer);
			}
			xmt_element_done(parser, "connect");
			e = Node_GetFieldByName(node, szNode, &nfield);
			if (e) {
				XMTLogErr2(parser, "Unknown node field", szNode);
				return;
			}
			pf = Proto_FindFieldByName(parser->parsing_proto, szProto);
			if (!pf) {
				XMTLogErr2(parser, "Unknown proto field", szProto);
				return;
			}
			ProtoField_GetField(pf, &pfield);
			e = Proto_SetISField(parser->parsing_proto, pfield.allIndex, node, nfield.allIndex);
			if (e) XMTLogErr2(parser, "Cannot set IS field", M4ErrToString(e));
		}
		else xmt_skip_element(parser, str);
	}
}


SFNode *xmt_proto_instance(XMTParser *parser, SFNode *parent)
{
	u32 ID;
	LPPROTO p;
	char szDEFName[1024], szProtoName[1024];
	Bool isUSE;
	SFNode *node;
	FieldInfo info;
	LPSCENEGRAPH sg;
	M4Err e;
	char *str;

	p = NULL;

	isUSE = 0;
	szDEFName[0] = 0;
	szProtoName[0] = 0;
	ID = 0;
	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		/*DEF node*/
		if (!stricmp(str, "DEF")) strcpy(szDEFName, parser->att_buffer);
		/*USE node*/
		else if (!stricmp(str, "USE")) {
			strcpy(szDEFName, parser->att_buffer);
			isUSE = 1;
		}
		/*proto name*/
		else if (!stricmp(str, "name")) strcpy(szProtoName, parser->att_buffer);
	}

	sg = parser->scene_graph;
	while (1) {
		p = SG_FindProto(sg, parser->bifs_es->next_proto_id + 1, szProtoName);
		if (p) break;
		sg = sg->parent_scene;
		if (!sg) break;
	}
	if (!p) {
		XMTLogErr2(parser, "not a valid/supported proto", szProtoName);
		return NULL;
	}

	if (isUSE) {
		node = SG_FindNodeByName(parser->scene_graph, parser->att_buffer);
		if (node && (node->sgprivate->NodeID > parser->bifs_es->max_node_id))
			parser->bifs_es->max_node_id = node->sgprivate->NodeID;

		if (!node) {
			/*create a temp node (undefined)*/
			node = Proto_CreateInstance(parser->scene_graph, p);
			str = parser->att_buffer;
			parser->att_buffer = szDEFName;
			ID = xmt_get_node_id(parser);
			Node_SetDEF(node, ID, str);
			parser->att_buffer = str;
		}
		Node_Register(node, parent);
		if (!xmt_element_done(parser, NULL)) {
			XMTLogErr1(parser, "Too many attributes - only USE=\"ID\" allowed");
		}
		return node;
	}

	if (strlen(szDEFName)) {
		node = SG_FindNodeByName(parser->scene_graph, szDEFName);
		if (node) {
			ID = node->sgprivate->NodeID;
		} else {
			ID = xmt_get_node_id(parser);
			node = Proto_CreateInstance(parser->scene_graph, p);
			Node_SetDEF(node, ID, szDEFName);
		}
	} else {
		node = Proto_CreateInstance(parser->scene_graph, p);
	}
	Node_Register(node, parent);

	while (!xmt_element_done(parser, "ProtoInstance")) {
		str = xmt_get_element(parser);
		if (!stricmp(str, "fieldValue")) {
			char szField[1024];
			char *szVal;
			szVal = NULL;
			while (xmt_has_attributes(parser)) {
				str = xmt_get_attribute(parser);
				if (!stricmp(str, "name")) strcpy(szField, parser->att_buffer);
				else if (strstr(str, "value") || strstr(str, "Value")) szVal = strdup(parser->att_buffer);
			}
			e = Node_GetFieldByName(node, szField, &info);
			if (e) {
				XMTLogErr2(parser, "Unknown proto field", szField);
				goto err_exit;
			}
			if (szVal) {
				str = parser->att_buffer;
				parser->att_buffer = parser->temp_att = szVal;
				if (SG_IsSFField(info.fieldType)) {
					xmt_sffield(parser, &info, node);
				} else {
					xmt_mffield(parser, &info, node);
				}
				parser->att_buffer = str;
				free(szVal);
			} else {
				xmt_parse_field_node(parser, node, &info);
				if (parser->last_error) goto err_exit;
			}
			xmt_element_done(parser, "fieldValue");
		}
		else xmt_skip_element(parser, str);
	}

	if (!parser->parsing_proto) Node_Init(node);
	return node;

err_exit:
	Node_Unregister(node, parent);
	return NULL;
}

Bool XMT_HasBeenDEF(XMTParser *parser, char *node_name)
{
	u32 i, count;
	count = ChainGetCount(parser->def_nodes);
	for (i=0; i<count; i++) {
		SFNode *n = ChainGetEntry(parser->def_nodes, i);
		if (!strcmp(n->sgprivate->NodeName, node_name)) return 1;
	}
	return 0;
}

SFNode *xmt_parse_node(XMTParser *parser, char *node_name, SFNode *parent)
{
	u32 tag, ID;
	Bool is_script, register_def;
	SFNode *node, *undef_node;
	FieldInfo info;
	char nName[100];
	char * str;

	if (node_name) {
		str = node_name;
	} else {
		str = xmt_get_element(parser);
	}
	ID = 0;
	if (!stricmp(str, "ProtoInstance")) return xmt_proto_instance(parser, parent);
	if (!stricmp(str, "NULL")) return NULL;
	tag = Node_GetTagByName(str);
	if (!tag) {
		sprintf(parser->szMsg, "Warning (line %d): \"%s\" not a valid/supported node\n", parser->line, str);
		XMT_REPORT_ERR(parser);
		xmt_skip_element(parser, str);
		return NULL;
	}
	strcpy(nName, str);
	node = SG_NewNode(parser->scene_graph, tag);
//	printf("create node %s\n", nName);
	
	is_script = 0;
#ifdef M4_DEF_Script
	if (tag==TAG_Script) is_script = 1;
#endif

	if (!node) {
		parser->last_error = M4UnknownNode;
		return NULL;
	}

	register_def = 0;
	/*parse all attributes*/
	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		/*DEF node*/
		if (!stricmp(str, "DEF")) {
			register_def = 1;
			undef_node = SG_FindNodeByName(parser->scene_graph, parser->att_buffer);
			if (undef_node) {
				ID = undef_node->sgprivate->NodeID;
				/*if we see twice a DEF N1 then force creation of a new node*/
				if (XMT_HasBeenDEF(parser, parser->att_buffer)) {
					undef_node = NULL;
				}
			} else {
				ID = xmt_get_node_id(parser);
			}
			Node_SetDEF(node, ID, parser->att_buffer);
			if (undef_node) Node_ReplaceAllInstances(undef_node, node, 0);
		}
		/*USE node*/
		else if (!stricmp(str, "USE")) {
			node = SG_FindNodeByName(parser->scene_graph, parser->att_buffer);
			if (node && (node->sgprivate->NodeID > parser->bifs_es->max_node_id))
				parser->bifs_es->max_node_id = node->sgprivate->NodeID;

			if (!node) {
				/*create a temp node (undefined)*/
				node = SG_NewNode(parser->scene_graph, tag);
				ID = xmt_get_node_id(parser);
				Node_SetDEF(node, ID, str);
			}
			Node_Register(node, parent);
			if (!xmt_element_done(parser, NULL)) {
				XMTLogErr1(parser, "Too many attributes - only USE=\"ID\" allowed");
				parser->last_error = M4BadParam;
			}
			return node;
		}
		/*all other fields*/
		else {
			parser->last_error = Node_GetFieldByName(node, str, &info);
			if (parser->last_error) {
				XMTLogErr2(parser, "Unknown/Invalid node field", str);
				Node_Register(node, parent);
				goto err;
			}
			if (SG_IsSFField(info.fieldType)) {
				parser->temp_att = parser->att_buffer;
				xmt_sffield(parser, &info, node);
			} else {
				xmt_mffield(parser, &info, node);
			}
		}
	}
	Node_Register(node, parent);

	if (register_def) ChainAddEntry(parser->def_nodes, node);

	/*sub-nodes or complex att*/
	while (!xmt_element_done(parser, nName)) {
		char subName[100];

		if (parser->last_error) goto err;
		
		str = xmt_get_element(parser);

		if (is_script && !stricmp(str, "field")) {
			xmt_parse_script_field(parser, node);
			continue;
		}
		else if (parser->parsing_proto && !stricmp(str, "IS")) {
			xmt_skip_attributes(parser);
			xmt_parse_ised(parser, node);
			continue;
		}

		parser->last_error = Node_GetFieldByName(node, str, &info);
		if (parser->last_error) {
			xmt_skip_element(parser, str);
			parser->last_error = M4OK;
			continue;
		}
		strcpy(subName, str);
		if (SG_GetSFType(info.fieldType) != FT_SFNode) {
			if (SG_IsSFField(info.fieldType)) {
				parser->temp_att = parser->att_buffer;
				xmt_sffield(parser, &info, node);
			} else {
				xmt_mffield(parser, &info, node);
			}
		} else {
			xmt_skip_attributes(parser);
			while (!xmt_element_done(parser, subName) && !parser->last_error) {
				SFNode *n = xmt_parse_node(parser, NULL, node);
				if (n) {
					switch (info.fieldType) {
					case FT_SFNode:
						if (!XMTCheckNDT(parser, &info, n, node)) goto err;
						* ((SFNode **)info.far_ptr) = n;
						break;
					case FT_MFNode:
						if (!XMTCheckNDT(parser, &info, n, node)) goto err;
						ChainAddEntry(*(Chain **)info.far_ptr, n);
						break;
					}
				}
			}
		}
	}
	if (!parser->parsing_proto) Node_Init(node);
	return node;

err:
	Node_Unregister(node, parent);
	return NULL;
}


Descriptor *xmt_parse_descriptor(XMTParser *parser, char *name, Descriptor *the_desc);

void xmt_parse_descr_field(XMTParser *parser, Descriptor *desc, char *name, char *value)
{
	char field[1024];
	u32 type;
	Descriptor *subdesc;
	/*regular field*/
	if (value) {
		parser->last_error = OD_SetDescriptorField(desc, name, parser->att_buffer);
	} 
	/*descriptor field*/
	else {
		strcpy(field, name);
		type = OD_DescriptorFieldType(desc, name);
		switch (type) {
		/*desc list*/
		case 2:
			xmt_skip_attributes(parser);
			while (!xmt_element_done(parser, field)) {
				subdesc = xmt_parse_descriptor(parser, NULL, NULL);
				if (!subdesc) break;
				OD_AddDescToDesc(desc, subdesc);
			}
			break;
		/*desc*/
		case 1:
			if (!stricmp(field, "StreamSource")) {
				subdesc = OD_NewDescriptor(MuxInfoDescriptor_Tag);
				subdesc = xmt_parse_descriptor(parser, field, subdesc);
			} else {
				xmt_skip_attributes(parser);
				subdesc = xmt_parse_descriptor(parser, NULL, NULL);
			}
			if (!subdesc) break;
			OD_AddDescToDesc(desc, subdesc);
			xmt_element_done(parser, field);
			break;
		}
	}
	if (parser->last_error) {
		sprintf(parser->szMsg, "Error line %d: unknown field %s\n", parser->line, name);
		XMT_REPORT_ERR(parser);
	}
}

Descriptor *xmt_parse_descriptor(XMTParser *parser, char *name, Descriptor *the_desc)
{
	char *str, desc_name[1024];
	Descriptor *desc;
	u8 tag;

	if (!the_desc) {
		if (name) {
			strcpy(desc_name,  name);
		} else {
			str = xmt_get_element(parser);
			if (!str) return NULL;
			strcpy(desc_name, str);
		}
		tag = OD_GetDescriptorTag(desc_name);
		if (!tag) {
			xmt_skip_element(parser, desc_name);
			return xmt_parse_descriptor(parser, name, NULL);
		}
		desc = OD_NewDescriptor(tag);

		if (!desc) return NULL;
	} else {
		strcpy(desc_name,  name);
		desc = the_desc;
	}

	/*parse attributes*/
	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		if (!stricmp(str, "objectDescriptorID")) {
			xmt_new_od_link(parser, (ObjectDescriptor *) desc, parser->att_buffer);
		} else if (!stricmp(str, "binaryID")) {
			if (desc->tag==ESDescriptor_Tag) {
				xmt_new_esd_link(parser, (ESDescriptor *) desc, NULL, atoi(parser->att_buffer));
			} else {
				xmt_new_od_link(parser, (ObjectDescriptor *) desc, parser->att_buffer);
			}
		} else if (!stricmp(str, "ES_ID")) {
			xmt_new_esd_link(parser, (ESDescriptor *) desc, parser->att_buffer, 0);
		} else {
			if (!stricmp(str, "value")) {
				xmt_parse_descr_field(parser, desc, name, parser->att_buffer);
			} else {
				xmt_parse_descr_field(parser, desc, str, parser->att_buffer);
			}
		}
		if (parser->last_error) {
			OD_DeleteDescriptor(&desc);
			return NULL;
		}
	}
	/*parse sub desc - WARNING: XMT defines some properties as elements in OD and also introduces dummy containers!!!*/
	while (!xmt_element_done(parser, desc_name) && !parser->last_error) {
		str = xmt_get_element(parser);
		/*special cases in IOD*/
		if (!stricmp(str, "Profiles")) xmt_parse_descriptor(parser, "Profiles", desc);
		/*special cases in OD/IOD*/
		else if (!stricmp(str, "Descr")) {
			xmt_skip_attributes(parser);
			while (!xmt_element_done(parser, "Descr") && !parser->last_error) {
				str = xmt_get_element(parser);
				if (str) xmt_parse_descr_field(parser, desc, str, NULL);
			}
		}
		/*special cases in BIFS config*/
		else if (!stricmp(str, "commandStream")) {
			((BIFSConfigDescriptor *)desc)->isCommandStream = 1;
			xmt_parse_descriptor(parser, "commandStream", desc);
		}
		/*special cases OD URL*/
		else if (!stricmp(str, "URL")) xmt_parse_descriptor(parser, "URL", desc);
		else if (!stricmp(str, "size")) xmt_parse_descriptor(parser, "size", desc);
		else if (!stricmp(str, "predefined")) xmt_parse_descriptor(parser, "predefined", desc);
		else if (!stricmp(str, "custom")) xmt_parse_descriptor(parser, "custom", desc);
		else if (!stricmp(str, "MP4MuxHints")) xmt_parse_descriptor(parser, "MP4MuxHints", desc);
		/*all other desc*/
		else {
			xmt_parse_descr_field(parser, desc, str, NULL);
		}
	}
	if (desc->tag == BIFSConfig_Tag) {
		BIFSConfigDescriptor *bcfg = (BIFSConfigDescriptor *)desc;
		parser->pixelMetrics = bcfg->pixelMetrics;
		parser->bifs_w = bcfg->pixelWidth;
		parser->bifs_h = bcfg->pixelHeight;
		/*for xmt->bt*/
		if (!bcfg->isCommandStream) bcfg->isCommandStream = 1;
		if (!bcfg->version) bcfg->version = 1;
	}
	else if (desc->tag==ESDescriptor_Tag) {
		ESDescriptor *esd  =(ESDescriptor*)desc;
		if (esd->decoderConfig) {
			switch (esd->decoderConfig->streamType) {
			case M4ST_BIFS:
			case M4ST_OD:
			{
				M4StreamContext *sc = M4SM_NewStream(parser->ctx, esd->ESID, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication);
				/*set default timescale for systems tracks (ignored for other)*/
				if (sc) sc->timeScale = (esd->slConfig && esd->slConfig->timestampResolution) ? esd->slConfig->timestampResolution : 1000;
				if (!parser->base_bifs_id && (esd->decoderConfig->streamType==M4ST_BIFS)) parser->base_bifs_id = esd->ESID;
				if (!parser->base_od_id && (esd->decoderConfig->streamType==M4ST_OD)) parser->base_od_id = esd->ESID;
			}
				break;
			}
		}
	}
	return desc;
}

/*
	locate node, if not defined yet parse ahead in current AU
*/
SFNode *xmt_peek_node(XMTParser *parser, char *defID)
{
	SFNode *n;
	u32 tag, ID;
	char *str, *prev_str;
	char nName[1000], elt[1000];
	u32 pos, line, line_pos, i;
	
	n = SG_FindNodeByName(parser->scene_graph, defID);
	if (n) {
		if (n->sgprivate->NodeID > parser->bifs_es->max_node_id) 
			parser->bifs_es->max_node_id = n->sgprivate->NodeID;
		return n;
	}

	pos = parser->line_start_pos;
	line_pos = parser->line_pos;
	line = parser->line;
	strcpy(nName, defID);

	/*this is ugly, assumes DEF attribute is in the same text line than element...*/
	xmt_skip_attributes(parser);
	n = NULL;
	while (!parser->done) {
next_line:
		str = parser->line_buffer;
		prev_str = str;
		while (str) {
			str = strstr(str, "DEF=\"");
			if (!str) {
				parser->line_pos = parser->line_size;
				xmt_check_line(parser);
				goto next_line;
			}
			if (!strnicmp(str+5, nName, strlen(nName)) ) break;
			str += 5;
			prev_str = str;
		}
		str = strrchr(prev_str, '<');
		if (!str || !stricmp(str+1, "ROUTE")) {
			parser->line_pos = parser->line_size;
			xmt_check_line(parser);
			continue;
		}
		str+=1;
		i = 0;
		while ((str[i] != ' ') && str[i]) {
			elt[i] = str[i];
			i++;
		}
		elt[i] = 0;
		if (!stricmp(elt, "ProtoInstance")) {
			str = strstr(str, "name=\"");
			if (!str) break;
			i = 0;
			str += 6;
			while ((str[i] != '\"') && str[i]) {
				elt[i] = str[i];
				i++;
			}
			elt[i] = 0;
		}
		tag = Node_GetTagByName(elt);
		if (!tag) {
			LPPROTO p;
			LPSCENEGRAPH sg = parser->scene_graph;
			while (1) {
				p = SG_FindProto(sg, parser->bifs_es->next_proto_id + 1, elt);
				if (p) break;
				sg = sg->parent_scene;
				if (!sg) break;
			}
			if (!p) {
				/*locate proto*/
				XMTLogErr2(parser, "not a valid/supported node", elt);
				parser->last_error = M4InvalidNode;
				return NULL;
			}
			n = Proto_CreateInstance(parser->scene_graph, p);
		} else {
			n = XMTNewNode(parser, tag);
		}
		strcpy(parser->att_buffer, nName);
		ID = xmt_get_node_id(parser);
		Node_SetDEF(n, ID, nName);
		/*NO REGISTER on peek*/
		break;
	}
	/*restore context*/
	fseek(parser->input, pos, SEEK_SET);
	parser->line_pos = parser->line_size;
	xmt_check_line(parser);
	parser->line = line;
	parser->line_pos = line_pos;
	return n;
}


void xmt_parse_route(XMTParser *parser, SGCommand *com)
{
	LPROUTE r;
	char *str, toN[1000], toNF[1000], fromN[1000], fromNF[1000], ID[1000];
	SFNode *orig, *dest;
	u32 rID;
	FieldInfo orig_field, dest_field;

	ID[0] = toN[0] = toNF[0] = fromN[0] = fromNF[0] = 0;

	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		if (!stricmp(str, "fromNode")) strcpy(fromN, parser->att_buffer);
		else if (!stricmp(str, "fromField")) strcpy(fromNF, parser->att_buffer);
		else if (!stricmp(str, "toNode")) strcpy(toN, parser->att_buffer);
		else if (!stricmp(str, "toField")) strcpy(toNF, parser->att_buffer);
		else if (!stricmp(str, "DEF")) strcpy(ID, parser->att_buffer);
	}
	xmt_element_done(parser, "ROUTE");

	orig = xmt_peek_node(parser, fromN);
	if (!orig) {
		XMTLogErr2(parser, "Cannot find node", fromN);
		return;
	}
	if (Node_GetFieldByName(orig, fromNF, &orig_field) != M4OK) {
		XMTLogErr2(parser, "Invalid node field", fromNF);
		return;
	}
	dest = xmt_peek_node(parser, toN);
	if (!dest) {
		XMTLogErr2(parser, "Cannot find node", toN);
		return;
	}
	if (Node_GetFieldByName(dest, toNF, &dest_field) != M4OK) {
		XMTLogErr2(parser, "Invalid node field", toNF);
		return;
	}
	rID = 0;
	if (strlen(ID)) {
		if (ID[0]=='R') rID = 1 + atoi(&ID[1]);
		if (rID) {
			/*fixme - check we're not conflicting with existing route*/
			if (parser->bifs_es->max_route_id < rID) parser->bifs_es->max_route_id = rID;
		} else {
			rID = 1 + parser->bifs_es->next_route_id;
			parser->bifs_es->next_route_id++;
		}
	}
	if (com) {
		/*for insert command*/
		if (rID) {
			com->RouteID = rID;
			com->def_name = strdup(ID);
		}
		com->fromNodeID = orig->sgprivate->NodeID;
		com->fromFieldIndex = orig_field.allIndex;
		com->toNodeID = dest->sgprivate->NodeID;
		com->toFieldIndex = dest_field.allIndex;
		return;
	}
	r = SG_NewRoute(parser->scene_graph, orig, orig_field.allIndex, dest, dest_field.allIndex);
	if (rID) {
		Route_SetID(r, rID);
		Route_SetName(r, ID);
	}
}

u32 xmt_get_route(XMTParser *parser, char *name) 
{
	u32 i;
	LPROUTE r = SG_FindRouteByName(parser->scene_graph, name);
	if (r) return r->ID;
	for (i=0; i<ChainGetCount(parser->inserted_routes); i++) {
		SGCommand *com = ChainGetEntry(parser->inserted_routes, i);
		if (com->def_name && !stricmp(com->def_name, name)) {
			ChainDeleteEntry(parser->inserted_routes, i);
			return com->RouteID;
		}
	}
	return 0;
}

void xmt_resolve_routes(XMTParser *parser)
{
	SGCommand *com;
	/*resolve all commands*/
	while(ChainGetCount(parser->unresolved_routes) ) {
		com = ChainGetEntry(parser->unresolved_routes, 0);
		ChainDeleteEntry(parser->unresolved_routes, 0);
		switch (com->tag) {
		case SG_RouteDelete:
		case SG_RouteReplace:
			com->RouteID = xmt_get_route(parser, com->unres_name);
			if (!com->RouteID) {
				XMTLogErr2(parser, "Cannot resolve Route DEF", com->unres_name);
				parser->last_error = M4BadParam;
			}
			free(com->unres_name);
			com->unres_name = NULL;
			com->unresolved = 0;
			break;
		}
	}
	while (ChainGetCount(parser->inserted_routes)) ChainDeleteEntry(parser->inserted_routes, 0);
}

u32 xmt_get_od_id(XMTParser *parser, char *od_name)
{
	u32 i, ID;
	if (sscanf(od_name, "%d", &ID)==1) return ID;

	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		ODLink *l = ChainGetEntry(parser->od_links, i);
		if (!l->od) continue;
		if (l->desc_name && !stricmp(l->desc_name, od_name)) return l->od->objectDescriptorID;
	}
	return 0;
}

u32 xmt_get_esd_id(XMTParser *parser, char *esd_name)
{
	u32 i, ID;
	if (sscanf(esd_name, "%d", &ID)==1) return ID;

	for (i=0; i<ChainGetCount(parser->esd_links); i++) {
		ESDLink *l = ChainGetEntry(parser->esd_links, i);
		if (!l->esd) continue;
		if (l->desc_name && !stricmp(l->desc_name, esd_name)) return l->esd->ESID;
	}
	return 0;
}

void xmt_parse_proto(XMTParser *parser, Chain *proto_list)
{
	FieldInfo info;
	LPPROTO proto, prevproto;
	LPPROTOFIELD pfield;
	LPSCENEGRAPH sg;
	char szName[1024];
	char *str, *extURL;
	u32 ID;

	extURL = NULL;
	while (xmt_has_attributes(parser)) {
		str = xmt_get_attribute(parser);
		if (!stricmp(str, "name")) strcpy(szName, parser->att_buffer);
		else if (!stricmp(str, "protoID")) ID = atoi(parser->att_buffer);
		else if (!stricmp(str, "locations")) extURL = strdup(parser->att_buffer);
	}
	proto = SG_NewProto(parser->scene_graph, parser->bifs_es->next_proto_id, szName, proto_list ? 1 : 0);
	if (proto_list) ChainAddEntry(proto_list, proto);
	parser->bifs_es->next_proto_id++;

	prevproto = parser->parsing_proto;
	sg = parser->scene_graph;
	parser->parsing_proto = proto;
	parser->scene_graph = Proto_GetSceneGraph(proto);

	/*parse all fields and proto code*/
	while (!xmt_element_done(parser, "ProtoDeclare")) {
		str = xmt_get_element(parser);
		/*proto field*/
		if (!stricmp(str, "field")) {
			char *szVal;
			u32 fType, eType;
			char szFieldName[1024];
			szVal = NULL;
			fType = eType = 0;
			while (xmt_has_attributes(parser)) {
				str = xmt_get_attribute(parser);
				if (!stricmp(str, "name")) strcpy(szFieldName, parser->att_buffer);
				else if (!stricmp(str, "type")) fType = GetXMTFieldTypeByName(parser->att_buffer);
				else if (!stricmp(str, "vrml97Hint")) eType = GetXMTEventTypeByName(parser->att_buffer);
				else if (strstr(str, "value") || strstr(str, "Value")) szVal = strdup(parser->att_buffer);
			}
			pfield = Proto_NewField(proto, fType, eType, szFieldName);
			if (szVal) {
				ProtoField_GetField(pfield, &info);
				str = parser->att_buffer;
				parser->temp_att = parser->att_buffer = szVal;
				if (SG_IsSFField(fType)) {
					xmt_sffield(parser, &info, NULL);
				} else {
					xmt_mffield(parser, &info, NULL);
				}
				parser->att_buffer = str;
				free(szVal);
			} else if (SG_GetSFType(fType)==FT_SFNode) {
				xmt_parse_field_node(parser, NULL, &info);
			}
			xmt_element_done(parser, "field");
		}
		/*sub proto*/
		else if (!stricmp(str, "ProtoDeclare")) {
			xmt_parse_proto(parser, NULL);
		}
		/*route*/
		else if (!stricmp(str, "ROUTE")) {
			xmt_parse_route(parser, NULL);
		}
		/*proto code*/
		else {
			SFNode *n = xmt_parse_node(parser, str, NULL);
			Proto_AddNodeCode(proto, n);
		}
	}

	if (extURL) {
		str = parser->att_buffer;
		parser->temp_att = parser->att_buffer = extURL;
		memset(&info, 0, sizeof(FieldInfo));
		info.fieldType = FT_MFURL;
		info.far_ptr = &proto->ExternProto;
		info.name = "ExternURL";
		xmt_mffield(parser, &info, NULL);
		parser->att_buffer = str;
		free(extURL);
	}

	xmt_resolve_routes(parser);
	parser->scene_graph = sg;
	parser->parsing_proto = prevproto;
}


static void xmt_set_com_node(SGCommand *com, SFNode *node)
{
	com->node = node;
	Node_Register(com->node, NULL);
}

void xmt_parse_command(XMTParser *parser, char *name, Chain *com_list)
{
	char *str;
	CommandFieldInfo *field;
	SGCommand *com;
	u32 com_type;
	FieldInfo inf;
	Bool hasField;
	s32 pos;
	char comName[50], fName[200], fVal[2000], extType[50];
	if (name) {
		strcpy(comName, name);
	} else {
		strcpy(comName, xmt_get_element(parser));
	}

	/*BIFS commands*/
	if (!stricmp(comName, "Replace") || !stricmp(comName, "Insert") || !stricmp(comName, "Delete") ) {
		com = SG_NewCommand(SG_UNDEFINED);

		if (!parser->stream_id) parser->stream_id = parser->base_bifs_id;

		if (!com_list) {
			if (!parser->bifs_es || (parser->bifs_es->ESID != parser->stream_id)) {
				M4StreamContext *prev = parser->bifs_es;
				parser->bifs_es = M4SM_NewStream(parser->ctx, (u16) parser->stream_id, M4ST_BIFS, 0);
				/*force new AU if stream changed*/
				if (parser->bifs_es != prev) parser->bifs_au = NULL;
			}
			if (!parser->bifs_au) parser->bifs_au = M4SM_NewAU(parser->bifs_es, 0, parser->au_time, parser->au_is_rap);
		}

		com_type = 0;
		if (!stricmp(comName, "Replace")) com_type = 0;
		else if (!stricmp(comName, "Insert")) com_type = 1;
		else if (!stricmp(comName, "Delete")) com_type = 2;

		/*parse attributes*/
		hasField = 0;
		/*default is eNd*/
		pos = -2;
		extType[0] = 0;
		while (xmt_has_attributes(parser)) {
			str = xmt_get_attribute(parser);
			if (!stricmp(str, "atNode")) {
				SFNode *atNode = xmt_peek_node(parser, parser->att_buffer);
				xmt_set_com_node(com, atNode);
			}
			else if (!stricmp(str, "atField")) {
				strcpy(fName, parser->att_buffer);
				hasField = 1;
			}
			else if (!stricmp(str, "position")) {
				if (!stricmp(parser->att_buffer, "BEGIN")) pos = 0;
				else if (!stricmp(parser->att_buffer, "END")) pos = -1;
				else pos = atoi(parser->att_buffer);
			}
			else if (!stricmp(str, "value")) strcpy(fVal, parser->att_buffer);
			else if (!stricmp(str, "atRoute")) {
				LPROUTE r = SG_FindRouteByName(parser->scene_graph, parser->att_buffer);
				if (!r) com->unres_name = strdup(parser->att_buffer);
				else {
					com->RouteID = r->ID;
					/*for bt<->xmt conversions*/
					com->def_name = strdup(parser->att_buffer);
				}
			}
			else if (!stricmp(str, "extended")) strcpy(extType, parser->att_buffer);

		}
		
		if (strlen(extType)) {
			if (!stricmp(extType, "globalQuant")) com->tag = SG_GlobalQuantizer;
			else if (!stricmp(extType, "fields")) com->tag = SG_MultipleReplace;
			else if (!stricmp(extType, "indices")) com->tag = SG_MultipleIndexedReplace;
			else if (!stricmp(extType, "deleteOrder")) com->tag = SG_NodeDeleteEx;
			else if (!stricmp(extType, "allProtos")) com->tag = SG_ProtoDeleteAll;
			else if (!stricmp(extType, "proto") || !stricmp(extType, "protos")) {
				if (com_type == 1) {
					com->tag = SG_ProtoInsert;
					com->new_proto_list = NewChain();
				} else {
					MFInt32 *IDs = SG_NewFieldPointer(FT_MFInt32);
					inf.fieldType = FT_MFInt32;
					inf.far_ptr = IDs;
					inf.name = "ProtoIDs";
					str = parser->att_buffer;
					parser->att_buffer = parser->temp_att = fVal;
					xmt_mffield(parser, &inf, NULL);
					parser->att_buffer = str;
					com->tag = SG_ProtoDelete;
					com->del_proto_list = IDs->vals;
					com->del_proto_list_size = IDs->count;
					free(IDs);
				}
			}
		} else {
			switch (com_type) {
			case 0:
				if (com->node) {
					if (hasField && (pos > -2)) com->tag = SG_IndexedReplace;
					else if (hasField) com->tag = SG_FieldReplace;
					else com->tag = SG_NodeReplace;
				} else if (com->RouteID || (com->unres_name && strlen(com->unres_name)) ) {
					com->tag = SG_RouteReplace;
					if (!com->RouteID) {
						com->unresolved = 1;
						ChainAddEntry(parser->unresolved_routes, com);
					}
				} else {
					com->tag = SG_SceneReplace;
				}
				break;
			case 1:
				if (com->node) {
					if (hasField) com->tag = SG_IndexedInsert;
					else com->tag = SG_NodeInsert;
				} else {
					com->tag = SG_RouteInsert;
				}
				break;
			case 2:
				if (com->node) {
					if (hasField) com->tag = SG_IndexedDelete;
					else com->tag = SG_NodeDelete;
				} else {
					com->tag = SG_RouteDelete;
					if (!com->RouteID) {
						com->unresolved = 1;
						ChainAddEntry(parser->unresolved_routes, com);
					}
				}
				break;
			}
		}

		field = NULL;
		if (com->node) {
			if (Node_GetFieldByName(com->node, fName, &inf) != M4OK) {
			}
			/*simple commands*/
			if (hasField && !strlen(extType)) {
				field = SG_NewFieldCommand(com);
				field->fieldType = inf.fieldType;
				field->fieldIndex = inf.allIndex;
				if (SG_GetSFType(inf.fieldType) != FT_SFNode) {
					if (pos==-2) {
						str = parser->att_buffer;
						parser->att_buffer = fVal;
		
						field->field_ptr = SG_NewFieldPointer(inf.fieldType);
						inf.far_ptr = field->field_ptr;
						if (SG_IsSFField(inf.fieldType)) {
							parser->temp_att = parser->att_buffer;
							xmt_sffield(parser, &inf, com->node);
						} else {
							xmt_mffield(parser, &inf, com->node);
						}
						parser->att_buffer = str;
					} else {
						field->fieldType = inf.fieldType = SG_GetSFType(inf.fieldType);
						field->pos = pos;
						if (com->tag != SG_IndexedDelete) {
							str = parser->att_buffer;
							parser->att_buffer = fVal;

							field->field_ptr = SG_NewFieldPointer(inf.fieldType);
							inf.far_ptr = field->field_ptr;
							parser->temp_att = parser->att_buffer;
							xmt_sffield(parser, &inf, com->node);
							
							parser->att_buffer = str;
						}
					}
				} else {
					field->fieldType = FT_SFNode;
					field->new_node = xmt_parse_node(parser, NULL, com->node);
					field->field_ptr = &field->new_node;
				}
			}
		}

		/*parse elements*/
		while (!xmt_element_done(parser, comName) && !parser->last_error) {
			str = xmt_get_element(parser);
			/*note that we register nodes*/
			switch (com->tag) {
			case SG_SceneReplace:
				if (stricmp(str, "Scene")) {
					XMTLogErr2(parser, "Unexpected symbol in scene replace", str);
					goto err;
				}
				/*if we have a previous scene*/
				xmt_resolve_routes(parser);
				while (ChainGetCount(parser->def_nodes)) ChainDeleteEntry(parser->def_nodes, 0);
				/*default scene replace*/
				parser->scene_graph = com->graph = NewSceneGraph(parser->XMTNodeInit, parser->xmt_cbck, NULL, NULL);
				SG_SetSizeInfo(parser->scene_graph, parser->bifs_w, parser->bifs_h, parser->pixelMetrics);

				parser->bifs_es->scene_context = parser->scene_graph;
				while (xmt_has_attributes(parser)) {
					str = xmt_get_attribute(parser);
					com->use_names = 0;
					if (!stricmp(str, "USENAMES") && !stricmp(parser->att_buffer, "true")) com->use_names = 1;
				}
				while (!xmt_element_done(parser, "Scene") && !parser->last_error) {
					str = xmt_get_element(parser);
					/*NULL*/
					if (!stricmp(str, "NULL")) {
					}
					/*proto */
					else if (!stricmp(str, "ProtoDeclare")) {
						xmt_parse_proto(parser, NULL);
					}
					/*route*/
					else if (!stricmp(str, "ROUTE")) {
						xmt_parse_route(parser, NULL);
					}
					/*top node*/
					else {
						SFNode *n = xmt_parse_node(parser, str, NULL);
						if (n) {
							SG_SetRootNode(parser->scene_graph, n);
						}
					}
				}
				break;
			case SG_NodeReplace:
				field = SG_NewFieldCommand(com);
				field->fieldType = FT_SFNode;
				if (!stricmp(str, "NULL")) {
					field->new_node = NULL;
				} else {
					field->new_node = xmt_parse_node(parser, str, com->node);
				}
				field->field_ptr = &field->new_node;
				break;
			case SG_IndexedReplace:
				assert(field);
				field->fieldType = FT_SFNode;
				field->pos = pos;
				if (!stricmp(str, "NULL")) {
					field->new_node = NULL;
				} else {
					field->new_node = xmt_parse_node(parser, str, com->node);
				}
				field->field_ptr = &field->new_node;
				break;
			case SG_NodeInsert:
				field = SG_NewFieldCommand(com);
				/*fall through*/
			case SG_IndexedInsert:
				field->fieldType = FT_SFNode;
				if (!stricmp(str, "NULL")) {
					field->new_node = NULL;
				} else {
					field->new_node = xmt_parse_node(parser, str, com->node);
				}
				field->field_ptr = &field->new_node;
				field->pos = pos;
				break;
			case SG_RouteReplace:
				xmt_parse_route(parser, com);
				break;
			case SG_RouteInsert:
				xmt_parse_route(parser, com);
				break;
			case SG_GlobalQuantizer:
				com->node = NULL;
				field = SG_NewFieldCommand(com);
				if (!stricmp(str, "NULL")) {
					field->new_node = NULL;
				} else {
					field->new_node = xmt_parse_node(parser, str, com->node);
				}
				field->field_ptr = &field->new_node;
				field->fieldType = FT_SFNode;
				break;
			case SG_MultipleReplace:
			{
				char szName[1024];
				char *szVal;
				if (stricmp(str, "repField")) {
					XMTLogErr2(parser, "Unexpected symbol in multiple replace", str);
					goto err;
				}
				szVal = NULL;
				while (xmt_has_attributes(parser)) {
					str = xmt_get_attribute(parser);
					if (!stricmp(str, "atField")) strcpy(szName, parser->att_buffer);
					if (!stricmp(str, "value")) szVal = strdup(parser->att_buffer);
				}
				if (szVal) {
					if (Node_GetFieldByName(com->node, szName, &inf) != M4OK) {
					} else {
						field = SG_NewFieldCommand(com);
						inf.far_ptr = field->field_ptr = SG_NewFieldPointer(inf.fieldType);
						field->fieldType = inf.fieldType;
						field->fieldIndex = inf.allIndex;
						str = parser->att_buffer;
						parser->temp_att = parser->att_buffer = szVal;
						if (SG_IsSFField(inf.fieldType)) {
							xmt_sffield(parser, &inf, NULL);
						} else {
							xmt_mffield(parser, &inf, NULL);
						}
						parser->att_buffer = str;
						free(szVal);
					}
				}
				/*nodes*/
				while (!xmt_element_done(parser, "repField")) {
					str = xmt_get_element(parser);
					strcpy(szName, str);
					if (Node_GetFieldByName(com->node, szName, &inf) != M4OK) {
						xmt_skip_element(parser, str);
					} else {
						xmt_skip_attributes(parser);
						field = SG_NewFieldCommand(com);
						field->fieldIndex = inf.allIndex;
						field->fieldType = inf.fieldType;
						if (inf.fieldType == FT_SFNode) {
							field->new_node = xmt_parse_node(parser, NULL, com->node);
							field->field_ptr = &field->new_node;
							xmt_element_done(parser, szName);
						} else {
							field->node_list = NewChain();
							field->field_ptr = &field->node_list;
							while (!xmt_element_done(parser, szName)) {
								SFNode *n = xmt_parse_node(parser, NULL, com->node);
								if (n) ChainAddEntry(field->node_list, n);
							}
						}
					}

				}
			}
				break;
			case SG_MultipleIndexedReplace:
			{
				char szName[1024];
				char *szVal;
				if (stricmp(str, "repValue")) {
					XMTLogErr2(parser, "Unexpected symbol in multiple replace", str);
					goto err;
				}
				szVal = NULL;
				while (xmt_has_attributes(parser)) {
					str = xmt_get_attribute(parser);
					if (!stricmp(str, "position")) {
						if (!stricmp(parser->att_buffer, "BEGIN")) pos = 0;
						else if (!stricmp(parser->att_buffer, "END")) pos = -1;
						else pos = atoi(parser->att_buffer);
					}
					if (!stricmp(str, "value")) szVal = strdup(parser->att_buffer);
				}
				if (szVal) {
					field = SG_NewFieldCommand(com);
					inf.fieldType = field->fieldType = SG_GetSFType(inf.fieldType);
					inf.far_ptr = field->field_ptr = SG_NewFieldPointer(field->fieldType);
					field->fieldIndex = inf.allIndex;
					field->pos = pos;
					str = parser->att_buffer;
					parser->temp_att = parser->att_buffer = szVal;
					xmt_sffield(parser, &inf, NULL);
					parser->att_buffer = str;
					free(szVal);
				}
				/*nodes*/
				while (!xmt_element_done(parser, "repValue")) {
					str = xmt_get_element(parser);
					strcpy(szName, str);
					if (Node_GetFieldByName(com->node, szName, &inf) != M4OK) {
						xmt_skip_element(parser, str);
					} else {
						xmt_skip_attributes(parser);
						field = SG_NewFieldCommand(com);
						field->pos = pos;
						field->fieldIndex = inf.allIndex;
						field->fieldType = FT_SFNode;
						field->new_node = xmt_parse_node(parser, NULL, com->node);
						field->field_ptr = &field->new_node;
						xmt_element_done(parser, szName);
					}

				}
			}
				break;
			case SG_ProtoInsert:
				if (stricmp(str, "ProtoDeclare")) {
					XMTLogErr2(parser, "Unexpected symbol in ProtoInsert", str);
					goto err;
				}
				xmt_parse_proto(parser, com->new_proto_list);
				break;
			}
		}
		if (com_list) ChainAddEntry(com_list, com);
		else ChainAddEntry(parser->bifs_au->commands, com);
		if (com->tag==SG_RouteInsert) ChainAddEntry(parser->inserted_routes, com);

		return;
err:
		SG_DeleteCommand(com);
		return;
	}
	/*OD commands*/
	if (!stricmp(comName, "ObjectDescriptorUpdate") || !stricmp(comName, "ObjectDescriptorRemove")
			|| !stricmp(comName, "ES_DescriptorUpdate") || !stricmp(comName, "ES_DescriptorRemove")
			|| !stricmp(comName, "IPMPDescriptorUpdate") || !stricmp(comName, "IPMPDescriptorRemove") ) {
		
		if (!parser->stream_id) parser->stream_id = parser->base_od_id;

		if (!parser->od_es) parser->od_es = M4SM_NewStream(parser->ctx, (u16) parser->stream_id, M4ST_OD, 0);
		if (!parser->od_au) parser->od_au = M4SM_NewAU(parser->od_es, 0, parser->au_time, parser->au_is_rap);

		if (!stricmp(comName, "ObjectDescriptorUpdate") ) {
			ObjectDescriptorUpdate *odU;
			xmt_skip_attributes(parser);
			while (1) {
				str = xmt_get_element(parser);
				if (!str) {
					XMTLogErr1(parser, "Expecting <OD> in <ObjectDescriptorUpdate>");
					return;
				}
				if (!stricmp(str, "OD")) break;
				xmt_skip_element(parser, str);
			}
			xmt_skip_attributes(parser);
			odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
			while (!xmt_element_done(parser, "OD")) {
				Descriptor *od = xmt_parse_descriptor(parser, NULL, NULL);
				if (od) ChainAddEntry(odU->objectDescriptors, od);
				else if (parser->last_error) {
					OD_DeleteCommand((ODCommand **) &odU);
					return;
				}
			}
			if (!xmt_element_done(parser, "ObjectDescriptorUpdate")) {
				XMTLogErr1(parser, "Expecting </ObjectDescriptorUpdate>");
				return;
			}
			ChainAddEntry(parser->od_au->commands, odU);
			return;
		}
		if (!stricmp(comName, "ObjectDescriptorRemove") ) {
			ObjectDescriptorRemove *odR;
			odR = (ObjectDescriptorRemove *) OD_NewCommand(ODRemove_Tag);
			while (xmt_has_attributes(parser)) {
				str = xmt_get_attribute(parser);
				if (!stricmp(str, "objectDescriptorId")) {
					u32 i, j, start;
					char szBuf[100];
					i = j = start = 0;
					while (parser->att_buffer[start+i]) {
						j = 0;
						while (parser->att_buffer[start+i] && (parser->att_buffer[start+i] == ' ')) i++;
						while (parser->att_buffer[start+i] && (parser->att_buffer[start+i] != ' ')) {
							szBuf[j] = parser->att_buffer[start+i];
							i++;
							j++;
						}
						szBuf[j] = 0;
						start += i;
						if (parser->att_buffer[start] == ' ') start ++;
						i = 0;
						j = xmt_get_od_id(parser, szBuf);
						if (!j) {
							XMTLogErr2(parser, "Cannot find ObjectDescriptor", szBuf);
							OD_DeleteCommand((ODCommand **) &odR);
							return;
						}
						odR->OD_ID = realloc(odR->OD_ID, sizeof(u16) * (odR->NbODs+1));
						odR->OD_ID[odR->NbODs] = j;
						odR->NbODs++;
					}
				}
			}
			xmt_element_done(parser, "ObjectDescriptorRemove");

			ChainAddEntry(parser->od_au->commands, odR);
			return;
		}
		if (!stricmp(comName, "ES_DescriptorRemove") ) {
			ESDescriptorRemove *esdR;
			esdR = (ESDescriptorRemove *) OD_NewCommand(ESDRemove_Tag);
			while (xmt_has_attributes(parser)) {
				str = xmt_get_attribute(parser);
				if (!stricmp(str, "objectDescriptorId")) {
					esdR->ODID = xmt_get_od_id(parser, parser->att_buffer);
				} else if (!stricmp(str, "ES_ID")) {
					u32 i, j, start;
					char szBuf[100];
					i = j = start = 0;
					while (parser->att_buffer[start+i]) {
						j = 0;
						while (parser->att_buffer[start+i] && (parser->att_buffer[start+i] == ' ')) i++;
						while (parser->att_buffer[start+i] && (parser->att_buffer[start+i] != ' ')) {
							szBuf[j] = parser->att_buffer[start+i];
							i++;
							j++;
						}
						szBuf[j] = 0;
						start += i;
						if (parser->att_buffer[start] == ' ') start ++;
						i = 0;
						j = xmt_get_esd_id(parser, szBuf);
						if (!j) {
							XMTLogErr2(parser, "Cannot find ESDescriptor", szBuf);
							OD_DeleteCommand((ODCommand **) &esdR);
							return;
						}
						esdR->ES_ID = realloc(esdR->ES_ID, sizeof(u16) * (esdR->NbESDs+1));
						esdR->ES_ID[esdR->NbESDs] = j;
						esdR->NbESDs++;
					}
				}
			}
			xmt_element_done(parser, "ES_DescriptorRemove");
			ChainAddEntry(parser->od_au->commands, esdR);
			return;
		}
	}
	/*not found*/
	xmt_skip_element(parser, comName);
}

Bool xmt_esid_available(XMTParser *parser, u16 ESID) 
{
	u32 i;
	for (i=0; i<ChainGetCount(parser->esd_links); i++) {
		ESDLink *esdl = ChainGetEntry(parser->esd_links, i);
		if (esdl->esd->ESID == ESID) return 0;
	}
	return 1;
}
Bool xmt_odid_available(XMTParser *parser, u16 ODID) 
{
	u32 i;
	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		ODLink *l = ChainGetEntry(parser->od_links, i);
		if (l->ID == ODID) return 0;
		if (l->od && l->od->objectDescriptorID == ODID) return 0;
	}
	return 1;
}

void xmt_resolve_od(XMTParser *parser)
{
	u32 i, j;
	ODLink *l;
	char szURL[5000];

	/*fix ESD IDs*/
	for (i=0; i<ChainGetCount(parser->esd_links); i++) {
		ESDLink *esdl = ChainGetEntry(parser->esd_links, i);
		if (!esdl->esd->ESID) {
			u16 ESID = 1;
			while (!xmt_esid_available(parser, ESID)) ESID++;
			esdl->esd->ESID = ESID;
		}
	}
	while (ChainGetCount(parser->esd_links)) {
		ESDLink *esdl = ChainGetEntry(parser->esd_links, 0);
		ChainDeleteEntry(parser->esd_links, 0);
		if (esdl->desc_name) free(esdl->desc_name);
		free(esdl);
	}

	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		l = ChainGetEntry(parser->od_links, i);
		if (l->od && !l->od->objectDescriptorID) {
			u16 ODID = 1;
			while (!xmt_odid_available(parser, ODID)) ODID++;
			l->od->objectDescriptorID = ODID;
		}
		if (l->od) {
			if (!l->ID) l->ID = l->od->objectDescriptorID;
			assert(l->ID == l->od->objectDescriptorID);
		}
	}

	/*unroll dep in case some URLs reference ODs by their binary IDs not their string ones*/
	for (i=0; i<ChainGetCount(parser->od_links); i++) {
		l = ChainGetEntry(parser->od_links, i);
		/*not OD URL*/
		if (!l->ID) continue;
		for (j=i+1; j<ChainGetCount(parser->od_links); j++) {
			ODLink *l2 = ChainGetEntry(parser->od_links, j);
			/*not OD URL*/
			if (!l2->ID) continue;
			if (l->ID == l2->ID) {
				while (ChainGetCount(l2->nodes)) {
					SFNode *n = ChainGetEntry(l2->nodes, 0);
					ChainDeleteEntry(l2->nodes, 0);
					ChainAddEntry(l->nodes, n);
				}
				ChainDeleteEntry(parser->od_links, j);
				j--;
				if (l2->desc_name) free(l2->desc_name);
				DeleteChain(l2->nodes);
				free(l2);
			}
		}
	}

	while (ChainGetCount(parser->od_links) ) {
		l = ChainGetEntry(parser->od_links, 0);
		if (!l->od) {
			/*if no ID found this is not an OD URL*/
			if (l->ID) {
				sprintf(parser->szMsg, "Error: OD %d (%s) not assigned\n", l->ID, l->desc_name ? l->desc_name : "");
				XMT_REPORT_ERR(parser);
				parser->last_error = M4BadParam;
			}
		} else {
			for (j=0; j<ChainGetCount(l->nodes); j++) {
				FieldInfo info;
				SFNode *n = ChainGetEntry(l->nodes, j);
				if (Node_GetFieldByName(n, "url", &info) == M4OK) {
					u32 k;
					MFURL *url = (MFURL *)info.far_ptr;
					for (k=0; k<url->count; k++) {
						char *seg = NULL;
						if (url->vals[k].url) seg = strstr(url->vals[k].url, "#");
						if (seg) {
							sprintf(szURL, "od:%d#%s", l->od->objectDescriptorID, seg+1);
							free(url->vals[k].url);
							url->vals[k].url = strdup(szURL);
						} else {
							if (url->vals[k].url) free(url->vals[k].url);
							url->vals[k].url = NULL;
							url->vals[k].OD_ID = l->od->objectDescriptorID;
						}
					}
				}
			}
		}

		if (l->desc_name) free(l->desc_name);
		DeleteChain(l->nodes);
		free(l);
		ChainDeleteEntry(parser->od_links, 0);
	}
}

M4Err M4SM_LoadContextFromXMT(M4SceneManager *ctx, FILE *input, void (*XMTNodeInit)(void *cbk, SFNode *n), void (*XMTOnError)(void *cbk, char *msg), void *xmt_cbck)
{
	u32 pos;
	unsigned char BOM[4];
	char *str;
	XMTParser parser;

	memset(&parser, 0, sizeof(XMTParser));
	parser.input = input;
	parser.ctx = ctx;

	
	pos = ftell(input);
	BOM[0] = fgetc(input);
	BOM[1] = fgetc(input);
	BOM[2] = fgetc(input);
	BOM[3] = fgetc(input);
	fseek(input, pos, SEEK_SET);

	/*0: no unicode, 1: UTF-16BE, 2: UTF-16LE*/
	if ((BOM[0]==0xFF) && (BOM[1]==0xFE)) {
		if (!BOM[2] && !BOM[3]) {
			fprintf(stdout, "UTF-32 Text Files not supported\n");
			return M4NotSupported;
		} else {
			parser.unicode_type = 2;
			fseek(input, 2, SEEK_CUR);
		}
	} else if ((BOM[0]==0xFE) && (BOM[1]==0xFF)) {
		if (!BOM[2] && !BOM[3]) {
			fprintf(stdout, "UTF-32 Text Files not supported\n");
			return M4NotSupported;
		} else {
			parser.unicode_type = 1;
			fseek(input, 2, SEEK_CUR);
		}
	} else if ((BOM[0]==0xEF) && (BOM[1]==0xBB) && (BOM[2]==0xBF)) {
		/*we handle UTF8 as asci*/
		parser.unicode_type = 0;
		fseek(input, 3, SEEK_CUR);
	}
	
	
	parser.unresolved_routes = NewChain();
	parser.inserted_routes = NewChain();
	parser.XMTNodeInit = XMTNodeInit;
	parser.XMTOnError = XMTOnError;
	parser.xmt_cbck = xmt_cbck;
	parser.od_links = NewChain();
	parser.esd_links = NewChain();
	parser.def_nodes = NewChain();

	/*create at least one empty BIFS stream*/
	parser.bifs_es = M4SM_NewStream(ctx, 0, M4ST_BIFS, 0);
	parser.bifs_au = M4SM_NewAU(parser.bifs_es, 0, 0, 1);


	/*check XML doc*/
	while (1) {
		str = xmt_get_element(&parser);
		if (!str || stricmp(str, "?xml")) {
			XMTLogErr1(&parser, "Invalid XML document - expecting <?xml> element");
			goto exit;
		}
		if (!stricmp(str, "?xml")) break;
		xmt_skip_element(&parser, str);
	}
	xmt_skip_attributes(&parser);

	/*check XMT-A doc*/
	while (1) {
		str = xmt_get_element(&parser);
		if (!str) {
			XMTLogErr1(&parser, "Invalid XMT-A document");
			goto exit;
		}
		if (!stricmp(str, "XMT-A")) break;
		xmt_skip_element(&parser, str);
	}
	xmt_skip_attributes(&parser);

	/*check header*/
	while (1) {
		str = xmt_get_element(&parser);
		if (!str) {
			XMTLogErr1(&parser, "Expecting <Header> in XMT-A document");
			goto exit;
		}
		if (!stricmp(str, "Header")) break;
		xmt_skip_element(&parser, str);
	}
	xmt_skip_attributes(&parser);
	parser.ctx->root_od = (ObjectDescriptor *) xmt_parse_descriptor(&parser, NULL, NULL);
	if (!xmt_element_done(&parser, "Header")) {
		XMTLogErr1(&parser, "Expecting </Header> in XMT-A document");
		goto exit;
	}

	/*check body*/
	while (1) {
		str = xmt_get_element(&parser);
		if (!str) {
			XMTLogErr1(&parser, "Expecting <Body> in XMT-A document");
			goto exit;
		}
		if (!stricmp(str, "Body")) break;
		xmt_skip_element(&parser, str);
	}
	xmt_skip_attributes(&parser);

	/*parse all commands*/
	parser.au_time = 0;
	parser.au_is_rap = 1;
	while (!xmt_element_done(&parser, "Body") && !parser.last_error) {
		str = xmt_get_element(&parser);
		/*explicit command*/
		if (!stricmp(str, "par")) {
			parser.stream_id = 0;
			while (xmt_has_attributes(&parser)) {
				str = xmt_get_attribute(&parser);
				if (!stricmp(str, "begin")) {
					parser.au_time = atof(parser.att_buffer);
				}
				else if (!stricmp(str, "atES_ID")) {
					parser.stream_id = atoi(parser.att_buffer);
				}
			}
			/*reset context*/
			if (parser.od_au && (parser.od_au->timing_sec != parser.au_time)) parser.od_au = NULL;
			if (parser.bifs_au && (parser.bifs_au->timing_sec != parser.au_time)) parser.bifs_au = NULL;
			/*parse all commands context*/
			while (!xmt_element_done(&parser, "par") && !parser.last_error) {
				xmt_parse_command(&parser, NULL, NULL);
			}
		} 
		/*implicit command*/
		else {
			parser.stream_id = 0;
			xmt_parse_command(&parser, str, NULL);
			parser.au_is_rap = 0;
		}
	}

	if (!parser.last_error && !xmt_element_done(&parser, "XMT-A")) {
		XMTLogErr1(&parser, "Expecting </XMT-A> in XMT-A document");
	}


exit:
	xmt_resolve_routes(&parser);
	DeleteChain(parser.unresolved_routes);
	DeleteChain(parser.inserted_routes);
	DeleteChain(parser.def_nodes);

	/*clean all OD references*/
	xmt_resolve_od(&parser);
	DeleteChain(parser.od_links);
	DeleteChain(parser.esd_links);
	if (parser.att_buffer) free(parser.att_buffer);
	fprintf(stdout, "Parsing done\n");
	return parser.last_error;
}
