/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"


static void DestroyCompositeTexture(SFNode *node)
{
	CompositeTextureStack *st = (CompositeTextureStack *) Node_GetPrivate(node);
	assert(!st->txh.data);
	texture_destroy(&st->txh);
	VS_Delete(st->surface);
	free(st);
}

static void Composite_GetPixelSize(SFNode *n, s32 *pw, s32 *ph)
{
	*pw = *ph = 0;
	if (!n) return;
	switch (Node_GetTag(n)) {
#ifdef M4_DEF_CompositeTexture3D
	case TAG_CompositeTexture3D:
		*pw = ((B_CompositeTexture3D*)n)->pixelWidth;
		*ph = ((B_CompositeTexture3D*)n)->pixelHeight;
		break;
#endif
#ifdef M4_DEF_CompositeTexture2D
	case TAG_CompositeTexture2D:
		*pw = ((B_CompositeTexture2D*)n)->pixelWidth;
		*ph = ((B_CompositeTexture2D*)n)->pixelHeight;
		break;
#endif
	}
	if (*pw<0) *pw = 0;
	if (*ph<0) *ph = 0;
}

static void Composite_SetPixelSize(SFNode *n, s32 pw, s32 ph)
{
	if (!n) return;
	switch (Node_GetTag(n)) {
#ifdef M4_DEF_CompositeTexture3D
	case TAG_CompositeTexture3D:
		((B_CompositeTexture3D*)n)->pixelWidth = pw;
		((B_CompositeTexture3D*)n)->pixelHeight = ph;
		break;
#endif
#ifdef M4_DEF_CompositeTexture2D
	case TAG_CompositeTexture2D:
		((B_CompositeTexture2D*)n)->pixelWidth = pw;
		((B_CompositeTexture2D*)n)->pixelHeight = ph;
		break;
#endif
	}
}

static void UpdateCompositeTexture(TextureHandler *txh)
{
	s32 w, h;
	RenderEffect *eff;
	CompositeTextureStack *st = (CompositeTextureStack *) Node_GetPrivate(txh->owner);

	if (st->txh.hwtx && st->txh.compositor->reset_graphics) tx_delete(&st->txh);

	Composite_GetPixelSize(st->txh.owner, &w, &h);
	if (!w || !h) return;

	if (!st->txh.hwtx || ((s32) st->txh.width != w) || ( (s32) st->txh.height != h) ) {
		tx_delete(&st->txh);

		if (st->txh.compositor->hw_caps.npot_texture) {
			st->txh.width = w;
			st->txh.height = h;
		} else {
			st->txh.width = 2;
			while (st->txh.width<(u32)w) st->txh.width*=2;
			st->txh.height = 2;
			while (st->txh.height<(u32)h) st->txh.height*=2;
		}
		Composite_SetPixelSize(st->txh.owner, st->txh.width, st->txh.height);

		/*generate texture*/
		st->txh.pixelformat = M4PF_RGBA;
		st->txh.active_window.x = st->txh.active_window.y = 0;
		st->txh.active_window.width = (Float) st->txh.width;
		st->txh.active_window.height = (Float) st->txh.height;
		st->txh.stride = st->txh.width*4;
		st->txh.transparent = 1;
		st->surface->width = st->txh.width;
		st->surface->height = st->txh.height;

		tx_allocate(&st->txh);
		
		st->txh.data = malloc(sizeof(unsigned char) * st->txh.stride * st->txh.height);
		tx_setup_format(&st->txh);
		tx_set_image(&st->txh, 0);
		free(st->txh.data);
		st->txh.data = NULL;
	}

	eff = malloc(sizeof(RenderEffect));
	memset(eff, 0, sizeof(RenderEffect));

	eff->surface = st->surface;
	eff->traversing_mode = TRAVERSE_RENDER;
	mx_init(eff->model_view);
	cmat_init(&eff->color_mat);

	VS_InitRender(st->surface, eff);
	VS3D_PushMatrix(st->surface);
	VS3D_ResetMatrix(st->surface);

	/*render*/
	Node_RenderChildren(st->txh.owner, eff);
	tx_copy_to_texture(&st->txh);
	VS3D_PopMatrix(st->surface);

	effect_delete(eff);
}

TextureHandler *r3d_composite_get_texture(SFNode *node)
{
	CompositeTextureStack *st = (CompositeTextureStack *) Node_GetPrivate(node);
	return &st->txh;
}


#ifdef M4_DEF_CompositeTexture3D

void R3D_InitCompositeTexture3D(Render3D *sr, SFNode *node)
{
	CompositeTextureStack *st = malloc(sizeof(CompositeTextureStack));
	memset(st, 0, sizeof(CompositeTextureStack));
	texture_setup(&st->txh, sr->compositor, node);
	st->txh.flags = TX_REPEAT_S | TX_REPEAT_T;

	/*create composite surface*/
	st->surface = VS_New(1);
	st->surface->render = sr;
	st->surface->is_pixel_metrics = sr->compositor->is_pixel_metrics;

	st->txh.update_texture_fcnt = UpdateCompositeTexture;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyCompositeTexture);
}

#endif

#ifdef M4_DEF_CompositeTexture2D

void R3D_InitCompositeTexture2D(Render3D *sr, SFNode *node)
{
	CompositeTextureStack *st = malloc(sizeof(CompositeTextureStack));
	memset(st, 0, sizeof(CompositeTextureStack));
	texture_setup(&st->txh, sr->compositor, node);
	st->txh.flags = TX_REPEAT_S | TX_REPEAT_T;

	/*create composite surface*/
	st->surface = VS_New(0);
	st->surface->render = sr;
	st->surface->is_pixel_metrics = sr->compositor->is_pixel_metrics;

	st->txh.update_texture_fcnt = UpdateCompositeTexture;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyCompositeTexture);
}

#endif


