/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/

#include                "render.h"
#include                "gd.h"
#include                "utils.h"
#include                <fcntl.h>
#ifdef MSWIN32
#include <io.h>
#endif

#ifdef _UWIN
#ifndef DEFAULT_FONTPATH
#define		DEFAULT_FONTPATH	"/win/fonts"
#endif  /* DEFAULT_FONTPATH */
#else
#ifndef MSWIN32
#ifndef DEFAULT_FONTPATH
#define		DEFAULT_FONTPATH	"/usr/share/ttf:/usr/local/share/ttf:/usr/share/fonts/ttf:/usr/local/share/fonts/ttf:/usr/lib/fonts:/usr/local/lib/fonts:/usr/lib/fonts/ttf:/usr/local/lib/fonts/ttf:/usr/common/graphviz/lib/fonts/ttf:/windows/fonts:/dos/windows/fonts:/usr/add-on/share/ttf:."
#endif /* DEFAULT_FONTPATH */
#else
#ifndef DEFAULT_FONTPATH
#define		DEFAULT_FONTPATH	"%WINDIR%/FONTS;C:/WINDOWS/FONTS;C:/WINNT/Fonts;C:/winnt/fonts"
#endif /* DEFAULT_FONTPATH */
#endif /* MSWIN32 */
#endif /* _UWIN */

#define BEZIERSUBDIVISION 10

/* fontsize at which text is omitted entirely */
#define FONTSIZE_MUCH_TOO_SMALL 0.15
/* fontsize at which text is rendered by a simple line */
#define FONTSIZE_TOO_SMALL 1.5

/* font modifiers */
#define REGULAR		0
#define BOLD		1
#define ITALIC		2

/* patterns */
#define P_SOLID		0
#define P_DOTTED	4
#define P_DASHED	11
#define P_NONE		15

/* bold line constant */
#define WIDTH_NORMAL	1
#define WIDTH_BOLD	3

/* static int	N_pages; */
/* static point	Pages; */
static double	Scale;
static double	Dpi;
static double	DevScale;
static double	CompScale;
static int	Rot;

static point	Viewport;
static pointf	GraphFocus;
static double   Zoom;

static gdImagePtr im;
static Dict_t	*ImageDict;

typedef struct context_t {
	int	pencolor, fillcolor;
        char	*fontfam, fontopt, font_was_set, pen, fill, penwidth;
	double   fontsz;
} context_t;

#define MAXNEST	4
static context_t cstk[MAXNEST];
static int	SP;
static node_t	*Curnode;

static int gd_resolve_color(char* name)
{
	color_t	color;
 
	if (!(strcmp(name,"transparent"))) {
	    	/* special case for "transparent" color */
		return gdImageGetTransparent(im);
	} 
	else {
		colorxlate(name,&color,RGBA_BYTE);
	/* seems gd alpha is "transparency" rather than the usual "opacity" */
		return gdImageColorResolveAlpha(im,
			color.u.rgba[0],
			color.u.rgba[1],
			color.u.rgba[2],
			(255 - color.u.rgba[3])*gdAlphaMax/255); 
	}
}

static int white, black, transparent, basecolor;

static void init_gd(graph_t *g)
{
	int bgcolor;
	char *str;

	SP = 0;

	white = gdImageColorResolveAlpha(im,
			gdRedMax, gdGreenMax, gdBlueMax, gdAlphaOpaque);
	black =  gdImageColorResolveAlpha(im,
			0, 0, 0, gdAlphaOpaque);
	transparent = gdImageColorResolveAlpha(im,
			gdRedMax-1, gdGreenMax, gdBlueMax, gdAlphaTransparent);
	gdImageColorTransparent(im, transparent);

	basecolor = white;
	if (((str = agget(g,"bgcolor")) != 0) && str[0]) {
		bgcolor = gd_resolve_color(str);
		if (gdTrueColorGetAlpha(bgcolor) != gdAlphaOpaque)
			basecolor = transparent;
	}
	/* Blending must be off to lay a transparent basecolor.
		Nothing to blend with anyway. */
	gdImageAlphaBlending(im, FALSE);
	gdImageFill (im, im->sx/2, im->sy/2, basecolor);
	/* Blend everything else together,
		especially fonts over non-transparent backgrounds */
	gdImageAlphaBlending(im, TRUE);

	cstk[0].pencolor = black;		/* set pen black*/
	cstk[0].fillcolor = black;		/* set fill black*/
	cstk[0].fontfam = "times";		/* font family name */
	cstk[0].fontopt = REGULAR;		/* modifier: REGULAR, BOLD or ITALIC */
	cstk[0].pen = P_SOLID;		/* pen pattern style, default is solid */
	cstk[0].fill = P_NONE;
	cstk[0].penwidth = WIDTH_NORMAL;
}

static pointf gdpt(pointf p)
{
	pointf		  rv;

	if (Rot == 0) {
		rv.x =  (p.x - GraphFocus.x) * CompScale + Viewport.x/2.;
		rv.y = -(p.y - GraphFocus.y) * CompScale + Viewport.y/2.;
	} else {
		rv.x = -(p.y - GraphFocus.y) * CompScale + Viewport.x/2.;
		rv.y = -(p.x - GraphFocus.x) * CompScale + Viewport.y/2.; 
	}
	return rv;
}

static void gd_begin_job(FILE *ofp, graph_t *g, char **lib, char *user,
char *info[], point pages)
{
/*	Pages = pages; */
/*	N_pages = pages.x * pages.y; */
#ifdef MYTRACE
fprintf(stderr,"gd_begin_job\n");
#endif
}

static void gd_end_job(void)
{
#ifdef MYTRACE
fprintf(stderr,"gd_end_job\n");
#endif
}

static void gd_begin_graph(GVC_t *gvc, graph_t* g, box bb, point pb)
{
	Dpi = gvc->dpi;
	if (Dpi < 1.0)
		Dpi = DEFAULT_DPI;
	DevScale = Dpi/POINTS_PER_INCH;

	Viewport = gvc->size;
	if (Viewport.x) {
		Zoom = gvc->zoom;
		GraphFocus = gvc->focus;
	}
	else {
		Viewport.x = (bb.UR.x - bb.LL.x + 2*GD_drawing(g)->margin.x) * DevScale + 2;
		Viewport.y = (bb.UR.y - bb.LL.y + 2*GD_drawing(g)->margin.y) * DevScale + 2;
		GraphFocus.x = (GD_bb(g).UR.x - GD_bb(g).LL.x) / 2.;
		GraphFocus.y = (GD_bb(g).UR.y - GD_bb(g).LL.y) / 2.;
		Zoom = 1.0;
	}
}

static void gd_begin_graph_to_file(GVC_t *gvc, graph_t* g, box bb, point pb)
{
#define TRUECOLOR_SWITCH 1

#ifdef TRUECOLOR_SWITCH
	char *truecolor_p;
#endif

	gd_begin_graph(gvc, g, bb, pb);

#ifdef TRUECOLOR_SWITCH
	truecolor_p = agget(g,"truecolor");
	/* automatically decide color mode if "truecolor" not specified */
	if (truecolor_p == NULL || *truecolor_p == '\0') {
		truecolor_p="false";  /* indexed color maps unless ... */

		if (agfindattr(g->proto->n, "shapefile")) {
			/* ...  custom shapefiles are used */
			truecolor_p = "true";
		}
	}
	if (mapbool(truecolor_p)) {
#endif
		if (Verbose)
			fprintf(stderr,"%s: allocating a %dK TrueColor GD image\n",
				CmdName, ROUND( Viewport.x * Viewport.y * 4 / 1024. ));
		im = gdImageCreateTrueColor(Viewport.x, Viewport.y);
#ifdef TRUECOLOR_SWITCH
	}
	else {
		if (Verbose)
			fprintf(stderr,"%s: allocating a %dK PaletteColor GD image\n",
				CmdName, ROUND( Viewport.x * Viewport.y / 1024. ));
		im = gdImageCreate(Viewport.x, Viewport.y);
	}
#endif
	init_gd(g);
#ifdef MYTRACE
fprintf(stderr,"gd_begin_graph_to_file\n");
#endif
}

static void gd_begin_graph_to_memory(GVC_t *gvc, graph_t* g, box bb, point pb)
{
	gd_begin_graph(gvc, g, bb, pb);
	if (Verbose)
		fprintf(stderr,"%s: using existing GD image\n", CmdName);
	im = *(gdImagePtr *)Output_file;
	init_gd(g);
#ifdef MYTRACE
fprintf(stderr,"gd_begin_graph_to_memory\n");
#endif
}

static void gd_end_graph_to_file(void)
{
/*
 * Windows will do \n -> \r\n  translations on stdout unless told otherwise.
 */
#ifdef HAVE_SETMODE
#ifdef O_BINARY
	setmode(fileno(Output_file), O_BINARY);
#endif
#endif

/*
 * Write IM to OUTFILE as a JFIF-formatted JPEG image, using quality
 * JPEG_QUALITY.  If JPEG_QUALITY is in the range 0-100, increasing values
 * represent higher quality but also larger image size.  If JPEG_QUALITY is
 * negative, the IJG JPEG library's default quality is used (which
 * should be near optimal for many applications).  See the IJG JPEG
 * library documentation for more details.  */

#define JPEG_QUALITY -1

	/* Only save the alpha channel in outputs that support it if
		the base color was transparent.   Otherwise everything
		was blended so there is no useful alpha info */
	gdImageSaveAlpha (im, (basecolor==transparent));
	if (Output_lang == GD) {
		gdImageGd(im, Output_file);
#ifdef HAVE_LIBZ
	} else if (Output_lang == GD2) {
#define GD2_CHUNKSIZE 128
#define GD2_RAW 1
#define GD2_COMPRESSED 2
		gdImageGd2(im, Output_file, GD2_CHUNKSIZE, GD2_COMPRESSED);
#endif
#ifdef HAVE_GD_GIF
	} else if (Output_lang == GIF) {
                gdImageTrueColorToPalette(im, 0, 256);
		gdImageGif(im, Output_file);
#endif
#ifdef HAVE_GD_JPEG
	} else if (Output_lang == JPEG) {
		gdImageJpeg(im, Output_file, JPEG_QUALITY);
#endif
#ifdef HAVE_GD_PNG
	} else if (Output_lang == PNG) {
		gdImagePng(im, Output_file);
#endif
	} else if (Output_lang == WBMP) {
        	/* Use black for the foreground color for the B&W wbmp image. */
		gdImageWBMP(im, black, Output_file);
#ifdef HAVE_GD_XPM
	} else if (Output_lang == XBM) {
		gdImageXbm(im, Output_file);
#endif
	}
	if (ImageDict) {
		dtclose(ImageDict); ImageDict = 0;
	}
	gdImageDestroy(im);
#ifdef MYTRACE
fprintf(stderr,"gd_end_graph_to_file\n");
#endif
}

static void gd_end_graph_to_memory(void)
{
/* leave image in memory to be handled by Gdtclft output routines */
#ifdef MYTRACE
fprintf(stderr,"gd_end_graph_to_memory\n");
#endif
}

static void
gd_begin_page(graph_t *g, point page, double scale, int rot, point offset)
{
/*	int		page_number; */
/*	point		sz; */

	Scale = scale;
	CompScale = Zoom * Scale * DevScale;
	Rot = rot;

/*	page_number = page.x + page.y * Pages.x + 1; */
#ifdef MYTRACE
fprintf(stderr,"gd_begin_page\n");
fprintf(stderr," page=%d,%d offset=%d,%d\n",page.x,page.y,offset.x,offset.y);
fprintf(stderr," page_number=%d\n",page_number);
#endif
}

void gd_end_page(void) {
#ifdef MYTRACE
fprintf(stderr,"gd_end_page\n");
#endif
}

static void
gd_begin_node(node_t* n)
{
        Curnode = n;
}

static void
gd_end_node()
{
        Curnode = NULL;
}

static  void
gd_begin_context(void)
{
	assert(SP + 1 < MAXNEST);
	cstk[SP + 1] = cstk[SP];
	SP++;
}

static  void
gd_end_context(void)
{
	int			 psp = SP - 1;
	assert(SP > 0);
	SP = psp;
}

static  void
gd_set_font(char* fontname, double fontsize)
{
	context_t	*cp;
#if 0
	char		*p;
	char		*q;
#endif

	cp = &(cstk[SP]);
	cp->fontsz = fontsize;
#if 0
	p = strdup(fontname);
	if ((q = strchr(p, '-'))) {
		*q++ = 0;
		if (strcasecmp(q, "italic") == 0)
			cp->fontopt = ITALIC;
		else if (strcasecmp(q, "bold") == 0)
			cp->fontopt = BOLD;
	}
#endif
	cp->fontfam = fontname;
}

static void gd_set_pencolor(char* name)
{
	cstk[SP].pencolor = gd_resolve_color(name);
}

static void gd_set_fillcolor(char* name)
{
	cstk[SP].fillcolor = gd_resolve_color(name);
}

static  void
gd_set_style(char** s)
{
	char		*line,*p;
	context_t	*cp;

	cp = &(cstk[SP]);
	while ((p = line = *s++)) {
		if (streq(line, "solid")) cp->pen = P_SOLID;
		else if (streq(line, "dashed")) cp->pen = P_DASHED;
                else if (streq(line, "dotted")) cp->pen = P_DOTTED;
		else if (streq(line, "invis")) cp->pen = P_NONE;
 		else if (streq(line, "bold")) cp->penwidth = WIDTH_BOLD;
		else if (streq(line, "setlinewidth")) {
			while (*p) p++;
			p++;
			cp->penwidth = atol(p);
		}
		else if (streq(line, "filled")) cp->fill = P_SOLID;
		else if (streq(line, "unfilled")) cp->fill = P_NONE;
		else agerr (AGWARN, "gd_set_style: unsupported style %s - ignoring\n",
			line);
	}
}

/* sometimes fonts are stored under a different name */
char *
gd_alternate_fontlist(char *font) 
{
#ifdef HAVE_GD_FONTCONFIG
	return font;
#else
	static char *fontbuf;
	static int fontbufsz;
	char *p, *fontlist;
	int len;

	len = strlen(font) + 1;
	if (len > fontbufsz) {
		fontbufsz = 2*len;
		if (fontbuf)
			fontbuf = malloc(fontbufsz);
		else
			fontbuf = realloc(fontbuf, fontbufsz);
	}

	/* fontbuf to contain font without style descriptions like -Roman or -Italic */
	strcpy(fontbuf,font);
	if ((p = strchr(fontbuf,'-')) || (p = strchr(fontbuf,'_')))
		*p = 0;

	fontlist = fontbuf;
	if      ((strcasecmp(font,"times-bold")==0)
	      || (strcasecmp(fontbuf,"timesb")==0))
		fontlist = "timesb;Timesb;TIMESB";

	else if ((strcasecmp(font,"times-italic")==0)
	      || (strcasecmp(fontbuf,"timesi")==0))
		fontlist = "timesi;Timesi;TIMESI";

	else if ((strcasecmp(font,"timesnewroman")==0)
	      || (strcasecmp(font,"timesnew")==0)
	      || (strcasecmp(font,"timesroman")==0)
	      || (strcasecmp(fontbuf,"times")==0))
		fontlist = "times;Times;TIMES";

	else if ((strcasecmp(font,"arial-bold")==0)
	      || (strcasecmp(fontbuf,"arialb")==0))
		fontlist = "arialb;Alialb;ARIALB";

	else if ((strcasecmp(font, "arial-italic")==0)
	      || (strcasecmp(fontbuf, "ariali")==0))
		fontlist = "ariali;Aliali;ARIALI";

	else if (strcasecmp(fontbuf,"helvetica")==0)
		fontlist = "helvetica;Helvetica;HELVETICA;arial;Arial;ARIAL";

	else if (strcasecmp(fontbuf,"arial")==0)
		fontlist = "arial;Arial;ARIAL";

	else if (strcasecmp(fontbuf,"courier")==0)
		fontlist = "courier;Courier;COURIER;cour";

	return fontlist;
#endif /* HAVE_GD_FONTCONFIG */
}

void gd_missingfont(char *err, char *fontreq)
{
	static char     *lastmissing = 0;
	static int      n_errors = 0;

	if (n_errors >= 20) return;
	if ((lastmissing == 0) || (strcmp(lastmissing,fontreq))) {
#if HAVE_GD_FONTCONFIG
		agerr(AGERR, "%s : %s\n",err,fontreq);
#else
		char *p=getenv("GDFONTPATH");
		if (!p) p = DEFAULT_FONTPATH;
		agerr(AGERR, "%s : %s in %s\n",err,fontreq,p);
#endif
		if (lastmissing) free(lastmissing);
		lastmissing = strdup(fontreq);
		n_errors++;
		if (n_errors >= 20) agerr(AGWARN, "(font errors suppressed)\n");
	}
}

extern gdFontPtr gdFontTiny, gdFontSmall, gdFontMediumBold, gdFontLarge, gdFontGiant;

static  void
gd_textline(point p, textline_t *line)
{
	char		*fontlist, *err;
	pointf		mp,ep;
	int		brect[8];
	char		*str = line->str;
	double		fontsz = cstk[SP].fontsz;
	gdFTStringExtra strex;

	strex.flags = gdFTEX_RESOLUTION;
	strex.hdpi = strex.vdpi = Dpi * Zoom * Scale;

	if (cstk[SP].pen == P_NONE)
		return;

	if (strstr(cstk[SP].fontfam,"/"))
		strex.flags |= gdFTEX_FONTPATHNAME;

	fontlist = gd_alternate_fontlist(cstk[SP].fontfam); 

	switch(line->just) {
		case 'l':
			mp.x = p.x;
			break;
		case 'r':
			mp.x = p.x - line->width;
			break;
		default:
		case 'n':
			mp.x = p.x - line->width / 2;
		break;
	}
	ep.y = mp.y = p.y;
        ep.x = mp.x + line->width;

	mp = gdpt(mp);
	if (fontsz <= FONTSIZE_MUCH_TOO_SMALL) {
                /* ignore entirely */
	} else if (fontsz <= FONTSIZE_TOO_SMALL) {
                /* draw line in place of text */ 
		ep = gdpt(ep);
		gdImageLine(im, ROUND(mp.x), ROUND(mp.y),
                        ROUND(ep.x), ROUND(ep.y),
			cstk[SP].pencolor);
        } else {
#ifdef HAVE_GD_FREETYPE
		err = gdImageStringFTEx(im, brect, cstk[SP].pencolor,
			fontlist, fontsz, Rot?(PI/2):0,
			ROUND(mp.x), ROUND(mp.y), str, &strex);
#if 0
gdImagePolygon (im, (gdPointPtr)brect, 4, cstk[SP].pencolor);
fprintf(stderr,"textline: font=%s size=%g width=%g dpi=%d width/dpi=%g\n",fontlist,fontsz,(double)(brect[4]-brect[0]),strex.hdpi,(((double)(brect[4]-brect[0]))/strex.hdpi));
#endif
		if (err) {
#endif
			/* revert to builtin fonts */
	    		gd_missingfont (err, cstk[SP].fontfam);
			if (fontsz <= 8.5) {
				gdImageString(im, gdFontTiny,
					ROUND(mp.x), ROUND(mp.y-9.),
					(unsigned char *)str, cstk[SP].pencolor);
			} else if (fontsz <= 9.5) {
				gdImageString(im, gdFontSmall,
					ROUND(mp.x), ROUND(mp.y-12.),
					(unsigned char *)str, cstk[SP].pencolor);
			} else if (fontsz <= 10.5) {
				gdImageString(im, gdFontMediumBold,
					ROUND(mp.x), ROUND(mp.y-13.),
					(unsigned char *)str, cstk[SP].pencolor);
			} else if (fontsz <= 11.5) {
				gdImageString(im, gdFontLarge,
					ROUND(mp.x), ROUND(mp.y-14.),
					(unsigned char *)str, cstk[SP].pencolor);
			} else {
				gdImageString(im, gdFontGiant,
					ROUND(mp.x), ROUND(mp.y-15.),
					(unsigned char *)str, cstk[SP].pencolor);
			}
#ifdef HAVE_GD_FREETYPE
		}
#endif
	}
}

/* return size in points (1/72 in) for the string in font at fontsz in points */
char * gd_textsize(textline_t *textline, char *fontname, double fontsz, char **fontpath)
{
	char		*fontlist,*err;
	int		brect[8];
	gdFTStringExtra strex;

	strex.flags = gdFTEX_XSHOW | gdFTEX_RESOLUTION;
	strex.xshow = NULL;
	strex.hdpi = strex.vdpi = 72;

	if (strstr(fontname,"/"))
		strex.flags |= gdFTEX_FONTPATHNAME;

	textline->width = 0.0;
	textline->xshow = NULL;

	fontlist = gd_alternate_fontlist(fontname);
	if (fontlist) {
		if (fontsz <= FONTSIZE_MUCH_TOO_SMALL) {
                	/* OK, but ignore text entirely */
			return NULL;
		} else if (fontsz <= FONTSIZE_TOO_SMALL) {
                	/* draw line in place of text */ 
                        /* fake a finite fontsize so that line length is calculated */
			fontsz = FONTSIZE_TOO_SMALL;
		} 
		
#ifdef HAVE_GD_FREETYPE
		/* call gdImageStringFT with null *im to get brect and to set font cache */
		err = gdImageStringFTEx(NULL, brect, -1, fontlist, 
			fontsz, 0, 0, 0, textline->str, &strex);

		if (err) 
			return err;

		if (strex.xshow) {
			/* transfer malloc'ed xshow string to textline */
			textline->xshow = strex.xshow;
			strex.xshow = NULL;
		}
		
		*fontpath = strex.fontpath;
			
		if (textline->str && textline->str[0]) {
			/* can't use brect on some archtectures if strlen 0 */
			textline->width = (double)(brect[4] - brect[0]);
		}
#else
	return "No Freetype support available";
#endif
	}
	return NULL;
}

static  void
gd_bezier(point* A, int n, int arrow_at_start, int arrow_at_end)
{
	pointf		p0, p1, V[4];
	int		i, j, step;
	int		style[20]; 
	int		pen, width;
	gdImagePtr	brush = NULL;

	if (cstk[SP].pen != P_NONE) {
		if (cstk[SP].pen == P_DASHED) {
			for (i = 0; i < 10; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 20; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 20);
			pen = gdStyled;
		} else if (cstk[SP].pen == P_DOTTED) {
			for (i = 0; i < 2; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 12; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 12);
			pen = gdStyled;
		} else {
			pen = cstk[SP].pencolor;
		}
#if 0
                if (cstk[SP].penwidth != WIDTH_NORMAL) {
			width=cstk[SP].penwidth;
                        brush = gdImageCreate(width,width);
                        gdImagePaletteCopy(brush, im);
                        gdImageFilledRectangle(brush,
                           0,0,width-1, width-1, cstk[SP].pencolor);
                        gdImageSetBrush(im, brush);
			if (pen == gdStyled) pen = gdStyledBrushed;      
			else pen = gdBrushed;      
		}
#else
		width = cstk[SP].penwidth;
		gdImageSetThickness(im, width);
#endif
		V[3].x = A[0].x; V[3].y = A[0].y;
		for (i = 0; i+3 < n; i += 3) {
			V[0] = V[3];
			for (j = 1; j <= 3; j++) {
				V[j].x  = A[i+j].x; V[j].y = A[i+j].y;
			}
			p0 = gdpt(V[0]); 
			for (step = 1; step <= BEZIERSUBDIVISION; step++) {
				p1 = gdpt(Bezier(V, 3, (double)step/BEZIERSUBDIVISION, NULL, NULL));
				gdImageLine(im, ROUND(p0.x), ROUND(p0.y),
					ROUND(p1.x), ROUND(p1.y), pen);
				p0 = p1;
			}
		}
		if (brush)
			gdImageDestroy(brush);
	}
}

static  void
gd_polygon(point *A, int n, int filled)
{
	pointf		p;
	int		i;
	gdPoint		*points;
	int		style[20];
	int		pen, width;
	gdImagePtr	brush = NULL;

	if (cstk[SP].pen != P_NONE) {
		if (cstk[SP].pen == P_DASHED) {
			for (i = 0; i < 10; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 20; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 20);
			pen = gdStyled;
		} else if (cstk[SP].pen == P_DOTTED) {
			for (i = 0; i < 2; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 12; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 12);
			pen = gdStyled;
		} else {
			pen = cstk[SP].pencolor;
		}
#if 1
		/* use brush instead of Thickness to improve end butts */
		gdImageSetThickness(im, WIDTH_NORMAL);
                if (cstk[SP].penwidth != WIDTH_NORMAL) {
			width=cstk[SP].penwidth * CompScale;
                        brush = gdImageCreate(width,width);
                        gdImagePaletteCopy(brush, im);
                        gdImageFilledRectangle(brush,
                           0,0,width-1, width-1, cstk[SP].pencolor);
                        gdImageSetBrush(im, brush);
			if (pen == gdStyled) pen = gdStyledBrushed;      
			else pen = gdBrushed;      
		}
#else
		width = cstk[SP].penwidth;
		gdImageSetThickness(im, width);
#endif
		points = N_GNEW(n,gdPoint);
		for (i = 0; i < n; i++) {
			p.x = A[i].x; p.y = A[i].y;
			p = gdpt(p);
			points[i].x = ROUND(p.x); points[i].y = ROUND(p.y);
		}
		if (filled) {
			gdImageFilledPolygon(im, points, n, cstk[SP].fillcolor);
		}

		gdImagePolygon(im, points, n, pen);
		free(points);
		if (brush)
			gdImageDestroy(brush);
	}
}

static  void
gd_ellipse(point p, int rx, int ry, int filled)
{
	pointf		mp;
	int		i;
	int		style[40];  /* need 2* size for arcs, I don't know why */
	int		pen, width;
	gdImagePtr	brush = NULL;

	if (cstk[SP].pen != P_NONE) {
		if (cstk[SP].pen == P_DASHED) {
			for (i = 0; i < 20; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 40; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 40);
			pen = gdStyled;
		} else if (cstk[SP].pen == P_DOTTED) {
			for (i = 0; i < 2; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 24; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 24);
			pen = gdStyled;
		} else {
			pen = cstk[SP].pencolor;
		}
#if 1
		/* use brush instead of Thickness to improve outline appearance */
		gdImageSetThickness(im, WIDTH_NORMAL);
                if (cstk[SP].penwidth != WIDTH_NORMAL) {
			width = cstk[SP].penwidth;
                        brush = gdImageCreate(width,width);
                        gdImagePaletteCopy(brush, im);
                        gdImageFilledRectangle(brush,
                           0,0,width-1, width-1, cstk[SP].pencolor);
                        gdImageSetBrush(im, brush);
			if (pen == gdStyled) pen = gdStyledBrushed;      
			else pen = gdBrushed;      
		}
#else
		width = cstk[SP].penwidth;
		gdImageSetThickness(im, width);
#endif
		if (Rot) {int t; t = rx; rx = ry; ry = t;}
		mp.x = p.x; mp.y = p.y;
		mp = gdpt(mp);
		if (filled) {
			gdImageFilledEllipse(im, ROUND(mp.x), ROUND(mp.y),
				ROUND(CompScale*(rx + rx)), ROUND(CompScale*(ry + ry)),
				cstk[SP].fillcolor);
		}
		gdImageArc(im, ROUND(mp.x), ROUND(mp.y),
			ROUND(CompScale*(rx + rx)), ROUND(CompScale*(ry + ry)), 0, 360, pen);
		if (brush)
			gdImageDestroy(brush);
	}
}

static  void
gd_polyline(point* A, int n)
{
	pointf		p, p1;
	int		i;
	int		style[20];
	int		pen, width;
	gdImagePtr	brush = NULL;

	if (cstk[SP].pen != P_NONE) {
		if (cstk[SP].pen == P_DASHED) {
			for (i = 0; i < 10; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 20; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 20);
			pen = gdStyled;
		} else if (cstk[SP].pen == P_DOTTED) {
			for (i = 0; i < 2; i++)
				style[i] = cstk[SP].pencolor;
			for (; i < 12; i++)
				style[i] = transparent;
			gdImageSetStyle(im, style, 12);
			pen = gdStyled;
		} else {
			pen = cstk[SP].pencolor;
		}
#if 0
                if (cstk[SP].penwidth != WIDTH_NORMAL) {
			width = cstk[SP].penwidth;
                        brush = gdImageCreate(width,width);
                        gdImagePaletteCopy(brush, im);
                        gdImageFilledRectangle(brush,
                           0,0,width-1,width-1,cstk[SP].pencolor);
                        gdImageSetBrush(im, brush);
			if (pen == gdStyled) pen = gdStyledBrushed;      
			else pen = gdBrushed;      
		}
#else
		width = cstk[SP].penwidth;
		gdImageSetThickness(im, width);
#endif
		p.x = A[0].x;
		p.y = A[0].y;
		p = gdpt(p);
		for (i = 1; i < n; i++) {
			p1.x = A[i].x;
			p1.y = A[i].y;
			p1 = gdpt(p1);
			gdImageLine(im, ROUND(p.x), ROUND(p.y),
				ROUND(p1.x), ROUND(p1.y), pen);
			p.x = p1.x;
			p.y = p1.y;
		}
		if (brush)
			gdImageDestroy(brush);
	}
}

static gdImagePtr loadshapeimage(char *name)
{
	gdImagePtr	rv = 0;
	char	*shapeimagefile,*suffix;
	FILE	*in = NULL;

	if ((shapeimagefile=safefile(name))) {
#ifndef MSWIN32
		in = fopen (shapeimagefile, "r");
#else
		in = fopen (shapeimagefile, "rb");
#endif
	}
	if (!in) 
		agerr(AGERR, "couldn't open image file %s\n",shapeimagefile);
	else {
		suffix = strrchr(shapeimagefile,'.');
		if (!suffix) suffix = shapeimagefile; else suffix++;
		if (!strcasecmp(suffix,"wbmp")) rv = gdImageCreateFromWBMP(in);
#ifdef HAVE_GD_GIF
		else if (!strcasecmp(suffix,"gif")) rv = gdImageCreateFromGif(in);
#endif
#ifdef HAVE_GD_JPEG
		else if (!strcasecmp(suffix,"jpeg")||!strcasecmp(suffix,"jpg")) rv = gdImageCreateFromJpeg(in);
#endif
#ifdef HAVE_GD_PNG
		else if (!strcasecmp(suffix,"png")) rv = gdImageCreateFromPng(in);
#endif
#ifdef HAVE_GD_XPM
		else if (!strcasecmp(suffix,"xbm")) rv = gdImageCreateFromXbm(in);
#endif
		else agerr(AGERR, "image file %s suffix not recognized\n",name);
		fclose(in);
		if (!rv) agerr(AGERR, "image file %s contents were not recognized\n",name);
	}
	return rv;
}

typedef struct imagerec_s {
    Dtlink_t        link;
    char            *name;
	gdImagePtr		im;
} imagerec_t;


static void imagerec_free(Dict_t *dict, Void_t *p, Dtdisc_t *disc)
{
	gdImagePtr im = ((imagerec_t*)p)->im;
	if (im) gdImageDestroy(im);
}

static Dtdisc_t ImageDictDisc = {
    offsetof(imagerec_t,name),   /* key */
    -1,                     /* size */
    0,                      /* link offset */
    NIL(Dtmake_f),
	imagerec_free,
    NIL(Dtcompar_f),
    NIL(Dthash_f),
    NIL(Dtmemory_f),
    NIL(Dtevent_f)
};

static gdImagePtr getshapeimage(char *name)
{
	imagerec_t	probe, *val;
    if (!name) return 0;  /* cdt does not like NULL keys */
	if (!ImageDict) ImageDict = dtopen(&ImageDictDisc,Dttree);
	probe.name = name;
	val = dtsearch(ImageDict,&probe);
	if (!val) {
		val = GNEW(imagerec_t);
		val->name = name;
		val->im = loadshapeimage(name);
		dtinsert(ImageDict,val);
	}
	return val->im;
}

static  void
gd_user_shape(char *name, point *A, int n, int filled)
{
	gdImagePtr im2 = 0;
	pointf	destul, destlr;
	pointf	ul, lr;		/* upper left, lower right */
	double	sx, sy;		/* target size */
	double  scalex, scaley;  /* scale factors */
	int		i;
	char *shapeimagefile;

	shapeimagefile = agget(Curnode,"shapefile");
	im2 = getshapeimage(shapeimagefile);
	if (im2) {
		/* compute dest origin and size */
		ul.x = lr.x = A[0].x; ul.y = lr.y = A[0].y;
		for (i = 1; i < n; i++) {
			if (ul.x > A[i].x) ul.x = A[i].x;
			if (ul.y < A[i].y) ul.y = A[i].y;
			if (lr.y > A[i].y) lr.y = A[i].y;
			if (lr.x < A[i].x) lr.x = A[i].x;
		}
		destul = gdpt(ul);
		destlr = gdpt(lr);
		scalex = (destlr.x - destul.x)/(double)(im2->sx);
		scaley = (destlr.y - destul.y)/(double)(im2->sy);
		/* keep aspect ratio fixed by just using the smaller scale */
		if (scalex < scaley) {
			sx = ROUND(im2->sx * scalex);
			sy = ROUND(im2->sy * scalex);
		}
		else {
			sx = ROUND(im2->sx * scaley);
			sy = ROUND(im2->sy * scaley);
		}
		gdImageCopyResized(im,im2,ROUND(destul.x),ROUND(destul.y),0,0,sx,sy,im2->sx,im2->sy);
	}
}

point gd_user_shape_size(node_t *n, char *shapeimagefile)
{
	point		rv;
	gdImagePtr	im;

	im = getshapeimage(shapeimagefile);
	if (im) {rv.x = im->sx / DevScale; rv.y = im->sy / DevScale; }
	else rv.x = rv.y = -1;
	return rv;
}

codegen_t	GD_CodeGen = {
	0, /* gd_reset */
	gd_begin_job, gd_end_job,
	gd_begin_graph_to_file, gd_end_graph_to_file, 
	gd_begin_page, gd_end_page,
	0, /* gd_begin_layer */ 0, /* gd_end_layer */
	0, /* gd_begin_cluster */ 0, /* gd_end_cluster */
	0, /* gd_begin_nodes */ 0, /* gd_end_nodes */
	0, /* gd_begin_edges */ 0, /* gd_end_edges */
	gd_begin_node, gd_end_node,
	0, /* gd_begin_edge */ 0, /* gd_end_edge */
	gd_begin_context, gd_end_context,
	0, /* gd_begin_anchor */ 0, /* gd_end_anchor */
	gd_set_font, gd_textline,
	gd_set_pencolor, gd_set_fillcolor, gd_set_style,
	gd_ellipse, gd_polygon,
	gd_bezier, gd_polyline,
	0, /* bezier_has_arrows */
	0, /* gd_comment */
	0, /* gd_textsize */
	gd_user_shape,
	0 /* gd_user_shape_size */
};

codegen_t	memGD_CodeGen = {		/* see tcldot */
	0, /* gd_reset*/
	gd_begin_job, gd_end_job,
	gd_begin_graph_to_memory, gd_end_graph_to_memory,
	gd_begin_page, gd_end_page,
	0, /* gd_begin_layer */ 0, /* gd_end_layer */
	0, /* gd_begin_cluster */ 0, /* gd_end_cluster */
	0, /* gd_begin_nodes */ 0, /* gd_end_nodes */
	0, /* gd_begin_edges */ 0, /* gd_end_edges */
	0, /* gd_begin_node */ 0, /* gd_end_node */
	0, /* gd_begin_edge */ 0, /* gd_end_edge */
	gd_begin_context, gd_end_context,
	0, /* gd_begin_anchor */ 0, /* gd_end_anchor */
	gd_set_font, gd_textline,
	gd_set_pencolor, gd_set_fillcolor, gd_set_style,
	gd_ellipse, gd_polygon,
	gd_bezier, gd_polyline,
	0, /* bezier_has_arrows */
	0, /* gd_comment */
	0, /* gd_textsize */
	gd_user_shape,
	0 /* gd_user_shape_size */
};
