/*
 * %W% %E%
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsclient;
import javax.jms.*;
import com.sun.messaging.jmq.io.*;

import com.sun.messaging.AdministeredObject;
import java.io.IOException;


public class WriteChannel {

    private boolean debug = Debug.debug;

    private ProtocolHandler protocolHandler = null;
    private ReadChannel readChannel = null;
    private InterestTable interestTable = null;
    private ConnectionImpl connection = null;
    /**
     * flow control vars.
     */
    public static final String JMQSize = "JMQSize";

    private boolean shouldPause = false;
    private int flowCount = -1;

    protected boolean turnOffFlowControl = false;

    public WriteChannel ( ConnectionImpl conn ) {
        this.connection = conn;
        this.protocolHandler = conn.getProtocolHandler();
        this.readChannel = conn.getReadChannel();
        this.interestTable = conn.getInterestTable();

        if ( System.getProperty("NoimqProducerFlowControl") != null ) {
            turnOffFlowControl = true;
        }
    }

    protected void
    setReadChannel (ReadChannel readChannel) {
        this.readChannel = readChannel;
    }

    /**
     * Register interest to the broker.
     */
    protected void
    addInterest (Consumer consumer) throws JMSException {
        protocolHandler.addInterest ( consumer );
    }

    protected void
    removeInterest (Consumer consumer) throws JMSException {
        protocolHandler.removeInterest(consumer);
    }

    protected void
    unsubscribe (String durableName) throws JMSException {
        protocolHandler.unsubscribe (durableName);
    }

    protected void
    writeJMSMessage ( Message message ) throws JMSException {
        if ( turnOffFlowControl &&
            connection.getBrokerProtocolLevel() < PacketType.VERSION350 ) {
            protocolHandler.writeJMSMessage (message);
        } else {
            sendWithFlowControl (message);
        }
    }

    /**
     * The follwing methods are for producer flow control.
     * This method is called by ReadChannel when it received
     * RESUME_FLOW packet from the broker.
     */
    protected void
    updateFlowControl (ReadOnlyPacket pkt) throws JMSException {

        int jmqSize = -1;

        try {
            Integer prop = (Integer) pkt.getProperties().get(JMQSize);
            if ( prop != null ) {
                jmqSize = prop.intValue();
            }
        } catch (IOException e) {
        connection.exceptionHandler.handleException(e, AdministeredObject.cr.X_PACKET_GET_PROPERTIES, true);

        } catch (ClassNotFoundException e) {
        connection.exceptionHandler.handleException(e, AdministeredObject.cr.X_PACKET_GET_PROPERTIES, true);
        }

        setFlowCount (jmqSize);
    }

    private synchronized void setFlowCount (int jmqSize) {
        flowCount = jmqSize;
        notifyAll();
    }

    /**
     * Send a message with flow control feature.
     */
    protected void sendWithFlowControl (Message message) throws JMSException {

        /**
         * wait until allow to send.
         */
        pause(message);

        /**
         * send message.
         */
        protocolHandler.writeJMSMessage (message);
    }

    protected synchronized void pause (Message message) {

        while (flowCount == 0) {

            if (debug) {
                Debug.println(
                    "WriteChannel : Waiting for RESUME_FLOW");
            }

            try {
                wait();
            }
            catch (InterruptedException e) {
                ;
            }
        }

        if (debug) {
            Debug.println("WriteChannel : wait() returned...");
        }

        if (flowCount > 0) {
            flowCount--;
        }

        if (flowCount == 0) {
            ( (MessageImpl) message).getPacket().setFlowPaused(true);
        }
        else {
            ( (MessageImpl) message).getPacket().setFlowPaused(false);
        }
    }

    protected void close() {
        if (debug) {
            Debug.println(
                "WriteChannel.close() : Waking up blocked producers");
        }

        // Set flow count to -1 to unblock the producers waiting for
        // RESUME_FLOW. There is no need to throw an exception
        // directly from the pause() method because
        // protocolHandler.writeJMSMessage() is guaranteed to fail
        // down the line...

        setFlowCount(-1);
    }

}
