/*
 * @(#)TransactionList.java	1.63 02/09/06
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data;

import java.util.*;
import java.io.*;

import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.persist.*;
import com.sun.messaging.jmq.jmsserver.data.handlers.TransactionHandler;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.service.ConnectionManager;
import com.sun.messaging.jmq.jmsserver.util.lists.RemoveReason;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.util.*;

public class TransactionList
{
    List inuse_translist = null;
    Hashtable translist = null;

    Logger logger = Globals.getLogger();

    Hashtable xidTable = null;  // Maps XIDs to UIDs
    Store store = null; // persistence access

    HashMap brokerToTxn = new HashMap();
    HashMap txnToBroker = new HashMap();


    public TransactionList(Store store)
    {
         this.store = store;
         this.translist = new Hashtable();
         this.xidTable = new Hashtable();
         this.inuse_translist = new ArrayList();
         try {
             loadTransactions();
         } catch (Exception ex) {
             logger.logStack(Logger.INFO,
                  BrokerResources.W_TRANS_LOAD_ERROR, ex);
         }
    }

    /**
     * called when a remote transaction (and its acks) are
     * added to the broker
     */
    public synchronized void addWatchedRemoteTxn(BrokerAddress bkr,
             TransactionUID tuid) {
        List l = (List)brokerToTxn.get(bkr);
        if (l == null) {
            l = new ArrayList();
            brokerToTxn.put(bkr, l);
        }
        l.add(tuid);
        txnToBroker.put(tuid, bkr);
    }

    /**
     * called when a remote transaction (and its acks) are
     * removed from the broker
     */
    public synchronized void removeWatchedRemoteTxn(TransactionUID tuid)
    {
        BrokerAddress addr = (BrokerAddress)txnToBroker.remove(tuid);
        if (addr == null) {
            logger.log(logger.DEBUG,"Unknown broker for " + tuid);
            return;
        }
        List l = (List)brokerToTxn.get(addr);
        l.remove(tuid);
        if (l.size() == 0)
            brokerToTxn.remove(addr);
    }
   
    /**
     * retrieves all txn's associated with this broker for cleanup
     */
    public synchronized List getRemoteBrokers(BrokerAddress addr)
    {
        return (List)brokerToTxn.get(addr);
    }

    /**
     * clears out all local txn data associated with this ack
     */
    public synchronized void clearRemoteBrokers(BrokerAddress addr)
    {
        List l = (List) brokerToTxn.remove(addr);
        Iterator itr = l.iterator();
        while (itr.hasNext()) {
            TransactionUID uid = (TransactionUID) itr.next();
            txnToBroker.remove(uid);
        }
    }



    public synchronized Hashtable getDebugState(TransactionUID id) {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) {
            Hashtable ht = new Hashtable();
            ht.put(id.toString(), "UNKNOWN TID");
            return ht;
         }
         return ti.getDebugState();         
    }

    public TransactionState addRemoteTxn(BrokerAddress bkr, TransactionUID tuid)
        throws BrokerException
    {
        logger.log(logger.DEBUG,"addRemoteTxn " + tuid);
        TransactionState state = new TransactionState(
                 AutoRollbackType.NOT_PREPARED, 0, false);
        state.setRemoteBroker(bkr);
        state.setState(TransactionState.STARTED);
        addTransactionID(tuid, state, !Globals.getHAEnabled());
        addWatchedRemoteTxn(bkr, tuid);
        return state;
    }


    public void rollbackRemoteTxn(TransactionUID tuid)
        throws BrokerException
    {
        logger.log(logger.DEBUG,"rollbackRemoteTxn " + tuid);
        // check state, if not rolledback, set it
        // remove transaction we dont have ANY produced messages
        //    since this is a remote transaction
        // re-process delivering each message
        //
        TransactionState state= retrieveState(tuid);
        state.setState(TransactionState.ROLLEDBACK);
        Map m = retrieveConsumedMessages(tuid);
        Iterator itr = new HashSet(m.keySet()).iterator();
        while (itr.hasNext()) {
            SysMessageID sysid = (SysMessageID)itr.next();
            ArrayList cuids = (ArrayList) m.get(sysid);
            PacketReference ref = Destination.get(sysid);
            if (sysid == null) continue;
            HashSet s = new HashSet();
            s.add(cuids);
            ref.getDestination().forwardMessage(s, ref);
        }
        removeTransactionAck(tuid);
        removeTransactionID(tuid);
        removeWatchedRemoteTxn(tuid);

    }

    public void commitRemoteTxn(TransactionUID tuid)
        throws BrokerException
    {
        logger.log(logger.DEBUG,"commitRemoteTxn " + tuid);
         TransactionState state= retrieveState(tuid);
         if (state == null) {
             logger.log(Logger.DEBUG,"XXX ACK - state is not there for " + tuid);
             throw new BrokerException("unknown txn " + tuid,
                             Status.GONE);
         } else if (state.getState() != TransactionState.PREPARED) {
             // ok
             state.setState(TransactionState.COMMITTED);
         }
         // ok now explicity ack messages
         // its consumer only
         Map m = retrieveConsumedMessages(tuid);
         Iterator itr = new HashSet(m.keySet()).iterator();
         while (itr.hasNext()) {
             SysMessageID sysid = (SysMessageID)itr.next();
             ArrayList cuids = (ArrayList) m.get(sysid);
             PacketReference ref = Destination.get(sysid);
             if (sysid == null) continue;
             Iterator citr = cuids.iterator();
             while (citr.hasNext()) {
                 ConsumerUID cuid = (ConsumerUID)citr.next();
                 Consumer c = Consumer.getConsumer(cuid);
                 try {
                     if (ref.acknowledged(cuid, 
                             c.getStoredConsumerUID(), true, true,
                             null)) {
                         Destination d = ref.getDestination();
                         d.removeMessage(sysid, RemoveReason.ACKNOWLEDGED);
                     }
                 } catch (IOException ex) {
                     logger.logStack(Logger.DEBUG,"IO exception", ex);
                 }
             }
         }
         removeTransactionAck(tuid);
         removeTransactionID(tuid);
         removeWatchedRemoteTxn(tuid);
    }

    // returns transaction acks
    public Map loadTakeoverTxns(List txns)
        throws BrokerException, IOException
    {
        logger.log(Logger.DEBUG,"Processing Transactions " +
                  txns.size() + " after takeover");
        // hey process through the states
        Iterator itr = txns.iterator();
        Map acks = new HashMap();
        while (itr.hasNext()) {
            TransactionUID tid = (TransactionUID) itr.next();
            TransactionState ts =null;
            try {
                ts = store.getTransactionState(tid);
            } catch (BrokerException ex) {
                itr.remove();
                continue;
            }
            // XXX make debug ?
            try {
                 logger.log(Logger.DEBUG, "Processing transaction " 
                               + tid + " state=" + ts);
                 if (tid == null) continue;
                 if (ts.getState() == TransactionState.COMMITTED) {
                     // remove txn
                     store.removeTransactionAck(tid, false);
                     store.removeTransaction(tid, false);
                     continue;
                 } else if (ts.getState() != TransactionState.PREPARED) {
                     store.removeTransactionAck(tid, false);
                 }
                 TransactionAcknowledgement  ta[] = store.
                                   getTransactionAcks(tid);
                 logger.log(Logger.DEBUG, "Processing transaction acks " 
                               + tid + " number=" + ta.length);
                 List l = Arrays.asList(ta);
                 acks.put(tid, l);
                 addTransactionID(tid, ts, false);
                 setTransactionIDStored(tid);
                 
            } catch (Exception ex) {
                 logger.logStack(Logger.ERROR,
                       BrokerResources.E_INTERNAL_BROKER_ERROR,
                       "error taking over " 
                             + tid, ex);
                 acks.remove(tid);
            
            }
        }
        return acks;
    }


    public Hashtable getDebugState() {
         Hashtable ht = new Hashtable();

         ht.put("TransactionCount", new Integer(translist.size()));
         Iterator itr = translist.entrySet().iterator();
         while (itr.hasNext()) {
             Map.Entry me = (Map.Entry)itr.next();
             String key = me.getKey().toString();
             TransactionInformation ti = (TransactionInformation)
                         me.getValue();
             if (ti == null) continue;
             ht.put(key, ti.getDebugState());
         }
         Vector v = new Vector();
         for (int i=0; i < inuse_translist.size(); i ++) {
             v.add(inuse_translist.get(i).toString());
         }
         if (v.size() > 0) {
             ht.put("inUse", v);
         } else {
             ht.put("inUse", "none");
         }

         Hashtable x = new Hashtable();

         itr = xidTable.entrySet().iterator();
         while (itr.hasNext()) {
             Map.Entry me = (Map.Entry)itr.next();
             x.put(me.getKey().toString(), me.getValue().toString());
         }
         if (x.size() > 0) {
             ht.put("XIDs", v);
         } else {
             ht.put("XIDs", "none");
         }
         return ht;

    }


    public void addTransactionID(TransactionUID id,
                                              TransactionState ts)
        throws BrokerException 
    {
        addTransactionID(id, ts, true);
    }

    public synchronized TransactionUID getTransaction(SysMessageID id) {
        Iterator itr = translist.values().iterator();
        while (itr.hasNext()) {
            TransactionInformation info = (TransactionInformation)itr.next();
            TransactionState ts = info.getState();
            SysMessageID creator = ts.getCreator();
            if (creator != null && creator.equals(id))
                return info.getTID();
        }
        return null;

    }

    public synchronized void setTransactionIDStored(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)
             translist.get(id);
        if (ti == null) return;
        ti.setPersisted(true);
    }
    public void addTransactionID(TransactionUID id,
                                              TransactionState ts,
                                              boolean persist) 
        throws BrokerException 
    {
        synchronized(this) {
            if (inuse_translist.contains(id)) {
                throw new BrokerException(
                    Globals.getBrokerResources().getKString(
                        BrokerResources.X_TRANSACTIONID_INUSE,
                        id.toString()),
                    BrokerResources.X_TRANSACTIONID_INUSE,
                    (Throwable) null,
                    Status.CONFLICT);
            }
    
            // If transaction is an XA (has an xid) save it for reverse mapping
            JMQXid xid = ts.getXid();
            if (xid != null) {
                if (xidTable.get(xid) != null) {
                    // Xid already in use
                    throw new BrokerException(
                        Globals.getBrokerResources().getKString(
                        BrokerResources.X_TRANSACTIONID_INUSE,
                        id.toString() + "[Xid=" + xid.toString() + "]"),
                        BrokerResources.X_TRANSACTIONID_INUSE,
                        (Throwable) null,
                        Status.CONFLICT);
                } else {
                    xidTable.put(xid, id);
                }
            }
        }

        try {
            if (persist) {
                store.storeTransaction(id, ts, false);
            }
        } catch (Exception ex) {
            throw new BrokerException(
                Globals.getBrokerResources().getKString(
                    BrokerResources.X_TRANSACTION_STORE_ERROR,
                    id.toString()),
                BrokerResources.X_TRANSACTION_STORE_ERROR,
                (Throwable) ex,
                Status.ERROR);
        }
        synchronized(this) {
            inuse_translist.add(id);
            translist.put(id, new TransactionInformation(id, ts, persist));
        }
    }

    public void removeTransactionID(TransactionUID id)
        throws BrokerException 
    {
        // If XA (has Xid) remove it from reverse mapping
        TransactionState ts = retrieveState(id);
        if (ts != null && ts.getXid() != null) {
            xidTable.remove(ts.getXid());
        }
        TransactionInformation state = 
                (TransactionInformation)translist.remove(id);

        if (state == null) return;
        try {
            if (state.getPersisted()) {
                store.removeTransaction(id, false);
            }
        } catch (IOException ex) {
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"unable to remove the transaction id " + id), ex);
        }
    }

    public void removeTransactionAck(TransactionUID id)
        throws BrokerException 
    {
        try {
            TransactionInformation tinfo = (TransactionInformation)
                   translist.get(id);
            if (tinfo != null && tinfo.getPersisted()) {
                store.removeTransactionAck(id, false);
            }
        } catch (Exception ex) {
            throw new BrokerException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION,": unable to remove the transaction ack for " + id), ex);
        }

        // remove the transaction from the list of valid ids
        synchronized(this) {
            inuse_translist.remove(id);
        }
    }

    public void addMessage(TransactionUID id, SysMessageID sysid)
        throws BrokerException
    {
        addMessage(id, sysid, false);
    }

    public synchronized void addMessage(TransactionUID id, SysMessageID sysid, boolean anyState)
        throws BrokerException
    {
        TransactionInformation info = (TransactionInformation)translist.get(id);
        if (info == null) {
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"received message with Unknown Transaction ID "+ id + ": ignoring message"),
                Status.GONE);
        }
        if (info.getState().getState() == TransactionState.TIMED_OUT) {
            // bad state
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id + ": is has timed out "),
                Status.TIMEOUT);

        }
        if (!anyState && info.getState().getState() != TransactionState.STARTED) {
            // bad state
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id + ": is not started, ignoring"),
                Status.PRECONDITION_FAILED);

        }
        info.addPublishedMessage(sysid);
    }

    public Hashtable getTransactionMap(TransactionUID tid, boolean ext) 
         throws BrokerException
    {
        TransactionInformation info = (TransactionInformation)translist.get(
              tid);
        if (info == null) {
            throw new BrokerException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION,"received acknowledgement with Unknown Transaction ID "+ tid ),
                  Status.GONE);
        }
        TransactionState ts = info.getState();
        if (ts == null) {
            throw new BrokerException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION,"received acknowledgement with Unknown Transaction state "+ tid ),
                  Status.ERROR);
        }

        Hashtable ht = new Hashtable();
        ht.put("JMQAutoRollback", new Integer(ts.getType().intValue()));
        if (ts.getXid() != null)
            ht.put("JMQXid", ts.getXid().toString());
        ht.put("JMQSessionLess", new Boolean(ts.isSessionLess()));
        ht.put("JMQCreateTime", new Long(ts.getCreationTime()));
        ht.put("JMQLifetime", new Long(ts.getLifetime()));
        if (ext) { // client protocol specifies +1A
            ht.put("State", new Integer(ts.getState() + 1));
        } else {
            ht.put("State", new Integer(ts.getState()));
        }

        return ht;
    }

    public synchronized boolean checkAcknowledgement(TransactionUID tid, 
       SysMessageID sysid, ConsumerUID id)
         throws BrokerException
    {
        TransactionInformation info = (TransactionInformation)translist.get(tid);
        if (info == null) {
            throw new BrokerException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION,"received acknowledgement with Unknown Transaction ID "+ tid + ": ignoring message"),
                  Status.GONE);
        }
        if (info.getState().getState() == TransactionState.TIMED_OUT) {
            // bad state
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id + ": is has timed out "),
                Status.TIMEOUT);

        }
        
        return info.checkConsumedMessage(sysid, id);
    }


    public synchronized void addAcknowledgement(TransactionUID tid, 
       SysMessageID sysid, ConsumerUID id, ConsumerUID sid)
         throws BrokerException
    {
        PacketReference pr = Destination.get(sysid);
        boolean persist = sid.shouldStore() && pr != null &&
               pr.isPersistent()&& pr.isLocal();
        addAcknowledgement(tid, sysid, id, sid, persist);
    }


    public void addAcknowledgement(TransactionUID tid, 
       SysMessageID sysid, ConsumerUID id, ConsumerUID sid, boolean persist)
         throws BrokerException
    {
        synchronized(this) {
            TransactionInformation info = (TransactionInformation)translist.get(tid);
            if (info == null) {
                throw new BrokerException(Globals.getBrokerResources().getString(
               BrokerResources.X_INTERNAL_EXCEPTION,"received acknowledgement with Unknown Transaction ID "+ tid + ": ignoring message"),
                      Status.GONE);
            }
            if (info.getState().getState() == TransactionState.TIMED_OUT) {
                // bad state
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id + ": is has timed out "),
                    Status.TIMEOUT);
    
            }
            if (info.getState().getState() != TransactionState.STARTED) {
                // bad state
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id + ": is not started, ignoring"),
                    Status.PRECONDITION_FAILED);

            }
            info.addConsumedMessage(sysid, id, sid);
        }

        if (persist) {
            store.storeTransactionAck(tid, 
                 new TransactionAcknowledgement(sysid, id, sid), false);
        }
    }

    public synchronized List retrieveSentMessages(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return null;
        return ti.getPublishedMessages();
    }

    public synchronized int retrieveNSentMessages(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return 0;
        return ti.getNPublishedMessages();
    }

    public synchronized HashMap retrieveConsumedMessages(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return null;
        return ti.getConsumedMessages();
    }

    // this retrieves a mapping of consumerUID to stored consumerUID
    // the stored consumerUId is the ID we use to store ack info
    // for a durable
    public synchronized HashMap retrieveStoredConsumerUIDs(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return null;
        return ti.getStoredConsumerUIDs();
    }
    public synchronized int retrieveNConsumedMessages(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return 0;
        return ti.getNConsumedMessages();
    }

    // Get the state of a transaction
    public synchronized TransactionState retrieveState(TransactionUID id) {
        if (id == null) return null;
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti == null) return null;
        return ti.getState();
    }

    // Update the state of a transaction.
    // If persist is true, then the state is updated in the persistent store
    // as well.
    public TransactionState
                updateState(TransactionUID id, int state, boolean persist)
        throws BrokerException {

        TransactionState ts = null;
        synchronized(this) {
            TransactionInformation ti = (TransactionInformation)translist.get(id);
            if (ti == null) {
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,
                    "updateState(): Unknown transaction: " + id));
            }
    
            ts = ti.getState();
            if (ts == null) {
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,
                    "updateState(): No state for transaction: " + id),
                    Status.GONE);
            }
            // timed out
            if (ts.getState() == TransactionState.TIMED_OUT) {
                // bad state
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,"Transaction "+ id 
                          + ": is has timed out"),
                    Status.TIMEOUT);

            }
            ts.setState(state);
        }

        // Update state in persistent store
        if (persist) {
            try {
                store.updateTransactionState(id, ts,
                    Destination.PERSIST_SYNC);
            } catch (IOException e) {
                throw new BrokerException(null, e);
            }
        }

        return ts;
    }

    /**
     * Given an Xid this routine converts it to an internal Transaction
     * Resource ID.
     */
    public synchronized TransactionUID xidToUID(JMQXid xid) {
        return (TransactionUID)xidTable.get(xid);
    }

    /**
     * Given an Xid this routine converts it to an internal Transaction
     * Resource ID. This method performs a linear search of all active
     * transactions, so it is not fast.
     */
    public synchronized TransactionUID slowXidToUID(JMQXid xid) {
        Set entries = translist.entrySet();
        Iterator iter = entries.iterator();
        TransactionState ts = null;
        JMQXid keyxid = null;

        while (iter.hasNext()) {
            Map.Entry entry = (Map.Entry)iter.next();
            TransactionInformation ti = (TransactionInformation)entry.getValue();
            ts = ti.getState();
            keyxid = ts.getXid();
            if (keyxid != null && xid.equals(keyxid)) {
                return (TransactionUID)entry.getKey();
            }
        }
        return null;
    }

    /**
     * Given a TransactionUID this routine returns the corresponding Xid.
     */
    public synchronized JMQXid UIDToXid(TransactionUID uid) {
        TransactionState ts = retrieveState(uid);
        if (ts != null) {
            return ts.getXid();
        } else {
            return null;
        }
    }

    /**
     * Get a list of transactions that are in the specified state.
     * This method is a bit expensive, so it shouldn't be called often.
     * It is expected that this will only be called when processing a
     * "recover" or an admin request.
     *
     * If state is < 0 get all transaction IDs.
     *
     * Returns a vector of TransactionUIDs.
     */
    public synchronized Vector getTransactions(int state) {
        Set keyset = translist.keySet();
        Iterator iter = keyset.iterator();
        TransactionUID tid = null;
        TransactionState ts = null;
        Vector v = new Vector();

        while (iter.hasNext()) {
            tid = (TransactionUID)iter.next();
            if (state < 0) {
                v.add(tid);
            } else {
                ts = retrieveState(tid);
                if (ts != null && ts.getState() == state) {
                    v.add(tid);
                }
            }
        }
        return v;
    }

    public void addOrphanAck(TransactionUID id,
           SysMessageID sysid, ConsumerUID uid)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti != null) {
            ti.addOrphanAck(sysid, uid);
        }
    }

    public Map getOrphanAck(TransactionUID id)
    {
        TransactionInformation ti = (TransactionInformation)translist.get(id);
        if (ti != null) {
            return ti.getOrphanAck();
        }
        return null;
    }


//LKS - look @ removing ack in future release
    public synchronized void loadTransactions() 
        throws BrokerException, IOException
    {

        boolean autorollback = Globals.getConfig().getBooleanProperty(
                                Globals.IMQ + ".transaction.autorollback", 
                                false);



        // before we do anything else, make sure we dont have any
        // unexpected exceptions

        // FIRST ... look at transaction table (tid,state)
        LoadException load_ex = Globals.getStore().getLoadTransactionException();

        if (load_ex != null) {
            // some messages could not be loaded
            LoadException processing = load_ex;
            while (processing != null) {
                TransactionUID tid = (TransactionUID)processing.getKey();
                TransactionState trans = (TransactionState)processing.getValue();
                if (tid == null && trans == null) {
                    logger.log(Logger.WARNING, 
                         BrokerResources.E_INTERNAL_ERROR, 
                         "both key and value for a Transactions entry"
                       + " are corrupted");
                    continue;
                }
                if (tid == null ) { 
                    // at this point, there is nothing we can do ...
                    // store with valid key
                    // improve when we address 5060661
                    logger.log(Logger.WARNING,
                        BrokerResources.W_TRANS_ID_CORRUPT,
                            trans.toString());

                } else { // trans == null
                    // if we dont know ... so make it prepared
                    logger.log(Logger.WARNING,
                        BrokerResources.W_TRANS_STATE_CORRUPT, tid);
                    TransactionState ts = new TransactionState(
                        AutoRollbackType.NOT_PREPARED, 0, true);
                    ts.setState(TransactionState.PREPARED);
                    try {
                        store.storeTransaction(tid, ts, false);
                    } catch (Exception ex) {
                        logger.logStack(Logger.WARNING, 
                         BrokerResources.E_INTERNAL_ERROR, 
                            "Error updating transaction " + tid, ex);
                    }
                } // end if
                processing = processing.getNextException();
            } // end while
        } 

        // now look at acks
        load_ex = Globals.getStore().getLoadTransactionAckException();

        if (load_ex != null) {
            // some messages could not be loaded
            LoadException processing = load_ex;
            while (processing != null) {
                TransactionUID tid = (TransactionUID)processing.getKey();
                TransactionAcknowledgement ta[] = 
                         (TransactionAcknowledgement[])processing.getValue();
                if (tid == null && ta == null) {
                    logger.log(Logger.WARNING, 
                         BrokerResources.E_INTERNAL_ERROR, 
                         "both key and value for a Transaction Ack entry"
                       + " are corrupted");
                    continue;
                }
                if (tid == null ) { 
                    // at this point, there is nothing we can do ...
                    // store with valid key
                    // improve when we address 5060661
                    logger.log(Logger.WARNING,
                        BrokerResources.W_TRANS_ID_CORRUPT,
                            ta.toString());
                } else { // ta == null
                    // nothing we can do, remove it
                    logger.log(Logger.WARNING,
                        BrokerResources.W_TRANS_ACK_CORRUPT,
                            tid.toString());
                    try {
                        store.removeTransactionAck(tid, false);
                    } catch (Exception ex) {
                        logger.logStack(Logger.WARNING, 
                         BrokerResources.E_INTERNAL_ERROR, 
                            "Error updating transaction ack " + tid, ex);
                    }
                } // end if
                processing = processing.getNextException();
            } // end while
        } 

        logger.log(Logger.INFO, BrokerResources.I_PROCESSING_TRANS);
        // OK -> first load the list of pending
        // transactions
        HashMap trans = store.getAllTransactionStates();

        // Write some info about the transactions to the log file
        // for informational purposes.
        logTransactionInfo(trans, autorollback);

        if (trans.size() <= 0) 
            return;

        // list of transactions which need to be cleaned up
        HashSet clearTrans = new HashSet(trans.size());

        HashMap openTransactions = new HashMap();
        HashMap inprocessAcks = new HashMap();

        logger.log(Logger.DEBUG,"Broker left open transactions"
              + " when it crashed");

        // loop through the list of transactions
        // placing each on the various lists

        Iterator itr = trans.keySet().iterator();
        while (itr.hasNext()) {
          try {
             TransactionUID tid = (TransactionUID)itr.next();
             TransactionState ts = (TransactionState)trans.get(tid);
             TransactionAcknowledgement  ta[] = store.
                       getTransactionAcks(tid);
             int state = ts.getState();
             switch (state) {
                 // no longer valid, ignore
                 case TransactionState.CREATED:
                     clearTrans.add(tid);
                     break;
                 case TransactionState.PREPARED:
                     // if autorollback, fall through to rollback
                     if (!autorollback) {
                         // nothing happens w/ preparedTransactions 
                         // they go back into the list
                         // We don't persist this because it is already
                         // in the store
                         addTransactionID(tid, ts, false);
                         openTransactions.put(tid, Boolean.TRUE);
                         // put messages in the orphan list
                         break; 
                     }
                 // rollback -> we didnt complete
                 case TransactionState.STARTED:
                 case TransactionState.COMPLETE:
                 case TransactionState.ROLLEDBACK:
                 case TransactionState.FAILED:
                 case TransactionState.INCOMPLETE:
                     addTransactionID(tid, ts, false);
                     setTransactionIDStored(tid);
                     openTransactions.put(tid, Boolean.FALSE);
                     clearTrans.add(tid);
                     ts.setState(TransactionState.ROLLEDBACK);
                     break; 
                 case TransactionState.COMMITTED:
                     clearTrans.add(tid);
                     break; 
            }

            for (int i=0; i < ta.length; i ++) {
                ConsumerUID cuid = ta[i].getConsumerUID();
                ConsumerUID scuid = ta[i].getStoredConsumerUID();
                SysMessageID sysid = ta[i].getSysMessageID();
                Map imap = (Map)inprocessAcks.get(sysid);
                if (cuid == null) {
                     logger.log(Logger.WARNING,"Internal Error: " +
                         " Unable to locate stored ConsumerUID :" + ta);
                     cuid =scuid;
                     if (ta == null) {
                         logger.log(Logger.WARNING, "Internal Error: "
                            + " no consumerUID stored w/ the transaction:" 
                            + ta);
                          continue;
                      }
                }
                    
                if (imap == null) {
                    imap = new HashMap();
                    inprocessAcks.put(sysid, imap);
                }
                imap.put(ta[i].getStoredConsumerUID(), tid);
                if (openTransactions.get(tid) != null) { 
                    TransactionInformation ti = (TransactionInformation)
                         translist.get(tid);
                    if (ti == null) {
                        logger.log(Logger.INFO, "Unable to retrieve "
                             + " transaction information " + ti +
                             " for " + tid + 
                            " we may be clearing the transaction");
                        continue;
                    }
                    if (openTransactions.get(tid) == Boolean.TRUE) {
                        ti.addConsumedMessage(sysid,
                            cuid, 
                            scuid);
                    }
                    ti.addOrphanAck(sysid,
                        cuid);
                 }

            }
          } catch (Exception ex) {
            logger.logStack(Logger.WARNING,
               BrokerResources.E_INTERNAL_BROKER_ERROR,
               "Error parsing transaction ", ex);
          }
                   
        } 

        // if we have ANY inprocess Acks or openTransactions we have to
        // load the database now and fix it
        if (openTransactions.size() > 0 || inprocessAcks.size() > 0) {
            Map m = Destination.processTransactions(inprocessAcks,
                         openTransactions);
            if (m != null && !m.isEmpty()) {
                Iterator meitr = m.entrySet().iterator();
                while (meitr.hasNext()) {
                    Map.Entry me = (Map.Entry)meitr.next();
                    TransactionInformation ti = (TransactionInformation)
                              translist.get(me.getValue());
                    ti.addPublishedMessage((SysMessageID)me.getKey());
                }
            }

        }


          // OK -> now clean up the cleared transactions
          // this removes them from the disk
          itr = clearTrans.iterator();
          while (itr.hasNext()) {
              TransactionUID uid = (TransactionUID)itr.next();
             TransactionState ts = (TransactionState)trans.get(uid);
              // ok rollback
              logger.log(Logger.DEBUG,"Clearing transaction "
                    + uid);
// XXX - doesnt not function correctly in HA
              removeTransactionAck(uid);
              removeTransactionID(uid);
          }

    }

    public static void logTransactionInfo(HashMap transactions, 
                                          boolean autorollback) {

        Logger logger = Globals.getLogger();

        /*
         * Loop through all transactions and count how many are in
         * what state. This is done strictly for informational purposes.
         */
        int nRolledBack = 0;    // Number of transactions rolledback
        int nPrepared = 0;      // Number of prepared transactions
        if (transactions != null && transactions.size() > 0) {
            Iterator itr = transactions.values().iterator();
            while (itr.hasNext()) {
                TransactionState _ts = (TransactionState)itr.next();
                if (_ts.getState() == TransactionState.PREPARED) {
                    nPrepared++;
                    if (autorollback) {
                        nRolledBack++;
                    }
                } else {
                    nRolledBack++;
                }
            }

            logger.log(Logger.INFO, BrokerResources.I_NTRANS, 
                       new Integer(transactions.size()), 
                       new Integer(nRolledBack));
            if (nPrepared > 0) {
                logger.log(Logger.INFO, BrokerResources.I_NPREPARED_TRANS, 
                       new Integer(transactions.size()), 
                       new Integer(nPrepared));
                if (autorollback) {
                    logger.log(Logger.INFO, 
                               BrokerResources.I_PREPARED_ROLLBACK);
                } else {
                    logger.log(Logger.INFO, 
                               BrokerResources.I_PREPARED_NOROLLBACK);
                }

            }
        }
    }


}

// OK .. we know that the only time we will ever be retrieving
// messages with these apis is after a commit or rollback
//
// that message will always occurs after all updates have happened
//
// there are never 2 connections sharing a single transaction, and
// access from a connection is on a single thread
//
// SO .. dont have to lock

class TransactionInformation
{
    ArrayList published;
    HashMap consumed;
    TransactionState state;
    HashMap cuidToStored;
    HashMap orphanedMessages;
    TransactionUID tid = null;
    boolean persisted = false;

    public TransactionInformation(TransactionUID uid, TransactionState state,
             boolean persisted)
    {
        published = new ArrayList();
        consumed = new HashMap();
        cuidToStored = new HashMap();
        orphanedMessages = new HashMap();
        this.state = state;
        this.tid = uid;
        this.persisted = persisted;
    }

    public void setPersisted(boolean persisted)
    {
        this.persisted = persisted;
    }

    public boolean getPersisted()
    {
        return this.persisted;
    }

    public String toString() {
        return "TransactionInfo["+tid+"]";
    }

    public void addOrphanAck(SysMessageID sysid, ConsumerUID sid)
    {
        List l = (List)orphanedMessages.get(sysid);
        if (l == null) {
            l = new ArrayList();
            orphanedMessages.put(sysid, l);
         }
         l.add(sid);
    }

    public Map getOrphanAck() {
        return orphanedMessages;
    }


    public Hashtable getDebugState() {
        Hashtable ht = new Hashtable();
        ht.put("state", state.getDebugState());
        ht.put("consumed#", new Integer(consumed.size()));
        ht.put("published#", new Integer(published.size()));
        ht.put("cuidToStored#", new Integer(cuidToStored.size()));
        if (cuidToStored.size() > 0) {
           Hashtable cid = new Hashtable();
           Iterator itr = ht.entrySet().iterator();
           while (itr.hasNext()) {
               Map.Entry me = (Map.Entry)itr.next();
               cid.put(me.getKey().toString(), 
                   me.getValue().toString());
           }
           ht.put("cuidToStored", cid);
        }
        if (consumed.size() > 0) {
            Hashtable m = new Hashtable();
            Iterator itr = consumed.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry me = (Map.Entry)itr.next();
                String id = me.getKey().toString();
                ArrayList ids = (ArrayList)me.getValue();
                if (ids.size() == 0) continue;
                if (ids.size() == 1) {
                    m.put(id, ids.get(0).toString());
                } else {
                    Vector v = new Vector();
                    for (int i=0; i < ids.size(); i ++)
                        v.add(ids.get(i).toString());
                    m.put(id, v);
                }
 
             }
             if (m.size() > 0) {
                 ht.put("consumed", m);           
             }
        }
        if (published.size() > 0) {
            Vector v = new Vector();
            for (int i =0; i < published.size(); i ++) {
                v.add(published.get(i).toString());
            } 
            ht.put("published", v);           
        }
        return ht;

    }

    public List getPublishedMessages() {
        return published;
    }

    public int getNPublishedMessages() {
        if (published != null) {
            return published.size();
        } else {
            return 0;
        }
    }

    public HashMap getConsumedMessages() {
        return consumed;
    }

    public HashMap getStoredConsumerUIDs() {
        return cuidToStored;
    }

    public int getNConsumedMessages() {
        if (consumed != null) {
            return consumed.size();
        } else {
            return 0;
        }
    }

    public TransactionState getState() {
        return state;
    }

    public void addPublishedMessage(SysMessageID id) {
        published.add(id);
    }

    public boolean checkConsumedMessage(SysMessageID sysid, ConsumerUID id)
    {
        List l = (List)consumed.get(sysid);
        if (l == null) {
            return false;
        }
        return l.contains(id);
    }

    public void addConsumedMessage(SysMessageID sysid, ConsumerUID id,
          ConsumerUID sid)
    {
        List l = (List)consumed.get(sysid);
        if (l == null) {
            l = new ArrayList();
            consumed.put(sysid, l);
         }
         l.add(id);
         cuidToStored.put(id, sid);
    }

    public TransactionUID getTID() {
        return tid;
    }

}
