/*
 * @(#)Packet.java	1.6 04/04/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.fullyconnected;

import java.io.*;

/**
 * This class encapsulates the packet format for standard
 * fully connected broker topology.
 */
class Packet {
    private static final short VERSION = 100;
    private static final int HEADER_SIZE = 16;

    /* packetType */
    public static final int UNICAST = 1;
    public static final int BROADCAST = 2;
    public static final int BROKER_INFO = 3;
    public static final int LINK_INIT = 4;
    public static final int STOP_FLOW = 5;
    public static final int RESUME_FLOW = 6;
    public static final int PING = 7;
    public static final int BROKER_INFO_REPLY = 9;

    /* bitFlags */
    public static final int USE_FLOW_CONTROL = 0x0001;

    private short version = VERSION;
    private short packetType = 0;
    private int packetSize = 0;
    private int destId = 0;
    private int bitFlags = 0;

    private byte[] packetBuffer = null;

    public void readPacket(InputStream is)
        throws IOException, EOFException {

        DataInputStream dis = new DataInputStream(is);

        version = dis.readShort();
        packetType = dis.readShort();
        packetSize = dis.readInt();
        destId = dis.readInt();
        bitFlags = dis.readInt();

        try {
            packetBuffer = new byte[packetSize - HEADER_SIZE];
        }
        catch (OutOfMemoryError oom) {
            dis.skip(packetSize - HEADER_SIZE);
            throw oom;
        }
        dis.readFully(packetBuffer);
    }

    public void writePacket(OutputStream os)
        throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        dos.writeShort(version);
        dos.writeShort(packetType);
        dos.writeInt(packetSize);
        dos.writeInt(destId);
        dos.writeInt(bitFlags);
        dos.flush();
        bos.flush();

        byte[] headerBuffer = bos.toByteArray();

        os.write(headerBuffer, 0, HEADER_SIZE);
        if (packetBuffer != null)
            os.write(packetBuffer, 0, packetSize - HEADER_SIZE);
        os.flush();
    }

    public int getPacketType() {
        return packetType;
    }

    public int getPacketSize() {
        return packetSize;
    }

    public int getDestId() {
        return destId;
    }

    public byte[] getPacketBody() {
        return packetBuffer;
    }

    public boolean getFlag(int flag) {
        return ((bitFlags & flag) == flag);
    }

    public void setPacketType(int packetType) {
        this.packetType = (short) packetType;
    }

    public void setDestId(int destId) {
        this.destId = destId;
    }

    public void setPacketBody(byte[] data) {
        packetBuffer = data;
        packetSize = HEADER_SIZE;
        if (packetBuffer != null)
            packetSize += packetBuffer.length;
    }

    public void setFlag(int flag, boolean on) {
        if (on)
            bitFlags = bitFlags | flag;
        else
            bitFlags = bitFlags & ~flag;
    }

    public String toString() {
        return "PacketType = " + packetType +
            ", DestId = " + destId + ", DATA :\n" +
            hexdump(packetBuffer, 128);
    }

    public static String hexdump(byte[] buffer, int maxlen) {
        if (buffer == null)
            return "";

        String ret = "";
        int addr = 0;
        int buflen = buffer.length;
        if (buflen > maxlen)
            buflen = maxlen;

        while (buflen > 0) {
            int count = buflen < 16 ? buflen : 16;
            ret = ret + "\n" + i2hex(addr, 6, "0");

            String tmp = "";

            int i;
            for (i = 0; i < count; i++) {
                byte b = buffer[addr + i];

                if (i == 8)
                    ret = ret + "-";
                else
                    ret = ret + " ";
                ret = ret + i2hex(b, 2, "0");
                if (b >= 32 && b <= 127)
                    tmp = tmp + ((char) b);
                else
                    tmp = tmp + ".";
            }
            for (; i < 16; i++)
                ret = ret + "   ";

            ret = ret + "   " + tmp;

            addr += count;
            buflen -= count;
        }
        return ret + "\n";
    }

    public static String i2hex(int i, int len, String filler) {
        String str = Integer.toHexString(i);
        if (str.length() == len)
            return str;
        if (str.length() > len)
            return str.substring(str.length() - len);
        while (str.length() < len)
            str = filler + str;
        return str;
    }
}

/*
 * EOF
 */
