/*
 * @(#)ClusterSubscriptionInfo.java	1.6 05/20/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker.raptor;

import java.util.*;
import java.io.*;
import java.nio.*;
import com.sun.messaging.jmq.io.GPacket;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.Subscription;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.multibroker.Cluster;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;

/**
 * An instance of this class is intended to be used one direction only
 * either Subscription/Consumer -> GPacket or GPacket -> Subscription/Consumer
 * (see assertions)
 */

public class ClusterSubscriptionInfo
{
    private Subscription subscription =  null;
    private Consumer consumer = null;
    private Cluster c = null;

    private GPacket pkt = null;

    private ClusterSubscriptionInfo(Subscription sub) {
        this.subscription = sub;
    }

    private ClusterSubscriptionInfo(Subscription sub, Consumer cs, Cluster c) {
        this.subscription = sub;
        this.consumer = cs;
        this.c = c;
    }
    
    private  ClusterSubscriptionInfo(GPacket pkt) {
        this.pkt = pkt;
    }

    private  ClusterSubscriptionInfo(GPacket pkt, Cluster c) {
        this.pkt = pkt;
        this.c = c;
    }

    public static ClusterSubscriptionInfo newInstance(Subscription sub) {
        return new ClusterSubscriptionInfo(sub);
    }

    public static ClusterSubscriptionInfo newInstance(Subscription sub, Consumer cs, Cluster c) {
        return new ClusterSubscriptionInfo(sub, cs, c);
    }

    public static ClusterSubscriptionInfo newInstance(GPacket pkt) { 
        return new ClusterSubscriptionInfo(pkt);
    }

    public static ClusterSubscriptionInfo newInstance(GPacket pkt, Cluster c) { 
        return new ClusterSubscriptionInfo(pkt, c);
    }

    public GPacket getGPacket(short protocol) {
        return getGPacket(protocol, false);
    }

    public GPacket getGPacket(short protocol, boolean changeRecord) {
        assert ( subscription != null );

        assert ( protocol == ProtocolGlobals.G_NEW_INTEREST ||  
                 protocol == ProtocolGlobals.G_DURABLE_ATTACH ||
                 protocol == ProtocolGlobals.G_REM_DURABLE_INTEREST );

        if (changeRecord ) {
        assert ( protocol == ProtocolGlobals.G_NEW_INTEREST ||  
                 protocol == ProtocolGlobals.G_REM_DURABLE_INTEREST );
        }

        GPacket gp = null;

        switch (protocol) {

            case ProtocolGlobals.G_NEW_INTEREST:

            ClusterConsumerInfo cci = ClusterConsumerInfo.newInstance(subscription, null);
            gp = cci.getGPacket(protocol);

            if (protocol == ProtocolGlobals.G_NEW_INTEREST && changeRecord) {
                gp.putProp("N", new String(subscription.getDurableName()));
                gp.putProp("I", new String(subscription.getClientID()));
                gp.putProp("M", new Boolean(true));
            }
            break;


            case ProtocolGlobals.G_DURABLE_ATTACH:
            {
            assert ( consumer != null );

            gp = GPacket.getInstance();
            gp.setType(protocol);

            String dname = subscription.getDurableName();
            if (dname != null) {
            gp.putProp("N", new String(dname));
            }

            String clientID = subscription.getClientID();
            gp.putProp("I", new String(clientID));

            gp.putProp("allowsNonDurable", new Boolean(true));
            c.marshalBrokerAddress(c.getSelfAddress(), gp);

            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            DataOutputStream dos = new DataOutputStream(bos);
            try {
                ClusterConsumerInfo.writeConsumer(consumer, dos);
                dos.flush();
                bos.flush();
            }
            catch (IOException e) { /* Ignore */ }

            gp.setPayload(ByteBuffer.wrap(bos.toByteArray()));

            }
            break;


            case ProtocolGlobals.G_REM_DURABLE_INTEREST:
            {

            gp = GPacket.getInstance();
            gp.setType(protocol);
			gp.putProp("C", new Integer(1));

            String dname = subscription.getDurableName();
            String clientID = subscription.getClientID();
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            DataOutputStream dos = new DataOutputStream(bos);
            try {
                dos.writeUTF(dname);
                dos.writeUTF(clientID);
                dos.flush();
                bos.flush();
            }
            catch (IOException e) { /* Ignore */ }

            gp.setPayload(ByteBuffer.wrap(bos.toByteArray()));

            if (changeRecord) {
                gp.putProp("N", new String(dname));
                gp.putProp("I", new String(clientID));
                gp.putProp("M", new Boolean(true));
            }

            }
            break;
        }

        return gp;
    }

    public int getConsumerCount() {
        assert ( pkt != null);

        short type = pkt.getType(); 
        assert ( type == ProtocolGlobals.G_NEW_INTEREST ||
                 type == ProtocolGlobals.G_INTEREST_UPDATE ||
                 type == ProtocolGlobals.G_REM_DURABLE_INTEREST );

        int count = ((Integer)pkt.getProp("C")).intValue();
        assert ( count == 1);
        return count;
    }

    public boolean isConfigSyncResponse() { 
        assert ( pkt != null );

        boolean b = false;
        if (pkt.getProp("M") != null) {
            b = ((Boolean) pkt.getProp("M")).booleanValue();
        }
        return b;
    }

    public String getDurableName() {
        assert ( pkt != null );
        return (String)pkt.getProp("N");
    }

    public String getClientID() {
        assert ( pkt != null );
        return (String)pkt.getProp("I");
    }

    public Boolean allowsNonDurable() { 
        assert ( pkt != null );
        return (Boolean)pkt.getProp("allowsNonDurable");
    }

    public Consumer getConsumer() throws Exception {
        assert ( pkt != null );
        
        short type = pkt.getType();
        assert ( type ==  ProtocolGlobals.G_DURABLE_ATTACH );
        
        ByteArrayInputStream bis = new ByteArrayInputStream(pkt.getPayload().array());
        DataInputStream dis = new DataInputStream(bis);
        Consumer cs = ClusterConsumerInfo.readConsumer(dis);
        BrokerAddress from = c.unmarshalBrokerAddress(pkt);
        if (from != null) cs.getConsumerUID().setBrokerAddress(from);
		return cs;
    }

    public Iterator getSubscriptions() {
        assert ( pkt != null );

        short type = pkt.getType();
        assert ( type ==  ProtocolGlobals.G_REM_DURABLE_INTEREST );
        return new SubscriptionIterator(pkt.getPayload().array(), getConsumerCount()); 
    }

    public boolean needReply() {
        assert ( pkt != null );
        return pkt.getBit(pkt.A_BIT);
    }

    public static GPacket getReplyGPacket(short protocol, int status) {
        GPacket gp = GPacket.getInstance();
        gp.setType(protocol);
        gp.putProp("S", new Integer(status));
        return gp;
    }

}


class SubscriptionIterator implements Iterator
{
    private int count = 0;
    private int count_read = 0;
    private DataInputStream dis = null;

    public SubscriptionIterator(byte[] payload, int count) {
        ByteArrayInputStream bis = new ByteArrayInputStream(payload);
        dis = new DataInputStream(bis);
        this.count = count;
        this.count_read = 0;
    }

    public boolean hasNext() {
        if (count_read < 0) throw new IllegalStateException("SubscriptionIterator");
        return count_read < count;
    }

    /**
     * Caller must catch RuntimeException and getCause
     */
    public Object next() throws RuntimeException {
        try {
            String dname = dis.readUTF();
            String cname = dis.readUTF();
            Subscription sub = Subscription.findDurableSubscription(cname, dname);
            count_read++;
            return sub;
        } catch (IOException e) {
            count_read = -1;
            throw new RuntimeException(e);
        }
    }

    public void remove() {
        throw new UnsupportedOperationException("Not supported");
    }
}


