/*
 * @(#)HABrokers.java	1.0 05/19/05
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.file;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.jmsserver.persist.HABrokerInfo;


import java.io.*;
import java.util.*;


/**
 * Keep track of all HA brokers.
 * File format: 4 byte magic number then HABrokerInfo is appended
 * to the backing file in this order:
 /**
 *         Broker ID
 *         Broker ID taken over the store
 *         the broker's URL
 *         the broker's version
 *         the broker's state
 *         the broker's session ID
 *         broker's last heartbeat
 */

class HABrokers {

    Logger logger = Globals.getLogger();
    BrokerResources br = Globals.getBrokerResources();
    protected BrokerConfig config = Globals.getConfig();

    static final String BASENAME = "habrokers"; // basename of data file

    // to make sure we are reading a good file
    static final int MAGIC = 0x12345678;

    // cache all persisted records
    // list of broker ids
    private ArrayList bkrIDList = new ArrayList();
 
    // list of broker ids which have taken over the store
    private ArrayList takeOverBkrIDList = new ArrayList();
    
    // list of broker urls
    private ArrayList urlList = new ArrayList();

    // list of broker versions
    private ArrayList bkrVersionList = new ArrayList();

    // list of broker states
    private ArrayList bkrStateList = new ArrayList();
    
    // list of broker session ids
    private ArrayList bkrSessionIDList = new ArrayList();

    // list of broker heartbeats
    private ArrayList bkrHeartBeatList = new ArrayList();

    private File backingFile = null;
    private RandomAccessFile raf = null;

    // when instantiated, all data are loaded
    HABrokers(File topDir, boolean clear) throws BrokerException {

	try {
	    backingFile = new File(topDir, BASENAME);
	    raf = new RandomAccessFile(backingFile, "rw");

	    if (clear) {
		clearAll(false);
		if (Store.DEBUG) {
		    logger.log(logger.DEBUGHIGH,
			"HABrokers initialized with clear option");
		}
	    } else {
		// true=initialize new file
		loadData(backingFile, raf, bkrIDList, takeOverBkrIDList, urlList, bkrVersionList, 
                         bkrStateList, bkrSessionIDList, bkrHeartBeatList, true);
	    }
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_LOAD_HABROKERS_FAILED, e);
	    throw new BrokerException(
			br.getString(br.X_LOAD_HABROKERS_FAILED), e);
	}
    }

    /**
     * Append a new record to the habrokers record store.
     *
    /**
     * @param id Broker ID
     * @param takeoverBrokerID Broker ID taken over the store
     * @param url the broker's URL
     * @param version the broker's version
     * @param state the broker's state
     * @param sessionID the broker's session ID
     * @param heartbeat broker's last heartbeat
     * @exception BrokerException if an error occurs while persisting the data
     * @exception NullPointerException	if <code>recordData</code> is
     *			<code>null</code>
     */
    public void addBrokerInfo(String brokerID, String takeoverBkrID, String URL, 
                int state, int version, long sessionID, long heartbeat)
                throws BrokerException {

	bkrIDList.add(brokerID);
        takeOverBkrIDList.add(takeoverBkrID);
        urlList.add(URL);
        bkrStateList.add(new Integer(state));
        bkrVersionList.add(new Integer(version));
        bkrSessionIDList.add(new Long(sessionID));
        bkrHeartBeatList.add(new Long(heartbeat));

	try {
		appendFile(brokerID, takeoverBkrID, URL, state, version, sessionID, heartbeat, true);
	} catch (IOException e) {
	    bkrIDList.remove(bkrIDList.size() - 1);
	    takeOverBkrIDList.remove(takeOverBkrIDList.size() - 1);
            urlList.remove(urlList.size() - 1);
            bkrStateList.remove(bkrStateList.size() - 1);
            bkrVersionList.remove(bkrVersionList.size() - 1);
            bkrSessionIDList.remove(bkrSessionIDList.size() - 1);
            bkrHeartBeatList.remove(bkrHeartBeatList.size() - 1);
                
	    logger.log(logger.ERROR, br.X_PERSIST_CONFIGRECORD_FAILED);
		throw new BrokerException(
				br.getString(br.X_PERSIST_CONFIGRECORD_FAILED),
				e);
	}
    }

    /**
     * Return all broker records.
     * The returned data is an array of Object with 7 elements.
     * The first element contains an ArrayList of all brokerIDs,
     * the second element contains an ArrayList of all takeoverBrokerIDs,
     * third element contains an ArrayList of all broker urls, fourth
     * element contains an ArrayList of all broker states, fifth 
     * element contains an ArrayList of all broker versions, sixth
     * element contains an ArrayList of all broker session ids, 
     * seventh element contains an ArrayList of all broker heartbeats.
     *
     * @return a HashMap of HABrokerInfo objects.
     *
     * @exception BrokerException if an error occurs while getting the data
     */

    public HashMap getAllBrokerInfos() throws BrokerException {
	HashMap data = new HashMap();

        for(int i = 0; i < bkrIDList.size(); i++)
        {
            HABrokerInfo haBkrInfo = new HABrokerInfo((String) bkrIDList.get(i), 
	    		(String) takeOverBkrIDList.get(i), (String) urlList.get(i),
                        ((Integer) bkrStateList.get(i)).intValue(), ((Integer)
			bkrVersionList.get(i)).intValue(), ((Long) bkrSessionIDList.get(i)).longValue(),
			((Long) bkrHeartBeatList.get(i)).longValue());
            
            data.put(bkrIDList.get(i), haBkrInfo);
        }
	return data;
    }

    /**
     * Clear all records
     */
    // clear the store; when this method returns, the store has a state
    // that is the same as an empty store 
    void clearAll(boolean sync) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH,
			"HABrokers.clearAll() called");
	}

        bkrIDList.clear();
        takeOverBkrIDList.clear();
        urlList.clear();
        bkrStateList.clear();
        bkrVersionList.clear();
        bkrSessionIDList.clear();
        bkrHeartBeatList.clear();

	try {
	    raf.setLength(0);

	    // write magic number
	    raf.writeInt(MAGIC);

	    if (sync) {
		sync();
	    }

	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_CLEAR_HABROKERTABLE_FAILED, e);
	}
    }

    void close(boolean cleanup) {
	if (Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH,
			"HABrokers : closing, " + bkrIDList.size() +
			" persisted records");
	}

	try {
	    raf.close();
	} catch (IOException e) {
	    if (Store.DEBUG) {
		logger.log(logger.DEBUG,
			"Got IOException while closing:" + backingFile, e);
		e.printStackTrace();
	    }
	}
    }

    /**
     * Get debug information about the store.
     * @return A Hashtable of name value pair of information
     */  
    Hashtable getDebugState() {
	Hashtable t = new Hashtable();
	t.put("HA broker records", String.valueOf(bkrIDList.size()));
	return t;
    }

    void printInfo(PrintStream out) {
	out.println("\n HA broker Records");
	out.println("---------------------------");
	out.println("backing file: " + backingFile);
	out.println("number of records: " + bkrIDList.size());
    }

    void sync() throws BrokerException {
	try {
	    // don't sync meta data for performance reason
	    raf.getChannel().force(false);
	} catch (IOException e) {
	    throw new BrokerException(
		"Failed to synchronize file: " + backingFile, e);
	}
    }

    /**
     * load data data from file
     */
     private void loadData(File filename, RandomAccessFile dataraf,
	ArrayList bkrIDList, ArrayList takeOverBkrIDList, ArrayList urlList,
        ArrayList bkrVersionList, ArrayList bkrStateList, ArrayList bkrSessionIDList, 
        ArrayList bkrHeartBeatList, boolean init) throws IOException {

	if (dataraf.length() == 0) {
	    if (init) {
		// write magic number
		dataraf.writeInt(MAGIC);
		if (Store.DEBUG) {
		    logger.log(logger.DEBUGHIGH,
				"initialized new file with magic number, "
				+ filename);
		}
	    }
	    return;
	}

	loadData(filename, dataraf, bkrIDList, takeOverBkrIDList, urlList,
                    bkrVersionList, bkrStateList, bkrSessionIDList, 
                    bkrHeartBeatList);
    }

    /**
     * load data data from file
     * file format:
     *	4 byte magic number (0x12345678)
     *  records....
     */
    private void loadData(File filename, RandomAccessFile dataraf,
	ArrayList bkrIDList, ArrayList takeOverBkrIDList, ArrayList urlList,
        ArrayList bkrVersionList, ArrayList bkrStateList, ArrayList bkrSessionIDList, 
        ArrayList bkrHeartBeatList) throws IOException {

	boolean done = false;

	// no record persisted yet 
	if (dataraf.length() == 0) {
	    return;
	}

	// check magic number
	int magic = dataraf.readInt();
	if (magic != MAGIC) {
	    throw new StreamCorruptedException(
		br.getString(br.E_BAD_HABROKERS_FILE, filename));
	}

	long pos = dataraf.getFilePointer();
	// read until the end of file
	while (!done) {
	    try {
	    	int len;
		byte[] b;
		// read until end of file
		// broker id
                len = dataraf.readInt();
                b = new byte[len];
		dataraf.readFully(b);
                bkrIDList.add(new String(b));
                
                // takeover broker id
                len = dataraf.readInt();
                b = new byte[len];
		dataraf.readFully(b);
                takeOverBkrIDList.add(new String(b));
                
                // broker url
                len = dataraf.readInt();
                b = new byte[len];
		dataraf.readFully(b);
                urlList.add(new String(b));
                
                // broker version
                bkrVersionList.add(new Integer(dataraf.readInt()));
                
                // broker state
                bkrStateList.add(new Integer(dataraf.readInt()));
                
                // broker session id
                bkrSessionIDList.add(new Long(dataraf.readLong()));
                
                // broker heart beat
                bkrHeartBeatList.add(new Long(dataraf.readLong()));

		pos = dataraf.getFilePointer();
	    } catch (EOFException e) {
		if (pos != dataraf.getFilePointer()) {
		    // in case broker failed while part of a record
		    // is being written
		    dataraf.setLength(pos);

		    logger.log(logger.WARNING,
			br.W_BAD_CONFIGRECORD, filename, new Long(pos));
		}
		done = true;
	    } catch (IOException e) {
		logger.log(logger.WARNING,
			br.W_EXCEPTION_LOADING_HABROKERS,
			new Integer(bkrIDList.size()), new Long(pos), e);

		// truncate rest of the file
		dataraf.setLength(pos);

		done = true;
	    }
	}

	if (Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH,
		"loaded " + bkrIDList.size() + " records from " + filename);
	}
    }

    // append data to file
    private void appendFile(String bkrID, String takeOverBkrID, String url,
        int bkrVersion, int bkrState, long bkrSessionID, long bkrHeartBeat, 
        boolean sync) throws IOException, BrokerException {

	// file position before write
	long pos = raf.getFilePointer();

	try {
	    // write data
            // broker id
	    byte[] b;
            b = bkrID.getBytes();
	    raf.writeInt(b.length);
            raf.write(b);
            
            // takeover broker id
            b = takeOverBkrID.getBytes();
	    raf.writeInt(b.length);
            raf.write(b);
            
            // broker url
            b = url.getBytes();
	    raf.writeInt(b.length);
            raf.write(b);
            
            // broker version
            raf.writeInt(bkrVersion);
            
            // broker state
            raf.writeInt(bkrState);
            
            // broker session id
            raf.writeLong(bkrSessionID);
            
            // broker heartbeat
            raf.writeLong(bkrHeartBeat);
            
	    if (sync) {
		sync();
	    }

	} catch (IOException e) {
	    // truncate file to end of previous record
	    raf.setLength(pos);
	    throw e;
	}

	if (Store.DEBUG) {
	    logger.log(logger.DEBUG, "habrokers : appended for broker id = " + bkrID + ".");
	}
    }
}
