/*
 * @(#)BaseDAOImpl.java	1.9 01/30/06
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.config.BrokerConfig;
import com.sun.messaging.jmq.util.log.Logger;

import java.sql.*;
import java.util.List;
import java.util.HashMap;
import java.util.Iterator;

/**
 * The DAO base class which provides methods for creating and dropping table.
 *
 * @version	1.9
 */
public abstract class BaseDAOImpl implements DBConstants, BaseDAO {

    // Private variables
    protected Logger logger = Globals.getLogger();
    protected BrokerResources br = Globals.getBrokerResources();
    protected BrokerConfig config = Globals.getConfig();

    /**
     * Create the table.
     * @param conn database connection
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void createTable( Connection conn ) throws BrokerException {

        String tableName = getTableName();
        logger.logToAll( Logger.INFO,
            br.getString( BrokerResources.I_CREATE_TABLE, tableName ) );

        // Make sure table definition is specified
        DBManager dbMgr = DBManager.getDBManager();
        HashMap schemas = dbMgr.getTableSchema();
        DBManager.TableSchema tableSchema =
            (DBManager.TableSchema)schemas.get( tableName );
        if ( tableSchema == null || tableSchema.tableSQL.length() == 0 ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_NO_JDBC_TABLE_PROP, getTableNamePrefix() ) );
        }

        boolean myConn = false;
        Statement stmt = null;
        String sql = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            // Create the table
            stmt = conn.createStatement();

            sql = tableSchema.tableSQL;
            stmt.executeUpdate( sql );

            // Create the table index if any
            Iterator itr = tableSchema.indexIterator();
            while ( itr.hasNext() ) {
                String indexName = (String)itr.next();
                sql = tableSchema.getIndex( indexName );

                logger.logToAll( Logger.INFO,
                    br.getString( BrokerResources.I_CREATE_TABLE_INDEX, indexName ) );

                stmt.executeUpdate( sql );
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_CREATE_TABLE_FAILED, tableName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, stmt, conn );
            } else {
                Util.close( null, stmt, null );
            }
        }
    }

    /**
     * Drop the table.
     * @param conn database connection
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void dropTable( Connection conn ) throws BrokerException {

        String tableName = getTableName();
        logger.logToAll( Logger.INFO,
            br.getString( BrokerResources.I_DROP_TABLE, tableName ) );

        boolean myConn = false;
        String dropSQL = "DROP TABLE " + tableName;
        Statement stmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            stmt = conn.createStatement();
            stmt.executeUpdate( dropSQL );
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + dropSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_DROP_TABLE_FAILED, tableName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, stmt, conn );
            } else {
                Util.close( null, stmt, null );
            }
        }
    }

    /**
     * Delete all entries.
     * @param conn database connection
     * @throws BrokerException
     */
    public void deleteAll( Connection conn ) throws BrokerException {
        deleteAll( conn, null );
    }

    /**
     * Convenience method to delete all entries. The intended purpose of this
     * method is to provided the ability for the extended class to overide it
     * and specified the timestamp column and chunk size which will be used
     * to chunk the data to be deleted. This is a work-around for HADB running
     * out of lock sets when deleting large number of records. For all other
     * databases, timestampColumn parameter should be set to null and chunkSize
     * should be set to 0.
     * @param conn
     * @param whereClause the where clause for the SQL command
     * @param timestampColumn the timestamp column which will be used to chunk
     *   the data to be deleted
     * @param chunkSize the size of each chunk
     * @throws BrokerException
     */
    protected void deleteAll( Connection conn, String whereClause,
        String timestampColumn, int chunkSize ) throws BrokerException {

        if ( chunkSize > 0 ) {
            deleteAllInChunk( conn, whereClause, timestampColumn, chunkSize );
        } else {
            deleteAll( conn, whereClause );
        }
    }

    /**
     * Delete all entries using the specified where clause.
     * @param conn database connection
     * @param whereClause the where clause for the SQL command
     * @throws BrokerException
     */
    private void deleteAll( Connection conn, String whereClause )
        throws BrokerException {

        DBManager dbMgr = DBManager.getDBManager();
        String tableName = getTableName();
        String sql = null;

        if ( dbMgr.isOracle() ) {
            sql = new StringBuffer(128)
                .append( "TRUNCATE TABLE " ).append( tableName )
                .toString();
        } else {
            sql = new StringBuffer(128)
                .append( "DELETE FROM " ).append( tableName )
                .append( (whereClause != null && whereClause.length() > 0)
                         ? " WHERE " + whereClause : "" )
                .toString();
        }

        boolean myConn = false;
        Statement stmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            stmt = conn.createStatement();
            stmt.executeUpdate( sql );
        } catch ( SQLException e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            SQLException ex = DBManager.wrapSQLException("[" + sql + "]", e);
            throw new BrokerException(
                br.getKString( BrokerResources.X_JDBC_CLEAR_TABLE_FAILED, tableName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, stmt, conn );
            } else {
                Util.close( null, stmt, null );
            }
        }
    }

    /**
     * Delete all entries in chunk to work-around HADB running out of
     * lock sets when deleting large number of records.
     * @param conn database connection
     * @param whereClause the where clause for the SQL command
     * @param timestampColumn the timestamp column which will be used to chunk
     *   the data to be deleted
     * @param chunkSize the size of each chunk
     * @throws BrokerException
     */
    private void deleteAllInChunk( Connection conn, String whereClause,
        String timestampColumn, int chunkSize ) throws BrokerException {

        // Set whereClause to empty if not specified, i.e. null
        if ( whereClause == null ) {
            whereClause = "";
        }

        String tableName = getTableName();
        
        boolean myConn = false;
        PreparedStatement pstmt = null;
        String sql = null;
        int txnIsolation = -1;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            } else {
                if ( !conn.getAutoCommit() ) {
                    conn.setAutoCommit( true );
                }
            }

            // Get the number of rows to be deleted to see if we need to delete
            // data in multiple chunks, i.e. # rowToBeDeleted > chunkSize
            sql = new StringBuffer(128)
                .append( "SELECT COUNT(*)" )
                .append( " FROM " ).append( tableName )
                .append( whereClause.length() > 0 ? " WHERE " + whereClause : "" )
                .toString();

            pstmt = conn.prepareStatement( sql );
            ResultSet rs = pstmt.executeQuery();
            int rowCount = 0;
            if ( rs.next() ) {
                rowCount = rs.getInt( 1 );
            }
            Util.close( rs, pstmt, null );

            if ( rowCount == 0 ) {
                return; // Nothing to delete!
            } else if ( rowCount < chunkSize ) {
                deleteAll( conn, whereClause ); // Don't have to delete in chunk
                return;
            }

            // We will need to delete all records in multiple chunks; so
            // generate the timestamp delimeter for each chunk.
            sql = new StringBuffer(128)
                .append( "SELECT " )
                .append( timestampColumn )
                .append( " FROM " ).append( tableName )
                .append( whereClause.length() > 0 ? " WHERE " + whereClause : "" )
                .append( " ORDER BY " )
                .append( timestampColumn )
                .toString();
            pstmt = conn.prepareStatement( sql );
            rs = pstmt.executeQuery();
            List chunkList = Util.getChunkDelimiters( rs, 1, chunkSize );
            Util.close( rs, pstmt, null );

            // For HADB, change txn isolation level to reduce the # of lock sets
            txnIsolation = conn.getTransactionIsolation();
            if ( txnIsolation != Connection.TRANSACTION_READ_COMMITTED ) {
                conn.setTransactionIsolation( Connection.TRANSACTION_READ_COMMITTED );
            }

            // Now, we delete all data in multiple chunks...
            sql = new StringBuffer(128)
                .append( "DELETE FROM " ).append( tableName )
                .append( " WHERE " )
                .append( timestampColumn ).append( " < ?" )
                .append( whereClause.length() > 0 ? " AND " + whereClause : "" )
                .toString();
            pstmt = conn.prepareStatement( sql );

            Long[] a = (Long[])chunkList.toArray(new Long[0]);
            for ( int i = 0, len = a.length; i < len; i++ )  {
                pstmt.setLong( 1, a[i].longValue() );
                pstmt.executeUpdate();
            }
        } catch ( SQLException e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            SQLException ex = DBManager.wrapSQLException("[" + sql + "]", e);
            throw new BrokerException(
                br.getKString( BrokerResources.X_JDBC_CLEAR_TABLE_FAILED, tableName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }

            // Restore txn isolation level to original value
            if ( txnIsolation != -1 ) {
                try {
                    conn.setTransactionIsolation( txnIsolation );
                } catch ( Exception e ) {}
            }
        }
    }
}
