/*
 * @(#)ConfigRecordDAOImpl.java	1.8 09/28/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;

import java.util.*;
import java.sql.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;

/**
 * This class implement a generic ConfigRecordDAO.
 *
 * @version	1.8
 */
class ConfigRecordDAOImpl extends BaseDAOImpl implements ConfigRecordDAO {

    protected String tableName;

    // SQLs
    protected String insertSQL;
    protected String selectCountSQL;
    protected String selectRecordsSinceSQL;
    protected String selectAllSQL;

    /**
     * Constructor
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    ConfigRecordDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( ") VALUES ( ?, ? )" )
            .toString();

        selectCountSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*)" )
            .append( " FROM " ).append( tableName )
            .toString();

        selectRecordsSinceSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( CREATED_TS_COLUMN ).append( " > ?" )
            .toString();

        selectAllSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public final String getTableName() {
        return tableName;
    }

    /**
     * Insert a new entry.
     * @param conn database connection
     * @param recordData the record data
     * @param timeStamp
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void insert( Connection conn, byte[] recordData, long timeStamp )
        throws BrokerException {

        boolean myConn = false;
        ByteArrayInputStream bais = null;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( insertSQL );
            bais = Util.setBytes( pstmt, 1, recordData );
            pstmt.setLong( 2, timeStamp );
            pstmt.executeUpdate();
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + insertSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_CONFIGRECORD_FAILED, 
                    String.valueOf(timeStamp) ), ex );
        } finally {
            if ( bais != null ) {
                try {
                    bais.close();
                } catch ( IOException e ) {}
            }

            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Get all records created since the specified timestamp.
     * @param conn database connection
     * @param timestamp the timestamp
     * @return a List of records.
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public List getRecordsSince( Connection conn, long timestamp )
        throws BrokerException {

        ArrayList list = new ArrayList();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectRecordsSinceSQL );
            pstmt.setLong( 1, timestamp );
            rs = pstmt.executeQuery();
            while ( rs.next() ) {
                try {
                    byte[] buf = Util.readBytes( rs, 1 );
                    list.add( buf );
                } catch (IOException e) {
                    // fail to load one record; just log the record TS
                    IOException ex = DBManager.wrapIOException(
                        "[" + selectRecordsSinceSQL + "]", e );
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_CONFIGRECORD_FAILED,
                        rs.getString( 2 ), ex);
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectRecordsSinceSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_CONFIGRECORDS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Return all records together with their corresponding timestamps.
     * @param conn database connection
     * @return an array of Object whose first element contains a List of
     *	 timestamps and the second element contains a List of config records of
     *   type byte[].
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public Object[] getAllRecords( Connection conn ) throws BrokerException {

        ArrayList timeList = new ArrayList();
        ArrayList recordList = new ArrayList();
        Object[] data = { timeList, recordList };

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAllSQL );
            rs = pstmt.executeQuery();
            while ( rs.next() ) {
                long createdTS = -1;
                try {
                    createdTS = rs.getLong( 2 );
                    timeList.add( new Long( createdTS ) );

                    byte[] buf = Util.readBytes( rs, 1 );
                    recordList.add( buf );
                } catch (IOException e) {
                    // fail to load one record; just log the record TS
                    IOException ex = DBManager.wrapIOException(
                        "[" + selectAllSQL + "]", e );
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_CONFIGRECORD_FAILED,
                        String.valueOf( createdTS ), ex);
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_CONFIGRECORDS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return data;
    }

    /**
     * Get debug information about the store.
     * @param conn database connection
     * @return a HashMap of name value pair of information
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int size = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountSQL );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                size = rs.getInt( 1 );
            }
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, BrokerResources.X_JDBC_QUERY_FAILED,
                selectCountSQL, e );
        } finally {
            try {
                if ( myConn ) {
                    Util.close( rs, pstmt, conn );
                } else {
                    Util.close( rs, pstmt, null );
                }
            } catch ( BrokerException be ) {
                logger.log( Logger.ERROR, be.getMessage(), be.getCause() );
            }
        }

        map.put( "Config Change Records(" + tableName + ")", String.valueOf( size ) );
        return map;
    }
}
