/*
 * @(#)MessageDAOImpl.java	1.32 01/30/06
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.io.SysMessageID;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.DestMetricsCounters;
import com.sun.messaging.jmq.io.Status;

import java.util.*;
import java.sql.*;
import java.io.*;

/**
 * This class implement a generic MessageDAO.
 *
 * @version	1.32
 */
class MessageDAOImpl extends BaseDAOImpl implements MessageDAO {

    protected String tableName;
    protected static int msgColumnType = -Integer.MAX_VALUE;

    // SQLs
    protected String insertSQL;
    protected String updateDestinationSQL;
    protected String deleteSQL;
    protected String deleteByDstSQL;
    protected String selectSQL;
    protected String selectForUpdateSQL;
    protected String selectBrokerSQL;
    protected String selectCountSQL;
    protected String selectCountByDstSQL;
    protected String selectCountByBrokerSQL;
    protected String selectCountByConsumerAckedSQL;
    protected String selectIDsByDstSQL;
    protected String selectExistSQL;
    protected String selectTakeoverSQL;
    protected String takeoverSQL;

    /**
     * Constructor
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    MessageDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( ID_COLUMN ).append( ", " )
            .append( MESSAGE_SIZE_COLUMN ).append( ", " )
            .append( BROKER_ID_COLUMN ).append( ", " )
            .append( DESTINATION_ID_COLUMN ).append( ", " )
            .append( TRANSACTION_ID_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN ).append( ", " )
            .append( SYSMESSAGE_ID_COLUMN ).append( ", " )
            .append( MESSAGE_COLUMN )
            .append( ") VALUES ( ?, ?, ?, ?, ?, ?, ?, ? )" )
            .toString();

        updateDestinationSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( DESTINATION_ID_COLUMN ).append( " = ?, " )
            .append( MESSAGE_SIZE_COLUMN ).append( " = ?, " )
            .append( MESSAGE_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        deleteSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        deleteByDstSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " AND " )
            .append( DESTINATION_ID_COLUMN ).append( " = ?" )
            .toString();

        selectCountSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*) " )
            .append( " FROM " ).append( tableName )
            .toString();

        selectCountByBrokerSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*) " )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        // This query is a bit more complex because we want the SQL to count
        // the # of msgs, calculate the total size, and also determine if the
        // destination does exists
        selectCountByDstSQL = new StringBuffer(128)
            .append( "SELECT totalmsg, totalsize, " )
            .append( DestinationDAO.ID_COLUMN )
            .append( " FROM " )
            .append( dbMgr.getTableName( DestinationDAO.TABLE_NAME_PREFIX  ) )
            .append( ", (SELECT COUNT(" )
            .append(     ID_COLUMN ).append( ") totalmsg, SUM(" )
            .append(     MESSAGE_SIZE_COLUMN ).append( ") totalsize")
            .append(   " FROM " ).append( tableName )
            .append(   " WHERE " )
            .append(     BROKER_ID_COLUMN ).append( " = ?" )
            .append(   " AND " )
            .append(     DESTINATION_ID_COLUMN ).append( " = ?) msgtable" )
            .append( " WHERE " )
            .append( DestinationDAO.ID_COLUMN ).append( " = ?" )
            .toString();

        // For this SQL, we can just use the consumer state table but
        // joining it w/ the message table ensure the data integrity
        selectCountByConsumerAckedSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*) total, SUM(CASE WHEN " )
            .append( ConsumerStateDAO.STATE_COLUMN )
            .append( " = " ).append( Store.INTEREST_STATE_ACKNOWLEDGED )
            .append( " THEN 1 ELSE 0 END) totalAcked" )
            .append( " FROM " )
            .append( tableName ).append( ", " )
            .append( dbMgr.getTableName( ConsumerStateDAO.TABLE_NAME_PREFIX ) )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .append( " AND " )
            .append( ID_COLUMN ).append( " = " )
            .append( ConsumerStateDAO.MESSAGE_ID_COLUMN )
            .toString();

        selectSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( MESSAGE_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectForUpdateSQL = new StringBuffer(128)
            .append( selectSQL )
            .append( " FOR UPDATE" )
            .toString();

        selectBrokerSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( BROKER_ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectIDsByDstSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " AND " )
            .append( DESTINATION_ID_COLUMN ).append( " = ?" )
            .toString();

        selectExistSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectTakeoverSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( MESSAGE_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        takeoverSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public String getTableName() {
        return tableName;
    }

    /**
     * Insert a new entry.
     * @param conn database connection
     * @param message the message to be persisted
     * @param conUIDs an array of interest ids whose states are to be
     *      stored with the message
     * @param states an array of states
     * @param brokerID the broker ID that owns the msg
     * @param createdTime timestamp
     * @param checkMsgExist check if message & destination exist in the store
     * @exception BrokerException if a message with the same id exists
     *  in the store already
     */
    public void insert( Connection conn, Packet message, ConsumerUID[] conUIDs,
        int[] states, String brokerID, long createdTime, boolean checkMsgExist )
        throws BrokerException {

        SysMessageID sysMsgID = (SysMessageID)message.getSysMessageID();
        String id = sysMsgID.getUniqueName();
        String dstID = DestinationUID.getUniqueString(
            message.getDestination(), message.getIsQueue() );
        int size = message.getPacketSize();
        long txnID = message.getTransactionID();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true;
            }

            if ( checkMsgExist ) {
                if ( hasMessage( conn, id ) ) {
                    throw new BrokerException(
                        br.getKString(BrokerResources.E_MSG_EXISTS_IN_STORE, id ) );
                }

                dbMgr.getDAOFactory().getDestinationDAO().checkDestination( conn, dstID );
            }

            if ( brokerID == null ) {
                brokerID = dbMgr.getBrokerID();
            }

            try {
                // Get SysMessageID as bytes array since it is not serializable
                ByteArrayOutputStream baos = new ByteArrayOutputStream( SysMessageID.ID_SIZE );
                DataOutputStream dos = new DataOutputStream( baos );
                sysMsgID.writeID( dos );
                dos.flush();
                baos.flush();
                byte[] msgIDByteArray = baos.toByteArray();
                dos.close();
                baos.close();

                // Get the msg as bytes array
                baos = new ByteArrayOutputStream( size );
                message.writePacket( baos );
                byte[] data = baos.toByteArray();
                baos.close();
                baos = null;

                pstmt = conn.prepareStatement( insertSQL );
                pstmt.setString( 1, id );
                pstmt.setInt( 2, size );
                pstmt.setString( 3, brokerID );
                pstmt.setString( 4, dstID );
                Util.setLong( pstmt, 5, (( txnID == 0 ) ? -1 : txnID) );
                pstmt.setLong( 6, createdTime );
                pstmt.setBytes( 7, msgIDByteArray );

                ByteArrayInputStream bais = Util.setBytes( pstmt, 8, data );
                bais.close();

                pstmt.executeUpdate();

                // Store the consumer's states if any
                if ( conUIDs != null ) {
                    dbMgr.getDAOFactory().getConsumerStateDAO().insert(
                        conn, sysMsgID, conUIDs, states, false );
                }

                // Commit all changes
                if ( myConn ) {
                    conn.commit();
                }
            } catch ( Exception e ) {
                try {
                    if ( (conn != null) && !conn.getAutoCommit() ) {
                        conn.rollback();
                    }
                } catch ( SQLException rbe ) {
                    logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
                }

                Exception ex;
                if ( e instanceof BrokerException ) {
                    throw (BrokerException)e;
                } else if ( e instanceof IOException ) {
                    ex = DBManager.wrapIOException("[" + insertSQL + "]", (IOException)e);
                } else if ( e instanceof SQLException ) {
                    ex = DBManager.wrapSQLException("[" + insertSQL + "]", (SQLException)e);
                } else {
                    ex = e;
                }

                throw new BrokerException(
                    br.getKString( BrokerResources.X_PERSIST_MESSAGE_FAILED,
                    id ), ex );
            }
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Move a message to another destination.
     * @param conn database connection
     * @param message the message
     * @param from the destination
     * @param to the destination to move to
     * @param conUIDs an array of interest ids whose states are to be
     *      stored with the message
     * @param states an array of states
     * @throws IOException
     * @throws BrokerException
     */
    public void moveMessage( Connection conn, Packet message,
        DestinationUID from, DestinationUID to, ConsumerUID[] conUIDs,
        int[] states ) throws IOException, BrokerException {

	SysMessageID sysMsgID = (SysMessageID)message.getSysMessageID().clone();
        String id = sysMsgID.getUniqueName();
        int size = message.getPacketSize();

/*      TODO phuong - should we verify the message exist in the destination?
	DestinationInfo fromDstInfo = getDestinationInfo( from.toString() );
	DestinationInfo toDstInfo = getDestinationInfo( to.toString() );
	if ( !fromDstInfo.containsMsg( sysMsgID ) ) {
	    logger.log( Logger.ERROR, BrokerResources.E_MSG_NOT_FOUND_IN_DST,
                id, from );
	    throw new BrokerException(
                br.getString( BrokerResources.E_MSG_NOT_FOUND_IN_DST,
                id, to ));
	}
*/
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true;
            }

            // Get the msg as bytes array
            ByteArrayOutputStream baos = new ByteArrayOutputStream( size );
            message.writePacket( baos );
            byte[] data = baos.toByteArray();
            baos.close();
            baos = null;

            pstmt = conn.prepareStatement( updateDestinationSQL );
            pstmt.setString( 1, to.toString() );
            pstmt.setInt( 2, size );

            ByteArrayInputStream bais = Util.setBytes( pstmt, 3, data );
            bais.close();

            pstmt.setString( 4, id );

            if ( pstmt.executeUpdate() == 0 ) {
                // We're assuming the entry does not exist
                throw new BrokerException(
                    br.getKString( BrokerResources.E_MSG_NOT_FOUND_IN_STORE, id ),
                    Status.NOT_FOUND );
            }

            /**
             * Update consumer states:
             * 1. remove the old states
             * 2. re-insert the states
             */
            ConsumerStateDAO conStateDAO = dbMgr.getDAOFactory().getConsumerStateDAO();
            conStateDAO.deleteByMessageID( conn, sysMsgID );
            if ( conUIDs != null || states != null ) {
                conStateDAO.insert( conn, sysMsgID, conUIDs, states, false );
            }

            // Commit all changes
            if ( myConn ) {
                conn.commit();
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateDestinationSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            Object[] args = { id, from, to };
            throw new BrokerException(
                br.getKString( BrokerResources.X_MOVE_MESSAGE_FAILED,
                args ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete an existing entry.
     * @param conn Database Connection
     * @param sysMsgID the SysMessageID
     * @throws BrokerException
     */
    public void delete( Connection conn, SysMessageID sysMsgID,
        DestinationUID dstUID ) throws BrokerException {

        String id = sysMsgID.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true; // Set to true since this is our connection
            }

            checkMessage( conn, id );

            // Delete states
            dbMgr.getDAOFactory().getConsumerStateDAO().deleteByMessageID( conn, sysMsgID );

            // Now delete the message
            pstmt = conn.prepareStatement( deleteSQL );
            pstmt.setString( 1, id );
            pstmt.executeUpdate();

            // Check whether to commit or not
            if ( myConn ) {
                conn.commit();
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + deleteSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_REMOVE_MESSAGE_FAILED,
                id ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete all messages from a destination for the current broker.
     * @param conn Database Connection
     * @param dst the destination
     * @throws BrokerException
     */
    public void deleteByDestination( Connection conn, Destination dst )
        throws BrokerException {

        String dstID = dst.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true; // Set to true since this is our connection
            }

            dbMgr.getDAOFactory().getDestinationDAO().checkDestination( conn, dstID );

            // First delete consumer states for the destination since
            // Consumer State table is a child table, i.e. it has to join with
            // the Message table to select message IDs for the destination that
            // will be deleted.
            dbMgr.getDAOFactory().getConsumerStateDAO().deleteByDestination( conn, dst );

            // Now delete all msgs associated with the destination
            pstmt = conn.prepareStatement( deleteByDstSQL );
            pstmt.setString( 1, dbMgr.getBrokerID() );
            pstmt.setString( 2, dstID );
            pstmt.executeUpdate();

            // Check whether to commit or not
            if ( myConn ) {
                conn.commit();
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + deleteByDstSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_REMOVE_MESSAGES_FOR_DST_FAILED,
                dstID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete all entries.
     * @param conn Database Connection
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void deleteAll( Connection conn )
        throws BrokerException {

        String whereClause = null;        
        if ( Globals.getHAEnabled() ) {
            // Only delete messages that belong to the running broker,
            // construct the where clause for the delete statement:
            //   DELETE FROM mqmsg40cmycluster WHERE broker_id = 'mybroker'
            whereClause = new StringBuffer(128)
                .append( BROKER_ID_COLUMN ).append( " = '")
                .append( DBManager.getDBManager().getBrokerID() ).append( "'" )
                .toString();
        }

        deleteAll( conn, whereClause, null, 0 );
    }

    /**
     * Take over the messages.
     * @param conn database connection
     * @param brokerID the current or local broker ID
     * @param targetBrokerID the broker ID of the store being taken over
     * @return a List of all messages the target broker owns
     * @throws BrokerException
     */
    public List takeover( Connection conn, String brokerID, String targetBrokerID )
        throws BrokerException {

        List list = Collections.EMPTY_LIST;

        String sql = null;
        PreparedStatement pstmt = null;
        try {
            // First retrieve all messages for the target broker
            sql = selectTakeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, targetBrokerID );
            ResultSet rs = pstmt.executeQuery();
            list = (List)loadData( rs, false );
            rs.close();
            pstmt.close();

            // Now takeover those destinations
            sql = takeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, brokerID );
            pstmt.setString( 2, targetBrokerID );
            int count = pstmt.executeUpdate();

            // Verify that we are able to takeover all destinations
            if ( count != list.size() ) {
                // This shouldn't occur but just being safe
                String[] args = { targetBrokerID,
                                  String.valueOf( list.size() ),
                                  String.valueOf( count ) };
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TAKEOVER_MSG_FAILED, args ) );
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + sql + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.E_TAKEOVER_MSG_2_FAILED,
                    targetBrokerID ), ex );
        } finally {
            Util.close( null, pstmt, null );
        }

        return list;
    }

    /**
     * Get the message.
     * @param conn database connection
     * @param sysMsgID the SysMessageID
     * @return Packet the message
     * @throws BrokerException if message does not exist in the store
     */
    public Packet getMessage( Connection conn, SysMessageID sysMsgID )
        throws BrokerException {

        String id = sysMsgID.getUniqueName();
        Packet msg = getMessage( conn, id );
        if ( msg == null ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_MSG_NOT_FOUND_IN_STORE, id ),
                Status.NOT_FOUND );
        }

        return msg;
    }

    /**
     * Get a Message.
     * @param conn database connection
     * @param id the system message id of the message
     * @return Packet the message
     * @throws BrokerException
     */
    public Packet getMessage( Connection conn, String id ) throws BrokerException {

        Packet msg = null;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectSQL );
            pstmt.setString( 1, id );
            rs = pstmt.executeQuery();
            msg = (Packet)loadData( rs, true );
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + selectSQL + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_MESSAGE_FAILED, id), ex);
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return msg;
    }

    /**
     * Get the broker ID that owns the specified message.
     * @param conn database connection
     * @param id the system message id of the message
     * @return the broker ID
     * @throws BrokerException
     */
    public String getBroker( Connection conn, String id ) throws BrokerException {

        String brokerID = null;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectBrokerSQL );
            pstmt.setString( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                brokerID = rs.getString( 1 );
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_MSG_NOT_FOUND_IN_STORE, id ),
                    Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectBrokerSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_MESSAGE_FAILED, id), ex);
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return brokerID;
    }

    /**
     * Get all message IDs for a destination and current/local broker.
     * @param conn database connection
     * @param dst the destination
     * @return a List of all persisted destination names.
     * @throws BrokerException
     */
    public List getMessagesByDestination( Connection conn, Destination dst )
        throws BrokerException {

        ArrayList list = new ArrayList();

        String dstID = dst.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }
            pstmt = conn.prepareStatement( selectIDsByDstSQL );
            pstmt.setString( 1, dbMgr.getBrokerID() );
            pstmt.setString( 2, dstID );
            rs = pstmt.executeQuery();

            while ( rs.next() ) {
                String msgID = rs.getString( 1 );
                list.add( msgID );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectIDsByDstSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_MESSAGES_FOR_DST_FAILED ),
                dstID, ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Return an enumeration of all persisted messages for the given destination.
     * Use the Enumeration methods on the returned object to fetch and load
     * each message sequentially.
     *
     * This method is to be used at broker startup to load persisted
     * messages on demand.
     *
     * @return an enumeration of all persisted messages, an empty
     *		enumeration will be returned if no messages exist for the
     *		destionation
     * @exception BrokerException if an error occurs while getting the data
     */
    public Enumeration messageEnumeration( Destination dst ) throws BrokerException {

        Connection conn = null;
        try {
            DBManager dbMgr = DBManager.getDBManager();
            conn = dbMgr.getConnection( true );

            // Verify destination exists
            dbMgr.getDAOFactory().getDestinationDAO().checkDestination(
                conn, dst.getUniqueName() );

            Iterator msgIDItr = getMessagesByDestination( conn, dst ).iterator();
            return new MsgEnumeration( this, msgIDItr );
        } finally {
            Util.close( null, null, conn );
        }
    }

    /**
     * Check if a a message has been acknowledged by all interests, i.e. consumers.
     * @param conn database connection
     * @param sysMsgID sysMsgID the SysMessageID
     * @return true if all interests have acknowledged
     * @throws BrokerException
     */
    public boolean hasMessageBeenAcked( Connection conn, SysMessageID sysMsgID )
        throws BrokerException {

        int total = -1;
        int totalAcked = -1;
        String id = sysMsgID.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountByConsumerAckedSQL );
            pstmt.setString( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                total = rs.getInt( 1 );
                totalAcked = rs.getInt( 2 );
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_MSG_NOT_FOUND_IN_STORE, id ),
                    Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectCountByConsumerAckedSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_MESSAGE_FAILED,
                id ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        // Return true if all interests have acknowledged
        if ( total != -1 && totalAcked != -1 && total == totalAcked ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Check whether the specified message exists.
     * @param conn database connection
     * @param id the system message id of the message to be checked
     * @return return true if the specified message exists
     */
    public boolean hasMessage( Connection conn, String id ) throws BrokerException {

        boolean found = false;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectExistSQL );
            pstmt.setString( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                found = true;
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectExistSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_MESSAGE_FAILED,
                id ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return found;
    }

    /**
     * Check whether the specified message exists.
     * @param conn
     * @param id the system message id of the message to be checked
     * @throws BrokerException if the message does not exist in the store
     */
    public void checkMessage( Connection conn, String id ) throws BrokerException {

        if ( !hasMessage( conn, id ) ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_MSG_NOT_FOUND_IN_STORE, id ),
                Status.NOT_FOUND );
        }
    }

    /**
     * Get debug information about the store.
     * @param conn database connection
     * @return A HashMap of name value pair of information
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int size = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountSQL );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                size = rs.getInt( 1 );
            }
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, BrokerResources.X_JDBC_QUERY_FAILED,
                selectCountSQL, e );
        } finally {
            try {
                if ( myConn ) {
                    Util.close( rs, pstmt, conn );
                } else {
                    Util.close( rs, pstmt, null );
                }
            } catch ( BrokerException be ) {
                logger.log( Logger.ERROR, be.getMessage(), be.getCause() );
            }
        }

        map.put( "Messages(" + tableName + ")", String.valueOf( size ) );

        return map;
    }

    /**
     * Return the message count for the given broker.
     * @param conn database connection
     * @param brokerID the broker ID
     * @return the message count
     */
    public int getMessageCount( Connection conn, String brokerID ) throws BrokerException {

        int size = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountByBrokerSQL );
            pstmt.setString( 1, brokerID );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                size = rs.getInt( 1 );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectCountByBrokerSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString(
                    BrokerResources.X_GET_MSG_COUNTS_FOR_BROKER_FAILED,
                    brokerID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return size;
    }

    /**
     * Return the number of persisted messages and total number of bytes for
     * the given destination.
     * @param conn database connection
     * @param dst the destination
     * @return a HashMap
     */
    public HashMap getMessageStorageInfo( Connection conn, Destination dst )
        throws BrokerException {

        HashMap data = new HashMap( 2 );

        String dstID = dst.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountByDstSQL );
            pstmt.setString( 1, dbMgr.getBrokerID() );
            pstmt.setString( 2, dstID );
            pstmt.setString( 3, dstID );

            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                data.put( DestMetricsCounters.CURRENT_MESSAGES,
                    new Integer( rs.getInt( 1 ) ) );
                data.put( DestMetricsCounters.CURRENT_MESSAGE_BYTES,
                    new Long( rs.getLong( 2 ) ) );
            } else {
                // Destination doesn't exists
                throw new BrokerException(
                    br.getKString( BrokerResources.E_DESTINATION_NOT_FOUND_IN_STORE,
                    dstID ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectCountByDstSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_GET_COUNTS_FROM_DATABASE_FAILED,
                dstID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return data;
    }

    /**
     * Load a single message or messages from a ResultSet.
     * @param rs the ResultSet
     * @param isSingleRow specify interesed in only the 1st row of the ResultSet
     * @return a message or a List of messages
     * @throws IOException
     * @throws SQLException
     */
    protected Object loadData( ResultSet rs, boolean isSingleRow )
        throws IOException, SQLException {

        ArrayList list = null;
        if ( !isSingleRow ) {
            list = new ArrayList( 100 );
        }

        while ( rs.next() ) {
            Packet msg = new Packet(false);
            msg.generateTimestamp(false);
            msg.generateSequenceNumber(false);

            InputStream is = null;
            if ( getMsgColumnType(rs, 1) == Types.BLOB ) {
                Blob blob = rs.getBlob( 1 );
                is = blob.getBinaryStream();
            } else {
                is = rs.getBinaryStream( 1 );
            }

            msg.readPacket(is);
            is.close();

            if (Store.DEBUG) {
                logger.log(Logger.DEBUG,
                    "Loaded message from database for "+ msg.getMessageID() );
            }

            if ( isSingleRow ) {
                return msg;
            } else {
                list.add( msg );
            }
        }

        return list;
    }

    /**
     * Get Message column type (e.g. is it a Blob?)
     * @param rs the ResultSet
     * @param msgColumnIndex the index of the Message column
     * @return column type
     */
    protected int getMsgColumnType( ResultSet rs, int msgColumnIndex )
        throws SQLException {

        // Cache the result
        if ( msgColumnType == -Integer.MAX_VALUE ) {
            msgColumnType = rs.getMetaData().getColumnType( msgColumnIndex );
        }

        return msgColumnType;
    }

    /**
     * Message Enumeration class.
     */
    private static class MsgEnumeration implements Enumeration {

        MessageDAO msgDAO = null;
        Iterator msgIDItr = null;
        Object msgToReturn = null;

        MsgEnumeration( MessageDAO dao, Iterator itr ) {
            msgDAO = dao;
            msgIDItr = itr;
        }

        public boolean hasMoreElements() {
            Packet msg = null;
            while ( msgIDItr.hasNext() ) {
                String mid = null;
                try {
                    mid = (String)msgIDItr.next();
                    msg = msgDAO.getMessage( null, mid );
                    msgToReturn = msg;
                    return true;
                } catch ( Exception e ) {
                    Globals.getLogger().logStack( Logger.ERROR,
                        BrokerResources.X_LOAD_MESSAGE_FAILED, mid, e );
                }
            }

            // no more
            msgToReturn = null;
            return false;
        }

        public Object nextElement() {
            if ( msgToReturn != null ) {
                return msgToReturn;
            } else {
                throw new NoSuchElementException();
            }
        }
    }
 }
