/******************************************************************************
  
  Copyright(c) 2003 - 2004 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of version 2 of the GNU General Public License as 
  published by the Free Software Foundation.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

  Portions of this file are based on the sample_* files provided by Wireless
  Extensions 0.26 package and copyright (c) 1997-2003 Jean Tourrilhes 
  <jt@hpl.hp.com>

  Portions of this file are based on the Host AP project, 
  Copyright (c) 2001-2002, SSH Communications Security Corp and Jouni Malinen
    <jkmaline@cc.hut.fi>
  Copyright (c) 2002-2003, Jouni Malinen <jkmaline@cc.hut.fi>

  Portions of ipw2100_mod_firmware_load, ipw2100_do_mod_firmware_load, and 
  ipw2100_fw_load are loosely based on drivers/sound/sound_firmware.c 
  available in the 2.4.25 kernel sources, and are copyright (c) Alan Cox

******************************************************************************/
/*

 Initial driver on which this is based was developed by Janusz Gorycki, 
 Maciej Urbaniak, and Maciej Sosnowski.

 Promiscuous mode support added by Jacek Wysoczynski and Maciej Urbaniak.

Theory of Operation

Tx - Commands and Data

Firmware and host share a circular queue of Transmit Buffer Descriptors (TBDs)
Each TBD contains a pointer to the physical (dma_addr_t) address of data being
sent to the firmware as well as the length of the data.

The host writes to the TBD queue at the WRITE index.  The WRITE index points
to the _next_ packet to be written and is advanced when after the TBD has been
filled.

The firmware pulls from the TBD queue at the READ index.  The READ index points
to the currently being read entry, and is advanced once the firmware is
done with a packet.

When data is sent to the firmware, the first TBD is used to indicate to the
firmware if a Command or Data is being sent.  If it is Command, all of the 
command information is contained within the physical address referred to by the
TBD.  If it is Data, the first TBD indicates the type of data packet, number
of fragments, etc.  The next TBD then referrs to the actual packet location.

The Tx flow cycle is as follows:

1) ipw2100_tx() is called by kernel with SKB to transmit
2) Packet is move from the tx_free_list and appended to the transmit pending 
   list (tx_pend_list)
3) work is scheduled to move pending packets into the shared circular queue.
4) when placing packet in the circular queue, the incoming SKB is DMA mapped
   to a physical address.  That address is entered into a TBD.  Two TBDs are
   filled out.  The first indicating a data packet, the second referring to the
   actual payload data.
5) the packet is removed from tx_pend_list and placed on the end of the 
   firmware pending list (fw_pend_list)
6) firmware is notified that the WRITE index has 
7) Once the firmware has processed the TBD, INTA is triggered.  
8) For each Tx interrupt received from the firmware, the READ index is checked
   to see which TBDs are done being processed.
9) For each TBD that has been processed, the ISR pulls the oldest packet 
   from the fw_pend_list.  
10)The packet structure contained in the fw_pend_list is then used
   to unmap the DMA address and to free the SKB originally passed to the driver
   from the kernel.
11)The packet structure is placed onto the tx_free_list

The above steps are the same for commands, only the msg_free_list/msg_pend_list
are used instead of tx_free_list/tx_pend_list

...

Critical Sections / Locking :

There are two locks utilized.  The first is the low level lock (priv->low_lock)
that protects the following:

- Access to the Tx/Rx queue lists via priv->low_lock. The lists are as follows:

  tx_free_list : Holds pre-allocated Tx buffers.
    TAIL modified in __ipw2100_tx_process()
    HEAD modified in ipw2100_tx()

  tx_pend_list : Holds used Tx buffers waiting to go into the TBD ring 
    TAIL modified ipw2100_tx()
    HEAD modified by X__ipw2100_tx_send_data()

  msg_free_list : Holds pre-allocated Msg (Command) buffers
    TAIL modified in __ipw2100_tx_process()
    HEAD modified in ipw2100_hw_send_command()

  msg_pend_list : Holds used Msg buffers waiting to go into the TBD ring
    TAIL modified in ipw2100_hw_send_command()
    HEAD modified in X__ipw2100_tx_send_commands()

  The flow of data on the TX side is as follows:

  MSG_FREE_LIST + COMMAND => MSG_PEND_LIST => TBD => MSG_FREE_LIST
  TX_FREE_LIST + DATA => TX_PEND_LIST => TBD => TX_FREE_LIST

  The methods that work on the TBD ring are protected via priv->low_lock.

- The internal data state of the device itself
- Access to the firmware read/write indexes for the BD queues
  and associated logic

...

TODO

- Deprecate the proc filesystem code (how to cleanly support 2.4 and 2.6 in
  one driver base?)  ethtool_ops ?
-   

...

*/

#include <linux/compiler.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/if_arp.h>
#include <linux/in6.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/skbuff.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#define __KERNEL_SYSCALLS__
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/unistd.h>
#include <linux/stringify.h>
#include <linux/tcp.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/time.h>
#ifndef CONFIG_IPW2100_LEGACY_FW_LOAD
#include <linux/firmware.h>
#endif

#include "ipw2100.h"

#define IPW2100_VERSION "0.55"
#define IPW2100_VERSION "0.55"

#define DRV_NAME	"ipw2100"
#define DRV_VERSION	IPW2100_VERSION
#define DRV_DESCRIPTION	"Intel(R) PRO/Wireless 2100 Network Driver"
#define DRV_COPYRIGHT	"Copyright(c) 2003-2004 Intel Corporation"


/* Debugging stuff */
#ifdef CONFIG_IPW_DEBUG
#define CONFIG_IPW2100_RX_DEBUG   /* Reception debugging */
#endif

static int debug = -1;

/**
 * Debug level structure--keeps the @name and the value/mask for
 * it.
 */
struct ipw2100_dl {
	const char *name;
	unsigned long value;
};

#define IPW_DL(v) { .name = #v, .value = v }

/** List of debug levels */
const struct ipw2100_dl ipw2100_debug_levels[] = {
#ifdef CONFIG_IPW_DEBUG 
	IPW_DL(IPW_DL_ERROR),
	IPW_DL(IPW_DL_WARNING),
	IPW_DL(IPW_DL_INFO),
	IPW_DL(IPW_DL_WIRELESS_EXT),
	IPW_DL(IPW_DL_ISR),
	IPW_DL(IPW_DL_SCAN),
	IPW_DL(IPW_DL_STATUS),
	IPW_DL(IPW_DL_TRACE),
	IPW_DL(IPW_DL_RX),
	IPW_DL(IPW_DL_TX),
	IPW_DL(IPW_DL_MANAGE),
	IPW_DL(IPW_DL_WEP),
	IPW_DL(IPW_DL_FW_COMMAND),
	IPW_DL(IPW_DL_FRAG),
	IPW_DL(IPW_DL_FW),
	IPW_DL(IPW_DL_RF_KILL),
	IPW_DL(IPW_DL_DROP),
	IPW_DL(IPW_DL_IO),
	IPW_DL(IPW_DL_IOCTL),
	IPW_DL(IPW_DL_TEST),
#endif
};


MODULE_DESCRIPTION(DRV_DESCRIPTION);
MODULE_AUTHOR(DRV_COPYRIGHT);
MODULE_LICENSE("GPL");

static char *if_name = NULL;
static int mode = 0;
static int channel = 1;
static int associate = 1;
static int disable = 0;
#ifdef CONFIG_PM
static struct ipw2100_fw ipw2100_firmware;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

MODULE_PARM(debug, "i");
MODULE_PARM(if_name, "s");
MODULE_PARM(mode, "i");
MODULE_PARM(channel, "i");
MODULE_PARM(associate, "i");
MODULE_PARM(disable, "i");

#else /* LINUX_VERSION_CODE < 2.6.0 */

#include <linux/moduleparam.h>
module_param(debug, int, -1);
module_param(if_name, charp, 0);
module_param(mode, int, 0);
module_param(channel, int, 1);
module_param(associate, int, 1);
module_param(disable, int, 1);

#endif /* LINUX_VERSION_CODE < 2.6.0 */

MODULE_PARM_DESC(debug, "debug level");
MODULE_PARM_DESC(if_name, "interface name (default 'eth%d')");
MODULE_PARM_DESC(mode, "network mode (0=BSS,1=IBSS,2=Monitor)");
MODULE_PARM_DESC(channel, "channel");
MODULE_PARM_DESC(associate, "auto associate when scanning (default on)");
MODULE_PARM_DESC(disable, "manually disable the radio (default 0 [radio on])");

unsigned long ipw2100_debug_level = IPW_DL_NONE;

const char *port_type_str[] = { 
	"Managed", "Monitor", "Ad-Hoc"
};

#ifdef CONFIG_IPW_DEBUG
const char *command_types[] = {
	"undefined",
	"unused", /* HOST_ATTENTION */ 
	"HOST_COMPLETE",
	"unused", /* SLEEP */
	"unused", /* HOST_POWER_DOWN */
	"unused", 
	"SYSTEM_CONFIG",
	"unused", /* SET_IMR */
	"SSID",
	"MANDATORY_BSSID",
	"AUTHENTICATION_TYPE",
	"ADAPTER_ADDRESS",
	"PORT_TYPE",
	"INTERNATIONAL_MODE",
	"CHANNEL",
	"RTS_THRESHOLD",
	"FRAG_THRESHOLD",
	"POWER_MODE",
	"TX_RATES",
	"BASIC_TX_RATES",
	"WEP_KEY_INFO",
	"unused",
	"unused",
	"unused",
	"unused",
	"WEP_KEY_INDEX",
	"WEP_FLAGS",
	"ADD_MULTICAST",
	"CLEAR_ALL_MULTICAST",
	"BEACON_INTERVAL",
	"ATIM_WINDOW",
	"CLEAR_STATISTICS",
	"undefined",
	"undefined",
	"undefined",
	"undefined",
	"TX_POWER_INDEX",
	"undefined",
	"undefined",
	"undefined",
	"undefined",
	"undefined",
	"undefined",
	"BROADCAST_SCAN",
	"CARD_DISABLE",
	"PREFERRED_BSSID",
	"SET_SCAN_OPTIONS",
	"SCAN_DWELL_TIME",
	"SWEEP_TABLE",
	"AP_OR_STATION_TABLE",
	"GROUP_ORDINALS",
	"SHORT_RETRY_LIMIT",
	"LONG_RETRY_LIMIT",
	"unused", /* SAVE_CALIBRATION */
	"unused", /* RESTORE_CALIBRATION */
	"undefined",
	"undefined",
	"undefined",
	"HOST_PRE_POWER_DOWN",
	"unused", /* HOST_INTERRUPT_COALESCING */
	"undefined",
	"CARD_DISABLE_PHY_OFF",
	"MSDU_TX_RATES"
	"undefined",
	"undefined",
	"SET_STATION_STAT_BITS",
	"CLEAR_STATIONS_STAT_BITS",
	"LEAP_ROGUE_MODE",
	"SET_SECURITY_INFORMATION",
	"DISASSOCIATION_BSSID",
	"SET_WPA_ASS_IE"
};
#endif


/* Pre-decl until we get the code solid and then we can clean it up */
static void X__ipw2100_tx_send_commands(struct ipw2100_priv *priv);
static void X__ipw2100_tx_send_data(struct ipw2100_priv *priv);
static int ipw2100_adapter_setup(struct ipw2100_priv *priv);

static void ipw2100_queues_initialize(struct ipw2100_priv *priv);
static void ipw2100_queues_free(struct ipw2100_priv *priv);
static int ipw2100_queues_allocate(struct ipw2100_priv *priv);

static inline int ipw2100_hw_is_adapter_in_system(struct net_device *dev)
{
	return (dev->base_addr && 
		(readl((void *)(dev->base_addr + IPW_REG_DOA_DEBUG_AREA_START))
		 == IPW_DATA_DOA_DEBUG_VALUE));
}

int ipw2100_get_ordinal(struct ipw2100_priv *priv, u32 ord, 
			void *val, u32 *len)
{
	struct ipw2100_ordinals *ordinals = &priv->ordinals;
	u32 addr;
	u32 field_info;
	u16 field_len;
	u16 field_count;
	u32 total_length;

	if (ordinals->table1_addr == 0) {
		printk(KERN_WARNING DRV_NAME ": attempt to use fw ordinals "
		       "before they have been loaded.\n");
		return -EINVAL;
	}

	if (IS_ORDINAL_TABLE_ONE(ordinals, ord)) {
		if (*len < IPW_ORD_TAB_1_ENTRY_SIZE) {
			*len = IPW_ORD_TAB_1_ENTRY_SIZE;

			printk(KERN_WARNING DRV_NAME 
			       ": ordinal buffer length too small, need %d\n",
			       IPW_ORD_TAB_1_ENTRY_SIZE);

			return -EINVAL;
		}

		read_nic_dword(priv->ndev, ordinals->table1_addr + (ord << 2), 
			       &addr);
		read_nic_dword(priv->ndev, addr, val);

		*len = IPW_ORD_TAB_1_ENTRY_SIZE;
		
		return 0;
	} 
	
	if (IS_ORDINAL_TABLE_TWO(ordinals, ord)) {

		ord -= IPW_START_ORD_TAB_2;
		
		/* get the address of statistic */
		read_nic_dword(priv->ndev, ordinals->table2_addr + (ord << 3), 
			       &addr);
		
		/* get the second DW of statistics ; 
		 * two 16-bit words - first is length, second is count */
		read_nic_dword(priv->ndev, 
			       ordinals->table2_addr + (ord << 3) + sizeof(u32),
			       &field_info);
		
		/* get each entry length */
		field_len = *((u16 *)&field_info);
		
		/* get number of entries */
		field_count = *(((u16 *)&field_info) + 1);
		
		/* abort if no enought memory */
		total_length = field_len * field_count;
		if (total_length > *len) {
			*len = total_length;
			return -EINVAL;
		}
		
		*len = total_length;
		if (!total_length)
			return 0;
		
		/* read the ordinal data from the SRAM */
		read_nic_memory(priv->ndev, addr, total_length, val);
		
		return 0;
	}

	printk(KERN_WARNING DRV_NAME ": ordinal %d neither in table 1 nor "
	       "in table 2\n", ord);

	return -EINVAL;
}

static int ipw2100_set_ordinal(struct ipw2100_priv *priv, u32 ord, u32 *val, 
			       u32 *len)
{
	struct ipw2100_ordinals *ordinals = &priv->ordinals;
	u32 addr;

	if (IS_ORDINAL_TABLE_ONE(ordinals, ord)) {
		if (*len != IPW_ORD_TAB_1_ENTRY_SIZE) {
			*len = IPW_ORD_TAB_1_ENTRY_SIZE;
			IPW_DEBUG_INFO("wrong size\n");
			return -EINVAL;
		}

		read_nic_dword(priv->ndev, ordinals->table1_addr + (ord << 2), 
			       &addr);

		write_nic_dword(priv->ndev, addr, *val);

		*len = IPW_ORD_TAB_1_ENTRY_SIZE;

		return 0;
	} 

	IPW_DEBUG_INFO("wrong table\n");
	if (IS_ORDINAL_TABLE_TWO(ordinals, ord)) 
		return -EINVAL;

	return -EINVAL;
}

/*
 * 1. check if message in progress, ret EBUSY if so
 * 2. issue command
 * 3. wait for command to complete; timeout in 3s
 */
static int ipw2100_hw_send_command(struct ipw2100_priv *priv,
				   struct host_command * cmd) 
{
	struct list_head *element;
	struct ipw2100_tx_packet *packet;
	unsigned long flags;
	int err = 0;

	IPW_DEBUG_INFO("enter - '%s' (%d)\n",
	       command_types[cmd->host_command],
	       cmd->host_command);
	
	spin_lock_irqsave(&priv->low_lock, flags);

	if (priv->fatal_error) {
		IPW_DEBUG_INFO("Attempt to send command while hardware in fatal error condition.");
		err = -EIO;
		goto fail_unlock;
	}

	if (priv->stopped) {
		IPW_DEBUG_INFO("Attempt to send command while hardware is not running.");
		err = -EIO;
		goto fail_unlock;
	}

	if (priv->message_sending) {
		IPW_DEBUG_INFO("Attempt to send command while hardware another command is pending.");
		err = -EBUSY;
		goto fail_unlock;
	}

	if (list_empty(&priv->msg_free_list)) {
		IPW_DEBUG_INFO("no available msg buffers\n");
		goto fail_unlock;
	}

	priv->message_sending = 1;
	priv->messages_sent++;

	element = priv->msg_free_list.next;

	packet = list_entry(element, struct ipw2100_tx_packet, list);
	packet->jiffy_start = jiffies;

	/* initialize the firmware command packet */
	packet->info.c_struct.cmd->host_command_reg = cmd->host_command;
	packet->info.c_struct.cmd->host_command_reg1 = cmd->host_command1;
	packet->info.c_struct.cmd->host_command_len_reg = cmd->host_command_length;
	packet->info.c_struct.cmd->sequence = cmd->host_command_sequence;

	memcpy(packet->info.c_struct.cmd->host_command_params_reg,
	       cmd->host_command_parameters,
	       sizeof(packet->info.c_struct.cmd->host_command_params_reg));

	list_del(element);
	DEC_STAT(&priv->msg_free_stat);

	list_add_tail(element, &priv->msg_pend_list);
	INC_STAT(&priv->msg_pend_stat);

	X__ipw2100_tx_send_commands(priv);
	X__ipw2100_tx_send_data(priv);

	spin_unlock_irqrestore(&priv->low_lock, flags);

	/*
	 * We must wait for this command to complete before another
	 * command can be sent...  but if we wait more than 3 seconds
	 * then there is a problem.
	 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	err = interruptible_sleep_on_timeout(
		&priv->wait_command_queue, 3 * HZ);
#else
	err = wait_event_interruptible_timeout(
		priv->wait_command_queue, !priv->message_sending, 3 * HZ);
#endif
	if (err == 0) {
		printk(KERN_WARNING 
		       "%s: ipw2100_hw_send_command timed out.\n",
		       priv->ndev->name);
		priv->fatal_error = IPW2100_ERR_MSG_TIMEOUT;
		priv->message_sending = 0;
		return -EIO;
	}

	if (priv->fatal_error) {
		printk(KERN_WARNING "%s: firmware fatal error\n",
		       priv->ndev->name);
		return -EIO;
	}

	IPW_DEBUG_INFO("exit - command %d\n", cmd->host_command);

	/* !!!!! HACK TEST !!!!! 
	 * When lots of debug trace statements are enabled, the driver 
	 * doesn't seem to have as many firmware restart cycles...
	 * 
	 * As a test, we're sticking in a 1/100s delay here */
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(HZ / 100);
	
	return 0;

 fail_unlock:
	spin_unlock_irqrestore(&priv->low_lock, flags);

	return err;
}


/*
 * Verify the values and data access of the hardware
 * No locks needed or used.  No functions called.
 */
static int ipw2100_verify(struct ipw2100_priv *priv)
{
	u32 data1, data2;
	u32 address;

	u32 val1 = 0x76543210;
	u32 val2 = 0xFEDCBA98;

	/* Domain 0 check - all values should be DOA_DEBUG */
	for (address = IPW_REG_DOA_DEBUG_AREA_START;
	     address < IPW_REG_DOA_DEBUG_AREA_END;
	     address += sizeof(u32)) {
		read_register(priv->ndev, address, &data1);
		if (data1 != IPW_DATA_DOA_DEBUG_VALUE)
			return -EIO;
	}

	/* Domain 1 check - use arbitrary read/write compare  */
	for (address = 0; address < 5; address++) {
		/* The memory area is not used now */
		write_register(priv->ndev, IPW_REG_DOMAIN_1_OFFSET + 0x32, 
			       val1);
		write_register(priv->ndev, IPW_REG_DOMAIN_1_OFFSET + 0x36, 
			       val2);
		read_register(priv->ndev, IPW_REG_DOMAIN_1_OFFSET + 0x32, 
			      &data1);
		read_register(priv->ndev, IPW_REG_DOMAIN_1_OFFSET + 0x36, 
			      &data2);
		if (val1 == data1 && val2 == data2)
			return 0;
	}

	return -EIO;
}

/*
 * 
 * Loop until the CARD_DISABLED bit is the same value as the
 * supplied parameter
 *
 * TODO: See if it would be more efficient to do a wait/wake
 *       cycle and have the completion event trigger the wakeup
 * 
 */
static int ipw2100_wait_for_card_state(struct ipw2100_priv *priv, int disabled)
{
	int i;
	u32 card_disabled;
	u32 len = sizeof(card_disabled);
	int err;
	
	for (i = 0; i <= IPW_CARD_DISABLE_COMPLETE_WAIT * 1000; i += 50) {
		err = ipw2100_get_ordinal(priv, IPW_ORD_CARD_DISABLED, 
					  &card_disabled, &len);
		if (err) {
			IPW_DEBUG_INFO("query failed.\n");
			break;
		}

		/* We'll break out if either the HW state says it is
		 * in the state we want, or if HOST_COMPLETE command
		 * finishes */
		if (card_disabled == disabled ||
		    priv->disabled == disabled) {
			priv->disabled = disabled;
			return 0;
		}

		udelay(50);
	}

	if (i >= IPW_CARD_DISABLE_COMPLETE_WAIT * 1000) {
		IPW_DEBUG_INFO("ipw2100_wait_for_card_state to %s state "
				   "timed out\n", 
				   disabled ? "DISABLED" : "ENABLED");
		return -EIO;
	}

	return 0;
}


/*********************************************************************
    Procedure   :   sw_reset_and_clock
    Purpose     :   Asserts s/w reset, asserts clock initialization 
                    and waits for clock stabilization
 ********************************************************************/
static int sw_reset_and_clock(struct ipw2100_priv *priv)
{
	int i;
	u32 r;

	// assert s/w reset
	write_register(priv->ndev, IPW_REG_RESET_REG, 
		       IPW_AUX_HOST_RESET_REG_SW_RESET);

	// wait for clock stabilization
	for (i = 0; i < 1000; i++) {
		udelay(IPW_WAIT_RESET_ARC_COMPLETE_DELAY);

		// check clock ready bit
		read_register(priv->ndev, IPW_REG_RESET_REG, &r);
		if (r & IPW_AUX_HOST_RESET_REG_PRINCETON_RESET)
			break;
	}

	if (i == 1000)
		return -EIO;	// TODO: better error value

	/* set "initialization complete" bit to move adapter to 
	 * D0 state */
	write_register(priv->ndev, IPW_REG_GP_CNTRL,
		       IPW_AUX_HOST_GP_CNTRL_BIT_INIT_DONE);

	/* wait for clock stabilization */
	for (i = 0; i < 10000; i++) {
		udelay(IPW_WAIT_CLOCK_STABILIZATION_DELAY * 4);

		/* check clock ready bit */
		read_register(priv->ndev, IPW_REG_GP_CNTRL, &r);
		if (r & IPW_AUX_HOST_GP_CNTRL_BIT_CLOCK_READY)
			break;
	}

	if (i == 10000)
		return -EIO;	/* TODO: better error value */

//#if CONFIG_IPW2100_D0ENABLED
	/* set D0 standby bit */
	read_register(priv->ndev, IPW_REG_GP_CNTRL, &r);
	write_register(priv->ndev, IPW_REG_GP_CNTRL,
		       r | IPW_AUX_HOST_GP_CNTRL_BIT_HOST_ALLOWS_STANDBY);
//#endif

	return 0;
}

/*********************************************************************
    Procedure   :   ipw2100_ipw2100_download_firmware
    Purpose     :   Initiaze adapter after power on.
                    The sequence is:
                    1. assert s/w reset first!
                    2. awake clocks & wait for clock stabilization
                    3. hold ARC (don't ask me why...)
                    4. load Dino ucode and reset/clock init again
                    5. zero-out shared mem
                    6. download f/w
 *******************************************************************/
static int ipw2100_download_firmware(struct ipw2100_priv *priv)
{
	u32 address;
	int err;

#ifndef CONFIG_PM
	/* Fetch the firmware and microcode */
	struct ipw2100_fw ipw2100_firmware;
#endif
	
	if (priv->fatal_error) {
		printk(KERN_ERR "%s: ipw2100_download_firmware called after "
		       "fatal error %d.  Interface must be brought down.\n",
		       priv->ndev->name, priv->fatal_error);
		return -EINVAL;
	}

#ifdef CONFIG_PM
	if (!ipw2100_firmware.version) {
		err = ipw2100_get_firmware(priv, &ipw2100_firmware);
		if (err) {
			printk(KERN_ERR "%s: ipw2100_get_firmware failed: %d\n",
			       priv->ndev->name, err);
			priv->fatal_error = IPW2100_ERR_FW_LOAD;
			goto fail;
		}
	}
#else
	err = ipw2100_get_firmware(priv, &ipw2100_firmware);
	if (err) {
		printk(KERN_ERR "%s: ipw2100_get_firmware failed: %d\n",
		       priv->ndev->name, err);
		priv->fatal_error = IPW2100_ERR_FW_LOAD;
		goto fail;
	}
#endif
	priv->firmware_version = ipw2100_firmware.version;

	/* s/w reset and clock stabilization */
	err = sw_reset_and_clock(priv);
	if (err) {
		printk(KERN_ERR "%s: sw_reset_and_clock failed: %d\n",
		       priv->ndev->name, err);
		goto fail;
	}

	err = ipw2100_verify(priv);
	if (err) {
		printk(KERN_ERR "%s: ipw2100_verify failed: %d\n",
		       priv->ndev->name, err);
		goto fail;
	}

	/* Hold ARC */
	write_nic_dword(priv->ndev,
			IPW_INTERNAL_REGISTER_HALT_AND_RESET,
			0x80000000);

	/* allow ARC to run */
	write_register(priv->ndev, IPW_REG_RESET_REG, 0);

	/* load microcode */
	err = ipw2100_ucode_download(priv, &ipw2100_firmware);
	if (err) {
		printk(KERN_ERR "%s: Error loading microcode: %d\n",
		       priv->ndev->name, err);
		goto fail;
	}

	/* release ARC */
	write_nic_dword(priv->ndev,
			IPW_INTERNAL_REGISTER_HALT_AND_RESET,
			0x00000000);

	/* s/w reset and clock stabilization (again!!!) */
	err = sw_reset_and_clock(priv);
	if (err) {
		printk(KERN_ERR "%s: sw_reset_and_clock failed: %d\n",
		       priv->ndev->name, err);
		goto fail;
	}

	/* load f/w */
	err = ipw2100_fw_download(priv, &ipw2100_firmware);
	if (err) {
		printk(KERN_ERR "%s: Error loading firmware: %d\n",
		       priv->ndev->name, err);
		goto fail;
	}
	
#ifndef CONFIG_PM
	/*
	 * When the .resume method of the driver is called, the other
	 * part of the system, i.e. the ide driver could still stay in
	 * the suspend stage. This prevents us from loading the firmware
	 * from the disk.  --YZ
	 */

	/* free any storage allocated for firmware image */
	ipw2100_release_firmware(priv, &ipw2100_firmware);
#endif

	/* zero out Domain 1 area indirectly (Si requirement) */
	for (address = IPW_HOST_FW_SHARED_AREA0; 
	     address < IPW_HOST_FW_SHARED_AREA0_END; address += 4) 
		write_nic_dword(priv->ndev, address, 0);
	for (address = IPW_HOST_FW_SHARED_AREA1; 
	     address < IPW_HOST_FW_SHARED_AREA1_END; address += 4)
		write_nic_dword(priv->ndev, address, 0);
	for (address = IPW_HOST_FW_SHARED_AREA2; 
	     address < IPW_HOST_FW_SHARED_AREA2_END; address += 4)
		write_nic_dword(priv->ndev, address, 0);
	for (address = IPW_HOST_FW_SHARED_AREA3; 
	     address < IPW_HOST_FW_SHARED_AREA3_END; address += 4)
		write_nic_dword(priv->ndev, address, 0);
	for (address = IPW_HOST_FW_INTERRUPT_AREA; 
	     address < IPW_HOST_FW_INTERRUPT_AREA_END; address += 4)
		write_nic_dword(priv->ndev, address, 0);

	return 0;

 fail:
	ipw2100_release_firmware(priv, &ipw2100_firmware);
	return err;
}

static inline void ipw2100_hw_disable_interrupt(struct ipw2100_priv *priv)
{
	if (!priv->irq_ignore) {
		write_register(priv->ndev, IPW_REG_INTA_MASK, 0x0);
		priv->irq_ignore = 1;
	}
}

static inline void ipw2100_hw_enable_interrupt(struct ipw2100_priv *priv)
{
	if (priv->irq_ignore) {
		priv->irq_ignore = 0;
		write_register(priv->ndev, IPW_REG_INTA_MASK, 
			       IPW_INTERRUPT_MASK);
	}
}

static void ipw2100_initialize_ordinals(struct ipw2100_priv *priv)
{
	struct ipw2100_ordinals *ord = &priv->ordinals;

	IPW_DEBUG_INFO("enter\n");

	read_register(priv->ndev, IPW_MEM_HOST_SHARED_ORDINALS_TABLE_1,
		      &ord->table1_addr);

	read_register(priv->ndev, IPW_MEM_HOST_SHARED_ORDINALS_TABLE_2,
		      &ord->table2_addr);

	read_nic_dword(priv->ndev, ord->table1_addr, &ord->table1_size);
	read_nic_dword(priv->ndev, ord->table2_addr, &ord->table2_size);

	ord->table2_size &= 0x0000FFFF;

	IPW_DEBUG_INFO("table 1 size: %d\n", ord->table1_size);
	IPW_DEBUG_INFO("table 2 size: %d\n", ord->table2_size);
	IPW_DEBUG_INFO("exit\n");
}

static inline void ipw2100_hw_set_gpio(struct ipw2100_priv *priv)
{
	u32 reg = 0;
	/*
	 * Set GPIO 3 writable by FW; GPIO 1 writable 
	 * by driver and enable clock
	 */
	reg = (IPW_BIT_GPIO_GPIO3_MASK | IPW_BIT_GPIO_GPIO1_ENABLE |
	       IPW_BIT_GPIO_LED_OFF);
	write_register(priv->ndev, IPW_REG_GPIO, reg);
}

static inline int ipw2100_get_rf_switch(struct ipw2100_priv *priv)
{
#define MAX_RF_KILL_CHECKS 5
#define RF_KILL_CHECK_DELAY 40
#define RF_KILL_CHECK_THRESHOLD 3
	
	unsigned short value = 0;
	u32 reg = 0;
	int i;
	
	if (!(priv->hw_features & HW_FEATURE_RFKILL)) 
		return 0;

	for (i = 0; i < MAX_RF_KILL_CHECKS; i++) {
		udelay(RF_KILL_CHECK_DELAY);
		read_register(priv->ndev, IPW_REG_GPIO, &reg);
		value = (value << 1) | ((reg & IPW_BIT_GPIO_RF_KILL) ? 0 : 1);
	}

	return (value == 0);
}

static int ipw2100_get_hw_features(struct ipw2100_priv *priv)
{
	u32 addr, len;
	u32 val;

	/* 
	 * EEPROM_SRAM_DB_START_ADDRESS using ordinal in ordinal table 1
	 */
	len = sizeof(addr);
	if (ipw2100_get_ordinal(
		    priv, IPW_ORD_EEPROM_SRAM_DB_BLOCK_START_ADDRESS,
		    &addr, &len)) {
		printk(KERN_INFO "failed querying ordinals at line %d\n", 
		       __LINE__);
		return -EIO;
	}

	IPW_DEBUG_INFO("EEPROM address: %08X\n", addr);
 
	/*
	 * EEPROM version is the byte at offset 0xfd in firmware
	 * We read 4 bytes, then shift out the byte we actually want */ 
	read_nic_dword(priv->ndev, addr + 0xFC, &val);
	priv->eeprom_version = (val >> 24) & 0xFF;
	IPW_DEBUG_INFO("EEPROM version: %d\n", priv->eeprom_version);

        /*
	 *  HW RF Kill enable is bit 0 in byte at offset 0x21 in firmware
	 *
	 *  notice that the EEPROM bit is reverse polarity, i.e.
	 *     bit = 0  signifies HW RF kill switch is supported
	 *     bit = 1  signifies HW RF kill switch is NOT supported
	 */
	read_nic_dword(priv->ndev, addr + 0x20, &val);
	if (!((val >> 24) & 0x01))
		priv->hw_features |= HW_FEATURE_RFKILL;

	IPW_DEBUG_INFO("HW RF Kill: %ssupported.\n",
			   (priv->hw_features & HW_FEATURE_RFKILL) ?
			   "" : "not ");

	return 0;
}

/*
 * Start firmware execution after power on and intialization
 * The sequence is:
 *  1. Release ARC
 *  2. Wait for f/w initialization completes;
 */
static int ipw2100_start_adapter(struct ipw2100_priv *priv)
{
#define IPW_WAIT_FW_INIT_COMPLETE_DELAY (40 * HZ / 1000)
	int i;
	u32 inta, inta_mask, gpio;

	IPW_DEBUG_INFO("enter\n");

	if (!priv->stopped)
		return 0;

	/*
	 * Initialize the hw - drive adapter to DO state by setting 
	 * init_done bit. Wait for clk_ready bit and Download 
	 * fw & dino ucode
	 */
	if (ipw2100_download_firmware(priv)) {
		printk(KERN_ERR "%s: Failed to power on the adapter.\n",
		       priv->ndev->name);
		return -EIO;
	}
	
	/* Clear the Tx, Rx and Msg queues and the r/w indexes
	 * in the firmware RBD and TBD ring queue */
	ipw2100_queues_initialize(priv);

	ipw2100_hw_set_gpio(priv);

	/* TODO -- Look at disabling interrupts here to make sure none
	 * get fired during FW initialization */

	/* Release ARC - clear reset bit */
	write_register(priv->ndev, IPW_REG_RESET_REG, 0);

	/* wait for f/w intialization complete */
	IPW_DEBUG_FW("Waiting for f/w initialization to complete...\n");
	i = 5000;
	do {
  		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(IPW_WAIT_FW_INIT_COMPLETE_DELAY);
		/* Todo... wait for sync command ... */

		read_register(priv->ndev, IPW_REG_INTA, &inta);

		/* check "init done" bit */
		if (inta & IPW2100_INTA_FW_INIT_DONE) {
			/* reset "init done" bit */
			write_register(priv->ndev, IPW_REG_INTA,
				       IPW2100_INTA_FW_INIT_DONE);
			break;
		}

		/* check error conditions : we check these after the firmware
		 * check so that if there is an error, the interrupt handler
		 * will see it and the adapter will be reset */
		if (inta & 
		    (IPW2100_INTA_FATAL_ERROR | IPW2100_INTA_PARITY_ERROR)) {
			/* clear error conditions */
			write_register(priv->ndev, IPW_REG_INTA,
				       IPW2100_INTA_FATAL_ERROR |
				       IPW2100_INTA_PARITY_ERROR);
		}
	} while (i--);

	/* Clear out any pending INTAs since we aren't supposed to have 
	 * interrupts enabled at this point... */
	read_register(priv->ndev, IPW_REG_INTA, &inta);
	read_register(priv->ndev, IPW_REG_INTA_MASK, &inta_mask);
	inta &= IPW_INTERRUPT_MASK;
	/* Clear out any pending interrupts */
	if (inta & inta_mask) 
		write_register(priv->ndev, IPW_REG_INTA, inta);

	IPW_DEBUG_FW("f/w initialization complete: %s\n",
		     i ? "SUCCESS" : "FAILED");

	/* Start firmware hang check timer */
	if (priv->stop_hang_check) {
		priv->stop_hang_check = 0;
		priv->hang_check_timer.expires = jiffies + 2 * HZ;
		add_timer(&priv->hang_check_timer);
	}
	
	if (!i) {
		printk(KERN_WARNING "%s: Firmware did not initialize.\n",
		       priv->ndev->name);
		return -EIO;
	}

	/* allow firmware to write to GPIO1 & GPIO3 */
	read_register(priv->ndev, IPW_REG_GPIO, &gpio);

	gpio |= (IPW_BIT_GPIO_GPIO1_MASK | IPW_BIT_GPIO_GPIO3_MASK);

	write_register(priv->ndev, IPW_REG_GPIO, gpio);

	priv->stopped = 0;

	/* The adapter has been reset; we are not associated */
	priv->connected = 0;

	IPW_DEBUG_INFO("exit\n");

	return 0;
}

static inline void ipw2100_reset_sw(struct ipw2100_priv *priv)
{
	/* Forget whatever we know about any associated SSIDs... */
	memset(priv->ctx->ssid, 0, priv->ctx->ssid_len);
	priv->ctx->channel = 1;
	
	ipw2100_hw_set_gpio(priv);

	/* SW Reset the adapter */
	write_register(priv->ndev, IPW_REG_RESET_REG, 0);
}

static inline void ipw2100_reset_fatalerror(struct ipw2100_priv *priv)
{
	if (!priv->fatal_error)
		return;

	priv->fatal_errors[priv->fatal_index++] = priv->fatal_error;
	priv->fatal_index %= IPW2100_ERROR_QUEUE;
	priv->fatal_error = 0;
}


/* NOTE: Our interrupt is disabled when this method is called */
static int ipw2100_power_cycle_adapter(struct ipw2100_priv *priv)
{
	u32 reg;
	int i;

	IPW_DEBUG_INFO("Power cycling the hardware.\n");

	ipw2100_hw_set_gpio(priv);

	/* Step 1. Stop Master Assert */
	write_register(priv->ndev, IPW_REG_RESET_REG, 
		       IPW_AUX_HOST_RESET_REG_STOP_MASTER);    
	
	/* Step 2. Wait for stop Master Assert 
	 *         (not more then 50us, otherwise ret error */
	i = 5;
	do {
		udelay(IPW_WAIT_RESET_MASTER_ASSERT_COMPLETE_DELAY);
		read_register(priv->ndev, IPW_REG_RESET_REG, &reg);
		
		if (reg & IPW_AUX_HOST_RESET_REG_MASTER_DISABLED)
			break;
	}  while(i--);

	priv->reset_pending = 0;

	if (!i) {
		IPW_DEBUG_INFO("exit - waited too long for master assert stop\n");
		return -EIO;
	}

	write_register(priv->ndev, IPW_REG_RESET_REG, 
		       IPW_AUX_HOST_RESET_REG_SW_RESET);

	
	/* Reset any fatal_error conditions */
	ipw2100_reset_fatalerror(priv);

	/* At this point, the adapter is now stopped and disabled */
	priv->stopped = 1;
	priv->disabled = 1;
	priv->connected = 0;

	return 0;
}

/*
 * Send the CARD_DISABLE_PHY_OFF comamnd to the card to disable it
 *
 * After disabling, if the card was associated, a STATUS_ASSN_LOST will be sent.
 *
 * STATUS_CARD_DISABLE_NOTIFICATION will be sent regardless of
 * if STATUS_ASSN_LOST is sent.
 */
static int ipw2100_hw_phy_off(struct ipw2100_priv *priv)
{

#define HW_PHY_OFF_LOOP_DELAY (HZ / 5000)

	struct host_command cmd = {
		.host_command = CARD_DISABLE_PHY_OFF,
		.host_command_sequence = 0,
		.host_command_length = 0,
	};
	int err, i;
	u32 val1, val2;

	if (priv->phy_off == 1)
		return 0;

	IPW_DEBUG_FW_COMMAND("CARD_DISABLE_PHY_OFF\n");

	/* Turn off the radio */
	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) 
		return err;

	for (i = 0; i < 2500; i++) {
		read_nic_dword(priv->ndev, IPW2100_CONTROL_REG, &val1);
		read_nic_dword(priv->ndev, IPW2100_COMMAND, &val2);

		if ((val1 & IPW2100_CONTROL_PHY_OFF) &&
		    (val2 & IPW2100_COMMAND_PHY_OFF)) {
			priv->phy_off = 1;
			return 0;
		}
		
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(HW_PHY_OFF_LOOP_DELAY);
	}

	return -EIO;
}


static int ipw2100_enable_adapter(struct ipw2100_priv *priv)
{	
	struct host_command cmd = {
		.host_command = HOST_COMPLETE,
		.host_command_sequence = 0,
		.host_command_length = 0
	};
	int err;

	IPW_DEBUG_FW_COMMAND("HOST_COMPLETE\n");
	
	if (!priv->disabled) 
		return 0;

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) {
		IPW_DEBUG_INFO("Failed to send HOST_COMPLETE command\n");
		return err;
	}

	err = ipw2100_wait_for_card_state(priv, STATE_ENABLED);
	if (err) {
		printk(KERN_WARNING 
		       "%s: card not responding to init command.\n",
		       priv->ndev->name);
		return err;
	}

	IPW_DEBUG_INFO("TODO: implement scan state machine\n");

	return 0;
}

static int ipw2100_hw_stop_adapter(struct ipw2100_priv *priv)
{
#define HW_POWER_DOWN_DELAY (HZ / 10)

	struct host_command cmd = {
		.host_command = HOST_PRE_POWER_DOWN,
		.host_command_sequence = 0,
		.host_command_length = 0,
	};
	int err, i;
	u32 reg;

	if (priv->stopped)
		return 0;

	/* We can only shut down the card if the firmware is operational.  So,
	 * if we haven't reset since a fatal_error, then we can not send the
	 * shutdown commands. */
	if (priv->fatal_error) {
		/* First, make sure the adapter is enabled so that the PHY_OFF 
		 * command can shut it down */
		ipw2100_enable_adapter(priv);

		err = ipw2100_hw_phy_off(priv);
		if (err) 
			IPW_DEBUG_WARNING("Error disabling radio %d\n", err);
		
		/*
		 * If in D0-standby mode going directly to D3 may cause a
		 * PCI bus violation.  Therefore we must change out of the D0 
		 * state.
		 *
		 * Sending the PREPARE_FOR_POWER_DOWN will restrict the 
		 * hardware from going into standby mode and will transition 
		 * out of D0-standy if it is already in that state.
		 * 
		 * STATUS_PREPARE_POWER_DOWN_COMPLETE will be sent by the
		 * driver upon completion.  Once received, the driver can 
		 * proceed to the D3 state.
		 *
		 * Prepare for power down command to fw.  This command would 
		 * take HW out of D0-standby and prepare it for D3 state.
		 * 
		 * Currently FW does not support event notification for this 
		 * event. Therefore, skip waiting for it.  Just wait a fixed 
		 * 100ms
		 */
		IPW_DEBUG_FW_COMMAND("HOST_PRE_POWER_DOWN\n");
		
		err = ipw2100_hw_send_command(priv, &cmd);
		if (err) 
			printk(KERN_WARNING 
			       "%s: Power down command failed: Error %d\n",
			       priv->ndev->name, err);
		else {
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(HW_POWER_DOWN_DELAY);
		}
	}
	
	priv->disabled = 1;

	/*
	 * Set GPIO 3 writable by FW; GPIO 1 writable 
	 * by driver and enable clock
	 */
	ipw2100_hw_set_gpio(priv);

	/*
	 * Power down adapter.  Sequence:
	 * 1. Stop master assert (RESET_REG[9]=1)
	 * 2. Wait for stop master (RESET_REG[8]==1)
	 * 3. S/w reset assert (RESET_REG[7] = 1)
	 */

	/* Stop master assert */
	write_register(priv->ndev, IPW_REG_RESET_REG,
		       IPW_AUX_HOST_RESET_REG_STOP_MASTER);

	/* wait stop master not more than 50 usec. 
	 * Otherwise return error. */
	for (i = 5; i > 0; i--) {
		udelay(10);

		/* Check master stop bit */
		read_register(priv->ndev, IPW_REG_RESET_REG, &reg);

		if (reg & IPW_AUX_HOST_RESET_REG_MASTER_DISABLED)
			break;
	}

	if (i == 0) 
		printk(KERN_WARNING DRV_NAME 
		       ": %s: Could now power down adapter.\n", 
		       priv->ndev->name);

	/* assert s/w reset */
	write_register(priv->ndev, IPW_REG_RESET_REG,
		       IPW_AUX_HOST_RESET_REG_SW_RESET);

	priv->stopped = 1;

	return 0;
}


static int ipw2100_disable_adapter(struct ipw2100_priv *priv)
{
	struct host_command cmd = {
		.host_command = CARD_DISABLE,
		.host_command_sequence = 0,
		.host_command_length = 0
	};
	int err;

	IPW_DEBUG_FW_COMMAND("CARD_DISABLE\n");

	if (priv->disabled) 
		return 0;

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) {
		IPW_DEBUG_WARNING("exit - failed to send CARD_DISABLE command\n");
		return err;
	}

	err = ipw2100_wait_for_card_state(priv, STATE_DISABLED);
	if (err) {
		IPW_DEBUG_WARNING("exit - card failed to change to DISABLED\n");
		return err;
	}

	IPW_DEBUG_INFO("TODO: implement scan state machine\n");


	return 0;
}

static int ipw2100_up(struct ipw2100_priv *priv, int deferred)
{
	unsigned long flags;
	int rc = 0;
	u32 lock;
	u32 ord_len = sizeof(lock);

	/* If the interrupt is enabled, turn it off... */
	spin_lock_irqsave(&priv->low_lock, flags);
	ipw2100_hw_disable_interrupt(priv);
	
	/* Reset any fatal_error conditions */
	ipw2100_reset_fatalerror(priv);
	spin_unlock_irqrestore(&priv->low_lock, flags);

	if (priv->been_powered || priv->reset_pending) {
		/* Power cycle the card ... */
		if (ipw2100_power_cycle_adapter(priv)) {
			printk(KERN_WARNING "%s: Could not cycle adapter.\n",
			       priv->ndev->name);
			rc = 1;
			goto exit;
		}
	} else 
		priv->been_powered = 1;
	
	/* Load the firmeware, start the clocks, etc. */
	if (ipw2100_start_adapter(priv)) {
	       	printk(KERN_ERR "%s: Failed to start the firmware.\n",
		       priv->ndev->name);
		rc = 1;
		goto exit;
	}

	ipw2100_initialize_ordinals(priv);
	
	/* Determine capabilities of this particular HW configuration */
	if (ipw2100_get_hw_features(priv)) {
		printk(KERN_ERR "%s: Failed to determine HW features.\n",
		       priv->ndev->name);
		rc = 1;
		goto exit;
	}
	
	lock = LOCK_NONE;
	if (ipw2100_set_ordinal(priv, IPW_ORD_PERS_DB_LOCK, &lock, &ord_len)) {
		printk(KERN_ERR "%s: Failed to clear ordinal lock.\n",
		       priv->ndev->name);
		rc = 1;
		goto exit;
	}

	priv->in_scan = 0;

	if (priv->hw_features & HW_FEATURE_RFKILL)
		priv->rf_kill = ipw2100_get_rf_switch(priv);
	else
		priv->rf_kill = 0;

	if (priv->rf_kill || priv->manual_disable) {
		if (priv->rf_kill)
			printk(KERN_INFO "%s: Radio is disabled by RF switch\n",
			       priv->ndev->name);
		if (priv->manual_disable)
			printk(KERN_INFO "%s: Radio is disabled by Manual Disable switch\n",
			       priv->ndev->name);

		if (priv->stop_rf_check) {
			priv->stop_rf_check = 0;
			priv->rf_switch_timer.expires = jiffies + HZ;
			add_timer(&priv->rf_switch_timer);
		}

		/* Since the RF switch is off, we will defer
		 * the sending of the HOST_COMPLETE */
		deferred = 1;
	}
	
	/* Turn on the interrupt so that commands can be processed */
	ipw2100_hw_enable_interrupt(priv);
	
	/* Send all of the commands that must be sent prior to 
	 * HOST_COMPLETE */
	if (ipw2100_adapter_setup(priv)) {
		printk(KERN_ERR "%s: Failed to start the card.\n", 
		       priv->ndev->name);
		rc = 1;
		goto exit;
	}

	if (!deferred) {
		/* Enable the adapter - sends HOST_COMPLETE */
		if (ipw2100_enable_adapter(priv)) {
			printk(KERN_ERR 
			       "%s: failed in call to enable adapter.\n",
			       priv->ndev->name);
			rc = 1;
			goto exit;
		}
		
		/* Start a scan . . . */
		priv->wx_ap_event_pending = 1;
		queue_work(priv->workqueue, &priv->wx_event_work);
		
		ipw2100_set_scan_options(priv);
		ipw2100_start_scan(priv);
	}

 exit:
	return rc;
}

/* Called by register_netdev() */
static int ipw2100_net_init(struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	return ipw2100_up(priv, 1);
}



static void ipw2100_down(struct ipw2100_priv *priv)
{
	unsigned long flags;

	/* Kill the RF switch timer */
	if (!priv->stop_rf_check) {
		priv->stop_rf_check = 1;
		del_timer_sync(&priv->rf_switch_timer);
	}
	
	/* Kill the firmare hang check timer */
	if (!priv->stop_hang_check) {
		priv->stop_hang_check = 1;
		del_timer_sync(&priv->hang_check_timer);
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	/* Kill any pending resets */
	if (priv->reset_pending == 2) /* 1 == instant reset, 2 == scheduled */
		cancel_delayed_work(&priv->reset_work);
#endif
	/* On 2.4 schedule_delayed_work isn't used for the backoff -- instead
	 * the work itself just goes to sleep for the backoff period.  */

	/* Make sure the interrupt is on so that FW commands will be 
	 * processed correctly */
	spin_lock_irqsave(&priv->low_lock, flags);
	ipw2100_hw_enable_interrupt(priv);
	spin_unlock_irqrestore(&priv->low_lock, flags);
	
	if (ipw2100_hw_stop_adapter(priv)) 
		printk(KERN_ERR "%s: Error stopping adapter.\n", 
		       priv->ndev->name);
	
	/* Do not disable the interrupt until _after_ we disable
	 * the adaptor.  Otherwise the CARD_DISABLE command will never
	 * be ack'd by the firmware */
	spin_lock_irqsave(&priv->low_lock, flags);
	ipw2100_hw_disable_interrupt(priv);
	spin_unlock_irqrestore(&priv->low_lock, flags);
}

void ipw2100_reset_adapter(struct ipw2100_priv *priv)
{
	unsigned long flags;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	unsigned long delay;
#endif
	spin_lock_irqsave(&priv->low_lock, flags);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	if (priv->reset_backoff) {
		delay = priv->reset_backoff * HZ;
		spin_unlock_irqrestore(&priv->low_lock, flags);

		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(delay);

		spin_lock_irqsave(&priv->low_lock, flags);
	}
#endif
	printk(KERN_INFO DRV_NAME ": %s: Restarting adapter.\n", 
	       priv->ndev->name);
	priv->resets++;
	priv->connected = 0;
	priv->security_updated = 1;
	
	/* Force a power cycle even if interface hasn't been opened
	 * yet */
	priv->reset_pending = 1;
	spin_unlock_irqrestore(&priv->low_lock, flags);

	ipw2100_up(priv, 0);

}


static void isr_indicate_associated(struct ipw2100_priv *priv, u32 status)
{

#define MAC_ASSOCIATION_READ_DELAY (HZ)
	int ret, len, ssid_len;
	char essid[IW_ESSID_MAX_SIZE + 1];
	u32 txrate;
	u32 chan;
	char *txratename;
 	u8 bssid[ETH_ALEN]; 	

	IPW_DEBUG_INFO("enter\n");

	/*
	 * TBD: BSSID is usually 00:00:00:00:00:00 here and not
	 *      an actual MAC of the AP. Seems like FW sets this
	 *      address too late. Read it later and expose through 
	 *      /proc or schedule a later task to query and update
	 */

	memset(essid, 0, IW_ESSID_MAX_SIZE + 1);

	ssid_len = IW_ESSID_MAX_SIZE;
	ret = ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_SSID, 
				  essid, &ssid_len);
	if (ret) {
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);
		return;
	}

	len = sizeof(u32);
	ret = ipw2100_get_ordinal(priv, IPW_ORD_CURRENT_TX_RATE, 
				  &txrate, &len);
	if (ret) {
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);
		return;
	}

	len = sizeof(u32);
	ret = ipw2100_get_ordinal(priv, IPW_ORD_OUR_FREQ, &chan, &len);
	if (ret) {
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);
		return;
	}
	len = ETH_ALEN;
        ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID, &bssid,  &len);
	if (ret) {
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);
		return;
	}
	memcpy(priv->ieee->bssid, bssid, ETH_ALEN);


	switch (txrate) {
	case TX_RATE_1_MBIT:
		txratename = "1Mbps";
		break;
	case TX_RATE_2_MBIT:
		txratename = "2Mbsp";
		break;
	case TX_RATE_5_5_MBIT:
		txratename = "5.5Mbps";
		break;
	case TX_RATE_11_MBIT:
		txratename = "11Mbps";
		break;
	default:
		txratename = "unknown rate";
		break;
	}

	printk(KERN_INFO "%s: Associated with '%s' at %s, channel %d (BSSID=" MACSTR ")\n",
	       priv->ndev->name, essid, txratename, chan, MAC2STR(bssid));

	/* now we copy read ssid into dev */
	memcpy(priv->ctx->ssid, essid, ssid_len);
	priv->ctx->ssid_len = ssid_len;
	priv->ctx->channel = chan;
	memcpy(priv->ctx->bssid, bssid, ETH_ALEN);

	priv->connected = 1;
	priv->connect_start = get_seconds();

	netif_carrier_on(priv->ndev);
	if (netif_queue_stopped(priv->ndev)) {
		IPW_DEBUG_INFO(KERN_INFO "%s: Waking net queue.\n",
				   priv->ndev->name);
		netif_wake_queue(priv->ndev);
	} else {
		IPW_DEBUG_INFO(KERN_INFO "%s: Starting net queue.\n",
				   priv->ndev->name);
		netif_start_queue(priv->ndev);
	}

	if (!priv->wx_ap_event_pending) {
		priv->wx_ap_event_pending = 1;
		queue_work(priv->workqueue, &priv->wx_event_work);
	}
}

static void isr_indicate_association_lost(struct ipw2100_priv *priv, u32 status)
{
#if 0
	struct host_command cmd = {
		.host_command = BROADCAST_SCAN,
		.host_command_sequence = 0,
		.host_command_length = 4,
	};
#endif

	printk(KERN_DEBUG "%s: Association lost.\n", priv->ndev->name);

	IPW_DEBUG_INFO("TODO: only scan if scanning enabled and radio is on\n");

	IPW_DEBUG_INFO("TODO: Move broadcast_scan initiation to scheduled work\n");

#if 0
	ipw2100_hw_send_command(priv, &cmd);
#endif
	memset(priv->ctx->bssid, 0, ETH_ALEN);
	memset(priv->ieee->bssid, 0, ETH_ALEN);
	priv->connected = 0;
	netif_carrier_off(priv->ndev);
	netif_stop_queue(priv->ndev);
	
	if (priv->security_updated)
		queue_work(priv->workqueue, &priv->security_work);
	
	priv->wx_ap_event_pending = 1;
	queue_work(priv->workqueue, &priv->wx_event_work);
}

static void isr_indicate_rf_kill(struct ipw2100_priv *priv, u32 status)
{
	printk(KERN_INFO "%s: RF Kill state changed to radio OFF.\n",
	       priv->ndev->name);

	/* RF_KILL is now enabled (else we wouldn't be here) */
	priv->rf_kill = 1;

	/* If not already running, we now fire up a timer that will poll
	 * the state of the RF switch on the hardware so we can re-enable
	 * the firmware if the switch is enabled */
	if (priv->stop_rf_check) {
		priv->stop_rf_check = 0;
		priv->rf_switch_timer.expires = jiffies + HZ;
		add_timer(&priv->rf_switch_timer);
	}
}

static void isr_scan_complete(struct ipw2100_priv *priv, u32 status)
{
	IPW_DEBUG_SCAN("scan complete\n");
	/* Age the scan results... */
	priv->ieee->scans++;
	priv->in_scan = 0;
}

#ifdef CONFIG_IPW_DEBUG
#define IPW2100_HANDLER(v, f) { v, f, # v }
struct ipw2100_status_indicator {
	int status;
	void (*cb)(struct ipw2100_priv *priv, u32 status);
	char *name;
};
#else
#define IPW2100_HANDLER(v, f) { v, f }
struct ipw2100_status_indicator {
	int status;
	void (*cb)(struct ipw2100_priv *priv, u32 status);
};
#endif /* CONFIG_IPW_DEBUG */

const struct ipw2100_status_indicator status_handlers[] = {
	IPW2100_HANDLER(STATUS_INIT_COMPLETE, 0),
	IPW2100_HANDLER(STATUS_COUNTRY_FOUND, 0),
	IPW2100_HANDLER(STATUS_ASSOCIATED, isr_indicate_associated),
	IPW2100_HANDLER(STATUS_ASSN_LOST, isr_indicate_association_lost),
	IPW2100_HANDLER(STATUS_ASSN_CHANGED, 0),
	IPW2100_HANDLER(STATUS_SCAN_COMPLETE, isr_scan_complete),
	IPW2100_HANDLER(STATUS_ENTERED_PSP, 0),
	IPW2100_HANDLER(STATUS_WAKEN_FROM_PSP, 0),
	IPW2100_HANDLER(STATUS_RF_KILL_SWITCH, isr_indicate_rf_kill),
	IPW2100_HANDLER(STATUS_CARD_DISABLE_COMPLETE, 0),
	IPW2100_HANDLER(STATUS_PREP_PWR_DOWN_COMPLETE, 0),
	IPW2100_HANDLER(STATUS_SCANNING, 0),
	IPW2100_HANDLER(-1, 0)
};

static void isr_status_change(struct ipw2100_priv *priv, int status)
{
	int i;

	for (i = 0; status_handlers[i].status != -1; i++) {
		if (status == status_handlers[i].status) {
			IPW_DEBUG_STATUS("Status change: %s\n", 
					     status_handlers[i].name);
			if (status_handlers[i].cb)
				status_handlers[i].cb(priv, status);
			priv->wstats.status = status;
			return;
		}
	}

	IPW_DEBUG_STATUS("unknown status received: %04x\n", status);
}

static void isr_rx_complete_command(
	struct ipw2100_priv *priv,
	struct ipw2100_cmd_header *cmd)
{
#ifdef CONFIG_IPW_DEBUG
	if (cmd->host_command_reg < sizeof(command_types) / 
	    sizeof(*command_types)) {
		IPW_DEBUG_RX("Command '%s (%d)' completed\n",
				 command_types[cmd->host_command_reg],
				 cmd->host_command_reg);
	}
#endif
	if (cmd->host_command_reg == HOST_COMPLETE) {
		priv->disabled = 0;
	}
	if (cmd->host_command_reg == CARD_DISABLE) {
		priv->disabled = 1;
	}
	if (cmd->host_command_reg == CARD_DISABLE_PHY_OFF) {
		priv->phy_off = 1;
	}

	priv->message_sending = 0;

	wake_up_interruptible(&priv->wait_command_queue);

#ifdef CONFIG_IPW_DEBUG
	IPW_DEBUG_INFO("received command status:\n");
	IPW_DEBUG_INFO("    CMD    = %d\n", cmd->host_command_reg);
	IPW_DEBUG_INFO("    CMD1   = %08x\n", cmd->host_command_reg1);
	IPW_DEBUG_INFO("    SEQ    = %d\n", cmd->sequence);
	IPW_DEBUG_INFO("    STATUS = %d\n", cmd->cmd_status_reg);
#endif

}

#ifdef CONFIG_IPW_DEBUG
const char *frame_types[] = {
	"COMMAND_STATUS_VAL",
	"STATUS_CHANGE_VAL",
	"P80211_DATA_VAL",
	"P8023_DATA_VAL",
	"HOST_NOTIFICATION_VAL"
};
#endif


static inline int ipw2100_alloc_skb(
	struct ipw2100_priv *priv,
	struct ipw2100_rx_packet *packet)
{
	packet->skb = dev_alloc_skb(sizeof(struct ipw2100_rx));
	if (!packet->skb)
		return -ENOMEM;

	packet->rxp = (struct ipw2100_rx *)packet->skb->data;
	packet->dma_addr = pci_map_single(priv->pdev, packet->skb->data,
					  sizeof(struct ipw2100_rx),
					  PCI_DMA_FROMDEVICE);
	/* NOTE: pci_map_single does not return an error code, and 0 is a valid
	 *       dma_addr */
	
	return 0;
}

#define MAX_RESET_BACKOFF 10

static inline void schedule_reset(struct ipw2100_priv *priv) 
{
	unsigned long now = get_seconds();

	/* If we haven't received a reset request within the backoff period,
	 * then we can reset the backoff interval so this reset occurs
	 * immediately */
	if (priv->reset_backoff && 
	    (now - priv->last_reset > priv->reset_backoff))
		priv->reset_backoff = 0;

	priv->last_reset = now;

	priv->last_reset = get_seconds();
	if (!priv->reset_pending) {
		printk(KERN_INFO "%s: Scheduling firmware restart (%ds).\n",
		       priv->ndev->name, priv->reset_backoff);
		netif_carrier_off(priv->ndev);
		netif_stop_queue(priv->ndev);
		if (0 == priv->reset_backoff) {
			priv->reset_backoff = 1;
			priv->reset_pending = 1;
			queue_work(priv->workqueue, &priv->reset_work);
		} else {
			priv->reset_pending = 2;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
			queue_delayed_work(priv->workqueue, &priv->reset_work, 
					   priv->reset_backoff * HZ);
#else
			queue_work(priv->workqueue, &priv->reset_work);
#endif
			if (priv->reset_backoff < MAX_RESET_BACKOFF)
				priv->reset_backoff++;
		}
		wake_up_interruptible(&priv->wait_command_queue);
	} else
		printk(KERN_INFO 
		       "%s: Firmware restart already in progress.\n",
		       priv->ndev->name);
}

#define SEARCH_ERROR   0xffffffff
#define SEARCH_FAIL    0xfffffffe
#define SEARCH_SUCCESS 0xfffffff0
#define SEARCH_DISCARD 0
#define SEARCH_SNAPSHOT 1

#define SNAPSHOT_ADDR(ofs) (priv->snapshot[((ofs) >> 12) & 0xff] + ((ofs) & 0xfff))
static inline int ipw2100_snapshot_alloc(struct ipw2100_priv *priv) 
{
	int i;
	if (priv->snapshot[0])
		return 1;
	for (i = 0; i < 0x30; i++) {
		priv->snapshot[i] = (u8*)kmalloc(0x1000, GFP_ATOMIC);
		if (!priv->snapshot[i]) {
			printk(KERN_DEBUG "%s: Error allocating snapshot "
			       "buffer %d\n", priv->ndev->name, i);
			while (i > 0)
				kfree(priv->snapshot[--i]);
			priv->snapshot[0] = NULL;
			return 0;
		}
	}

	return 1;
}

static inline void ipw2100_snapshot_free(struct ipw2100_priv *priv) 
{
	int i;
	if (!priv->snapshot[0])
		return;
	for (i = 0; i < 0x30; i++) 
		kfree(priv->snapshot[i]);
	priv->snapshot[0] = NULL;
}

static inline u32 ipw2100_match_buf(struct ipw2100_priv *priv, u8 *in_buf, 
				    size_t len, int mode)
{
	u32 i, j;
	u32 tmp;
	u8 *s, *d;
	u32 ret;

	s = in_buf;
	if (mode == SEARCH_SNAPSHOT) {
		if (!ipw2100_snapshot_alloc(priv))
			mode = SEARCH_DISCARD;
	}

	for (ret = SEARCH_FAIL, i = 0; i < 0x30000; i += 4) {
		read_nic_dword(priv->ndev, i, &tmp);
		if (mode == SEARCH_SNAPSHOT)
			*(u32 *)SNAPSHOT_ADDR(i) = tmp;
		if (ret == SEARCH_FAIL) {
			d = (u8*)&tmp;
			for (j = 0; j < 4; j++) {
				if (*s != *d) {
					s = in_buf;
					continue;
				}
				
				s++;
				d++;
				
				if ((s - in_buf) == len)
					ret = (i + j) - len + 1;
			}
		} else if (mode == SEARCH_DISCARD)
			return ret;
	}

	return ret;
}

/*
 *
 * 0) Disconnect the SKB from the firmware (just unmap)
 * 1) Pack the ETH header into the SKB
 * 2) Pass the SKB to the network stack
 *
 * When packet is provided by the firmware, it contains the following:
 * 
 * .  ieee80211_hdr
 * .  ieee80211_snap_hdr
 *
 * The size of the constructed ethernet 
 *
 */
#ifdef CONFIG_IPW2100_RX_DEBUG
u8 packet_data[IPW_RX_NIC_BUFFER_LENGTH];
#endif

static inline void ipw2100_corruption_detected(struct ipw2100_priv *priv,
					       int i)
{
	struct ipw2100_status *status = &priv->status_queue.drv[i];
#ifdef CONFIG_IPW_DEBUG_C3
	u32 match, reg;
	int j;
#endif

	printk(KERN_INFO "Corruption detected at 0x%04X:\n",
	       i * sizeof(struct ipw2100_status));
	printk_buf((u8*)status, sizeof(struct ipw2100_status));

#ifdef CONFIG_IPW_DEBUG_C3
	/* Halt the fimrware so we can get a good image */
	write_register(priv->ndev, IPW_REG_RESET_REG, 
		       IPW_AUX_HOST_RESET_REG_STOP_MASTER);    
	j = 5;
	do {
		udelay(IPW_WAIT_RESET_MASTER_ASSERT_COMPLETE_DELAY);
		read_register(priv->ndev, IPW_REG_RESET_REG, &reg);
		
		if (reg & IPW_AUX_HOST_RESET_REG_MASTER_DISABLED)
			break;
	}  while (j--);

	match = ipw2100_match_buf(priv, (u8*)status, 
				  sizeof(struct ipw2100_status),
				  SEARCH_SNAPSHOT);
	if (match < SEARCH_SUCCESS) 
		printk(KERN_INFO "%s: DMA status match in Firmware at "
		       "offset 0x%06X, length %d:\n",
		       priv->ndev->name, match, 
		       sizeof(struct ipw2100_status));
	else 
		printk(KERN_INFO "%s: No DMA status match in "
		       "Firmware.\n", priv->ndev->name);

	printk_buf((u8*)priv->status_queue.drv, 
		   sizeof(struct ipw2100_status) * RX_QUEUE_LENGTH);
#endif

	priv->fatal_error = IPW2100_ERR_C3_CORRUPTION;
	priv->ieee->stats.rx_errors++;
	schedule_reset(priv);
}
	
static inline void isr_rx(struct ipw2100_priv *priv, int i,
			  struct ieee80211_rx_stats *stats)
{
	struct ipw2100_status *status = &priv->status_queue.drv[i];
	struct ipw2100_rx_packet *packet = &priv->rx_buffers[i];

	IPW_DEBUG_RX("Handler...\n");

	if (unlikely(status->frame_size > skb_tailroom(packet->skb))) {
		printk(KERN_INFO "%s: frame_size (%u) > skb_tailroom (%u)!"
		       "  Dropping.\n",
		       priv->ndev->name,
		       status->frame_size, skb_tailroom(packet->skb));
		priv->ieee->stats.rx_errors++;
		return;
	}

	if (unlikely(priv->ieee->iw_mode == IW_MODE_MONITOR && 
		status->flags & IPW_STATUS_FLAG_CRC_ERROR)) {
		IPW_DEBUG_RX("CRC error in packet.  Dropping.\n");
		priv->ieee->stats.rx_errors++;
		return;
	}

	pci_unmap_single(priv->pdev,
			 packet->dma_addr,
			 sizeof(struct ipw2100_rx),
			 PCI_DMA_FROMDEVICE);

	skb_put(packet->skb, status->frame_size);

#ifdef CONFIG_IPW2100_RX_DEBUG
	/* Make a copy of the frame so we can dump it to the logs if 
	 * ieee80211_rx fails */
	memcpy(packet_data, packet->skb->data, 
	       min(status->frame_size, IPW_RX_NIC_BUFFER_LENGTH));
#endif
	
	if (!ieee80211_rx(priv->ieee, packet->skb, stats)) {
#ifdef CONFIG_IPW2100_RX_DEBUG
		printk(KERN_DEBUG "%s: Non consumed packet:\n",
		       priv->ndev->name);
		printk_buf(packet_data, status->frame_size);
#endif
		priv->ieee->stats.rx_errors++;

		/* ieee80211_rx failed, so it didn't free the SKB */
		dev_kfree_skb_any(packet->skb);
		packet->skb = NULL;
	}
	
	/* We need to allocate a new SKB and attach it to the RDB. */
	if (unlikely(ipw2100_alloc_skb(priv, packet))) {
		printk(KERN_WARNING 
		       "%s: Unable to allocate SKB onto RBD ring - disabling "
		       "adapter.\n", priv->ndev->name);
		/* TODO: schedule adapter shutdown */
		IPW_DEBUG_INFO("TODO: Shutdown adapter...\n");
	}

	/* Update the RDB entry */
	priv->rx_queue.drv[i].host_addr = packet->dma_addr;
}

static inline int ipw2100_corruption_check(struct ipw2100_priv *priv, int i)
{
	struct ipw2100_status *status = &priv->status_queue.drv[i];
	struct ipw2100_rx *u = priv->rx_buffers[i].rxp;
	u16 frame_type = status->status_fields & STATUS_TYPE_MASK;

	switch (frame_type) {
	case COMMAND_STATUS_VAL:
		return (status->frame_size != sizeof(u->rx_data.command));
	case STATUS_CHANGE_VAL:
		return (status->frame_size != sizeof(u->rx_data.status));
	case HOST_NOTIFICATION_VAL:
		return (status->frame_size < sizeof(u->rx_data.notification));
	case P80211_DATA_VAL:
	case P8023_DATA_VAL:
#ifdef CONFIG_IPW2100_PROMISC
		return 0;
#else
		switch (WLAN_FC_GET_TYPE(u->rx_data.header.frame_ctl)) {
		case IEEE80211_FTYPE_MGMT:
		case IEEE80211_FTYPE_CTL:
			return 0;
		case IEEE80211_FTYPE_DATA:
			return (status->frame_size > 
				IPW_MAX_802_11_PAYLOAD_LENGTH);
		}
#endif
	}

	return 1;
}

/*
 * ipw2100 interrupts are disabled at this point, and the ISR
 * is the only code that calls this method.  So, we do not need
 * to play with any locks.  
 *
 * RX Queue works as follows:
 * 
 * Read index - firmware places packet in entry identified by the 
 *              Read index and advances Read index.  In this manner, 
 *              Read index will always point to the next packet to 
 *              be filled--but not yet valid.
 *
 * Write index - driver fills this entry with an unused RBD entry.
 *               This entry has not filled by the firmware yet.
 *
 * In between the W and R indexes are the RBDs that have been received
 * but not yet processed.  
 *
 * The process of handling packets will start at WRITE + 1 and advance
 * until it reaches the READ index.  
 *
 * The WRITE index is cached in the variable 'priv->rx_queue.next'.
 *
 */
static inline void __ipw2100_rx_process(struct ipw2100_priv *priv)
{
	struct ipw2100_bd_queue *rxq = &priv->rx_queue;
	struct ipw2100_status_queue *sq = &priv->status_queue;
	struct ipw2100_rx_packet *packet;
	u16 frame_type;
	u32 frame_size;
	u32 r, w, i, s;
	struct ipw2100_rx *u;
	struct ieee80211_rx_stats stats = {
		.noise = IPW2100_RSSI_TO_DBM,
		.rate = 0,
		.mac_time = jiffies,
	};
	
	read_register(priv->ndev, IPW_MEM_HOST_SHARED_RX_READ_INDEX, &r);
	read_register(priv->ndev, IPW_MEM_HOST_SHARED_RX_WRITE_INDEX, &w);
	
	if (r >= rxq->entries) {
		IPW_DEBUG_RX("exit - bad read index\n");
		return;
	}


	i = (rxq->next + 1) % rxq->entries;
	s = i;
	while (i != r) {
		/* IPW_DEBUG_RX("r = %d : w = %d : processing = %d\n",
		   r, rxq->next, i); */
		
		packet = &priv->rx_buffers[i];

		/* Sync the DMA for the STATUS buffer so CPU is sure to get 
		 * the correct values */
		pci_dma_sync_single_for_cpu(
			priv->pdev, 
			sq->nic + sizeof(struct ipw2100_status) * i,
			sizeof(struct ipw2100_status),
			PCI_DMA_FROMDEVICE);

		/* Sync the DMA for the RX buffer so CPU is sure to get 
		 * the correct values */
		pci_dma_sync_single_for_cpu(priv->pdev, packet->dma_addr,
					    sizeof(struct ipw2100_rx),
					    PCI_DMA_FROMDEVICE);
		
		if (unlikely(ipw2100_corruption_check(priv, i))) {
			ipw2100_corruption_detected(priv, i);
			goto increment;
		}

		u = packet->rxp;
		frame_type = sq->drv[i].status_fields & 
			STATUS_TYPE_MASK;
		frame_size = sq->drv[i].frame_size;
		stats.rssi = sq->drv[i].rssi;
		stats.signal = sq->drv[i].rssi;
		
		IPW_DEBUG_RX(
			"%s: '%s' frame type received (%d).\n",
			priv->ndev->name, frame_types[frame_type],
			frame_size);
		
		switch (frame_type) {
		case COMMAND_STATUS_VAL:
			isr_rx_complete_command(
				priv, &u->rx_data.command);
			break;
				
		case STATUS_CHANGE_VAL:
			isr_status_change(priv, u->rx_data.status);
			break;
				
		case P80211_DATA_VAL:
		case P8023_DATA_VAL:
#ifdef CONFIG_IPW2100_PROMISC
			if (priv->ieee->iw_mode == IW_MODE_MONITOR) {
				isr_rx(priv, i, &stats);
				break;
			}
#endif
			if (frame_size < sizeof(u->rx_data.header))
				break;
			switch (WLAN_FC_GET_TYPE(u->rx_data.header.
						 frame_ctl)) {
			case IEEE80211_FTYPE_MGMT:
				ieee80211_rx_mgt(priv->ieee, 
						 &u->rx_data.header,
						 frame_size, 
						 &stats);
				break;
					
			case IEEE80211_FTYPE_CTL:
				break;
					
			case IEEE80211_FTYPE_DATA:
				/* We only process data packets if the 
				 * interface is open */
				if (!priv->open) {
					priv->wstats.discard.misc++;
					IPW_DEBUG_DROP(
						"Dropping packet "
						"while interface is "
						"not up.\n");
					break;
				}
					
				isr_rx(priv, i, &stats);
				break;
					
			}
			break;
		}

	increment:
		/* clear status field associated with this RBD */
		rxq->drv[i].status.info.field = 0;
		
		i = (i + 1) % rxq->entries;
	}
	
	if (i != s) {
		/* backtrack one entry, wrapping to end if at 0 */
		rxq->next = (i ? i : rxq->entries) - 1;
	
		write_register(priv->ndev,
			       IPW_MEM_HOST_SHARED_RX_WRITE_INDEX,
			       rxq->next);
	}
}


/*
 * __ipw2100_tx_process
 *
 * This routine will determine whether the next packet on
 * the fw_pend_list has been processed by the firmware yet.
 *
 * If not, then it does nothing and returns.
 *
 * If so, then it removes the item from the fw_pend_list, frees
 * any associated storage, and places the item back on the
 * free list of its source (either msg_free_list or tx_free_list)
 *
 * TX Queue works as follows:
 * 
 * Read index - points to the next TBD that the firmware will 
 *              process.  The firmware will read the data, and once
 *              done processing, it will advance the Read index.
 *
 * Write index - driver fills this entry with an constructed TBD 
 *               entry.  The Write index is not advanced until the
 *               packet has been configured.
 *
 * In between the W and R indexes are the TBDs that have NOT been 
 * processed.  Lagging behind the R index are packets that have
 * been processed but have not been freed by the driver.
 *
 * In order to free old storage, an internal index will be maintained
 * that points to the next packet to be freed.  When all used
 * packets have been freed, the oldest index will be the same as the
 * firmware's read index.
 *
 * The OLDEST index is cached in the variable 'priv->tx_queue.oldest'
 *
 * Because the TBD structure can not contain arbitrary data, the
 * driver must keep an internal queue of cached allocations such that
 * it can put that data back into the tx_free_list and msg_free_list
 * for use by future command and data packets.
 *
 */
static inline int __ipw2100_tx_process(struct ipw2100_priv *priv)
{
	struct ipw2100_bd_queue *txq = &priv->tx_queue;
        struct ipw2100_bd *tbd;
	struct list_head *element;
	struct ipw2100_tx_packet *packet;
	int descriptors_used;
	int e, i;
	u32 r, w, frag_num = 0;

	if (list_empty(&priv->fw_pend_list)) 
		return 0;
	
	element = priv->fw_pend_list.next;

	packet = list_entry(element, struct ipw2100_tx_packet, list);
        tbd = &txq->drv[packet->index];

	/* Determine how many TBD entries must be finished... */
	switch (packet->type) {
	case COMMAND:
		/* COMMAND uses only one slot; don't advance */
		descriptors_used = 1;
		e = txq->oldest;
		break;

	case DATA:
		/* DATA uses two slots; advance and loop position. */
		descriptors_used = tbd->num_fragments;
                frag_num = tbd->num_fragments - 1;
		e = txq->oldest + frag_num;
		e %= txq->entries;
		break;

	default:
		IPW_DEBUG_WARNING("%s: Bad fw_pend_list entry!\n", 
				   priv->ndev->name);
		return 0;
	}

	/* if the last TBD is not done by NIC yet, then packet is
	 * not ready to be released. 
	 * 
	 */
	read_register(priv->ndev, IPW_MEM_HOST_SHARED_TX_QUEUE_READ_INDEX,
		      &r);
	read_register(priv->ndev, IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX,
		      &w);
	if (w != txq->next) 
		printk(KERN_WARNING "%s: write index mismatch\n", 
		       priv->ndev->name);

        /*
	 * txq->next is the index of the last packet written txq->oldest is 
	 * the index of the r is the index of the next packet to be read by 
	 * firmware
	 */


	/* 
	 * Quick graphic to help you visualize the following
	 * if / else statement
	 * 
	 * ===>|                     s---->|===============
	 *                               e>|
	 * | a | b | c | d | e | f | g | h | i | j | k | l 
	 *       r---->|
	 *               w
	 *
	 * w - updated by driver
	 * r - updated by firmware
	 * s - start of oldest BD entry (txq->oldest)
	 * e - end of oldest BD entry
	 *
	 */
	if (!((r <= w && (e < r || e >= w)) || (e < r && e >= w))) { 
		IPW_DEBUG_TX("exit - no processed packets ready to release.\n");
		return 0;
	}

	list_del(element);
	DEC_STAT(&priv->fw_pend_stat);

#ifdef CONFIG_IPW_DEBUG
	{
		int i = txq->oldest;
		IPW_DEBUG_TX(
			"TX%d V=%p P=%p T=%p L=%d\n", i,
			&txq->drv[i],
			(void*)txq->nic + i * sizeof(struct ipw2100_bd),
			(void*)txq->drv[i].host_addr,
			txq->drv[i].buf_length);
		
		if (packet->type == DATA) {
			i = (i + 1) % txq->entries;
			
			IPW_DEBUG_TX(
				"TX%d V=%p P=%p T=%p L=%d\n", i,
				&txq->drv[i],
				(void*)txq->nic + i * 
				sizeof(struct ipw2100_bd),
				(void*)txq->drv[i].host_addr,
				txq->drv[i].buf_length);
		}
	}
#endif
 
	switch (packet->type) {
	case DATA:
		if (txq->drv[txq->oldest].status.info.fields.txType != 0) 
			printk(KERN_WARNING "%s: Queue mismatch.  "
			       "Expecting DATA TBD but pulled "
			       "something else: ids %d=%d.\n", 
			       priv->ndev->name, txq->oldest, packet->index);
		
		/* DATA packet; we have to unmap and free the SKB */
		priv->ieee->stats.tx_packets++;
		for (i = 0; i < frag_num; i++) {
			tbd = &txq->drv[(packet->index + 1 + i) % 
					txq->entries];
			
			IPW_DEBUG_TX(
				"TX%d P=%08x L=%d\n",
				(packet->index + 1 + i) % txq->entries,
				tbd->host_addr, tbd->buf_length);

			pci_unmap_single(priv->pdev,
					 tbd->host_addr,
					 tbd->buf_length,
					 PCI_DMA_TODEVICE);
		}

		priv->ieee->stats.tx_bytes += packet->info.d_struct.txb->payload_size;
		ieee80211_txb_free(packet->info.d_struct.txb);
		packet->info.d_struct.txb = NULL;

		list_add_tail(element, &priv->tx_free_list);
		INC_STAT(&priv->tx_free_stat);

		/* We have a free slot in the Tx queue, so wake up the
		 * transmit layer if it is stopped. */
		if (priv->open &&
		    priv->connected && netif_queue_stopped(priv->ndev)) {
			IPW_DEBUG_INFO(KERN_INFO 
					   "%s: Waking net queue.\n",
					   priv->ndev->name);
			netif_wake_queue(priv->ndev);
		}

		/* A packet was processed by the hardware, so update the 
		 * watchdog */
		priv->ndev->trans_start = jiffies;

		break;

	case COMMAND:
		if (txq->drv[txq->oldest].status.info.fields.txType != 1) 
			printk(KERN_WARNING "%s: Queue mismatch.  "
			       "Expecting COMMAND TBD but pulled "
			       "something else: ids %d=%d.\n", 
			       priv->ndev->name, txq->oldest, packet->index);

#ifdef CONFIG_IPW_DEBUG
		if (packet->info.c_struct.cmd->host_command_reg <
		    sizeof(command_types) / sizeof(*command_types)) 
			IPW_DEBUG_TX(
				"Command '%s (%d)' processed: %d.\n",
				command_types[packet->info.c_struct.cmd->host_command_reg],
				packet->info.c_struct.cmd->host_command_reg,
				packet->info.c_struct.cmd->cmd_status_reg);
#endif

		list_add_tail(element, &priv->msg_free_list);
		INC_STAT(&priv->msg_free_stat);
		break;
	}
        
	/* advance oldest used TBD pointer to start of next entry */
	txq->oldest = (e + 1) % txq->entries;
	/* increase available TBDs number */
	txq->available += descriptors_used;
	SET_STAT(&priv->txq_stat, txq->available);

	IPW_DEBUG_TX("packet latency (send to process)  %ld jiffies\n",
			 jiffies - packet->jiffy_start);
	
	return (!list_empty(&priv->fw_pend_list));
}


static inline void __ipw2100_tx_complete(struct ipw2100_priv *priv)
{
	int i = 0;

	while (__ipw2100_tx_process(priv) && i < 200) i++;

	if (i == 200) {
		printk(KERN_WARNING 
		       "%s: Driver is running slow (%d iters).\n",
		       priv->ndev->name, i);
	}
}


static void X__ipw2100_tx_send_commands(struct ipw2100_priv *priv)
{
	struct list_head *element;
	struct ipw2100_tx_packet *packet;
	struct ipw2100_bd_queue *txq = &priv->tx_queue;
	struct ipw2100_bd *tbd;
	int next = txq->next;

	while (!list_empty(&priv->msg_pend_list)) {
		/* if there isn't enough space in TBD queue, then
		 * don't stuff a new one in. 
		 * NOTE: 3 are needed as a command will take one,
		 *       and there is a minimum of 2 that must be
		 *       maintained between the r and w indexes
		 */
		if (txq->available <= 3) {
			IPW_DEBUG_TX("no room in tx_queue\n");
			break;
		}

		element = priv->msg_pend_list.next;
		list_del(element);
		DEC_STAT(&priv->msg_pend_stat);

		packet = list_entry(element, 
				    struct ipw2100_tx_packet, list);
		
		IPW_DEBUG_TX("using TBD at virt=%p, phys=%p\n",
				 &txq->drv[txq->next],
				 (void*)(txq->nic + txq->next * 
					 sizeof(struct ipw2100_bd)));

		packet->index = txq->next;

		tbd = &txq->drv[txq->next];

		/* initialize TBD */
		tbd->host_addr = packet->info.c_struct.cmd_phys;
		tbd->buf_length = sizeof(struct ipw2100_cmd_header);
		/* not marking number of fragments causes problems 
		 * with f/w debug version */
		tbd->num_fragments = 1;
		tbd->status.info.field =
			IPW_BD_STATUS_TX_FRAME_COMMAND |
			IPW_BD_STATUS_TX_INTERRUPT_ENABLE;

		/* update TBD queue counters */
		txq->next++;
		txq->next %= txq->entries;
		txq->available--;
		DEC_STAT(&priv->txq_stat);

		list_add_tail(element, &priv->fw_pend_list);
		INC_STAT(&priv->fw_pend_stat);
	}

	if (txq->next != next) {
		/* kick off the DMA by notifying firmware the 
		 * write index has moved; make sure TBD stores are sync'd */
		wmb();
		write_register(priv->ndev, 
			       IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX,
			       txq->next);
	}
}


/*
 * X__ipw2100_tx_send_data
 *
 */
static void X__ipw2100_tx_send_data(struct ipw2100_priv *priv)
{
	struct list_head *element;
	struct ipw2100_tx_packet *packet;
	struct ipw2100_bd_queue *txq = &priv->tx_queue;
	struct ipw2100_bd *tbd;
	int next = txq->next;
        int i = 0;

	while (!list_empty(&priv->tx_pend_list)) {
		/* if there isn't enough space in TBD queue, then
		 * don't stuff a new one in. 
		 * NOTE: 4 are needed as a data will take two,
		 *       and there is a minimum of 2 that must be
		 *       maintained between the r and w indexes
		 */
		element = priv->tx_pend_list.next;
                packet = list_entry(element, struct ipw2100_tx_packet, list);

		if (unlikely(1 + packet->info.d_struct.txb->nr_frags > IPW_MAX_BDS)) {
			/* TODO: Support merging buffers if more than 
			 * IPW_MAX_BDS are used */
			printk(KERN_DEBUG 
			       "%s: Maximum BD theshold exceeded.  "
			       "Increase fragmentation level.\n",
			       priv->ndev->name);
		}

		if (txq->available <= 3 + packet->info.d_struct.txb->nr_frags) {
			IPW_DEBUG_TX("no room in tx_queue\n");
			break;
		}

		list_del(element);
		DEC_STAT(&priv->tx_pend_stat);

		tbd = &txq->drv[txq->next];

		packet->index = txq->next;

		tbd->host_addr = packet->info.d_struct.data_phys;
		tbd->buf_length = sizeof(struct ipw2100_data_header);
		tbd->num_fragments = 1 + packet->info.d_struct.txb->nr_frags;
		tbd->status.info.field =
			IPW_BD_STATUS_TX_FRAME_802_3 |
			IPW_BD_STATUS_TX_FRAME_NOT_LAST_FRAGMENT;
		txq->next++;
		txq->next %= txq->entries;

		IPW_DEBUG_TX(
			"data header tbd TX%d P=%08x L=%d\n", 
			packet->index, tbd->host_addr, 
			tbd->buf_length);
#ifdef CONFIG_IPW_DEBUG
		if (packet->info.d_struct.txb->nr_frags > 1) 
			IPW_DEBUG_FRAG("fragment Tx: %d frames\n", 
					   packet->info.d_struct.txb->nr_frags);
#endif

                for (i = 0; i < packet->info.d_struct.txb->nr_frags; i++) {
		        tbd = &txq->drv[txq->next];
			if (i == packet->info.d_struct.txb->nr_frags - 1) 
				tbd->status.info.field = 
					IPW_BD_STATUS_TX_FRAME_802_3 |
					IPW_BD_STATUS_TX_INTERRUPT_ENABLE;
			 else 
				tbd->status.info.field =
					IPW_BD_STATUS_TX_FRAME_802_3 |
					IPW_BD_STATUS_TX_FRAME_NOT_LAST_FRAGMENT;
				
			tbd->buf_length = packet->info.d_struct.txb->fragments[i]->len;
 
                        tbd->host_addr = pci_map_single(
				priv->pdev, 
				packet->info.d_struct.txb->fragments[i]->data,
				tbd->buf_length, 
				PCI_DMA_TODEVICE);

			IPW_DEBUG_TX(
				"data frag tbd TX%d P=%08x L=%d\n", 
				txq->next, tbd->host_addr, tbd->buf_length);

			pci_dma_sync_single_for_device(
				priv->pdev, tbd->host_addr,
				tbd->buf_length, 
				PCI_DMA_TODEVICE);

			txq->next++;
			txq->next %= txq->entries;
                }
		
		txq->available -= 1 + packet->info.d_struct.txb->nr_frags;
		SET_STAT(&priv->txq_stat, txq->available);

		list_add_tail(element, &priv->fw_pend_list);
		INC_STAT(&priv->fw_pend_stat);
	}

	if (txq->next != next) {
		/* kick off the DMA by notifying firmware the 
		 * write index has moved; make sure TBD stores are sync'd */
		write_register(priv->ndev, 
			       IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX,
			       txq->next);
	}
        return;
}

static void ipw2100_irq_tasklet(struct ipw2100_priv *priv)
{
	struct net_device *dev = priv->ndev;
	unsigned long flags;
	u32 inta, tmp;

	spin_lock_irqsave(&priv->low_lock, flags);
	ipw2100_hw_disable_interrupt(priv);

	read_register(dev, IPW_REG_INTA, &inta);
			 
	IPW_DEBUG_ISR("enter - INTA: 0x%08lX\n", 
			  (unsigned long)inta & IPW_INTERRUPT_MASK);

	priv->in_isr++;
	priv->interrupts++;

	/* We do not loop and keep polling for more interrupts as this
	 * is frowned upon and doesn't play nicely with other potentially
	 * chained IRQs */
	IPW_DEBUG_ISR("INTA: 0x%08lX\n", 
			  (unsigned long)inta & IPW_INTERRUPT_MASK);

	if (inta & IPW2100_INTA_FATAL_ERROR) {
		printk(KERN_WARNING DRV_NAME 
		       ": Fatal interrupt. Scheduling firmware restart.\n");
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_FATAL_ERROR);

		read_nic_dword(dev, IPW_NIC_FATAL_ERROR, &priv->fatal_error);
		printk(KERN_INFO "%s: Fatal error value: 0x%08X\n",
		       priv->ndev->name, priv->fatal_error);

		read_nic_dword(dev, IPW_ERROR_ADDR(priv->fatal_error), &tmp);
		printk(KERN_INFO "%s: Fatal error address value: 0x%08X\n",
		       priv->ndev->name, tmp);
		
		/* Wake up any sleeping jobs */
		schedule_reset(priv);
	}
	
	if (inta & IPW2100_INTA_PARITY_ERROR) {
		IPW_DEBUG_ERROR("***** PARITY ERROR INTERRUPT !!!! \n");
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA, 
			IPW2100_INTA_PARITY_ERROR);
	}

	if (inta & IPW2100_INTA_RX_TRANSFER) {
		IPW_DEBUG_ISR("RX interrupt\n");

		priv->rx_interrupts++;

		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_RX_TRANSFER);
		
		__ipw2100_rx_process(priv);
		__ipw2100_tx_complete(priv);
	}
	
	if (inta & IPW2100_INTA_TX_TRANSFER) {
		IPW_DEBUG_ISR("TX interrupt\n");

		priv->tx_interrupts++;
		
		write_register(dev, IPW_REG_INTA,
			       IPW2100_INTA_TX_TRANSFER);
		
		__ipw2100_tx_complete(priv);
		X__ipw2100_tx_send_commands(priv);
		X__ipw2100_tx_send_data(priv);
	}
	
	if (inta & IPW2100_INTA_TX_COMPLETE) {
		IPW_DEBUG_ISR("TX complete\n");
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_TX_COMPLETE);
		
		__ipw2100_tx_complete(priv);
	}
	
	if (inta & IPW2100_INTA_EVENT_INTERRUPT) {
		/* ipw2100_handle_event(dev); */
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_EVENT_INTERRUPT);
	}
	
	if (inta & IPW2100_INTA_FW_INIT_DONE) {
		IPW_DEBUG_ISR("FW init done interrupt\n");
		priv->inta_other++;
		
		read_register(dev, IPW_REG_INTA, &tmp);
		if (tmp & (IPW2100_INTA_FATAL_ERROR |
			   IPW2100_INTA_PARITY_ERROR)) {
			write_register(
				dev, IPW_REG_INTA,
				IPW2100_INTA_FATAL_ERROR |
				IPW2100_INTA_PARITY_ERROR);
		}
		
		write_register(dev, IPW_REG_INTA,
			       IPW2100_INTA_FW_INIT_DONE);
	}
	
	if (inta & IPW2100_INTA_STATUS_CHANGE) {
		IPW_DEBUG_ISR("Status change interrupt\n");
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_STATUS_CHANGE);
	}
	
	if (inta & IPW2100_INTA_SLAVE_MODE_HOST_COMMAND_DONE) {
		IPW_DEBUG_ISR("slave host mode interrupt\n");
		priv->inta_other++;
		write_register(
			dev, IPW_REG_INTA,
			IPW2100_INTA_SLAVE_MODE_HOST_COMMAND_DONE);
	}
	
	priv->in_isr--;
	ipw2100_hw_enable_interrupt(priv);

	spin_unlock_irqrestore(&priv->low_lock, flags);

	IPW_DEBUG_ISR("exit\n");
}


static irqreturn_t ipw2100_interrupt(int irq, void *data, 
				     struct pt_regs *regs)
{
	struct ipw2100_priv *priv;
	u32 inta, inta_mask;
	unsigned long flags;

	if (!data)
		return IRQ_NONE;

	priv = (struct ipw2100_priv *)data;
 	spin_lock_irqsave(&priv->low_lock, flags);

	/* We check to see if we should be ignoring interrupts before
	 * we touch the hardware.  During ucode load if we try and handle
	 * an interrupt we can cause keyboard problems as well as cause
	 * the ucode to fail to initialize */
	if (priv->irq_ignore) {
		spin_unlock_irqrestore(&priv->low_lock, flags);
		return IRQ_NONE;
	}

	read_register(priv->ndev, IPW_REG_INTA_MASK, &inta_mask);
	read_register(priv->ndev, IPW_REG_INTA, &inta);

	if (inta == 0xFFFFFFFF) {
		/* Hardware disappeared */
		IPW_DEBUG_WARNING("IRQ INTA == 0xFFFFFFFF\n");
		spin_unlock_irqrestore(&priv->low_lock, flags);
		return IRQ_HANDLED;
	}

	inta &= IPW_INTERRUPT_MASK;

	if (!(inta & inta_mask)) {
		/* Shared interrupt */
		spin_unlock_irqrestore(&priv->low_lock, flags);
		return IRQ_NONE;
	}

	/* We disable the hardware interrupt here just to prevent unneeded
	 * calls to be made.  We disable this again within the actual
	 * work tasklet, so if another part of the code re-enables the
	 * interrupt, that is fine */
	ipw2100_hw_disable_interrupt(priv);

	tasklet_schedule(&priv->irq_tasklet);
 	spin_unlock_irqrestore(&priv->low_lock, flags);

	return IRQ_HANDLED;
}

static int ipw2100_tx(struct sk_buff *skb, struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct list_head *element;
	struct ipw2100_data_header *header = NULL;
	struct ipw2100_tx_packet *packet;
	unsigned long flags;
	
	spin_lock_irqsave(&priv->low_lock, flags);

	if (!priv->connected) { 
		/* TODO: check code to ensure that xmit disabled during
		 * any call that results in priv->connected == false */
		IPW_DEBUG_INFO("Can not transmit when not connected.\n");
		priv->ieee->stats.tx_carrier_errors++;
		netif_stop_queue(dev);
		goto fail_unlock;
	}

	if (list_empty(&priv->tx_free_list))
		goto fail_unlock;

	element = priv->tx_free_list.next;
	packet = list_entry(element, struct ipw2100_tx_packet, list);

	header = packet->info.d_struct.data;
	memcpy(header->dst_addr, skb->data, ETH_ALEN);
	memcpy(header->src_addr, skb->data + ETH_ALEN, ETH_ALEN);

	packet->info.d_struct.txb = ieee80211_skb_to_txb(priv->ieee, skb);
	if (packet->info.d_struct.txb == NULL) {
		IPW_DEBUG_DROP("Failed to Tx packet\n");
		priv->ieee->stats.tx_errors++;
		goto fail_unlock;
	} 

	header->host_command_reg = SEND;
	header->host_command_reg1 = 0;

	/* For now we only support host based encryption */
	header->needs_encryption = 0;
	header->encrypted = packet->info.d_struct.txb->encrypted;
	if (packet->info.d_struct.txb->nr_frags > 1)
		header->fragment_size = packet->info.d_struct.txb->frag_size;
	else
		header->fragment_size = 0;

	packet->jiffy_start = jiffies;

	list_del(element);
	DEC_STAT(&priv->tx_free_stat);

	list_add_tail(element, &priv->tx_pend_list);
	INC_STAT(&priv->tx_pend_stat);
	
	X__ipw2100_tx_send_data(priv);
	
	spin_unlock_irqrestore(&priv->low_lock, flags);
	IPW_DEBUG_INFO("exit\n");
	return 0;

 fail_unlock:
	netif_stop_queue(dev);
	IPW_DEBUG_INFO("exit - failed\n");
	spin_unlock_irqrestore(&priv->low_lock, flags);
	return 1;
}


static int ipw2100_msg_allocate(struct ipw2100_priv *priv)
{
	int i, j, err = -EINVAL;
	void *v;
	dma_addr_t p;
	     
	priv->msg_buffers = (struct ipw2100_tx_packet *)kmalloc(
		IPW_COMMAND_POOL_SIZE * sizeof(struct ipw2100_tx_packet), 
		GFP_KERNEL);
	if (!priv->msg_buffers) {
		printk(KERN_ERR "%s: PCI alloc failed for msg "
		       "buffers.\n", priv->ndev->name);
		return -ENOMEM;
	}

	for (i = 0; i < IPW_COMMAND_POOL_SIZE; i++) {
		v = pci_alloc_consistent(
			priv->pdev,
			sizeof(struct ipw2100_cmd_header), 
			&p);
		if (!v) {
			printk(KERN_ERR 
			       "%s: PCI alloc failed for msg "
			       "buffers.\n",
			       priv->ndev->name);
			err = -ENOMEM;
			break;
		} 

		memset(v, 0, sizeof(struct ipw2100_cmd_header));
		
		priv->msg_buffers[i].type = COMMAND;
		priv->msg_buffers[i].info.c_struct.cmd = 
			(struct ipw2100_cmd_header*)v;
		priv->msg_buffers[i].info.c_struct.cmd_phys = p;
	}

	if (i == IPW_COMMAND_POOL_SIZE)
		return 0;

	for (j = 0; j < i; j++) {
		pci_free_consistent(
			priv->pdev,
			sizeof(struct ipw2100_cmd_header), 
			priv->msg_buffers[j].info.c_struct.cmd,
			priv->msg_buffers[j].info.c_struct.cmd_phys);
	}
	
	kfree(priv->msg_buffers);
	priv->msg_buffers = NULL;

	return err;
}

static int ipw2100_msg_initialize(struct ipw2100_priv *priv)
{
	int i;

	INIT_LIST_HEAD(&priv->msg_free_list);
	INIT_LIST_HEAD(&priv->msg_pend_list);

	for (i = 0; i < IPW_COMMAND_POOL_SIZE; i++) 
		list_add_tail(&priv->msg_buffers[i].list, &priv->msg_free_list);
	SET_STAT(&priv->msg_free_stat, i);

	return 0;
}

static void ipw2100_msg_free(struct ipw2100_priv *priv)
{
	int i;

	if (!priv->msg_buffers) 
		return;

	for (i = 0; i < IPW_COMMAND_POOL_SIZE; i++) {
		pci_free_consistent(priv->pdev,
				    sizeof(struct ipw2100_cmd_header),
				    priv->msg_buffers[i].info.c_struct.cmd,
				    priv->msg_buffers[i].info.c_struct.cmd_phys);
	}
	
	kfree(priv->msg_buffers);
	priv->msg_buffers = NULL;
}

static ssize_t show_pci(struct device *d, char *buf)
{
	struct pci_dev *pdev = container_of(d, struct pci_dev, dev);
	char * out = buf;
	int i, j;
	u32 val;

	out += sprintf(out, "PCI Configuration Data\n");
	for (i = 0; i < 16; i++) {
		out += sprintf(out, "[%08X] ", i * 16);
		for (j = 0; j < 16; j += 4) {
			pci_read_config_dword(pdev, i * 16 + j, &val); 
			out += sprintf(out, "%08X ", val);
		}
		out += sprintf(out, "\n");
	}

	return out - buf;
}
static DEVICE_ATTR(pci, S_IRUGO, show_pci, NULL);


#define IPW2100_REG(x) { IPW_ ##x, #x }
const struct {
	u32 addr;
	const char *name;
} hw_data[] = {
	IPW2100_REG(REG_GP_CNTRL),
	IPW2100_REG(REG_GPIO),
	IPW2100_REG(REG_INTA),
	IPW2100_REG(REG_INTA_MASK),
	IPW2100_REG(REG_RESET_REG),
};
#define IPW2100_NIC(x, s) { x, #x, s }
const struct {
	u32 addr;
	const char *name;
	size_t size;
} nic_data[] = {
	IPW2100_NIC(IPW2100_CONTROL_REG, 2),
	IPW2100_NIC(0x210014, 1),
	IPW2100_NIC(0x210000, 1),
};
#define IPW2100_ORD(x, d) { IPW_ORD_ ##x, #x, d }
const struct {
	u8 index;
	const char *name;
	const char *desc;
} ord_data[] = {
	IPW2100_ORD(STAT_TX_HOST_REQUESTS, "requested Host Tx's (MSDU)"),
	IPW2100_ORD(STAT_TX_HOST_COMPLETE, "successful Host Tx's (MSDU)"),
	IPW2100_ORD(STAT_TX_DIR_DATA,	   "successful Directed Tx's (MSDU)"),
	IPW2100_ORD(STAT_TX_DIR_DATA1,	   "successful Directed Tx's (MSDU) @ 1MB"),
	IPW2100_ORD(STAT_TX_DIR_DATA2,	   "successful Directed Tx's (MSDU) @ 2MB"),
	IPW2100_ORD(STAT_TX_DIR_DATA5_5,   "successful Directed Tx's (MSDU) @ 5_5MB"),
	IPW2100_ORD(STAT_TX_DIR_DATA11,	   "successful Directed Tx's (MSDU) @ 11MB"),
	IPW2100_ORD(STAT_TX_NODIR_DATA1,   "successful Non_Directed Tx's (MSDU) @ 1MB"),
	IPW2100_ORD(STAT_TX_NODIR_DATA2,   "successful Non_Directed Tx's (MSDU) @ 2MB"),
	IPW2100_ORD(STAT_TX_NODIR_DATA5_5, "successful Non_Directed Tx's (MSDU) @ 5.5MB"),
	IPW2100_ORD(STAT_TX_NODIR_DATA11,  "successful Non_Directed Tx's (MSDU) @ 11MB"),
	IPW2100_ORD(STAT_NULL_DATA,	   "successful NULL data Tx's"),
	IPW2100_ORD(STAT_TX_RTS,	   "successful Tx RTS"),
	IPW2100_ORD(STAT_TX_CTS,	   "successful Tx CTS"),         
	IPW2100_ORD(STAT_TX_ACK,	   "successful Tx ACK"),
	IPW2100_ORD(STAT_TX_ASSN,	   "successful Association Tx's"),               
	IPW2100_ORD(STAT_TX_ASSN_RESP,	   "successful Association response Tx's"),   
	IPW2100_ORD(STAT_TX_REASSN,	   "successful Reassociation Tx's"),           
	IPW2100_ORD(STAT_TX_REASSN_RESP,   "successful Reassociation response Tx's"),
	IPW2100_ORD(STAT_TX_PROBE,	   "probes successfully transmitted"), 
	IPW2100_ORD(STAT_TX_PROBE_RESP,	   "probe responses successfully transmitted"),                
	IPW2100_ORD(STAT_TX_BEACON,	   "tx beacon"),             
	IPW2100_ORD(STAT_TX_ATIM,	   "Tx ATIM"),            
	IPW2100_ORD(STAT_TX_DISASSN,	   "successful Disassociation TX"),           
	IPW2100_ORD(STAT_TX_AUTH,	   "successful Authentication Tx"),                 
	IPW2100_ORD(STAT_TX_DEAUTH,	   "successful Deauthentication TX"),                
	IPW2100_ORD(STAT_TX_TOTAL_BYTES,   "Total successful Tx data bytes"),
	IPW2100_ORD(STAT_TX_RETRIES,       "Tx retries"),
	IPW2100_ORD(STAT_TX_RETRY1,        "Tx retries at 1MBPS"),
	IPW2100_ORD(STAT_TX_RETRY2,        "Tx retries at 2MBPS"),
	IPW2100_ORD(STAT_TX_RETRY5_5,	   "Tx retries at 5.5MBPS"),
	IPW2100_ORD(STAT_TX_RETRY11,	   "Tx retries at 11MBPS"),
	IPW2100_ORD(STAT_TX_FAILURES,	   "Tx Failures"),
	IPW2100_ORD(STAT_TX_MAX_TRIES_IN_HOP,"times max tries in a hop failed"),
	IPW2100_ORD(STAT_TX_DISASSN_FAIL,	"times disassociation failed"),
	IPW2100_ORD(STAT_TX_ERR_CTS,         "missed/bad CTS frames"),
	IPW2100_ORD(STAT_TX_ERR_ACK,	"tx err due to acks"),
	IPW2100_ORD(STAT_RX_HOST,	"packets passed to host"),
	IPW2100_ORD(STAT_RX_DIR_DATA,	"directed packets"),                
	IPW2100_ORD(STAT_RX_DIR_DATA1,	"directed packets at 1MB"),            
	IPW2100_ORD(STAT_RX_DIR_DATA2,	"directed packets at 2MB"),           
	IPW2100_ORD(STAT_RX_DIR_DATA5_5,	"directed packets at 5.5MB"),            
	IPW2100_ORD(STAT_RX_DIR_DATA11,	"directed packets at 11MB"),            
	IPW2100_ORD(STAT_RX_NODIR_DATA,"nondirected packets"),        
	IPW2100_ORD(STAT_RX_NODIR_DATA1,	"nondirected packets at 1MB"),            
	IPW2100_ORD(STAT_RX_NODIR_DATA2,	"nondirected packets at 2MB"),           
	IPW2100_ORD(STAT_RX_NODIR_DATA5_5,	"nondirected packets at 5.5MB"),            
	IPW2100_ORD(STAT_RX_NODIR_DATA11,	"nondirected packets at 11MB"),            
	IPW2100_ORD(STAT_RX_NULL_DATA,	"null data rx's"),            
	IPW2100_ORD(STAT_RX_RTS,	"Rx RTS"),                
	IPW2100_ORD(STAT_RX_CTS,	"Rx CTS"),                
	IPW2100_ORD(STAT_RX_ACK,	"Rx ACK"),                
	IPW2100_ORD(STAT_RX_CFEND,	"Rx CF End"),                
	IPW2100_ORD(STAT_RX_CFEND_ACK,	"Rx CF End + CF Ack"),             
	IPW2100_ORD(STAT_RX_ASSN,	"Association Rx's"),                    
	IPW2100_ORD(STAT_RX_ASSN_RESP,	"Association response Rx's"),            
	IPW2100_ORD(STAT_RX_REASSN,	"Reassociation Rx's"),            
	IPW2100_ORD(STAT_RX_REASSN_RESP,	"Reassociation response Rx's"),            
	IPW2100_ORD(STAT_RX_PROBE,	"probe Rx's"),                
	IPW2100_ORD(STAT_RX_PROBE_RESP,	"probe response Rx's"),            
	IPW2100_ORD(STAT_RX_BEACON,	"Rx beacon"),            
	IPW2100_ORD(STAT_RX_ATIM,	"Rx ATIM"),                    
	IPW2100_ORD(STAT_RX_DISASSN,	"disassociation Rx"),            
	IPW2100_ORD(STAT_RX_AUTH,	"authentication Rx"),                    
	IPW2100_ORD(STAT_RX_DEAUTH,	"deauthentication Rx"),            
	IPW2100_ORD(STAT_RX_TOTAL_BYTES,"Total rx data bytes received"),
	IPW2100_ORD(STAT_RX_ERR_CRC,	 "packets with Rx CRC error"),
	IPW2100_ORD(STAT_RX_ERR_CRC1,	 "Rx CRC errors at 1MB"),
	IPW2100_ORD(STAT_RX_ERR_CRC2,	 "Rx CRC errors at 2MB"),
	IPW2100_ORD(STAT_RX_ERR_CRC5_5,	 "Rx CRC errors at 5.5MB"),
	IPW2100_ORD(STAT_RX_ERR_CRC11,	 "Rx CRC errors at 11MB"),
	IPW2100_ORD(STAT_RX_DUPLICATE1, "duplicate rx packets at 1MB"),
	IPW2100_ORD(STAT_RX_DUPLICATE2,	 "duplicate rx packets at 2MB"),
	IPW2100_ORD(STAT_RX_DUPLICATE5_5,	 "duplicate rx packets at 5.5MB"),
	IPW2100_ORD(STAT_RX_DUPLICATE11,	 "duplicate rx packets at 11MB"),
	IPW2100_ORD(STAT_RX_DUPLICATE, "duplicate rx packets"),            
	IPW2100_ORD(PERS_DB_LOCK,	"locking fw permanent  db"),
	IPW2100_ORD(PERS_DB_SIZE,	"size of fw permanent  db"),           
	IPW2100_ORD(PERS_DB_ADDR,	"address of fw permanent  db"),           
	IPW2100_ORD(STAT_RX_INVALID_PROTOCOL,	"rx frames with invalid protocol"),        
	IPW2100_ORD(SYS_BOOT_TIME,	"Boot time"),
	IPW2100_ORD(STAT_RX_NO_BUFFER,	"rx frames rejected due to no buffer"), 
	IPW2100_ORD(STAT_RX_MISSING_FRAG,	"rx frames dropped due to missing fragment"),            
	IPW2100_ORD(STAT_RX_ORPHAN_FRAG,	"rx frames dropped due to non-sequential fragment"),        
	IPW2100_ORD(STAT_RX_ORPHAN_FRAME,	"rx frames dropped due to unmatched 1st frame"),            
	IPW2100_ORD(STAT_RX_FRAG_AGEOUT,	"rx frames dropped due to uncompleted frame"),        
	IPW2100_ORD(STAT_RX_ICV_ERRORS,	"ICV errors during decryption"),        
	IPW2100_ORD(STAT_PSP_SUSPENSION,"times adapter suspended"),        
	IPW2100_ORD(STAT_PSP_BCN_TIMEOUT,	"beacon timeout"),           
	IPW2100_ORD(STAT_PSP_POLL_TIMEOUT,	"poll response timeouts"),       
	IPW2100_ORD(STAT_PSP_NONDIR_TIMEOUT, "timeouts waiting for last {broad,multi}cast pkt"),
	IPW2100_ORD(STAT_PSP_RX_DTIMS,	"PSP DTIMs received"),
	IPW2100_ORD(STAT_PSP_RX_TIMS,	"PSP TIMs received"),
	IPW2100_ORD(STAT_PSP_STATION_ID,"PSP Station ID"),
	IPW2100_ORD(LAST_ASSN_TIME,	"RTC time of last association"),               
	IPW2100_ORD(STAT_PERCENT_MISSED_BCNS,"current calculation of % missed beacons"),       
	IPW2100_ORD(STAT_PERCENT_RETRIES,"current calculation of % missed tx retries"),            
	IPW2100_ORD(ASSOCIATED_AP_PTR,	"0 if not associated, else pointer to AP table entry"),
	IPW2100_ORD(AVAILABLE_AP_CNT,	"AP's decsribed in the AP table"),
	IPW2100_ORD(AP_LIST_PTR,	"Ptr to list of available APs"),
	IPW2100_ORD(STAT_AP_ASSNS,	"associations"),
	IPW2100_ORD(STAT_ASSN_FAIL,	"association failures"),       
	IPW2100_ORD(STAT_ASSN_RESP_FAIL,"failures due to response fail"),       
	IPW2100_ORD(STAT_FULL_SCANS,	"full scans"),
	IPW2100_ORD(CARD_DISABLED,	"Card Disabled"),
	IPW2100_ORD(STAT_ROAM_INHIBIT,	"times roaming was inhibited due to activity"),       
	IPW2100_ORD(RSSI_AT_ASSN,	"RSSI of associated AP at time of association"),           
	IPW2100_ORD(STAT_ASSN_CAUSE1,	"reassociation: no probe response or TX on hop"),
	IPW2100_ORD(STAT_ASSN_CAUSE2,	"reassociation: poor tx/rx quality"),            
	IPW2100_ORD(STAT_ASSN_CAUSE3,	"reassociation: tx/rx quality (excessive AP load"),
	IPW2100_ORD(STAT_ASSN_CAUSE4,	"reassociation: AP RSSI level"),
	IPW2100_ORD(STAT_ASSN_CAUSE5,	"reassociations due to load leveling"),            
	IPW2100_ORD(STAT_AUTH_FAIL,	"times authentication failed"),        
	IPW2100_ORD(STAT_AUTH_RESP_FAIL,"times authentication response failed"),       
	IPW2100_ORD(STATION_TABLE_CNT,	"entries in association table"),        
	IPW2100_ORD(RSSI_AVG_CURR,	"Current avg RSSI"),
	IPW2100_ORD(POWER_MGMT_MODE,	"Power mode - 0=CAM, 1=PSP"),       
	IPW2100_ORD(COUNTRY_CODE,	"IEEE country code as recv'd from beacon"),
	IPW2100_ORD(COUNTRY_CHANNELS,	"channels suported by country"),           
	IPW2100_ORD(RESET_CNT,	"adapter resets (warm)"),       
	IPW2100_ORD(BEACON_INTERVAL,	"Beacon interval"),       
	IPW2100_ORD(ANTENNA_DIVERSITY,	"TRUE if antenna diversity is disabled"),       
	IPW2100_ORD(DTIM_PERIOD,	"beacon intervals between DTIMs"),       
	IPW2100_ORD(OUR_FREQ,	"current radio freq lower digits - channel ID"),
	IPW2100_ORD(RTC_TIME,	"current RTC time"),           
	IPW2100_ORD(PORT_TYPE,	"operating mode"),       
	IPW2100_ORD(CURRENT_TX_RATE,	"current tx rate"),
	IPW2100_ORD(SUPPORTED_RATES,	"supported tx rates"),       
	IPW2100_ORD(ATIM_WINDOW,	"current ATIM Window"),
	IPW2100_ORD(BASIC_RATES,	"basic tx rates"),       
	IPW2100_ORD(NIC_HIGHEST_RATE,	"NIC highest tx rate"),
	IPW2100_ORD(AP_HIGHEST_RATE,	"AP highest tx rate"),
	IPW2100_ORD(CAPABILITIES,	"Management frame capability field"),
	IPW2100_ORD(AUTH_TYPE,	"Type of authentication"),       
	IPW2100_ORD(RADIO_TYPE,	"Adapter card platform type"),       
	IPW2100_ORD(RTS_THRESHOLD,	"Min packet length for RTS handshaking"),       
	IPW2100_ORD(INT_MODE,	"International mode"),           
	IPW2100_ORD(FRAGMENTATION_THRESHOLD,	"protocol frag threshold"),       
	IPW2100_ORD(EEPROM_SRAM_DB_BLOCK_START_ADDRESS,	"EEPROM offset in SRAM"),
	IPW2100_ORD(EEPROM_SRAM_DB_BLOCK_SIZE,	"EEPROM size in SRAM"),
	IPW2100_ORD(EEPROM_SKU_CAPABILITY,	"EEPROM SKU Capability"),
	IPW2100_ORD(EEPROM_IBSS_11B_CHANNELS,	"EEPROM IBSS 11b channel set"),       
	IPW2100_ORD(MAC_VERSION,	"MAC Version"),       
	IPW2100_ORD(MAC_REVISION,	"MAC Revision"),       
	IPW2100_ORD(RADIO_VERSION,	"Radio Version"),       
	IPW2100_ORD(NIC_MANF_DATE_TIME,	"MANF Date/Time STAMP"),
	IPW2100_ORD(UCODE_VERSION,	"Ucode Version"),       
};


static ssize_t show_registers(struct device *d, char *buf)
{
	int i;
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char * out = buf;
	u32 val = 0;

	out += sprintf(out, "%30s [Address ] : Hex\n", "Register");
	
	for (i = 0; i < (sizeof(hw_data) / sizeof(*hw_data)); i++) {
		read_register(dev, hw_data[i].addr, &val);
		out += sprintf(out, "%30s [%08X] : %08X\n",
			       hw_data[i].name, hw_data[i].addr, val);
	}

	return out - buf;
}
static DEVICE_ATTR(registers, S_IRUGO, show_registers, NULL);


static ssize_t show_hardware(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char * out = buf;
	int i;

	out += sprintf(out, "%30s [Address ] : Hex\n", "NIC entry");
	
	for (i = 0; i < (sizeof(nic_data) / sizeof(*nic_data)); i++) {
		u8 tmp8;
		u16 tmp16;
		u32 tmp32;
		
		switch (nic_data[i].size) {
		case 1:
			read_nic_byte(dev, nic_data[i].addr, &tmp8);
			out += sprintf(out, "%30s [%08X] : %02X\n",
				       nic_data[i].name, nic_data[i].addr,
				       tmp8);
			break;
		case 2:
			read_nic_word(dev, nic_data[i].addr, &tmp16);
			out += sprintf(out, "%30s [%08X] : %04X\n",
				       nic_data[i].name, nic_data[i].addr,
				       tmp16);
			break;
		case 4:
			read_nic_dword(dev, nic_data[i].addr, &tmp32);
			out += sprintf(out, "%30s [%08X] : %08X\n",
				       nic_data[i].name, nic_data[i].addr,
				       tmp32);
			break;
		}
	}
	return out - buf;
}
static DEVICE_ATTR(hardware, S_IRUGO, show_hardware, NULL);


static ssize_t show_memory(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	static unsigned long loop = 0;
	int len = 0;
	u32 buffer[4];
	int i;
	char line[81];

	if (loop >= 0x30000)
		loop = 0;

	/* sysfs provides us PAGE_SIZE buffer */
	while (len < PAGE_SIZE - 128 && loop < 0x30000) {

		if (priv->snapshot[0]) for (i = 0; i < 4; i++) 
			buffer[i] = *(u32 *)SNAPSHOT_ADDR(loop + i * 4);
		else for (i = 0; i < 4; i++) 
			read_nic_dword(dev, loop + i * 4, &buffer[i]);
		
		if (priv->dump_raw) 
			len += sprintf(buf + len,
				       "%c%c%c%c"
				       "%c%c%c%c"
				       "%c%c%c%c"
				       "%c%c%c%c",
				       ((u8*)buffer)[0x0], 
				       ((u8*)buffer)[0x1], 
				       ((u8*)buffer)[0x2], 
				       ((u8*)buffer)[0x3], 
				       ((u8*)buffer)[0x4], 
				       ((u8*)buffer)[0x5], 
				       ((u8*)buffer)[0x6], 
				       ((u8*)buffer)[0x7], 
				       ((u8*)buffer)[0x8], 
				       ((u8*)buffer)[0x9], 
				       ((u8*)buffer)[0xa], 
				       ((u8*)buffer)[0xb], 
				       ((u8*)buffer)[0xc], 
				       ((u8*)buffer)[0xd], 
				       ((u8*)buffer)[0xe], 
				       ((u8*)buffer)[0xf]);
		else
			len += sprintf(buf + len, "%s\n",
				       snprint_line(line, sizeof(line),
						    (u8*)buffer, 16, loop));
		loop += 16;
	}
	
	return len;
}

static ssize_t store_memory(struct device *d, const char *buf, size_t count)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	const char *p = buf;

	if (count < 1)
		return count;

	if (p[0] == '1' || 
	    (count >= 2 && tolower(p[0]) == 'o' && tolower(p[1]) == 'n')) {
		printk(KERN_INFO "%s: Setting memory dump to RAW mode.\n",
		       dev->name);
		priv->dump_raw = 1;

	} else if (p[0] == '0' || (count >= 2 && tolower(p[0]) == 'o' &&
				  tolower(p[1]) == 'f')) {
		printk(KERN_INFO "%s: Setting memory dump to HEX mode.\n",
		       dev->name);
		priv->dump_raw = 0;

	} else if (tolower(p[0]) == 'r') {
		printk(KERN_INFO "%s: Resetting firmware snapshot.\n",
		       dev->name);
		ipw2100_snapshot_free(priv);

	} else 
		printk(KERN_INFO "%s: Usage: 0|on = HEX, 1|off = RAW, "
		       "reset = clear memory snapshot\n",
		       dev->name);

	return count;
}
static DEVICE_ATTR(memory, S_IWUSR|S_IRUGO, show_memory, store_memory);


static ssize_t show_ordinals(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	u32 val = 0;
	int len = 0;
	u32 val_len;
	static int loop = 0;

	if (loop >= sizeof(ord_data) / sizeof(*ord_data))
		loop = 0;

	/* sysfs provides us PAGE_SIZE buffer */
	while (len < PAGE_SIZE - 128 &&
	       loop < (sizeof(ord_data) / sizeof(*ord_data))) {
		
		val_len = sizeof(u32);

		if (ipw2100_get_ordinal(priv, ord_data[loop].index, &val, 
					&val_len)) 
			len += sprintf(buf + len, "[0x%02X] = ERROR    %s\n",
				       ord_data[loop].index,
				       ord_data[loop].desc);
		else
			len += sprintf(buf + len, "[0x%02X] = 0x%08X %s\n",
				       ord_data[loop].index, val,
				       ord_data[loop].desc);
		loop++;
	}

	return len;
}
static DEVICE_ATTR(ordinals, S_IRUGO, show_ordinals, NULL);


static ssize_t show_version(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	char * out = buf;
	char tmp[MAX_FW_VERSION_LEN];
	int err;

	out += sprintf(out, "hardware     : 0x%04X\n", 
		       priv->pdev->subsystem_device);

	out += sprintf(out, "driver       : " DRV_VERSION " ["
#ifdef CONFIG_IEEE80211_NOWEP
		       " !WEP"
#else
		       " WEP"
#endif
#ifdef CONFIG_IPW2100_PROMISC
		       " RFMON"
#else
		       " !RFMON"
#endif
		       " ]\n");

	out += sprintf(out, "build date   : " __DATE__ "\n");
	out += sprintf(out, "build time   : " __TIME__ "\n");
	out += sprintf(out, "eeprom       : %d\n", priv->eeprom_version);

	err = ipw2100_get_ucodeversion(priv, tmp, sizeof(tmp));
	if (err < 0) 
		out += sprintf(out, "ucode        : error\n");
	else 
		out += sprintf(out, "ucode        : %s\n", tmp);

	err = ipw2100_get_fwversion(priv, tmp, sizeof(tmp));
	if (err < 0) 
		out += sprintf(out, "firmware     : error\n");
	else 
		out += sprintf(out, "firmware     : %s\n", tmp);

	out += sprintf(out, "firmware img : 0x%04X\n",
		       priv->firmware_version);

	out += sprintf(out, "firmware mode: %s\n",
		       priv->ctx->port_type == MONITOR ? "RFMON" :
		       priv->ctx->port_type == IBSS ? "IBSS" : "BSS");

	return out - buf;
}
static DEVICE_ATTR(version, S_IRUGO, show_version, NULL);


static ssize_t show_stats(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	char * out = buf;

	out += sprintf(out, "interrupts: %d {tx: %d, rx: %d, other: %d}\n",
		       priv->interrupts, priv->tx_interrupts,
		       priv->rx_interrupts, priv->inta_other);
	out += sprintf(out, "firmware resets: %d\n", priv->resets);
	out += sprintf(out, "firmware hangs: %d\n", priv->hangs);
#ifdef CONFIG_IPW_DEBUG
	out += sprintf(out, "packet mismatch image: %s\n",
		       priv->snapshot[0] ? "YES" : "NO");
#endif

	return out - buf;
}
static DEVICE_ATTR(stats, S_IRUGO, show_stats, NULL);


#ifdef CONFIG_IPW2100_PROMISC
int ipw2100_enable_monitor(struct ipw2100_priv *priv, u32 channel)
{
	int err;

	if (priv->ndev->type == ARPHRD_IEEE80211)
		return -EINVAL;

	err = ipw2100_disable_adapter(priv);
	if (err) {
		printk(KERN_ERR "%s: Could not disable adapter %d\n",
		       priv->ndev->name, err);
		return err;
	}

	memcpy(&priv->backup_ctx, priv->ctx, sizeof(struct ssid_context));

	priv->ctx->ssid[0] = '\0';
	priv->ctx->ssid_len = 0;
	priv->ctx->port_type = MONITOR;
	priv->ieee->iw_mode = IW_MODE_MONITOR;
	priv->ctx->channel = channel;

	/*
	 * Indicate ipw2100_download_firmware download firmware
	 * from disk instead of memory.
	 */
	ipw2100_firmware.version = 0;

	priv->reset_backoff = 0;
	ipw2100_reset_adapter(priv);

	priv->ndev->type = ARPHRD_IEEE80211;

	return 0;
}

int ipw2100_disable_monitor(struct ipw2100_priv *priv)
{
	int err;

	if (priv->ndev->type == ARPHRD_ETHER)
		return -EINVAL;

	err = ipw2100_disable_adapter(priv);
	if (err) {
		printk(KERN_ERR "%s: Could not disable adapter %d\n",
		       priv->ndev->name, err);
		return err;
	}

	memcpy(priv->ctx, &priv->backup_ctx, sizeof(struct ssid_context));
	if (priv->ctx->port_type == BSS)
		priv->ieee->iw_mode = IW_MODE_INFRA;
	else 
		priv->ieee->iw_mode = IW_MODE_ADHOC;

	/*
	 * Indicate ipw2100_download_firmware download firmware
	 * from disk instead of memory.
	 */
	ipw2100_firmware.version = 0;

	priv->reset_backoff = 0;
	ipw2100_reset_adapter(priv);

	priv->ndev->type = ARPHRD_ETHER;

	return 0;
}

#endif /* CONFIG_IPW2100_PROMISC */

int ipw2100_switch_mode(struct ipw2100_priv *priv, u32 mode, u32 channel)
{
	if (mode == BSS) {
		priv->ieee->iw_mode = IW_MODE_INFRA;
		priv->ctx->port_type = BSS; 
	} else {
		priv->ieee->iw_mode = IW_MODE_ADHOC;
		priv->ctx->port_type = IBSS;
		priv->ctx->channel = channel;
	}

	/*
	 * Indicate ipw2100_download_firmware download firmware
	 * from disk instead of memory.
	 */
	ipw2100_firmware.version = 0;

	priv->reset_backoff = 0;
	ipw2100_reset_adapter(priv);

	return 0;
}

static ssize_t show_internals(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	int len = 0;

#define DUMP_VAR(x,y) len += sprintf(buf + len, # x ": %" # y "\n", priv-> x)

	DUMP_VAR(connected, d);
	DUMP_VAR(connected ? get_seconds() - priv->connect_start : 0, lu);
	DUMP_VAR(auto_associate, d);
	DUMP_VAR(ieee->crypt[priv->ieee->tx_keyidx], p);
	DUMP_VAR(reset_pending, d);
	DUMP_VAR(stopped, d);
	DUMP_VAR(disabled, d);
	DUMP_VAR(phy_off, d);
	DUMP_VAR(manual_disable, d);
	DUMP_VAR(been_powered, d);
	DUMP_VAR(fatal_error, d);
	DUMP_VAR(rf_kill, d);
	DUMP_VAR(irq_ignore, d);
	DUMP_VAR(stop_hang_check, d);
	DUMP_VAR(stop_rf_check, d);
	DUMP_VAR(message_sending, d);
	DUMP_VAR(messages_sent, d);

	DUMP_VAR(tx_pend_stat.value, d);
	DUMP_VAR(tx_pend_stat.hi, d);

	DUMP_VAR(tx_free_stat.value, d);
	DUMP_VAR(tx_free_stat.lo, d);

	DUMP_VAR(msg_free_stat.value, d);
	DUMP_VAR(msg_free_stat.lo, d);

	DUMP_VAR(msg_pend_stat.value, d);
	DUMP_VAR(msg_pend_stat.hi, d);

	DUMP_VAR(fw_pend_stat.value, d);
	DUMP_VAR(fw_pend_stat.hi, d);

	DUMP_VAR(txq_stat.value, d);
	DUMP_VAR(txq_stat.lo, d);

	DUMP_VAR(ieee->scans, d);
	DUMP_VAR(reset_backoff, d);

	return len;
}
static DEVICE_ATTR(internals, S_IRUGO, show_internals, NULL);


static ssize_t show_bssinfo(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	char essid[IW_ESSID_MAX_SIZE + 1];
	u8 bssid[ETH_ALEN];
	u32 chan = 0;
	char * out = buf;
	int length;
	int ret;

	memset(essid, 0, sizeof(essid));
	memset(bssid, 0, sizeof(bssid));

	length = IW_ESSID_MAX_SIZE;
	ret = ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_SSID, essid, &length);
	if (ret) 
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);

	length = sizeof(bssid);
	ret = ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID,
				  bssid, &length);
	if (ret)
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);

	length = sizeof(u32);
	ret = ipw2100_get_ordinal(priv, IPW_ORD_OUR_FREQ, &chan, &length);
	if (ret) 
		IPW_DEBUG_INFO("failed querying ordinals at line %d\n", 
				   __LINE__);

	out += sprintf(out, "ESSID: %s\n", essid);
	out += sprintf(out, "BSSID:   %02x:%02x:%02x:%02x:%02x:%02x\n",
		       bssid[0], bssid[1], bssid[2],
		       bssid[3], bssid[4], bssid[5]);
	out += sprintf(out, "Channel: %d\n", chan);

	return out - buf;
}
static DEVICE_ATTR(bssinfo, S_IRUGO, show_bssinfo, NULL);


static ssize_t show_txqueue(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char * out = buf;
	u32 tbdr_r, tbdr_w;

	IPW_DEBUG_INFO("enter\n");

	read_register(dev, IPW_MEM_HOST_SHARED_TX_QUEUE_READ_INDEX,
		      &tbdr_r);
	read_register(dev, IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX,
		      &tbdr_w);

	IPW_DEBUG_INFO("after register read\n");

	out += sprintf(out,
		       "Tx Queue\nnic:\n\tread  index=%d\n\twrite index=%d\n",
		       tbdr_r, tbdr_w);
	out += sprintf(out, "drv:\n\tread index=%d\n\twrite index=%d\n",
		       priv->tx_queue.oldest,
		       priv->tx_queue.next);

	IPW_DEBUG_INFO("exit\n");
	return out - buf;
}
static DEVICE_ATTR(txqueue, S_IRUGO, show_txqueue, NULL);


static ssize_t show_rxqueue(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char * out = buf;
	u32 rbdr_r, rbdr_w;

	IPW_DEBUG_INFO("enter\n");

	read_register(dev, IPW_MEM_HOST_SHARED_RX_READ_INDEX, &rbdr_r);
	read_register(dev, IPW_MEM_HOST_SHARED_RX_WRITE_INDEX, &rbdr_w);

	IPW_DEBUG_INFO("after register read\n");

	out += sprintf(out,
		       "Rx Queue\nnic:\n\tread  index=%d\n\twrite index=%d\n",
		       rbdr_r, rbdr_w);
	out += sprintf(out,
		       "drv:\n\tread  index=NOT USED\n\twrite index=%d\n",
		       priv->rx_queue.next);

	IPW_DEBUG_INFO("exit\n");
	return out - buf;
}
static DEVICE_ATTR(rxqueue, S_IRUGO, show_rxqueue, NULL);


#ifdef CONFIG_IPW_DEBUG
static ssize_t show_debug_level(struct device_driver *d, char *buf)
{
	char * out = buf;
	int i;

	if (!IPW_DEBUG_ENABLED)
		return 0;
	
	out += sprintf(out, "%-25s\tHex        SET Decimal\n", "Description");
	for (i = 0; i < sizeof(ipw2100_debug_levels) / 
	     sizeof(struct ipw2100_dl); i++) {
		out += sprintf(
		       out, "%-25s\t0x%08lX [%c] %lu\n",
		       ipw2100_debug_levels[i].name,
		       ipw2100_debug_levels[i].value,
		       (ipw2100_debug_level & ipw2100_debug_levels[i].value) ?
		       '*' : ' ',
		       ipw2100_debug_levels[i].value);
	}
	
	out += sprintf(out, "debug_level = 0x%08lX (* = enabled)\n", 
		       ipw2100_debug_level);

	return out - buf;
}

static ssize_t store_debug_level(struct device_driver *d, const char *buf,
				 size_t count)
{
	char buffer[] = "0x00000000";
	unsigned long len =
	    (sizeof(buffer) - 1) > count ? count : sizeof(buffer) - 1;
	unsigned long val;
	char *p = buffer;
	int i;

	if (!IPW_DEBUG_ENABLED)
		return 0;

	strncpy(buffer, buf, len);
	buffer[len] = 0;

	if (p[1] == 'x' || p[1] == 'X' || p[0] == 'x' || p[0] == 'X') {
		p++;
		if (p[0] == 'x' || p[0] == 'X')
			p++;
		val = simple_strtoul(p, &p, 16);
	} else
		val = simple_strtoul(p, &p, 10);

	if (p == buffer) 
		printk(KERN_INFO DRV_NAME ": %s is not in hex [0x] or "
		       "decimal form.\n", buffer);
	else {
		printk(KERN_INFO DRV_NAME ": Debug level set to 0x%08lX\n", 
		       val);
		ipw2100_debug_level = val;

		for (i = 0; i < sizeof(ipw2100_debug_levels) / 
			     sizeof(struct ipw2100_dl); i++) {
			if ((ipw2100_debug_level & 
			    ipw2100_debug_levels[i].value) ==
			    ipw2100_debug_levels[i].value)
				printk(KERN_INFO DRV_NAME 
				       ": tracing enabled for %-25s (%08lX).\n",
				       ipw2100_debug_levels[i].name,
				       ipw2100_debug_levels[i].value);
		}
	}

	return len;
}
static DRIVER_ATTR(debug_level, S_IWUSR | S_IRUGO, show_debug_level,
		   store_debug_level);
#endif /* CONFIG_IPW_DEBUG */


static ssize_t show_fatal_error(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	char * out = buf;
	int errors = 0;
	int i;

	if (priv->fatal_error) 
		out += sprintf(out, "Current error: 0x%08X\n",
			       priv->fatal_error);
	else 
		out += sprintf(out, "No current error condition.\n");

	for (i = 1; i <= IPW2100_ERROR_QUEUE; i++) {
		if (!priv->fatal_errors[(priv->fatal_index - i) % 
					IPW2100_ERROR_QUEUE]) 
			continue;

		if (!errors)
			out += sprintf(out, "Up to last %d errors:\n",
				       IPW2100_ERROR_QUEUE);
		out += sprintf(out, "%d. Error: 0x%08X\n", i,
			       priv->fatal_errors[(priv->fatal_index - i) %
						  IPW2100_ERROR_QUEUE]);
		errors++;
	}

	if (!errors)
		out += sprintf(out, "No errors encountered.\n");

#ifdef CONFIG_IPW_DEBUG
	out += sprintf(out, "Packet mismatch image: %s\n",
		       priv->snapshot[0] ? "YES" : "NO");
#endif

	if (priv->fatal_error)
		out += sprintf(out,
			       "`echo 0 > fatal_error` to force firmware "
			       "restart and clear current error condition.\n");

	return out - buf;
}

static ssize_t store_fatal_error(struct device *d, const char *buf,
				 size_t count)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	schedule_reset(priv);
	return count;
}
static DEVICE_ATTR(fatal_error, S_IWUSR|S_IRUGO, show_fatal_error, store_fatal_error);


static ssize_t show_cardmem(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char * out = buf;
	u32 dword;

	read_nic_dword(dev, priv->proc_cardmemaddr, &dword);

	out += sprintf(out, "cardmem addr[0x%08x] = 0x%08x (%d)\n",
		       priv->proc_cardmemaddr, dword, dword);

	return out - buf;
}

static ssize_t store_cardmem(struct device *d, const char *buf, size_t count)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char buffer[] = "00000000";
	unsigned long len =
	    (sizeof(buffer) - 1) > count ? count : sizeof(buffer) - 1;
	char *p = buffer;
	u32 val;

	IPW_DEBUG_INFO("enter\n");

	strncpy(buffer, buf, len);
	buffer[len] = 0;

	if (p[1] == 'x' || p[1] == 'X' || p[0] == 'x' || p[0] == 'X') {
		p++;
		if (p[0] == 'x' || p[0] == 'X')
			p++;
		val = simple_strtoul(p, &p, 16);
	} else
		val = simple_strtoul(p, &p, 10);

	if (p == buffer) {
		printk(KERN_INFO "%s: user supplied invalid address value.\n",
		       dev->name);
	} else {
		priv->proc_cardmemaddr = val;
		IPW_DEBUG_INFO("set mem addr = 0x%08x\n", val);
	}
	
	IPW_DEBUG_INFO("exit\n");
	return len;
}
static DEVICE_ATTR(cardmem, S_IWUSR | S_IRUGO, show_cardmem, store_cardmem);


static ssize_t show_scan_age(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);

	return sprintf(buf, "Scan entries will expire after %u missed scans. "
		       "(default is %u, 0 = never expire)\n",
		       priv->ieee->scan_age, DEFAULT_MAX_SCAN_AGE);
}

static ssize_t store_scan_age(struct device *d, const char *buf, size_t count)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	struct net_device *dev = priv->ndev;
	char buffer[] = "00000000";
	unsigned long len =
	    (sizeof(buffer) - 1) > count ? count : sizeof(buffer) - 1;
	unsigned long val;
	char *p = buffer;

	IPW_DEBUG_INFO("enter\n");

	strncpy(buffer, buf, len);
	buffer[len] = 0;

	if (p[1] == 'x' || p[1] == 'X' || p[0] == 'x' || p[0] == 'X') {
		p++;
		if (p[0] == 'x' || p[0] == 'X')
			p++;
		val = simple_strtoul(p, &p, 16);
	} else
		val = simple_strtoul(p, &p, 10);
	if (p == buffer) {
		printk(KERN_INFO "%s: user supplied invalid value.\n",
		       dev->name);
	} else {
		priv->ieee->scan_age = val;
		IPW_DEBUG_INFO("set scan_age = %u\n", priv->ieee->scan_age);
	}
	
	IPW_DEBUG_INFO("exit\n");
	return len;
}
static DEVICE_ATTR(scan_age, S_IWUSR | S_IRUGO, show_scan_age, store_scan_age);


static ssize_t show_rf_kill(struct device *d, char *buf)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	char * out = buf;
	
	if (priv->manual_disable) 
		out += sprintf(out, "Radio disabled by manual switch.\n");
	else if (priv->hw_features & HW_FEATURE_RFKILL) 
		out += sprintf(out, "Radio is %s by RF switch\n",
			       ipw2100_get_rf_switch(priv) ? 
			       "disabled" : "enabled");
	else
		out += sprintf(out,
			       "Your hardware does not have an RF switch\n");
	
	return out - buf;
}

static ssize_t store_rf_kill(struct device *d, const char *buf, size_t count)
{
	struct ipw2100_priv *priv = dev_get_drvdata(d);
	int state;

	IPW_DEBUG_INFO("enter\n");

	sscanf(buf, "%d", &state);
	switch (state) {
	case 0: /* Turn off Manual Disable */
		if (priv->manual_disable) {
			printk(KERN_INFO "%s: Manual Disable turned OFF.\n",
			      priv->ndev->name);
			
			priv->manual_disable = 0;
			
			if (!priv->rf_kill) {
				/* Restart the NIC */
				schedule_reset(priv);		
			}
		}
		break;

	case 1: /* Turn on Manual Disable */
		if (!priv->manual_disable) {
			printk(KERN_INFO "%s: Manual Disable turned ON.\n",
			       priv->ndev->name);
			priv->manual_disable = 1;
			
			ipw2100_down(priv);
		}
		break;

	default:
		printk(KERN_INFO "%s: Invalid state request [%d].\n",
		       priv->ndev->name, state);
		break;
	}
	
	IPW_DEBUG_INFO("exit\n");
	return count;
}
static DEVICE_ATTR(rf_kill, S_IWUSR|S_IRUGO, show_rf_kill, store_rf_kill);


static struct attribute *ipw2100_sysfs_entries[] = {
	&dev_attr_hardware.attr,
	&dev_attr_registers.attr,
	&dev_attr_ordinals.attr,
	&dev_attr_pci.attr,
	&dev_attr_version.attr,
	&dev_attr_stats.attr,
	&dev_attr_internals.attr,
	&dev_attr_txqueue.attr,
	&dev_attr_rxqueue.attr,
	&dev_attr_bssinfo.attr,
	&dev_attr_memory.attr,
	&dev_attr_cardmem.attr,
	&dev_attr_scan_age.attr,
	&dev_attr_fatal_error.attr,
	&dev_attr_rf_kill.attr,
	NULL,
};

static struct attribute_group ipw2100_attribute_group = {
	.attrs = ipw2100_sysfs_entries,
};


static int status_queue_allocate(struct ipw2100_priv *priv, int entries)
{
	struct ipw2100_status_queue *q = &priv->status_queue;

	IPW_DEBUG_INFO("enter\n");

	q->size = entries * sizeof(struct ipw2100_status);
	q->drv = (struct ipw2100_status *)pci_alloc_consistent(
		priv->pdev, q->size, &q->nic);
	if (!q->drv) {
		printk(KERN_WARNING 
		       "Can not allocate status queue.\n");
		return -ENOMEM;
	}

	memset(q->drv, 0, q->size);

	IPW_DEBUG_INFO("exit\n");

	return 0;
}

static void status_queue_free(struct ipw2100_priv *priv)
{
	IPW_DEBUG_INFO("enter\n");

	if (priv->status_queue.drv) {
		pci_free_consistent(
			priv->pdev, priv->status_queue.size, 
			priv->status_queue.drv, priv->status_queue.nic);
		priv->status_queue.drv = NULL;
	}

	IPW_DEBUG_INFO("exit\n");
}

static int bd_queue_allocate(struct ipw2100_priv *priv,
			     struct ipw2100_bd_queue *q, int entries)
{
	IPW_DEBUG_INFO("enter\n");

	memset(q, 0, sizeof(struct ipw2100_bd_queue));

	q->entries = entries;
	q->size = entries * sizeof(struct ipw2100_bd);
	q->drv = pci_alloc_consistent(priv->pdev, q->size, &q->nic);
	if (!q->drv) {
		IPW_DEBUG_INFO("can't allocate shared memory for buffer descriptors\n");
		return -ENOMEM;
	}
	memset(q->drv, 0, q->size);

	IPW_DEBUG_INFO("exit\n");

	return 0;
}

static void bd_queue_free(struct ipw2100_priv *priv,
			  struct ipw2100_bd_queue *q)
{
	IPW_DEBUG_INFO("enter\n");

	if (!q)
		return;

	if (q->drv) {
		pci_free_consistent(priv->pdev,
				    q->size, q->drv, q->nic);
		q->drv = NULL;
	}

	IPW_DEBUG_INFO("exit\n");
}

static void bd_queue_initialize(
	struct ipw2100_priv *priv, struct ipw2100_bd_queue * q,
	u32 base, u32 size, u32 r, u32 w)
{
	IPW_DEBUG_INFO("enter\n");

	IPW_DEBUG_INFO("initializing bd queue at virt=%p, phys=%08x\n", q->drv, q->nic);

	write_register(priv->ndev, base, q->nic);
	write_register(priv->ndev, size, q->entries);
	write_register(priv->ndev, r, q->oldest);
	write_register(priv->ndev, w, q->next);

	IPW_DEBUG_INFO("exit\n");
}

static int ipw2100_tx_allocate(struct ipw2100_priv *priv)
{
	int i, j, err = -EINVAL;
	void *v;
	dma_addr_t p;

	IPW_DEBUG_INFO("enter\n");

	err = bd_queue_allocate(priv, &priv->tx_queue, TX_QUEUE_LENGTH);
	if (err) {
		printk(KERN_ERR "%s: failed bd_queue_allocate\n",
		       priv->ndev->name);
		return err;
	}

	priv->tx_buffers = (struct ipw2100_tx_packet *)kmalloc(
		TX_PENDED_QUEUE_LENGTH * sizeof(struct ipw2100_tx_packet), 
		GFP_KERNEL);
	if (!priv->tx_buffers) {
		printk(KERN_ERR "%s: alloc failed form tx buffers.\n",
		       priv->ndev->name);
		bd_queue_free(priv, &priv->tx_queue);
		return -ENOMEM;
	}

	for (i = 0; i < TX_PENDED_QUEUE_LENGTH; i++) {
		v = pci_alloc_consistent(
			priv->pdev, sizeof(struct ipw2100_data_header), &p);
		if (!v) {
			printk(KERN_ERR "%s: PCI alloc failed for tx "
			       "buffers.\n", priv->ndev->name);
			err = -ENOMEM;
			break;
		}

		priv->tx_buffers[i].type = DATA;
		priv->tx_buffers[i].info.d_struct.data = (struct ipw2100_data_header*)v;
		priv->tx_buffers[i].info.d_struct.data_phys = p;
		priv->tx_buffers[i].info.d_struct.txb = NULL;
	}
	
	if (i == TX_PENDED_QUEUE_LENGTH)
		return 0;

	for (j = 0; j < i; j++) {
		pci_free_consistent(
			priv->pdev,
			sizeof(struct ipw2100_data_header), 
			priv->tx_buffers[j].info.d_struct.data,
			priv->tx_buffers[j].info.d_struct.data_phys);
	}
	
	kfree(priv->tx_buffers);
	priv->tx_buffers = NULL;

	return err;
}

static void ipw2100_tx_initialize(struct ipw2100_priv *priv)
{
	int i;

	IPW_DEBUG_INFO("enter\n");

	/*
	 * reinitialize packet info lists
	 */
	INIT_LIST_HEAD(&priv->fw_pend_list);
	INIT_STAT(&priv->fw_pend_stat);

	/* 
	 * reinitialize lists
	 */
	INIT_LIST_HEAD(&priv->tx_pend_list);
	INIT_LIST_HEAD(&priv->tx_free_list);
	INIT_STAT(&priv->tx_pend_stat);
	INIT_STAT(&priv->tx_free_stat);

	for (i = 0; i < TX_PENDED_QUEUE_LENGTH; i++) {
		/* We simply drop any SKBs that have been queued for
		 * transmit */
		if (priv->tx_buffers[i].info.d_struct.txb) {
			ieee80211_txb_free(priv->tx_buffers[i].info.d_struct.txb);
			priv->tx_buffers[i].info.d_struct.txb = NULL;
		}
		
		list_add_tail(&priv->tx_buffers[i].list, &priv->tx_free_list);
	}

	SET_STAT(&priv->tx_free_stat, i);

	priv->tx_queue.oldest = 0;
	priv->tx_queue.available = priv->tx_queue.entries;
	priv->tx_queue.next = 0;
	INIT_STAT(&priv->txq_stat);
	SET_STAT(&priv->txq_stat, priv->tx_queue.available);

	bd_queue_initialize(priv, &priv->tx_queue,
			    IPW_MEM_HOST_SHARED_TX_QUEUE_BD_BASE,
			    IPW_MEM_HOST_SHARED_TX_QUEUE_BD_SIZE,
			    IPW_MEM_HOST_SHARED_TX_QUEUE_READ_INDEX,
			    IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX);

	IPW_DEBUG_INFO("exit\n");

}

static void ipw2100_tx_free(struct ipw2100_priv *priv)
{
	int i;

	IPW_DEBUG_INFO("enter\n");

	bd_queue_free(priv, &priv->tx_queue);

	if (!priv->tx_buffers)
		return;

	for (i = 0; i < TX_PENDED_QUEUE_LENGTH; i++) {
		if (priv->tx_buffers[i].info.d_struct.txb) {
			ieee80211_txb_free(priv->tx_buffers[i].info.d_struct.txb);
			priv->tx_buffers[i].info.d_struct.txb = NULL;
		}
		if (priv->tx_buffers[i].info.d_struct.data)
			pci_free_consistent(
				priv->pdev,
				sizeof(struct ipw2100_data_header),
				priv->tx_buffers[i].info.d_struct.data,
				priv->tx_buffers[i].info.d_struct.data_phys);
	}
	
	kfree(priv->tx_buffers);
	priv->tx_buffers = NULL;

	IPW_DEBUG_INFO("exit\n");
}



static int ipw2100_rx_allocate(struct ipw2100_priv *priv)
{
	int i, j, err = -EINVAL;

	IPW_DEBUG_INFO("enter\n");

	err = bd_queue_allocate(priv, &priv->rx_queue, RX_QUEUE_LENGTH);
	if (err) {
		IPW_DEBUG_INFO("failed bd_queue_allocate\n");
		return err;
	}

	err = status_queue_allocate(priv, RX_QUEUE_LENGTH);
	if (err) {
		IPW_DEBUG_INFO("failed status_queue_allocate\n");
		bd_queue_free(priv, &priv->rx_queue);
		return err;
	}

	/*
	 * allocate packets
	 */
	priv->rx_buffers = (struct ipw2100_rx_packet *)
	    kmalloc(RX_QUEUE_LENGTH * sizeof(struct ipw2100_rx_packet), 
		    GFP_KERNEL);
	if (!priv->rx_buffers) {
		IPW_DEBUG_INFO("can't allocate rx packet buffer table\n");

		bd_queue_free(priv, &priv->rx_queue);

		status_queue_free(priv);

		return -ENOMEM;
	}

	for (i = 0; i < RX_QUEUE_LENGTH; i++) {
		struct ipw2100_rx_packet *packet = &priv->rx_buffers[i];

		err = ipw2100_alloc_skb(priv, packet);
		if (unlikely(err)) {
			err = -ENOMEM;
			break;
		}

		/* The BD holds the cache aligned address */
		priv->rx_queue.drv[i].host_addr = packet->dma_addr;
		priv->rx_queue.drv[i].buf_length = IPW_RX_NIC_BUFFER_LENGTH;
		priv->status_queue.drv[i].status_fields = 0;
	}

	if (i == RX_QUEUE_LENGTH) 
		return 0;

	for (j = 0; j < i; j++) {
		pci_unmap_single(priv->pdev, priv->rx_buffers[j].dma_addr,
				 sizeof(struct ipw2100_rx_packet),
				 PCI_DMA_FROMDEVICE);
		dev_kfree_skb(priv->rx_buffers[j].skb);
	}
	
	kfree(priv->rx_buffers);
	priv->rx_buffers = NULL;

	bd_queue_free(priv, &priv->rx_queue);

	status_queue_free(priv);

	return err;
}

static void ipw2100_rx_initialize(struct ipw2100_priv *priv)
{
	IPW_DEBUG_INFO("enter\n");

	priv->rx_queue.oldest = 0;
	priv->rx_queue.available = priv->rx_queue.entries - 1;
	priv->rx_queue.next = priv->rx_queue.entries - 1;

	INIT_STAT(&priv->rxq_stat);
	SET_STAT(&priv->rxq_stat, priv->rx_queue.available);

	bd_queue_initialize(priv, &priv->rx_queue,
			    IPW_MEM_HOST_SHARED_RX_BD_BASE,
			    IPW_MEM_HOST_SHARED_RX_BD_SIZE,
			    IPW_MEM_HOST_SHARED_RX_READ_INDEX,
			    IPW_MEM_HOST_SHARED_RX_WRITE_INDEX);

	/* set up the status queue */
	write_register(priv->ndev, IPW_MEM_HOST_SHARED_RX_STATUS_BASE,
		       priv->status_queue.nic);

	IPW_DEBUG_INFO("exit\n");
}

static void ipw2100_rx_free(struct ipw2100_priv *priv)
{
	int i;

	IPW_DEBUG_INFO("enter\n");

	bd_queue_free(priv, &priv->rx_queue);
	status_queue_free(priv);

	if (!priv->rx_buffers)
		return;

	for (i = 0; i < RX_QUEUE_LENGTH; i++) {
		if (priv->rx_buffers[i].rxp) {
			pci_unmap_single(priv->pdev, 
					 priv->rx_buffers[i].dma_addr,
					 sizeof(struct ipw2100_rx),
					 PCI_DMA_FROMDEVICE);
			dev_kfree_skb(priv->rx_buffers[i].skb);
		}
	}
	
	kfree(priv->rx_buffers);
	priv->rx_buffers = NULL;

	IPW_DEBUG_INFO("exit\n");
}

static int ipw2100_read_mac_address(struct ipw2100_priv *priv)
{
	u32 length = ETH_ALEN;
	u8 mac[ETH_ALEN];

	int err;

	err = ipw2100_get_ordinal(priv, IPW_ORD_STAT_ADAPTER_MAC, 
				  mac, &length);
	if (err) {
		IPW_DEBUG_INFO("MAC address read failed\n");
		return -EIO;
	}
	IPW_DEBUG_INFO("card MAC is %02X:%02X:%02X:%02X:%02X:%02X\n",
	       mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	memcpy(priv->ndev->dev_addr, mac, ETH_ALEN);

	return 0;
}

/********************************************************************
 *
 * Firmware Commands
 *
 ********************************************************************/
int ipw2100_start_scan(struct ipw2100_priv *priv)
{
	struct host_command cmd = {
		.host_command = BROADCAST_SCAN,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	IPW_DEBUG_FW_COMMAND("START_SCAN\n");

	cmd.host_command_parameters[0] = 0;

	/* No scanning if in monitor mode */
	if (priv->ctx->port_type == MONITOR)
		return 1;

	if (priv->in_scan) {
		IPW_DEBUG_SCAN("Scan requested while already in scan...\n");
		return 0;
	}

	IPW_DEBUG_INFO("enter\n");

	/* Not clearing here; doing so makes iwlist always return nothing... 
	 *
	 * We should modify the table logic to use aging tables vs. clearing 
	 * the table on each scan start.
	 */
	IPW_DEBUG_SCAN("starting scan\n");
	err =  ipw2100_hw_send_command(priv, &cmd);

	IPW_DEBUG_INFO("exit\n");

	return err;
}

int ipw2100_set_mac_address(struct ipw2100_priv *priv, int batch_mode)
{
	struct host_command cmd = {
		.host_command = ADAPTER_ADDRESS,
		.host_command_sequence = 0,
		.host_command_length = ETH_ALEN
	};
	int err;

	IPW_DEBUG_FW_COMMAND("SET_MAC_ADDRESS\n");

	IPW_DEBUG_INFO("enter\n");

	if (priv->custom_addr) {
		memcpy(cmd.host_command_parameters, priv->mac_addr,
		       ETH_ALEN);
		memcpy(priv->ndev->dev_addr, priv->mac_addr, ETH_ALEN);
	} else
		memcpy(cmd.host_command_parameters, priv->ndev->dev_addr, 
		       ETH_ALEN);

	err = ipw2100_hw_send_command(priv, &cmd);

	IPW_DEBUG_INFO("exit\n");
	return err;
}

int ipw2100_set_port_type(struct ipw2100_priv *priv, u32 port_type, 
				 int batch_mode)
{
	struct host_command cmd = {
		.host_command = PORT_TYPE,
		.host_command_sequence = 0,
		.host_command_length = sizeof(u32)
	};
	int err;

	cmd.host_command_parameters[0] = port_type;

	IPW_DEBUG_FW_COMMAND("PORT_TYPE: %s\n",
				 port_type == IBSS ? "IBSS" : "BSS");
	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) {
			printk(KERN_ERR "%s: Could not disable adapter %d\n",
			       priv->ndev->name, err);
			return err;
		}
	}

	/* send cmd to firmware */
	err = ipw2100_hw_send_command(priv, &cmd);

	/* store if successful */
	if (!err) 
		priv->ctx->port_type = port_type;

	if (!batch_mode) 
		ipw2100_enable_adapter(priv);

	return err;
}


int ipw2100_set_channel(struct ipw2100_priv *priv, u32 channel, int batch_mode)
{
	struct host_command cmd = {
		.host_command = CHANNEL,
		.host_command_sequence = 0,
		.host_command_length = sizeof(u32)
	};
	int err;

	cmd.host_command_parameters[0] = channel;

	IPW_DEBUG_FW_COMMAND("CHANNEL: %d\n", channel);

	/* If not BSS then we don't support channel selection */
	if (priv->ctx->port_type != IBSS && priv->ctx->port_type != MONITOR)
		return 0;

	if ((priv->ctx->channel < REG_MIN_CHANNEL) ||
	    (priv->ctx->channel > REG_MAX_CHANNEL)) 
		return -EINVAL;
		
	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) 
			return err;
	}
	
		
	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) {
		IPW_DEBUG_INFO("Failed to set channel to %d",
			 channel);
		return err;
	}
	
	priv->ctx->channel = channel;

	if (!batch_mode) {
		err = ipw2100_enable_adapter(priv);
		if (err) 
			return err;
	}

	return 0;
}

int ipw2100_system_config(struct ipw2100_priv *priv, int batch_mode)
{
	struct host_command cmd = {
		.host_command = SYSTEM_CONFIG,
		.host_command_sequence = 0,
		.host_command_length = 12,
	};
	u32 ibss_mask, len = sizeof(u32);
	int err;

	/* Set system configuration */

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) 
			return err;
	}
	
	if (priv->ctx->port_type == IBSS) 
		cmd.host_command_parameters[0] |= CFG_IBSS_AUTO_START;

	cmd.host_command_parameters[0] |= CFG_IBSS_MASK | CFG_BSS_MASK | 
		CFG_802_1x_ENABLE;

	if (priv->preamble == IPW_PREAMBLE_AUTO)
		cmd.host_command_parameters[0] |= CFG_PREAMBLE_LEN;
	
	err = ipw2100_get_ordinal(priv, 
				  IPW_ORD_EEPROM_IBSS_11B_CHANNELS,
				  &ibss_mask,  &len);
	if (err)
		ibss_mask = IPW_IBSS_11B_DEFAULT_MASK;

	cmd.host_command_parameters[1] = REG_CHANNEL_MASK;
	cmd.host_command_parameters[2] = REG_CHANNEL_MASK & ibss_mask;
	
	/* 11b only */
	/*cmd.host_command_parameters[0] |= DIVERSITY_ANTENNA_A;*/

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err)
		return err;

/* If IPv6 is configured in the kernel then we don't want to filter out all
 * of the multicast packets as IPv6 needs some. */
#if !defined(CONFIG_IPV6) && !defined(CONFIG_IPV6_MODULE)
	cmd.host_command = ADD_MULTICAST;
	cmd.host_command_sequence = 0;
	cmd.host_command_length = 0;

	ipw2100_hw_send_command(priv, &cmd);
#endif
	if (!batch_mode) {
		err = ipw2100_enable_adapter(priv);
		if (err) 
			return err;
	}

	return 0;
}

int ipw2100_set_tx_rates(struct ipw2100_priv *priv, u32 rate, int batch_mode)
{
	struct host_command cmd = {
		.host_command = BASIC_TX_RATES,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = rate & TX_RATE_MASK;

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) 
			return err;
	}

	/* Set BASIC TX Rate first */
	ipw2100_hw_send_command(priv, &cmd);

	/* Set TX Rate */
	cmd.host_command = TX_RATES;
	ipw2100_hw_send_command(priv, &cmd);

	/* Set MSDU TX Rate */
	cmd.host_command = MSDU_TX_RATES;
	ipw2100_hw_send_command(priv, &cmd);

	if (!batch_mode) {
		err = ipw2100_enable_adapter(priv);
		if (err) 
			return err;
	}

	priv->tx_rates = rate;

	return 0;
}

int ipw2100_set_power_mode(struct ipw2100_priv *priv, 
			   int power_level)
{
	struct host_command cmd = {
		.host_command = POWER_MODE,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = power_level;

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err)
		return err;

	if (power_level == IPW_POWER_MODE_CAM) 
		priv->power_mode = IPW_POWER_LEVEL(priv->power_mode);
	else 
		priv->power_mode = IPW_POWER_ENABLED | power_level;

#ifdef CONFIG_IPW2100_TX_POWER
	if (priv->port_type == IBSS && 
	    priv->adhoc_power != DFTL_IBSS_TX_POWER) {
		/* Set beacon interval */
		cmd.host_command = TX_POWER_INDEX;
		cmd.host_command_parameters[0] = (u32)priv->adhoc_power;
		
		err = ipw2100_hw_send_command(priv, &cmd);
		if (err)
			return err;
	}
#endif

	return 0;
}


int ipw2100_set_rts_threshold(struct ipw2100_priv *priv, u32 threshold)
{
	struct host_command cmd = {
		.host_command = RTS_THRESHOLD,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = threshold;

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) 
		return err;

	priv->rts_threshold = threshold;

	return 0;
}

#if 0
int ipw2100_set_fragmentation_threshold(struct ipw2100_priv *priv,
					u32 threshold, int batch_mode)
{
	struct host_command cmd = {
		.host_command = FRAG_THRESHOLD,
		.host_command_sequence = 0,
		.host_command_length = 4,
		.host_command_parameters[0] = 0,
	};
	int err;

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) 
			return err;
	}

	if (threshold == 0) 
		threshold = DEFAULT_FRAG_THRESHOLD;
	else {
		threshold = max(threshold, MIN_FRAG_THRESHOLD);
		threshold = min(threshold, MAX_FRAG_THRESHOLD);
	}

	cmd.host_command_parameters[0] = threshold;

	IPW_DEBUG_FW_COMMAND("FRAG_THRESHOLD: %u\n", threshold);

	err = ipw2100_hw_send_command(priv, &cmd);

	if (!batch_mode)
		ipw2100_enable_adapter(priv);

	if (!err)
		priv->frag_threshold = threshold;

	return err;
}
#endif

int ipw2100_set_short_retry(struct ipw2100_priv *priv, u32 retry)
{
	struct host_command cmd = {
		.host_command = SHORT_RETRY_LIMIT,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = retry;
	
	err = ipw2100_hw_send_command(priv, &cmd);
	if (err)
		return err;

	priv->short_retry_limit = retry;

	return 0;
}

int ipw2100_set_long_retry(struct ipw2100_priv *priv, u32 retry)
{
	struct host_command cmd = {
		.host_command = LONG_RETRY_LIMIT,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = retry;

	err = ipw2100_hw_send_command(priv, &cmd);
	if (err) 
		return err;

	priv->long_retry_limit = retry;

	return 0;
}


int ipw2100_set_mandatory_bssid(struct ipw2100_priv *priv, u8 *bssid, 
				int batch_mode)
{
	struct host_command cmd = {
		.host_command = MANDATORY_BSSID,
		.host_command_sequence = 0,
		.host_command_length = (bssid == NULL) ? 0 : ETH_ALEN 
	};
	int err;

#ifdef CONFIG_IPW_DEBUG
	if (bssid != NULL)
		IPW_DEBUG_FW_COMMAND(
			"MANDATORY_BSSID: %02X:%02X:%02X:%02X:%02X:%02X\n",
			bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], 
			bssid[5]);
	else
		IPW_DEBUG_FW_COMMAND("MANDATORY_BSSID: <clear>\n");
#endif
	/* if BSSID is empty then we disable mandatory bssid mode */
	if (bssid != NULL)
		memcpy((u8 *)cmd.host_command_parameters, bssid, ETH_ALEN);
	
	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) 
			return err;
	}

	err = ipw2100_hw_send_command(priv, &cmd);
		
	if (!batch_mode) 
		ipw2100_enable_adapter(priv);

	return err;
}

#ifdef CONFIG_IEEE80211_WPA    
static int ipw2100_disassociate_bssid(struct ipw2100_priv *priv)
{
	struct host_command cmd = {
		.host_command = DISASSOCIATION_BSSID,
		.host_command_sequence = 0,
		.host_command_length = ETH_ALEN
	};
	int err;
	int len;

	IPW_DEBUG_FW_COMMAND("DISASSOCIATION_BSSID\n");

	len = ETH_ALEN;
	/* The Firmware currently ignores the BSSID and just disassociates from
	 * the currently associated AP -- but in the off chance that a future
	 * firmware does use the BSSID provided here, we go ahead and try and
	 * set it to the currently associated AP's BSSID */
	ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID, 
			    &priv->ctx->bssid,  &len);
	memcpy(cmd.host_command_parameters, priv->ctx->bssid, ETH_ALEN);

	err = ipw2100_hw_send_command(priv, &cmd);

	return err;
}
#endif /* CONFIG_IEEE80211_WPA */

/*
 * Pseudo code for setting up wpa_frame: 
 */
#if 0
void x(struct ieee80211_assoc_frame *wpa_assoc)
{
	struct ipw2100_wpa_assoc_frame frame;
	frame->fixed_ie_mask = IPW_WPA_CAPABILTIES |
		IPW_WPA_LISTENINTERVAL |
		IPW_WPA_AP_ADDRESS;
	frame->capab_info = wpa_assoc->capab_info;
	frame->lisen_interval = wpa_assoc->listent_interval;
	memcpy(frame->current_ap, wpa_assoc->current_ap, ETH_ALEN);

	/* UNKNOWN -- I'm not postivive about this part; don't have any WPA
	 * setup here to test it with.
	 *
	 * Walk the IEs in the wpa_assoc and figure out the total size of all
	 * that data.  Stick that into frame->var_ie_len.  Then memcpy() all of
	 * the IEs from wpa_frame into frame.
	 */
	frame->var_ie_len = calculate_ie_len(wpa_assoc);
	memcpy(frame->var_ie,  wpa_assoc->variable, frame->var_ie_len);

	ipw2100_set_wpa_ie(priv, &frame, 0);
}
#endif 




static int ipw2100_set_wpa_ie(struct ipw2100_priv *, 
			      struct ipw2100_wpa_assoc_frame *, int)
__attribute__ ((unused));

static int ipw2100_set_wpa_ie(struct ipw2100_priv *priv, 
			      struct ipw2100_wpa_assoc_frame *wpa_frame,
			      int batch_mode) 
{
	struct host_command cmd = {
		.host_command = SET_WPA_IE,
		.host_command_sequence = 0,
		.host_command_length = sizeof(struct ipw2100_wpa_assoc_frame),
	};
	int err;

	IPW_DEBUG_FW_COMMAND("SET_WPA_IE\n");

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err)
			return err;
	}	

	memcpy(cmd.host_command_parameters, wpa_frame,
	       sizeof(struct ipw2100_wpa_assoc_frame));

	err = ipw2100_hw_send_command(priv, &cmd);

	if (!batch_mode) {
		if (ipw2100_enable_adapter(priv)) 
			err = -EIO;
	}

	return err;
}
		
int ipw2100_set_essid(struct ipw2100_priv *priv, char *essid, 
		      int length, int batch_mode)
{
	int ssid_len = min(length, IW_ESSID_MAX_SIZE);
	struct host_command cmd = {
		.host_command = SSID,
		.host_command_sequence = 0,
		.host_command_length = ssid_len
	};
	int err;

	IPW_DEBUG_FW_COMMAND("SSID: %s (%d)\n", essid, ssid_len);
	
	if (ssid_len) 
		memcpy((char*)cmd.host_command_parameters, 
		       essid, ssid_len);

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err)
			return err;
	}

	/* Bug in FW currently doesn't honor bit 0 in SET_SCAN_OPTIONS to
	 * disable auto association -- so we cheat by setting a bogus SSID */
	if (!ssid_len && !priv->auto_associate) {
		int i;
		u8 *bogus = (u8*)cmd.host_command_parameters;
		for (i = 0; i < IW_ESSID_MAX_SIZE; i++)
			bogus[i] = 0x18 + i;
		cmd.host_command_length = IW_ESSID_MAX_SIZE;
	}
		
	/* NOTE:  We always send the SSID command even if the provided ESSID is
	 * the same as what we currently think is set. */
	
	err = ipw2100_hw_send_command(priv, &cmd);
	if (!err) {
		memset(priv->ctx->ssid, 0, IW_ESSID_MAX_SIZE);
		memcpy(priv->ctx->ssid, essid, ssid_len);
		priv->ctx->ssid_len = ssid_len;
	}

	if (!batch_mode) {
		if (ipw2100_enable_adapter(priv)) 
			err = -EIO;
	}

	return err;
}

struct security_info_params {
	u32 allowed_ciphers;
	u16 version;
	u8 auth_mode;
	u8 replay_counters_number;
	u8 unicast_using_group;
} __attribute__ ((packed));

int ipw2100_set_security_information(struct ipw2100_priv *priv, 
				     int auth_mode, 
				     int security_level,
				     int unicast_using_group,
				     int batch_mode)
{
	struct host_command cmd = {
		.host_command = SET_SECURITY_INFORMATION,
		.host_command_sequence = 0,
		.host_command_length = sizeof(struct security_info_params)
	};
	struct security_info_params *security = 
		(struct security_info_params *)&cmd.host_command_parameters;
	int err;
	memset(security, 0, sizeof(*security));

	/* If shared key AP authentication is turned on, then we need to 
	 * configure the firmware to try and use it.
	 *
	 * Actual data encryption/decryption is handled by the host. */
	security->auth_mode = auth_mode;
	security->unicast_using_group = unicast_using_group;

	switch (security_level) {
	case SEC_LEVEL_0:
		security->allowed_ciphers = IPW_NONE_CIPHER;
		break;
	case SEC_LEVEL_1:
		security->allowed_ciphers = IPW_WEP40_CIPHER | 
			IPW_WEP104_CIPHER;
		break;
	case SEC_LEVEL_2:
		security->allowed_ciphers = IPW_WEP40_CIPHER | 
			IPW_WEP104_CIPHER | IPW_TKIP_CIPHER;
		break;
	case SEC_LEVEL_2_CKIP:
		security->allowed_ciphers = IPW_WEP40_CIPHER | 
			IPW_WEP104_CIPHER | IPW_CKIP_CIPHER;
		break;
	case SEC_LEVEL_3:
		security->allowed_ciphers = IPW_WEP40_CIPHER | 
			IPW_WEP104_CIPHER | IPW_TKIP_CIPHER | IPW_CCMP_CIPHER;
		break;
	}

	IPW_DEBUG_FW_COMMAND(
		"SET_SECURITY_INFORMATION: auth:%d cipher:0x%02X\n",
		security->auth_mode, security->allowed_ciphers);

	security->replay_counters_number = 0;

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err)
			return err;
	}

	err = ipw2100_hw_send_command(priv, &cmd);
		
	if (!batch_mode) 
		ipw2100_enable_adapter(priv);

	return err;
}

int ipw2100_set_tx_power(struct ipw2100_priv *priv,
			 u32 tx_power)
{
	struct host_command cmd = {
	  .host_command = TX_POWER_INDEX,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err = 0;

	cmd.host_command_parameters[0] = tx_power;

	IPW_DEBUG_INFO("enter\n");
	
	if (priv->port_type == IBSS) 
		err = ipw2100_hw_send_command(priv, &cmd);
	if (!err)
		priv->tx_power = tx_power;
	
	IPW_DEBUG_INFO("exit\n");
	
	return 0;
}

int ipw2100_set_ibss_beacon_interval(struct ipw2100_priv *priv, 
				     u32 interval, int batch_mode)
{
	struct host_command cmd = {
		.host_command = BEACON_INTERVAL,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = interval;

	IPW_DEBUG_INFO("enter\n");

	if (priv->port_type == IBSS) {
		if (!batch_mode) {
			err = ipw2100_disable_adapter(priv);
			if (err)
				return err;
		}
		
		ipw2100_hw_send_command(priv, &cmd);
		
		if (!batch_mode) {
			err = ipw2100_enable_adapter(priv);
			if (err)
				return err;
		}
	}

	IPW_DEBUG_INFO("exit\n");

	return 0;
}

int ipw2100_set_scan_options(struct ipw2100_priv *priv)
{
	struct host_command cmd = {
		.host_command = SET_SCAN_OPTIONS,
		.host_command_sequence = 0,
		.host_command_length = 8
	};
	int err;

	IPW_DEBUG_INFO("enter\n");

	IPW_DEBUG_SCAN("setting scan options\n");

	cmd.host_command_parameters[0] = 0;

	if (!priv->auto_associate)
		cmd.host_command_parameters[0] |= IPW_SCAN_NOASSOCIATE;
	if ((priv->sec.flags & SEC_ENABLED) && priv->sec.enabled)
		cmd.host_command_parameters[0] |= IPW_SCAN_MIXED_CELL;
	if (priv->passive_scan)
		cmd.host_command_parameters[0] |= IPW_SCAN_PASSIVE;

	cmd.host_command_parameters[1] = priv->channel_mask;
		
	err = ipw2100_hw_send_command(priv, &cmd);

	IPW_DEBUG_FW_COMMAND("SET_SCAN_OPTIONS 0x%04X\n", 
				 cmd.host_command_parameters[0]);

	return err;
}


void ipw2100_queues_initialize(struct ipw2100_priv *priv)
{
	ipw2100_tx_initialize(priv);
	ipw2100_rx_initialize(priv);
	ipw2100_msg_initialize(priv);
}

void ipw2100_queues_free(struct ipw2100_priv *priv)
{
	ipw2100_tx_free(priv);
	ipw2100_rx_free(priv);
	ipw2100_msg_free(priv);
}

int ipw2100_queues_allocate(struct ipw2100_priv *priv)
{
	if (ipw2100_tx_allocate(priv) ||
	    ipw2100_rx_allocate(priv) ||
	    ipw2100_msg_allocate(priv))
		goto fail;

	return 0;

 fail:
	ipw2100_tx_free(priv);
	ipw2100_rx_free(priv);
	ipw2100_msg_free(priv);
	return -ENOMEM;
}

#define IPW_PRIVACY_CAPABLE 0x0008

static int ipw2100_set_wep_flags(struct ipw2100_priv *priv, u32 flags, 
				 int batch_mode)
{
	struct host_command cmd = {
		.host_command = WEP_FLAGS,
		.host_command_sequence = 0,
		.host_command_length = 4
	};
	int err;

	cmd.host_command_parameters[0] = flags;

	IPW_DEBUG_FW_COMMAND("WEP_FLAGS: flags = 0x%08X\n", flags);
	
	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) {
			printk(KERN_ERR "%s: Could not disable adapter %d\n",
			       priv->ndev->name, err);
			return err;
		}
	}

	/* send cmd to firmware */
	err = ipw2100_hw_send_command(priv, &cmd);

	if (!batch_mode) 
		ipw2100_enable_adapter(priv);

	return err;
}

struct ipw2100_wep_key {
	u8 idx;
	u8 len;
	u8 key[13];
};

/* Macros to ease up priting WEP keys */
#define WEP_FMT_64  "%02X%02X%02X%02X-%02X"
#define WEP_FMT_128 "%02X%02X%02X%02X-%02X%02X%02X%02X-%02X%02X%02X"
#define WEP_STR_64(x) x[0],x[1],x[2],x[3],x[4]
#define WEP_STR_128(x) x[0],x[1],x[2],x[3],x[4],x[5],x[6],x[7],x[8],x[9],x[10]


/**
 * Set a the wep key
 *
 * @priv: struct to work on
 * @idx: index of the key we want to set
 * @key: ptr to the key data to set
 * @len: length of the buffer at @key
 * @batch_mode: FIXME perform the operation in batch mode, not
 *              disabling the device.
 *
 * @returns 0 if OK, < 0 errno code on error.
 *
 * Fill out a command structure with the new wep key, length an
 * index and send it down the wire.
 */
static int ipw2100_set_key(struct ipw2100_priv *priv, 
			   int idx, char *key, int len, int batch_mode)
{
	int keylen = len ? (len <= 5 ? 5 : 13) : 0;
	struct host_command cmd = {
		.host_command = WEP_KEY_INFO,
		.host_command_sequence = 0,
		.host_command_length = sizeof(struct ipw2100_wep_key),
	};
	struct ipw2100_wep_key *wep_key = (void*)cmd.host_command_parameters;
	int err;
	
	IPW_DEBUG_FW_COMMAND("WEP_KEY_INFO: index = %d, len = %d/%d\n",
				 idx, keylen, len);

	/* NOTE: We don't check cached values in case the firmware was reset
	 * or some other problem is occuring.  If the user is setting the key,
	 * then we push the change */
	
	wep_key->idx = idx;
	wep_key->len = keylen;

	if (keylen) {
		memcpy(wep_key->key, key, len);
		memset(wep_key->key + len, 0, keylen - len);
	}

	/* Will be optimized out on debug not being configured in */
	if (keylen == 0)
		IPW_DEBUG_WEP("%s: Clearing key %d\n",
				  priv->ndev->name, wep_key->idx);
	else if (keylen == 5)
		IPW_DEBUG_WEP("%s: idx: %d, len: %d key: " WEP_FMT_64 "\n",
				  priv->ndev->name, wep_key->idx, wep_key->len,
				  WEP_STR_64(wep_key->key));
	else 
		IPW_DEBUG_WEP("%s: idx: %d, len: %d key: " WEP_FMT_128 
				  "\n",
				  priv->ndev->name, wep_key->idx, wep_key->len,
				  WEP_STR_128(wep_key->key));

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		/* FIXME: IPG: shouldn't this prink be in _disable_adapter()? */
		if (err) {
			printk(KERN_ERR "%s: Could not disable adapter %d\n",
			       priv->ndev->name, err);
			return err;
		}
	}

	/* send cmd to firmware */
	err = ipw2100_hw_send_command(priv, &cmd);
	
	if (!batch_mode) {
		int err2 = ipw2100_enable_adapter(priv);
		if (err == 0)
			err = err2;
	}
	return err;
}

#if 0
static int ipw2100_set_key_index(struct ipw2100_priv *priv, 
				 int idx, int batch_mode)
{
	struct host_command cmd = {
		.host_command = WEP_KEY_INDEX,
		.host_command_sequence = 0,
		.host_command_length = 4,
		.host_command_parameters[0] = idx,
	};
	int err;

	IPW_DEBUG_FW_COMMAND("WEP_KEY_INDEX: index = %d\n", idx);
	
	if (idx < 0 || idx > 3)
		return -EINVAL;

	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err) {
			printk(KERN_ERR "%s: Could not disable adapter %d\n",
			       priv->ndev->name, err);
			return err;
		}
	}

	/* send cmd to firmware */
	err = ipw2100_hw_send_command(priv, &cmd);

	if (!batch_mode) 
		ipw2100_enable_adapter(priv);

	return err;
}
#endif


static int ipw2100_configure_security(struct ipw2100_priv *priv, 
				      int batch_mode)
{
	int i, err, auth_mode, sec_level, use_group;
	
	if (!batch_mode) {
		err = ipw2100_disable_adapter(priv);
		if (err)
			return err;
	}

	if (!priv->sec.enabled) {
		err = ipw2100_set_security_information(
			priv, IPW_AUTH_OPEN, SEC_LEVEL_0, 0, 1);
	} else {
		auth_mode = IPW_AUTH_OPEN;
		if ((priv->sec.flags & SEC_AUTH_MODE) &&
		    (priv->sec.auth_mode == WLAN_AUTH_SHARED_KEY)) 
			auth_mode = IPW_AUTH_SHARED;
		
		sec_level = SEC_LEVEL_0;
		if (priv->sec.flags & SEC_LEVEL) 
			sec_level = priv->sec.level;

		use_group = 0;
		if (priv->sec.flags & SEC_UNICAST_GROUP)
			use_group = priv->sec.unicast_uses_group;

		err = ipw2100_set_security_information(
			    priv, auth_mode, sec_level, use_group, 1);
	}

	if (err)
		goto exit;

	if (priv->sec.enabled) {
		for (i = 0; i < 4; i++) { 
			if (!(priv->sec.flags & (1 << i))) {
				memset(priv->sec.keys[i], 0, WEP_KEY_LEN);
				priv->sec.key_sizes[i] = 0;
			} else {
				err = ipw2100_set_key(priv, i, 
						      priv->sec.keys[i], 
						      priv->sec.key_sizes[i], 
						      1);
				if (err)
					goto exit;
			}
		}   
	}

	/* Always enable privacy so the Host can filter WEP packets if
	 * encrypted data is sent up */
	err = ipw2100_set_wep_flags(
		priv, priv->sec.enabled ? IPW_PRIVACY_CAPABLE : 0, 1);
	if (err)
		goto exit;
	
	priv->security_updated = 0;

 exit:
	if (!batch_mode)
		ipw2100_enable_adapter(priv);
	
	return err;
}
 
static void ipw2100_security_work(struct ipw2100_priv *priv)
{
	/* If we happen to have reconnected before we get a chance to
	 * process this, then update the security settings--which causes
	 * a disassociation to occur */
	if (!priv->connected && priv->security_updated)
		ipw2100_configure_security(priv, 0);
}

static void shim__set_security(struct ieee80211_device *ieee, 
			       struct ieee80211_security *sec)
{
	struct ipw2100_priv *priv = ieee->priv;
	int i;

	for (i = 0; i < 4; i++) { 
		if (sec->flags & (1 << i)) {
			priv->sec.key_sizes[i] = sec->key_sizes[i];
			if (sec->key_sizes[i] == 0)
				priv->sec.flags &= ~(1 << i);
			else
				memcpy(priv->sec.keys[i], sec->keys[i], 
				       sec->key_sizes[i]);
			priv->sec.flags |= (1 << i);
			priv->security_updated = 1;
		} 
	}

	if ((sec->flags & SEC_ACTIVE_KEY) &&
	    priv->sec.active_key != sec->active_key) {
		if (sec->active_key <= 3) {
			priv->sec.active_key = sec->active_key;
			priv->sec.flags |= SEC_ACTIVE_KEY;
		} else 
			priv->sec.flags &= ~SEC_ACTIVE_KEY;
			
		priv->security_updated = 1;
	}

	if ((sec->flags & SEC_AUTH_MODE) &&
	    (priv->sec.auth_mode != sec->auth_mode)) {
		priv->sec.auth_mode = sec->auth_mode;
		priv->sec.flags |= SEC_AUTH_MODE;
		priv->security_updated = 1;
	}
	
	if (sec->flags & SEC_ENABLED &&
	    priv->sec.enabled != sec->enabled) {
		priv->sec.flags |= SEC_ENABLED;
		priv->sec.enabled = sec->enabled;
		priv->security_updated = 1;
	}
	
	if (sec->flags & SEC_LEVEL &&
	    priv->sec.level != sec->level) {
		priv->sec.level = sec->level;
		priv->sec.flags |= SEC_LEVEL;
		priv->security_updated = 1;
	}

	IPW_DEBUG_WEP("Security flags: %c %c%c%c%c %c%c%c%c\n",
			  priv->sec.flags & BIT(8) ? '1' : '0',
			  priv->sec.flags & BIT(7) ? '1' : '0',
			  priv->sec.flags & BIT(6) ? '1' : '0',
			  priv->sec.flags & BIT(5) ? '1' : '0',
			  priv->sec.flags & BIT(4) ? '1' : '0',
			  priv->sec.flags & BIT(3) ? '1' : '0',
			  priv->sec.flags & BIT(2) ? '1' : '0',
			  priv->sec.flags & BIT(1) ? '1' : '0',
			  priv->sec.flags & BIT(0) ? '1' : '0');

	if (!priv->connected)
		ipw2100_configure_security(priv, 0);
}

static struct ieee80211_helper_functions ipw2100_ieee_callbacks = {
	.set_security = shim__set_security,
	.card_present = NULL,
	.cor_sreset = NULL,
	.dev_open = NULL,
	.dev_close = NULL,
	.genesis_reset = NULL,
	.set_unencrypted_filter = NULL,
	.hw_enable = NULL,
	.hw_config = NULL,
	.hw_reset = NULL,
	.hw_shutdown = NULL,
	.reset_port = NULL,
	.tx = NULL,
	.schedule_reset = NULL,
	.tx_80211 = NULL
};

static int ipw2100_adapter_setup(struct ipw2100_priv *priv)
{
	int err;
	int batch_mode = 1;
	u8 *bssid;

	IPW_DEBUG_INFO("enter\n");

	err = ipw2100_disable_adapter(priv);
	if (err)
		return err;
#ifdef CONFIG_IPW2100_PROMISC	
	if (priv->ctx->port_type == MONITOR) {
		err = ipw2100_set_channel(priv, priv->ctx->channel, batch_mode);
		if (err)
			return err;

		IPW_DEBUG_INFO("exit\n");

		return 0;
	}
#endif /* CONFIG_IPW2100_PROMISC */

	err = ipw2100_read_mac_address(priv);
	if (err)
		return -EIO;

	err = ipw2100_set_mac_address(priv, batch_mode);
	if (err)
		return err;
	
	err = ipw2100_set_port_type(priv, priv->ctx->port_type, batch_mode);
	if (err)
		return err;

	if (priv->ctx->port_type == IBSS) {
		err = ipw2100_set_channel(priv, priv->ctx->channel, batch_mode);
		if (err)
			return err;
	}
		
	err  = ipw2100_system_config(priv, batch_mode);
	if (err)
		return err;

	err = ipw2100_set_tx_rates(priv, priv->tx_rates, batch_mode);
	if (err)
		return err;

	/* Default to power mode OFF */
	err = ipw2100_set_power_mode(priv, IPW_POWER_MODE_CAM);
	if (err)
		return err;

	err = ipw2100_set_rts_threshold(priv, priv->rts_threshold);
	if (err)
		return err;

	if (priv->mandatory_bssid) 
		bssid = priv->mandatory_bssid_mac;
	else
		bssid = NULL;
	err = ipw2100_set_mandatory_bssid(priv, bssid, batch_mode);
	if (err)
		return err;

	err = ipw2100_set_essid(priv, priv->ctx->ssid, priv->ctx->ssid_len, 
				batch_mode);
	if (err)
		return err;

	err = ipw2100_configure_security(priv, batch_mode);
	if (err)
		return err;

	if (priv->ctx->port_type == IBSS) {
		err = ipw2100_set_ibss_beacon_interval(
			priv, priv->beacon_interval, batch_mode);
		if (err)
			return err;
		
		err = ipw2100_set_tx_power(priv, priv->tx_power);
		if (err) 
			return err; 
	}

	/*
	  err = ipw2100_set_fragmentation_threshold(
	  priv, priv->frag_threshold, batch_mode);
	  if (err) 
	  return err;
	*/
	
	IPW_DEBUG_INFO("exit\n");

	return 0;
}


/*************************************************************************
 * 
 * EXTERNALLY CALLED METHODS
 *
 *************************************************************************/

/* This method is called by the network layer -- not to be confused with
 * ipw2100_set_mac_address() declared above called by this driver (and this
 * method as well) to talk to the firmware */
static int ipw2100_set_address(struct net_device *dev, void *p)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct sockaddr *addr = p;
	int err;

	if (!is_valid_ether_addr(addr->sa_data))
		return -EADDRNOTAVAIL;

	priv->custom_addr = 1;
	memcpy(priv->mac_addr, addr->sa_data, ETH_ALEN);

	err = ipw2100_set_mac_address(priv, 0);
	if (err)
		return err;

	priv->reset_backoff = 0;
	ipw2100_reset_adapter(priv);

	return 0;
}

static int ipw2100_open(struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	unsigned long flags;

	IPW_DEBUG_INFO("enter\n");
	
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_INC_USE_COUNT;
#endif

	spin_lock_irqsave(&priv->low_lock, flags);
	priv->open = 1;
	spin_unlock_irqrestore(&priv->low_lock, flags);

	IPW_DEBUG_INFO("exit\n");
	return 0;
}

static int ipw2100_close(struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	unsigned long flags;
	struct list_head *element;
	struct ipw2100_tx_packet *packet;

	IPW_DEBUG_INFO("enter\n");

	spin_lock_irqsave(&priv->low_lock, flags);

	if (priv->connected) 
		netif_carrier_off(dev);


	priv->open = 0;
	
	/* Flush the TX queue ... */
	while (!list_empty(&priv->tx_pend_list)) {
		element = priv->tx_pend_list.next;
                packet = list_entry(element, struct ipw2100_tx_packet, list);

		list_del(element);
		DEC_STAT(&priv->tx_pend_stat);

		ieee80211_txb_free(packet->info.d_struct.txb);
		packet->info.d_struct.txb = NULL;

		list_add_tail(element, &priv->tx_free_list);
		INC_STAT(&priv->tx_free_stat);
	}
	spin_unlock_irqrestore(&priv->low_lock, flags);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT;
#endif

	IPW_DEBUG_INFO("exit\n");

	return 0;
}



/*
 * TODO:  Fix this function... its just wrong
 */
static void ipw2100_tx_timeout(struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	IPW_DEBUG_TX("enter (connected=%d)\n", priv->connected);

	priv->ieee->stats.tx_errors++;

#ifdef CONFIG_IPW2100_PROMISC
	// we do nothing as we should not be transmitting anyway
	if (priv->ctx->port_type == MONITOR) {
		IPW_DEBUG_TX("exit - monitor mode\n");
		return;
	}
#endif

	/* TODO: Make sure we shut off the network interface on 
	 * disassociation */
	printk(KERN_INFO "%s: TX timed out.  Scheduling firmware restart.\n",
	       dev->name);
	schedule_reset(priv);
	

	IPW_DEBUG_TX("exit\n");
}


/*
 * TODO: reimplement it so that it reads statistics
 *       from the adapter using ordinal tables
 *       instead of/in addition to collecting them
 *       in the driver
 */
static struct net_device_stats *ipw2100_stats(struct net_device *dev)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	
	return &priv->ieee->stats;
}

static int ipw2100_change_mtu(struct net_device *dev, int new_mtu)
{
	if (new_mtu < ETH_ZLEN || new_mtu > ETH_DATA_LEN)
		return -EINVAL;

	dev->mtu = new_mtu;
	return 0;
}

/* Support for wpa_supplicant. Will be replaced with WEXT once
 * they get WPA support. */
#ifdef CONFIG_IEEE80211_WPA    

/* following definitions must match definitions in driver_ipw2100.c */

#define IPW2100_IOCTL_WPA_SUPPLICANT		SIOCIWFIRSTPRIV+30

#define IPW2100_CMD_SET_WPA_PARAM		1
#define	IPW2100_CMD_SET_WPA_IE			2
#define IPW2100_CMD_SET_ENCRYPTION		3
#define IPW2100_CMD_MLME			4

#define IPW2100_PARAM_WPA_ENABLED		1
#define IPW2100_PARAM_TKIP_COUNTERMEASURES	2
#define IPW2100_PARAM_DROP_UNENCRYPTED		3
#define IPW2100_PARAM_PRIVACY_INVOKED		4
#define IPW2100_PARAM_AUTH_ALGS			5
#define IPW2100_PARAM_IEEE_802_1X		6

#define IPW2100_MLME_STA_DEAUTH			1
#define IPW2100_MLME_STA_DISASSOC		2

#define IPW2100_CRYPT_ERR_UNKNOWN_ALG		2
#define IPW2100_CRYPT_ERR_UNKNOWN_ADDR		3
#define IPW2100_CRYPT_ERR_CRYPT_INIT_FAILED	4
#define IPW2100_CRYPT_ERR_KEY_SET_FAILED	5
#define IPW2100_CRYPT_ERR_TX_KEY_SET_FAILED	6
#define IPW2100_CRYPT_ERR_CARD_CONF_FAILED	7

#define	IPW2100_CRYPT_ALG_NAME_LEN		16

struct ipw2100_param {
	u32 cmd;
	u8 sta_addr[ETH_ALEN];
        union {
		struct {
			u8 name;
			u32 value;
		} wpa_param;
		struct {
			u32 len;
			u8 *data;
		} wpa_ie;
	        struct{
			int command;
    			int reason_code;
		} mlme;
		struct {
			u8 alg[IPW2100_CRYPT_ALG_NAME_LEN];
			u8 set_tx;
			u32 err;
			u8 idx;
			u8 seq[8]; /* sequence counter (set: RX, get: TX) */
			u16 key_len;
			u8 key[0];
		} crypt;

	} u;
};

/* end of driver_ipw2100.c code */

static int ipw2100_wpa_enable(struct ipw2100_priv *priv, int value){

	struct ieee80211_device *ieee = priv->ieee;
	struct ieee80211_security sec = {
		.flags = SEC_LEVEL | SEC_ENABLED,
	};	
	int ret = 0;
    
	ieee->wpa_enabled = value;
	
	if (value){
		sec.level = SEC_LEVEL_3;
		sec.enabled = 1;
	} else {
		sec.level = SEC_LEVEL_0;
		sec.enabled = 0;
	}

	if (ieee->func && ieee->func->set_security)
		ieee->func->set_security(ieee, &sec);
	else
		ret = -EOPNOTSUPP;
		
	return ret;
}

#define AUTH_ALG_OPEN_SYSTEM			0x1
#define AUTH_ALG_SHARED_KEY			0x2

static int ipw2100_wpa_set_auth_algs(struct ipw2100_priv *priv, int value){

	struct ieee80211_device *ieee = priv->ieee;
	struct ieee80211_security sec = {
		.flags = SEC_AUTH_MODE,
	};	
	int ret = 0;
    	
	if (value & AUTH_ALG_SHARED_KEY){
		sec.auth_mode = WLAN_AUTH_SHARED_KEY;
		ieee->open_wep = 0;
	} else {
		sec.auth_mode = WLAN_AUTH_OPEN;
		ieee->open_wep = 1;
	}

	if (ieee->func && ieee->func->set_security)
		ieee->func->set_security(ieee, &sec);
	else
		ret = -EOPNOTSUPP;
		
	return ret;
}


static int ipw2100_wpa_set_param(struct net_device *dev, u8 name, u32 value){

	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int ret=0;

	switch(name){
		case IPW2100_PARAM_WPA_ENABLED:
			ret = ipw2100_wpa_enable(priv, value);
			break;

		case IPW2100_PARAM_TKIP_COUNTERMEASURES:
			priv->ieee->tkip_countermeasures=value;	    
			break;

		case IPW2100_PARAM_DROP_UNENCRYPTED:
			priv->ieee->drop_unencrypted=value;	    
			break;

		case IPW2100_PARAM_PRIVACY_INVOKED:
			priv->ieee->privacy_invoked=value;	    
			break;

		case IPW2100_PARAM_AUTH_ALGS:
			ret = ipw2100_wpa_set_auth_algs(priv, value);	    
			break;

		case IPW2100_PARAM_IEEE_802_1X:
			priv->ieee->pass_unencrypted_eapol=value;
			break;

		default:
			IPW_DEBUG_ERROR("%s: Unknown WPA param: %d\n", 
					    dev->name, name);
			ret = -EOPNOTSUPP;
	}

	return ret;
}

static int ipw2100_wpa_mlme(struct net_device *dev, int command, int reason){

	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int ret=0;

	switch(command){
		case IPW2100_MLME_STA_DEAUTH:
			// silently ignore
			break;

		case IPW2100_MLME_STA_DISASSOC:
			ipw2100_disassociate_bssid(priv);	    
			break;

		default:
			IPW_DEBUG_ERROR("%s: Unknown MLME request: %d\n", 
					    dev->name, command);
			ret = -EOPNOTSUPP;
	}

	return ret;
}


void ipw2100_wpa_assoc_frame(struct ipw2100_priv *priv, 
			     char *wpa_ie, int wpa_ie_len){

	struct ipw2100_wpa_assoc_frame frame;

	frame.fixed_ie_mask = 0;
	
	/* copy WPA IE */
	memcpy(frame.var_ie, wpa_ie, wpa_ie_len);
	frame.var_ie_len = wpa_ie_len;

	/* make sure WPA is enabled */
	ipw2100_wpa_enable(priv, 1);
	ipw2100_set_wpa_ie(priv, &frame, 0);
}


static int ipw2100_wpa_set_wpa_ie(struct net_device *dev, 
				struct ipw2100_param *param, int plen){

	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct ieee80211_device *ieee = priv->ieee;
	u8 *buf;
		
	if (! ieee->wpa_enabled)
	    return -EOPNOTSUPP;

	if (param->u.wpa_ie.len > MAX_WPA_IE_LEN || 
	   (param->u.wpa_ie.len && 
		param->u.wpa_ie.data==NULL))
		return -EINVAL;

	if (param->u.wpa_ie.len){
		buf = kmalloc(param->u.wpa_ie.len, GFP_KERNEL);
		if (buf == NULL)
			return -ENOMEM;

		memcpy(buf, param->u.wpa_ie.data, param->u.wpa_ie.len);

		kfree(ieee->wpa_ie);
		ieee->wpa_ie = buf;
		ieee->wpa_ie_len = param->u.wpa_ie.len;
		
	} else {
		kfree(ieee->wpa_ie);
		ieee->wpa_ie = NULL;
		ieee->wpa_ie_len = 0;
	}

	ipw2100_wpa_assoc_frame(priv, ieee->wpa_ie, ieee->wpa_ie_len);

	return 0;
}

/* implementation borrowed from hostap driver */

static int ipw2100_wpa_set_encryption(struct net_device *dev, 
				struct ipw2100_param *param, int param_len){

	int ret = 0;
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct ieee80211_device *ieee = priv->ieee;
	struct ieee80211_crypto_ops *ops;
	struct ieee80211_crypt_data **crypt;

	struct ieee80211_security sec = {
		.flags = 0,
	};	

	param->u.crypt.err = 0;
	param->u.crypt.alg[IPW2100_CRYPT_ALG_NAME_LEN - 1] = '\0';

	if (param_len !=
	    (int) ((char *) param->u.crypt.key - (char *) param) +
	    param->u.crypt.key_len){
		printk(KERN_DEBUG "Len mismatch %d, %d\n", param_len, param->u.crypt.key_len);
		return -EINVAL;
	}
	if (param->sta_addr[0] == 0xff && param->sta_addr[1] == 0xff &&
	    param->sta_addr[2] == 0xff && param->sta_addr[3] == 0xff &&
	    param->sta_addr[4] == 0xff && param->sta_addr[5] == 0xff) {
		if (param->u.crypt.idx >= WEP_KEYS)
			return -EINVAL;
		crypt = &ieee->crypt[param->u.crypt.idx];
	} else {
		return -EINVAL;
	}

	if (strcmp(param->u.crypt.alg, "none") == 0) {
		if (crypt){
			sec.enabled = 0;
			sec.level = SEC_LEVEL_0;
			sec.flags |= SEC_ENABLED | SEC_LEVEL;
			ieee80211_crypt_delayed_deinit(ieee, crypt);
		}
		goto done;
	}
	sec.enabled = 1;
	sec.flags |= SEC_ENABLED;		

	ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	if (ops == NULL && strcmp(param->u.crypt.alg, "WEP") == 0) {
		request_module("ieee80211_crypt_wep");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	} else if (ops == NULL && strcmp(param->u.crypt.alg, "TKIP") == 0) {
		request_module("ieee80211_crypt_tkip");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	} else if (ops == NULL && strcmp(param->u.crypt.alg, "CCMP") == 0) {
		request_module("ieee80211_crypt_ccmp");
		ops = ieee80211_get_crypto_ops(param->u.crypt.alg);
	}
	if (ops == NULL) {
		printk(KERN_DEBUG "%s: unknown crypto alg '%s'\n",
		       dev->name, param->u.crypt.alg);
		param->u.crypt.err = IPW2100_CRYPT_ERR_UNKNOWN_ALG;
		ret = -EINVAL;
		goto done;
	} 

	if (*crypt == NULL || (*crypt)->ops != ops) {
		struct ieee80211_crypt_data *new_crypt;

		ieee80211_crypt_delayed_deinit(ieee, crypt);

		new_crypt = (struct ieee80211_crypt_data *)
			kmalloc(sizeof(struct ieee80211_crypt_data), GFP_KERNEL);
		if (new_crypt == NULL) {
			ret = -ENOMEM;
			goto done;
		}
		memset(new_crypt, 0, sizeof(struct ieee80211_crypt_data));
		new_crypt->ops = ops;
		new_crypt->priv = new_crypt->ops->init(param->u.crypt.idx);
		if (new_crypt->priv == NULL) {
			kfree(new_crypt);
			param->u.crypt.err =
				IPW2100_CRYPT_ERR_CRYPT_INIT_FAILED;
			ret = -EINVAL;
			goto done;
		}

		*crypt = new_crypt;
	}

	if (param->u.crypt.key_len > 0 && (*crypt)->ops->set_key &&
	    (*crypt)->ops->set_key(param->u.crypt.key,
				   param->u.crypt.key_len, param->u.crypt.seq,
				   (*crypt)->priv) < 0) {
		printk(KERN_DEBUG "%s: key setting failed\n",
		       dev->name);
		param->u.crypt.err = IPW2100_CRYPT_ERR_KEY_SET_FAILED;
		ret = -EINVAL;
		goto done;
	}

	if (param->u.crypt.set_tx){
		ieee->tx_keyidx = param->u.crypt.idx;
		sec.active_key = param->u.crypt.idx;
		sec.flags |= SEC_ACTIVE_KEY;
	}

	if (ops->name != NULL){

		if (strcmp(ops->name, "WEP") == 0) {
			memcpy(sec.keys[param->u.crypt.idx], param->u.crypt.key, param->u.crypt.key_len);
			sec.key_sizes[param->u.crypt.idx] = param->u.crypt.key_len;
			sec.flags |= (1 << param->u.crypt.idx);
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_1;
		} else if (strcmp(ops->name, "TKIP") == 0) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_2;
		} else if (strcmp(ops->name, "CCMP") == 0) {
			sec.flags |= SEC_LEVEL;
			sec.level = SEC_LEVEL_3;
		}
	}
 done:
	if (ieee->func && ieee->func->set_security)
		ieee->func->set_security(ieee, &sec);

	/* Do not reset port if card is in Managed mode since resetting will
	 * generate new IEEE 802.11 authentication which may end up in looping
	 * with IEEE 802.1X.  If your hardware requires a reset after WEP
	 * configuration (for example... Prism2), implement the reset_port in 
	 * the callbacks structures used to initialize the 802.11 stack. */
	if (ieee->reset_on_keychange &&
	    ieee->iw_mode != IW_MODE_INFRA && 
	    ieee->func->reset_port &&
	    ieee->func->reset_port(dev)) {
		printk(KERN_DEBUG "%s: reset_port failed\n", dev->name);
		param->u.crypt.err = IPW2100_CRYPT_ERR_CARD_CONF_FAILED;
		return -EINVAL;
	}

	return ret;
}


static int ipw2100_wpa_supplicant(struct net_device *dev, struct iw_point *p){

	struct ipw2100_param *param;
	int ret=0;

	IPW_DEBUG_IOCTL("wpa_supplicant: len=%d\n", p->length);
	
	if (p->length < sizeof(struct ipw2100_param) || !p->pointer)
		return -EINVAL;

	param = (struct ipw2100_param *)kmalloc(p->length, GFP_KERNEL);
	if (param == NULL)
		return -ENOMEM;
	
	if (copy_from_user(param, p->pointer, p->length)){
		kfree(param);
		return -EFAULT;
	}

	switch (param->cmd){
	
	case IPW2100_CMD_SET_WPA_PARAM:
		ret = ipw2100_wpa_set_param(dev, param->u.wpa_param.name, 
					    param->u.wpa_param.value);
		break;

	case IPW2100_CMD_SET_WPA_IE:    
		ret = ipw2100_wpa_set_wpa_ie(dev, param, p->length);
		break;
	
	case IPW2100_CMD_SET_ENCRYPTION:    
		ret = ipw2100_wpa_set_encryption(dev, param, p->length);
		break;

	case IPW2100_CMD_MLME:    
		ret = ipw2100_wpa_mlme(dev, param->u.mlme.command,
				       param->u.mlme.reason_code);
		break;
	
	default:
		IPW_DEBUG_ERROR("%s: Unknown WPA supplicant request: %d\n", 
				dev->name, param->cmd);
		ret = -EOPNOTSUPP;
	    
	}

	if (ret == 0 && copy_to_user(p->pointer, param, p->length))
		ret = -EFAULT;

	kfree(param);
	return ret;
}
#endif /* CONFIG_IEEE80211_WPA */

static int ipw2100_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
#ifdef CONFIG_IEEE80211_WPA    
	struct iwreq *wrq = (struct iwreq *) rq;
	int ret=-1;
	switch (cmd){
	    case IPW2100_IOCTL_WPA_SUPPLICANT:
		ret = ipw2100_wpa_supplicant(dev, &wrq->u.data);
		return ret;

	    default:
		return -EOPNOTSUPP;
	}

#endif /* CONFIG_IEEE80211_WPA */
	return -EOPNOTSUPP;
}


static u32 ipw2100_ethtool_get_link(struct net_device *dev)
{
    struct ipw2100_priv *priv = ipw2100_netdev(dev);
    return priv->connected;
}


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,23)
static struct ethtool_ops ipw2100_ethtool_ops = {
    .get_link        = ipw2100_ethtool_get_link,
};
#endif

static void ipw2100_hang_check(unsigned long arg)
{
	struct ipw2100_priv *priv = (struct ipw2100_priv *)arg;
	unsigned long flags;
	u32 rtc = 0xa5a5a5a5;
	u32 len = sizeof(rtc);
	int restart = 0;

	spin_lock_irqsave(&priv->low_lock, flags);

	if (priv->fatal_error != 0) {
		/* If fatal_error is set then we need to restart */
		printk(KERN_INFO "%s: Hardware fatal error detected.\n",
		       priv->ndev->name);

		restart = 1;

	} else if (ipw2100_get_ordinal(priv, IPW_ORD_RTC_TIME, &rtc, &len) ||
		   (rtc == priv->last_rtc)) {
		/* Check if firmware is hung */
		printk(KERN_INFO "%s: Firmware RTC stalled.\n",
		       priv->ndev->name);
		restart = 1;
	}

	if (restart) {
		/* Kill timer */
		priv->stop_hang_check = 1;
		priv->hangs++;
		
		/* Restart the NIC */
		schedule_reset(priv);
	} 

	priv->last_rtc = rtc;

	/* Check again in two seconds */
	if (!priv->stop_hang_check)
		mod_timer(&priv->hang_check_timer, jiffies + 2 * HZ);

	spin_unlock_irqrestore(&priv->low_lock, flags);
}


static void ipw2100_rf_switch_check(unsigned long arg)
{
	struct ipw2100_priv *priv = (struct ipw2100_priv *)arg;
	unsigned long flags;

	spin_lock_irqsave(&priv->low_lock, flags);
	if (priv->rf_kill != ipw2100_get_rf_switch(priv)) {
		/* Toggle the switch to match what the HW told us */
		priv->rf_kill = !priv->rf_kill;
		
		printk(KERN_INFO "%s: RF Kill state changed to radio %s.\n",
		       priv->ndev->name,
		       priv->rf_kill ? "OFF" : "ON");
		
		if (!priv->rf_kill) {
                        /* Kill timer; we don't need to poll for RF going
			 * off since the firmware will tell us */
			priv->stop_rf_check = 1;
			
			/* Restart the NIC */
			schedule_reset(priv);
		} 

		/* In the event we detect that the RF kill switch turned
		 * on, we do nothing as the firmware will shutdown the
		 * radio for us, and then notify us about it */
	} 
	
	/* Check again in one second */
	if (!priv->stop_rf_check)
		mod_timer(&priv->rf_switch_timer, jiffies + HZ);

	spin_unlock_irqrestore(&priv->low_lock, flags);
}

static void ipw2100_irq_tasklet(struct ipw2100_priv *priv);

static struct net_device *ipw2100_alloc_device(
	struct pci_dev *pdev,
	char *base_addr, 
	unsigned long mem_start,
	unsigned long mem_len)
{
	struct ipw2100_priv *priv;
	struct net_device *dev;

	dev = alloc_etherdev(sizeof(struct ipw2100_priv));
	if (!dev)
		return NULL;

	dev->type = ARPHRD_ETHER;
	dev->open = ipw2100_open;
	dev->stop = ipw2100_close;
	dev->init = ipw2100_net_init;
	dev->hard_start_xmit = ipw2100_tx;
	dev->do_ioctl = ipw2100_ioctl;
	dev->get_stats = ipw2100_stats;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,23)
	dev->ethtool_ops = &ipw2100_ethtool_ops;
#endif
	dev->change_mtu = ipw2100_change_mtu;
	dev->tx_timeout = ipw2100_tx_timeout;
	dev->wireless_handlers = &ipw2100_wx_handler_def;
	dev->get_wireless_stats = ipw2100_wx_wireless_stats;
	dev->set_mac_address = ipw2100_set_address;
	dev->watchdog_timeo = 3*HZ;
	dev->irq = 0;

	dev->base_addr = (unsigned long)base_addr;
	dev->mem_start = mem_start;
	dev->mem_end = dev->mem_start + mem_len - 1;
	
	if (!if_name || if_name[0] == '\0')
                strncpy(dev->name, "eth%d", IFNAMSIZ - 1);
        else 
                strncpy(dev->name, if_name, IFNAMSIZ - 1);

	/* NOTE: We don't use the wireless_handlers hook
	 * in dev as the system will start throwing WX requests
	 * to us before we're actually initialized and it just
	 * ends up causing problems.  So, we just handle
	 * the WX extensions through the ipw2100_ioctl interface */

	priv = ipw2100_netdev(dev);
	memset(priv, 0, sizeof(struct ipw2100_priv));

	priv->ctx = &priv->active_ctx;
	priv->pdev = pdev;
	priv->ndev = dev;

	/* memset() puts everything to 0, so we only have explicitely set
	 * those values that need to be something else */
	priv->disabled = 1;
	priv->stopped = 1;

	priv->preamble = IPW_PREAMBLE_AUTO;

	/* If power management is turned on, default to AUTO mode */
	priv->power_mode = IPW_POWER_AUTO;



	/* Initialize IEEE80211 stack */
	priv->ieee = ieee80211_alloc(dev, priv);
	if (!priv->ieee) {
		printk(KERN_WARNING DRV_NAME 
		       ": Unable to allocate IEEE stack\n");
		free_netdev(dev);
		return NULL;
	}

	priv->ieee->func = &ipw2100_ieee_callbacks;
	priv->ieee->tx_payload_only = 1;

#ifdef CONFIG_IEEE80211_WPA
	priv->ieee->wpa_enabled = 0;
	priv->ieee->tkip_countermeasures = 0;
	priv->ieee->drop_unencrypted = 0;
	priv->ieee->privacy_invoked = 0;
	priv->ieee->pass_unencrypted_eapol = 0;
#endif /* CONFIG_IEEE80211_WPA */

	/* Set module parameters */
	switch (mode) {
	case 1:
		priv->ctx->port_type = IBSS;
		priv->ieee->iw_mode = IW_MODE_ADHOC;
		break;
#ifdef CONFIG_IPW2100_PROMISC	
	case 2:
		priv->ctx->port_type = MONITOR;
		priv->ieee->iw_mode = IW_MODE_MONITOR;
		break;
#endif
	default:
	case 0:
		priv->ctx->port_type = BSS;
		priv->ieee->iw_mode = IW_MODE_INFRA;
		break;
	}

	if (disable == 1)
		priv->manual_disable = 1;
	else
		priv->manual_disable = 0;

	if ((channel < REG_MIN_CHANNEL) ||
	    (channel > REG_MAX_CHANNEL)) 
		priv->ctx->channel = 1;
	else
		priv->ctx->channel = channel;
	
	priv->auto_associate = (associate == 0) ? 0 : 1;

	priv->beacon_interval = DEFAULT_BEACON_INTERVAL;
	priv->short_retry_limit = DEFAULT_SHORT_RETRY_LIMIT;
	priv->long_retry_limit = DEFAULT_LONG_RETRY_LIMIT;
	priv->rts_threshold = DEFAULT_RTS_THRESHOLD;

	priv->tx_power = IPW_TX_POWER_DEFAULT;
	priv->tx_rates = DEFAULT_TX_RATES;

	strcpy(priv->nick, "ipw2100");


	IPW_DEBUG_INFO("Port type set to %s\n", 
			   port_type_str[priv->ctx->port_type - 1]);

	spin_lock_init(&priv->low_lock);

	init_waitqueue_head(&priv->wait_command_queue);

	priv->connected = 0;
	netif_carrier_off(dev);

	INIT_LIST_HEAD(&priv->msg_free_list);
	INIT_LIST_HEAD(&priv->msg_pend_list);
	INIT_STAT(&priv->msg_free_stat);
	INIT_STAT(&priv->msg_pend_stat);

	INIT_LIST_HEAD(&priv->tx_free_list);
	INIT_LIST_HEAD(&priv->tx_pend_list);
	INIT_STAT(&priv->tx_free_stat);
	INIT_STAT(&priv->tx_pend_stat);

	INIT_LIST_HEAD(&priv->fw_pend_list);
	INIT_STAT(&priv->fw_pend_stat);
	

#ifdef CONFIG_SOFTWARE_SUSPEND2
	priv->workqueue = create_workqueue(DRV_NAME, 0);
#else
	priv->workqueue = create_workqueue(DRV_NAME);
#endif	
	INIT_WORK(&priv->reset_work, 
		  (void (*)(void *))ipw2100_reset_adapter, priv);
	INIT_WORK(&priv->security_work, 
		  (void (*)(void *))ipw2100_security_work, priv);
	INIT_WORK(&priv->wx_event_work,
		  (void (*)(void *))ipw2100_wx_event_work, priv);
	priv->wx_ap_event_pending = 0;
	
	tasklet_init(&priv->irq_tasklet, (void (*)(unsigned long))
		     ipw2100_irq_tasklet, (unsigned long)priv);

	/* NOTE:  We do not start the timer yet; we only run the timer if the
	 *        the RF kill switch has been triggered */
	init_timer(&priv->rf_switch_timer);
	priv->rf_switch_timer.data = (unsigned long)priv;
	priv->rf_switch_timer.function = ipw2100_rf_switch_check;
	priv->stop_rf_check = 1;   /* Timer not running */

	init_timer(&priv->hang_check_timer);
	priv->hang_check_timer.data = (unsigned long)priv;
	priv->hang_check_timer.function = ipw2100_hang_check;
	priv->stop_hang_check = 1; /* Timer not running */

	return dev;
}



#define PCI_DMA_32BIT   0x00000000ffffffffULL

static int ipw2100_pci_init_one(struct pci_dev *pdev, 
				const struct pci_device_id *ent)
{
	unsigned long mem_start, mem_len, mem_flags;
	char *base_addr = NULL;
	struct net_device *dev = NULL;
	struct ipw2100_priv *priv = NULL;
	int err = 0;
	int registered = 0;
	u32 val;

	IPW_DEBUG_INFO("enter\n");

	mem_start = pci_resource_start(pdev, 0);
	mem_len = pci_resource_len(pdev, 0);
	mem_flags = pci_resource_flags(pdev, 0);

	if ((mem_flags & IORESOURCE_MEM) != IORESOURCE_MEM) {
		IPW_DEBUG_INFO("weird - resource type is not memory\n");
		err = -ENODEV;
		goto fail;
	}

	base_addr = ioremap_nocache(mem_start, mem_len);
	if (!base_addr) {
		err = -EIO;
		goto fail;
	}

	/* allocate and initialize our net_device */
	dev = ipw2100_alloc_device(pdev, base_addr, mem_start, mem_len);
	if (!dev) {
		err = -ENOMEM;
		goto fail;
	}

	/* set up PCI mappings for device */
	err = pci_enable_device(pdev);
	if (err)
		return err;

	priv = ipw2100_netdev(dev);

	pci_set_master(pdev);
	pci_set_drvdata(pdev, priv);

	err = pci_set_dma_mask(pdev, PCI_DMA_32BIT);
	if (err) {
		IPW_DEBUG_INFO("failed pci_set_dma_mask!\n");
		pci_disable_device(pdev);
		return err;
	}

	err = pci_request_regions(pdev, DRV_NAME);
	if (err) {
		pci_disable_device(pdev);
		return err;
	}

        /* We disable the RETRY_TIMEOUT register (0x41) to keep 
	 * PCI Tx retries from interfering with C3 CPU state */
	pci_read_config_dword(pdev, 0x40, &val); 
	if ((val & 0x0000ff00) != 0) 
		pci_write_config_dword(pdev, 0x40, val & 0xffff00ff);

	pci_set_power_state(pdev, 0);
	pci_restore_state(pdev, NULL);

	if (!ipw2100_hw_is_adapter_in_system(dev)) {
		err = -ENODEV;
		goto fail;
	}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	SET_NETDEV_DEV(dev, &pdev->dev);
#endif

	/* begin initializing the hardware */
	ipw2100_hw_disable_interrupt(priv);

	/* Allocate and initialize the Tx/Rx queues and lists */
	if (ipw2100_queues_allocate(priv)) {
		err = -ENOMEM;
		goto fail;
	}
	ipw2100_queues_initialize(priv);

	err = request_irq(pdev->irq,
			  ipw2100_interrupt, SA_SHIRQ, 
			  dev->name, priv);
	if (err) {
		printk(KERN_ERR
		       "ipw2100: Error allocating IRQ %d.\n",
		       pdev->irq);
		err = -EBUSY;
		goto fail;
	}
	dev->irq = pdev->irq;

	IPW_DEBUG_INFO("Attempting to register device...\n");

	SET_MODULE_OWNER(dev);

	printk(KERN_INFO DRV_NAME 
	       ": %s: Detected at mem: 0x%lX-0x%lX -> %p, irq: %d\n",
	       pci_name(pdev), 
	       dev->mem_start, dev->mem_end, base_addr, pdev->irq);

	/* Bring up the interface.  Pre 0.46, after we registered the 
	 * network device we would call ipw2100_up.  This introduced a race
	 * condition with newer hotplug configurations (network was coming
	 * up and making calls before the device was initialized).
	 *
	 * If we called ipw2100_up before we registered the device, then the
	 * device name wasn't registered.  So, we instead use the net_dev->init
	 * member to call a function that then just turns and calls ipw2100_up.
	 * net_dev->init is called after name allocation but before the 
	 * notifier chain is called */
	err = register_netdev(dev);
	if (err) {
		err = -ENODEV;
		goto fail;
	}
	registered = 1;

	printk(KERN_INFO "%s: Bound to %s\n", dev->name, pci_name(pdev));

	/* perform this after register_netdev so that dev->name is set */
	sysfs_create_group(&pdev->dev.kobj, &ipw2100_attribute_group);
	netif_carrier_off(dev);
	
	/* If the RF Kill switch is disabled, go ahead and complete the
	 * startup sequence */
	if (!priv->rf_kill && !priv->manual_disable) {
		/* Enable the adapter - sends HOST_COMPLETE */
		if (ipw2100_enable_adapter(priv)) {
			printk(KERN_ERR 
			       "%s: failed in call to enable adapter.\n",
			       priv->ndev->name);
			err = -EIO;
			goto fail;
		}
		
		/* Start a scan . . . */
		ipw2100_set_scan_options(priv);
		ipw2100_start_scan(priv);
	}

	IPW_DEBUG_INFO("exit\n");

	return 0;
	
 fail:
	if (dev) {
		if (!priv->stop_rf_check) {
			priv->stop_rf_check = 1;
			del_timer_sync(&priv->rf_switch_timer);
		}

		if (!priv->stop_hang_check) {
			priv->stop_hang_check = 1;
			del_timer_sync(&priv->hang_check_timer);
		}

		if (registered) 
			unregister_netdev(dev);

		if (dev->irq) {
			free_irq(dev->irq, priv);
		}

		if (priv->workqueue) {
			destroy_workqueue(priv->workqueue);
			priv->workqueue = NULL;
		}

		/* These are safe to call even if they weren't allocated */
		ipw2100_queues_free(priv);
		sysfs_remove_group(&pdev->dev.kobj, &ipw2100_attribute_group);

		free_netdev(dev);
		pci_set_drvdata(pdev, NULL);
	}

	if (base_addr) 
		iounmap((char*)base_addr);

	pci_release_regions(pdev);
	pci_disable_device(pdev);

	return err;
}

static void __devexit ipw2100_pci_remove_one(struct pci_dev *pdev)
{
	struct ipw2100_priv *priv = pci_get_drvdata(pdev);
	struct net_device *dev;

	if (priv) {
		dev = priv->ndev;
		sysfs_remove_group(&pdev->dev.kobj, &ipw2100_attribute_group);

#ifdef CONFIG_PM
		if (ipw2100_firmware.version)
			ipw2100_release_firmware(priv, &ipw2100_firmware);
#endif
		/* Unregister the device first - this results in close() 
		 * being called if the device is open.  If we free storage
		 * first, then close() will crash. */
		unregister_netdev(dev);

		/* Take down the hardware */
		ipw2100_down(priv);

		/* ipw2100_down will ensure that there is no more pending work
		 * in the workqueue's, so we can safely remove them now. */
		destroy_workqueue(priv->workqueue);
		priv->workqueue = NULL;
		
		ieee80211_free(priv->ieee);
		priv->ieee = NULL;

		ipw2100_queues_free(priv);

		/* Free potential debugging firmware snapshot */
		ipw2100_snapshot_free(priv);

		if (dev->irq)
			free_irq(dev->irq, priv);

		if (dev->base_addr)
			iounmap((unsigned char *)dev->base_addr);

		free_netdev(dev);
	}

	pci_release_regions(pdev);
	pci_disable_device(pdev);

	IPW_DEBUG_INFO("exit\n");
}


#ifdef CONFIG_PM
static int ipw2100_suspend(struct pci_dev *pdev, u32 state)
{
	struct ipw2100_priv *priv = pci_get_drvdata(pdev);
	struct net_device *dev = priv->ndev;

	printk(KERN_INFO "%s: Going into suspend...\n",
	       dev->name);

 	/* Take down the device; powers it off, etc. */
	ipw2100_down(priv);

	/* Remove the PRESENT state of the device */
	netif_device_detach(dev);

	pci_save_state(pdev, priv->pm_state);
	pci_disable_device(pdev);       // needed?
	pci_set_power_state(pdev, state);
	
	return 0;
}

static int ipw2100_resume(struct pci_dev *pdev)
{
	struct ipw2100_priv *priv = pci_get_drvdata(pdev);
	struct net_device *dev = priv->ndev;
	u32 val;
	
	if (IPW2100_PM_DISABLED)
		return 0;
	
	printk(KERN_INFO "%s: Coming out of suspend...\n",
	       dev->name);

	pci_set_power_state(pdev, 0);
	pci_restore_state(pdev, priv->pm_state);

	/*
	 * Suspend/Resume resets the PCI configuration space, so we have to
	 * re-disable the RETRY_TIMEOUT register (0x41) to keep PCI Tx retries
	 * from interfering with C3 CPU state. pci_restore_state won't help
	 * here since it only restores the first 64 bytes pci config header.
	 */
	pci_read_config_dword(pdev, 0x40, &val); 
	if ((val & 0x0000ff00) != 0) 
		pci_write_config_dword(pdev, 0x40, val & 0xffff00ff);

	/* Set the device back into the PRESENT state; this will also wake
	 * the queue of needed */
	netif_device_attach(dev);

	/* Bring the device back up */
	ipw2100_up(priv, 0);
	
	return 0;
}
#endif


#define IPW2100_DEV_ID(x) { PCI_VENDOR_ID_INTEL, 0x1043, 0x8086, x }

static struct pci_device_id ipw2100_pci_id_table[] __devinitdata = {
	IPW2100_DEV_ID(0x2520), /* IN 2100A mPCI 3A */
	IPW2100_DEV_ID(0x2521), /* IN 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2524), /* IN 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2525), /* IN 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2526), /* IN 2100A mPCI Gen A3 */
	IPW2100_DEV_ID(0x2522), /* IN 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2523), /* IN 2100 mPCI 3A */
	IPW2100_DEV_ID(0x2527), /* IN 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2528), /* IN 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2529), /* IN 2100 mPCI 3B */
	IPW2100_DEV_ID(0x252B), /* IN 2100 mPCI 3A */
	IPW2100_DEV_ID(0x252C), /* IN 2100 mPCI 3A */
	IPW2100_DEV_ID(0x252D), /* IN 2100 mPCI 3A */

	IPW2100_DEV_ID(0x2550), /* IB 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2551), /* IB 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2553), /* IB 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2554), /* IB 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2555), /* IB 2100 mPCI 3B */

	IPW2100_DEV_ID(0x2560), /* DE 2100A mPCI 3A */
	IPW2100_DEV_ID(0x2562), /* DE 2100A mPCI 3A */
	IPW2100_DEV_ID(0x2563), /* DE 2100A mPCI 3A */
	IPW2100_DEV_ID(0x2561), /* DE 2100 mPCI 3A */
	IPW2100_DEV_ID(0x2565), /* DE 2100 mPCI 3A */
	IPW2100_DEV_ID(0x2566), /* DE 2100 mPCI 3A */
	IPW2100_DEV_ID(0x2567), /* DE 2100 mPCI 3A */

	IPW2100_DEV_ID(0x2570), /* GA 2100 mPCI 3B */

	IPW2100_DEV_ID(0x2580), /* TO 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2582), /* TO 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2583), /* TO 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2581), /* TO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2585), /* TO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2586), /* TO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2587), /* TO 2100 mPCI 3B */

	IPW2100_DEV_ID(0x2590), /* SO 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2592), /* SO 2100A mPCI 3B */
	IPW2100_DEV_ID(0x2591), /* SO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2593), /* SO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2596), /* SO 2100 mPCI 3B */
	IPW2100_DEV_ID(0x2598), /* SO 2100 mPCI 3B */
	
	IPW2100_DEV_ID(0x25A0), /* HP 2100 mPCI 3B */
	{0,},
};

MODULE_DEVICE_TABLE(pci, ipw2100_pci_id_table);

static struct pci_driver ipw2100_pci_driver = {
	.name = DRV_NAME,
	.id_table = ipw2100_pci_id_table,
	.probe = ipw2100_pci_init_one,
	.remove = __devexit_p(ipw2100_pci_remove_one),
#ifdef CONFIG_PM
	.suspend = ipw2100_suspend,
	.resume = ipw2100_resume,
#endif
};


/**
 * Initialize the ipw2100 driver/module
 *
 * @returns 0 if ok, < 0 errno node con error.
 *
 * Note: we cannot init the /proc stuff until the PCI driver is there,
 * or we risk an unlikely race condition on someone accessing
 * uninitialized data in the PCI dev struct through /proc.
 */
static int __init ipw2100_init(void)
{
	int ret;

	printk(KERN_INFO DRV_NAME ": %s, %s\n", DRV_DESCRIPTION, DRV_VERSION);
	printk(KERN_INFO DRV_NAME ": %s\n", DRV_COPYRIGHT);

#ifdef CONFIG_IEEE80211_NOWEP
	printk(KERN_INFO DRV_NAME ": Compiled with WEP disabled.\n");
#endif

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD
	printk(KERN_INFO DRV_NAME ": Compiled with LEGACY FW load.\n");
#endif

	ret = pci_module_init(&ipw2100_pci_driver);

	/* If debug module parameter declared, set debug_level to that */
	if (debug != -1)
		ipw2100_debug_level = debug;

#ifdef CONFIG_IPW_DEBUG
	driver_create_file(&ipw2100_pci_driver.driver,
			   &driver_attr_debug_level);
#endif

	return ret;
}


/**
 * Cleanup ipw2100 driver registration
 */
static void __exit ipw2100_exit(void)
{
	/* FIXME: IPG: check that we have no instances of the devices open */
#ifdef CONFIG_IPW_DEBUG
	driver_remove_file(&ipw2100_pci_driver.driver,
			   &driver_attr_debug_level);
#endif
	pci_unregister_driver(&ipw2100_pci_driver);
}

module_init(ipw2100_init);
module_exit(ipw2100_exit);

#define WEXT_USECHANNELS 1

const long ipw2100_frequencies[] = {  
	2412, 2417, 2422, 2427, 
	2432, 2437, 2442, 2447, 
	2452, 2457, 2462, 2467, 
	2472, 2484  
};

#define FREQ_COUNT (sizeof(ipw2100_frequencies) / \
                    sizeof(ipw2100_frequencies[0]))

const long ipw2100_rates_11b[] = {
	1000000,
	2000000,
	5500000,
	11000000
};

#define RATE_COUNT (sizeof(ipw2100_rates_11b) / sizeof(ipw2100_rates_11b[0]))

static int ipw2100_wx_get_name(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	strcpy(wrqu->name, "IEEE 802.11b");
	IPW_DEBUG_WX("Name: %s\n", wrqu->name);
	return 0;
}


static int ipw2100_wx_set_freq(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra) 
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct iw_freq *fwrq = &wrqu->freq;
	
	/* we only set channel in adhoc and promiscuous mode */
	switch (priv->ctx->port_type) {
#ifdef CONFIG_IPW2100_PROMISC
	case MONITOR:
#endif
	case IBSS:
		/* if setting by freq convert to channel */
		if (fwrq->e == 1) {
			if ((fwrq->m >= (int) 2.412e8 &&
			     fwrq->m <= (int) 2.487e8)) {
				int f = fwrq->m / 100000;
				int c = 0;
				
				while ((c < REG_MAX_CHANNEL) &&
				       (f != ipw2100_frequencies[c]))
					c++;
				
				/* hack to fall through */
				fwrq->e = 0;
				fwrq->m = c + 1;
			}
		}
		
		if (fwrq->e > 0 || fwrq->m > 1000) 
			return -EOPNOTSUPP;
		else { /* Set the channel */
			IPW_DEBUG_WX("SET Freq/Channel -> %d \n", fwrq->m);
			return ipw2100_set_channel(priv, fwrq->m, 0);
		}
		break;

	case BSS:
	default:
		return -EOPNOTSUPP;
		break;
	}

	return 0;
}


static int ipw2100_wx_get_freq(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct iw_freq *fwrq = &wrqu->freq;
	u32 chan;
	u32 len = sizeof(u32);
	int err;

	/* feed empty data if we are not associated */
#ifdef CONFIG_IPW2100_PROMISC
	if (priv->disabled || priv->rf_kill) {
#else
	if (priv->disabled || priv->rf_kill || priv->ctx->ssid[0] == '\0') {
#endif
		fwrq->m = 0;
		fwrq->e = 0;
		return 0;
	}

	err = ipw2100_get_ordinal(priv, IPW_ORD_OUR_FREQ, &chan, &len);
	if (err) {
		IPW_DEBUG_WX("failed querying ordinals.\n");
		return err;
	}

	/* we keep it */
	priv->ctx->channel = chan;

#ifdef WEXT_USECHANNELS
	/* return CHANNEL no if requested so */
	fwrq->m = chan;
	fwrq->e = 0;
#else
	/* otherwise return freq */
	fwrq->m = ipw2100_frequencies[chan - 1] * 100000;
	fwrq->e = 1;
#endif	/* WEXT_USECHANNELS */

	IPW_DEBUG_WX("GET Freq/Channel -> %d \n", fwrq->m);
	return 0;
}

static int ipw2100_wx_set_mode(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err;

	if (wrqu->mode == priv->ieee->iw_mode)
		return 0;

#ifdef CONFIG_IPW2100_PROMISC
	if (priv->ieee->iw_mode == IW_MODE_MONITOR)
		ipw2100_disable_monitor(priv);
#endif /* CONFIG_IPW2100_PROMISC */
	
	switch (wrqu->mode) {
#ifdef CONFIG_IPW2100_PROMISC
	case IW_MODE_MONITOR:
		err = ipw2100_enable_monitor(priv, priv->ctx->channel);
		break;
#endif /* CONFIG_IPW2100_PROMISC */
	case IW_MODE_ADHOC:
		err = ipw2100_switch_mode(priv, IBSS, priv->ctx->channel);
		break;
	default:
		err = ipw2100_switch_mode(priv, BSS, 0);
		break;
	}

	if (err)
		return err;

	priv->ieee->iw_mode = wrqu->mode;
	
	IPW_DEBUG_WX("SET Mode -> %s \n",
			 port_type_str[priv->ctx->port_type - 1]);

 	return 0;
}

static int ipw2100_wx_get_mode(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	switch (priv->ctx->port_type) {
	case IBSS:
		wrqu->mode = IW_MODE_ADHOC;
		break;
#ifdef CONFIG_IPW2100_PROMISC
	case MONITOR:
		wrqu->mode = IW_MODE_MONITOR;
		break;
#endif /* CONFIG_IPW2100_PROMISC */
	case BSS:
	default:
		wrqu->mode = IW_MODE_INFRA;
		break;
	}

	IPW_DEBUG_WX("GET Mode -> %s \n",
	       port_type_str[priv->ctx->port_type - 1]);

	return 0;
}


#define POWER_MODES 5

/* Values are in microsecond */
const s32 timeout_duration[POWER_MODES] = {
	350000,
	250000,
	75000,
	37000,
	25000,
};

const s32 period_duration[POWER_MODES] = {
	400000,
	700000,
	1000000,
	1000000,
	1000000
};

static int ipw2100_wx_get_range(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct iw_range *range = (struct iw_range *)extra;
	u16 val;
	int i;

	wrqu->data.length = sizeof(*range);
	memset(range, 0, sizeof(*range));

	/* Let's try to keep this struct in the same order as in
	 * linux/include/wireless.h
	 */
	
	/* TODO: See what values we can set, and remove the ones we can't
	 * set, or fill them with some default data.
	 */

	/* ~5 Mb/s real (802.11b) */
	range->throughput = 5 * 1000 * 1000;     

	// TODO: Not used in 802.11b?
//	range->min_nwid;	/* Minimal NWID we are able to set */
	// TOOD: Not used in 802.11b?
//	range->max_nwid;	/* Maximal NWID we are able to set */

//	range->old_num_channels;	/* Old Frequency (backward compat - moved lower ) */
//	range->old_num_frequency;	/* Old Frequency (backward compat - moved lower ) */
//	range->old_freq[6];		/* Filler to keep "version" at the same offset */

//	range->sensitivity;	/* signal level threshold range */
	
	range->max_qual.qual = 100;
	/* TODO: Find real max RSSI and stick here */
	range->max_qual.level = 0;
	range->max_qual.noise = IPW2100_RSSI_TO_DBM;
	range->max_qual.updated = 7; /* Updated all three */

	range->avg_qual.qual = 70; /* > 8% missed beacons is 'bad' */
	/* TODO: Find real 'good' to 'bad' threshol value for RSSI */
	range->avg_qual.level = 20 + IPW2100_RSSI_TO_DBM;
	range->avg_qual.noise = 0;
	range->avg_qual.updated = 7; /* Updated all three */

	range->num_bitrates = RATE_COUNT;

	for (i = 0; i < RATE_COUNT && i < IW_MAX_BITRATES; i++) {
		range->bitrate[i] = ipw2100_rates_11b[i];
	}
	
//	range->min_rts = 0;	// FIXME: What about min_rts? and max_rts=DEFAULT below?
	range->max_rts = DEFAULT_RTS_THRESHOLD;
	range->min_frag = MIN_FRAG_THRESHOLD;
	range->max_frag = MAX_FRAG_THRESHOLD;

	range->min_pmp = period_duration[0];	/* Minimal PM period */
	range->max_pmp = period_duration[POWER_MODES-1];/* Maximal PM period */
	range->min_pmt = timeout_duration[POWER_MODES-1];	/* Minimal PM timeout */
	range->max_pmt = timeout_duration[0];/* Maximal PM timeout */

        /* How to decode max/min PM period */
	range->pmp_flags = IW_POWER_PERIOD;
        /* How to decode max/min PM period */
	range->pmt_flags = IW_POWER_TIMEOUT;
	/* What PM options are supported */
	range->pm_capa = IW_POWER_TIMEOUT | IW_POWER_PERIOD;

	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;           /* Different token sizes */
	range->num_encoding_sizes = 2;		/* Number of entry in the list */
	range->max_encoding_tokens = WEP_KEYS;  /* Max number of tokens */
//	range->encoding_login_index;		/* token index for login token */

//	range->txpower_capa;		/* What options are supported */
//	range->num_txpower;		/* Number of entries in the list */
//	range->txpower[IW_MAX_TXPOWER];	/* list, in bps */
		
	/* Set the Wireless Extension versions */
	range->we_version_compiled = WIRELESS_EXT;
	range->we_version_source = 16;

//	range->retry_capa;	/* What retry options are supported */
//	range->retry_flags;	/* How to decode max/min retry limit */
//	range->r_time_flags;	/* How to decode max/min retry life */
//	range->min_retry;	/* Minimal number of retries */
//	range->max_retry;	/* Maximal number of retries */
//	range->min_r_time;	/* Minimal retry lifetime */
//	range->max_r_time;	/* Maximal retry lifetime */
											
        range->num_channels = FREQ_COUNT;

	val = 0;
	for (i = 0; i < FREQ_COUNT; i++) {
		// TODO: Include only legal frequencies for some countries
//		if (local->channel_mask & (1 << i)) {
			range->freq[val].i = i + 1;
			range->freq[val].m = ipw2100_frequencies[i] * 100000;
			range->freq[val].e = 1;
			val++;
//		}
		if (val == IW_MAX_FREQUENCIES)
		break;
	}
	range->num_frequency = val;

	IPW_DEBUG_WX("GET Range\n");

	return 0;
}

static int ipw2100_wx_set_wap(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err;

	static const unsigned char any[] = {
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff
	};
	static const unsigned char off[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	// sanity checks
	if (wrqu->ap_addr.sa_family != ARPHRD_ETHER) 
		return -EINVAL;

	if (!memcmp(any, wrqu->ap_addr.sa_data, ETH_ALEN) ||
	    !memcmp(off, wrqu->ap_addr.sa_data, ETH_ALEN)) {
		/* we disable mandatory BSSID association */
		IPW_DEBUG_WX("exit - disable mandatory BSSID\n");
		priv->mandatory_bssid = 0;
		return ipw2100_set_mandatory_bssid(priv, NULL, 0);
	}

	priv->mandatory_bssid = 1;
	memcpy(priv->mandatory_bssid_mac, wrqu->ap_addr.sa_data, ETH_ALEN);

	err = ipw2100_set_mandatory_bssid(
		priv, wrqu->ap_addr.sa_data, 0);
	if (err)
		return err;

	IPW_DEBUG_WX("SET BSSID -> %02X:%02X:%02X:%02X:%02X:%02X\n",
	       wrqu->ap_addr.sa_data[0] & 0xff,
	       wrqu->ap_addr.sa_data[1] & 0xff,
	       wrqu->ap_addr.sa_data[2] & 0xff,
	       wrqu->ap_addr.sa_data[3] & 0xff,
	       wrqu->ap_addr.sa_data[4] & 0xff, 
	       wrqu->ap_addr.sa_data[5] & 0xff);

	return 0;
}

static int ipw2100_wx_get_wap(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int len = ETH_ALEN;

	wrqu->ap_addr.sa_family = ARPHRD_ETHER;

	/* Fetch from the hardware */
	if (priv->disabled || priv->rf_kill || 
	    ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID, 
				&priv->ctx->bssid,  &len) ||
	    priv->ctx->ssid[0] == '\0') {
		memset(wrqu->ap_addr.sa_data, 0, ETH_ALEN);
		return 0;
	}

	memcpy(wrqu->ap_addr.sa_data, &priv->ctx->bssid, ETH_ALEN);
	memcpy(&priv->ieee->bssid, &priv->ctx->bssid, ETH_ALEN);

	IPW_DEBUG_WX("GET BSSID -> %02X:%02X:%02X:%02X:%02X:%02X\n",
			 (u8)wrqu->ap_addr.sa_data[0],
			 (u8)wrqu->ap_addr.sa_data[1],
			 (u8)wrqu->ap_addr.sa_data[2],
			 (u8)wrqu->ap_addr.sa_data[3],
			 (u8)wrqu->ap_addr.sa_data[4],
			 (u8)wrqu->ap_addr.sa_data[5]);

	return 0;
}

static int ipw2100_wx_set_essid(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	char *essid = ""; /* ANY */
	int length = 0;
	int err;
  
	IPW_DEBUG_WX("enter\n");
	if (wrqu->essid.flags && wrqu->essid.length) {
		length = wrqu->essid.length; // - 1;
		essid = extra;
	}
	
        if (priv->disabled) {
		IPW_DEBUG_WX("Caching ESSID: %s\n", essid);
		memcpy(priv->ctx->ssid, essid, length);
		priv->ctx->ssid_len = length;
                return 0;
        }
	
	err = ipw2100_set_essid(priv, essid, length, 0);
	if (err) {
		IPW_DEBUG_WX("SET SSID failed\n");
		return err;
	}
	
	IPW_DEBUG_WX("exit\n");
	
	return 0;
}

static int ipw2100_wx_get_essid(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	wrqu->essid.length = priv->ctx->ssid_len;
	memcpy(extra, priv->ctx->ssid, wrqu->essid.length);
	wrqu->essid.flags = 1; /* active */

	IPW_DEBUG_WX("GET ESSID -> %s (%d)\n", extra, priv->ctx->ssid_len);
	return 0;
}

static int ipw2100_wx_set_nick(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	if (wrqu->data.length > IW_ESSID_MAX_SIZE)
		return -E2BIG;

	wrqu->data.length = min((size_t)wrqu->data.length, sizeof(priv->nick));
	memset(priv->nick, 0, sizeof(priv->nick));
	memcpy(priv->nick, extra,  wrqu->data.length);

	IPW_DEBUG_WX("SET Nickname -> %s \n", priv->nick);

	return 0;
}

static int ipw2100_wx_get_nick(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	
	wrqu->data.length = strlen(priv->nick) + 1;
	memcpy(extra, priv->nick, wrqu->data.length);
	wrqu->data.flags = 1; /* active */
	
	IPW_DEBUG_WX("GET Nickname -> %s \n", extra);

	return 0;
}

static int ipw2100_wx_set_rate(struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	u32 target_rate = wrqu->bitrate.value;
	u32 rate;
	int err;
	
	rate = 0;

	if (target_rate == 1000000 || 
	    (!wrqu->bitrate.fixed && target_rate > 1000000))
		rate |= TX_RATE_1_MBIT;
	if (target_rate == 2000000 || 
	    (!wrqu->bitrate.fixed && target_rate > 2000000))
		rate |= TX_RATE_2_MBIT;
	if (target_rate == 5500000 || 
	    (!wrqu->bitrate.fixed && target_rate > 5500000))
		rate |= TX_RATE_5_5_MBIT;
	if (target_rate == 11000000 || 
	    (!wrqu->bitrate.fixed && target_rate > 11000000))
		rate |= TX_RATE_11_MBIT;
	if (rate == 0)
		rate = DEFAULT_TX_RATES;
	
	err = ipw2100_set_tx_rates(priv, rate, 0);
	if (err) {
		IPW_DEBUG_WX("failed setting tx rates.\n");
		return err;
	}
	
	IPW_DEBUG_WX("SET Rate -> %04X \n", rate);
	return 0;
}


static int ipw2100_wx_get_rate(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int val;
	int len = sizeof(val);
	int err;

	if (priv->disabled || priv->rf_kill || priv->ctx->ssid[0] == '\0') {
		wrqu->bitrate.value = 0;
		return 0;
	}

	err = ipw2100_get_ordinal(priv, IPW_ORD_CURRENT_TX_RATE, &val, &len);
	if (err) {
		IPW_DEBUG_WX("failed querying ordinals.\n");
		return err;
	}

	switch (val & TX_RATE_MASK) {
	case TX_RATE_1_MBIT:
		wrqu->bitrate.value = 1000000;
		break;
	case TX_RATE_2_MBIT:
		wrqu->bitrate.value = 2000000;
		break;
	case TX_RATE_5_5_MBIT:
		wrqu->bitrate.value = 5500000;
		break;
	case TX_RATE_11_MBIT:
		wrqu->bitrate.value = 11000000;
		break;
	default:
		wrqu->bitrate.value = 0;
	}

	IPW_DEBUG_WX("GET Rate -> %d \n", wrqu->bitrate.value);
	return 0;
}

static int ipw2100_wx_set_rts(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err;

	if (wrqu->rts.value < 1 || 
	    wrqu->rts.value > 2304 || 
	    wrqu->rts.disabled)
		return -EINVAL;

	err = ipw2100_set_rts_threshold(priv, wrqu->rts.value);
	if (err)
		return err;

	IPW_DEBUG_WX("SET RTS Threshold -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_get_rts(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int len = sizeof(wrqu->rts.value);
	int err;

	if (priv->disabled || priv->rf_kill)
		wrqu->rts.value = priv->rts_threshold;
	else {
		err = ipw2100_get_ordinal(priv, IPW_ORD_RTS_THRESHOLD, 
					  &wrqu->rts.value, &len);
		if (err) {
			IPW_DEBUG_WX("query ordinal failed.\n");
			return err;
		}
	}

	wrqu->rts.fixed = 1; /* no auto select */
	wrqu->rts.disabled = 0; /* can't be disabled */

	IPW_DEBUG_WX("GET RTS Threshold -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_set_txpow(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	IPW_DEBUG_WX("TODO: Power management by wireless extension...\n");

	IPW_DEBUG_WX("SET TX Power -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_get_txpow(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	wrqu->power.value = priv->tx_power;
	wrqu->power.fixed = 1;
	wrqu->power.flags = IW_TXPOW_DBM;
	wrqu->power.disabled = 0;

	IPW_DEBUG_WX("GET TX Power -> %d \n", wrqu->power.value);

	return 0;
}

static int ipw2100_wx_set_frag(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	if (wrqu->frag.disabled)
		priv->ieee->fts = DEFAULT_FRAG_THRESHOLD;
	else {
		if (wrqu->frag.value < MIN_FRAG_THRESHOLD ||
		    wrqu->frag.value > MAX_FRAG_THRESHOLD)
			return -EINVAL;
		
		priv->ieee->fts = wrqu->frag.value & ~0x1;
	}

	IPW_DEBUG_WX("SET Frag Threshold -> %d \n", 
	       priv->ieee->fts);

	return 0;
}

static int ipw2100_wx_get_frag(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	wrqu->frag.value = priv->ieee->fts;
	wrqu->frag.fixed = 0;	/* no auto select */
	wrqu->frag.disabled = 
		(wrqu->frag.value == DEFAULT_FRAG_THRESHOLD);

	IPW_DEBUG_WX("GET Frag Threshold -> %d \n", wrqu->frag.value);

	return 0;
}

static int ipw2100_wx_set_retry(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err;

	if (wrqu->retry.flags & IW_RETRY_LIFETIME || 
	    wrqu->retry.disabled)
		return -EINVAL;

	if (!wrqu->retry.flags & IW_RETRY_LIMIT)
		return 0;

	if (wrqu->retry.flags & IW_RETRY_MIN) {
		err = ipw2100_set_short_retry(priv, wrqu->retry.value);
		if (err)
			return err;

		IPW_DEBUG_WX("SET Short Retry Limit -> %d \n",
		       wrqu->retry.value);

		return 0;
	} 

	if (wrqu->retry.flags & IW_RETRY_MAX) {
		err = ipw2100_set_long_retry(priv, wrqu->retry.value);
		if (err)
			return err;

		IPW_DEBUG_WX("SET Long Retry Limit -> %d \n",
		       wrqu->retry.value);

		return 0;
	}

	err = ipw2100_set_short_retry(priv, wrqu->retry.value);
	if (err)
		return err;
	err = ipw2100_set_long_retry(priv, wrqu->retry.value);
	if (err)
		return err;

	IPW_DEBUG_WX("SET Both Retry Limits -> %d \n", wrqu->retry.value);

	return 0;
}

static int ipw2100_wx_get_retry(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	wrqu->retry.disabled = 0; /* can't be disabled */

	if ((wrqu->retry.flags & IW_RETRY_TYPE) == 
	    IW_RETRY_LIFETIME) 
		return -EINVAL;
	
	if (wrqu->retry.flags & IW_RETRY_MAX) {
		wrqu->retry.flags = IW_RETRY_LIMIT & IW_RETRY_MAX;
		wrqu->retry.value = priv->long_retry_limit;
	} else {
		wrqu->retry.flags =
		    (priv->short_retry_limit !=
		     priv->long_retry_limit) ?
		    IW_RETRY_LIMIT & IW_RETRY_MIN : IW_RETRY_LIMIT;

		wrqu->retry.value = priv->short_retry_limit;
	}

	IPW_DEBUG_WX("GET Retry -> %d \n", wrqu->retry.value);

	return 0;
}

static int ipw2100_wx_set_scan(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	IPW_DEBUG_WX("Initiating scan...\n");
	if (ipw2100_set_scan_options(priv) ||
	    ipw2100_start_scan(priv)) {
		IPW_DEBUG_WX("Start scan failed.\n");

		/* TODO: Mark a scan as pending so when hardware initialized
		 *       a scan starts */
	}

	return 0;
}

static int ipw2100_wx_get_scan(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	return ieee80211_wx_get_scan(priv->ieee, info, wrqu, extra);
}


/*
 * Implementation based on code in hostap-driver v0.1.3 hostap_ioctl.c
 */
static int ipw2100_wx_set_encode(struct net_device *dev, 
				 struct iw_request_info *info, 
				 union iwreq_data *wrqu, char *key)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	return ieee80211_wx_set_encode(priv->ieee, info, wrqu, key);
}

static int ipw2100_wx_get_encode(struct net_device *dev, 
				 struct iw_request_info *info, 
				 union iwreq_data *wrqu, char *key)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	return ieee80211_wx_get_encode(priv->ieee, info, wrqu, key);
}

static int ipw2100_wx_set_power(struct net_device *dev, 
			        struct iw_request_info *info, 
			        union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err;

	if (wrqu->power.disabled) {
		err = ipw2100_set_power_mode(priv, IPW_POWER_MODE_CAM);
		if (err) {
			IPW_DEBUG_WX("failed setting power mode.\n");
			return err;
		}

		IPW_DEBUG_WX("SET Power Management Mode -> off\n");

		return 0;
	} 

	switch (wrqu->power.flags & IW_POWER_MODE) {
	case IW_POWER_ON: break;
	/* not supported */
	case IW_POWER_UNICAST_R:
	case IW_POWER_MULTICAST_R:
	case IW_POWER_ALL_R:
	case IW_POWER_FORCE_S:
	case IW_POWER_REPEATER:
	default:
		return -EINVAL;
	}
	
	switch (wrqu->power.flags & IW_POWER_TYPE) {
	case IW_POWER_ON: break;
	/* fail (use private ioctl instead) */
	case IW_POWER_PERIOD:
	case IW_POWER_TIMEOUT:
	default:
		return -EINVAL;
	}
	
	err = ipw2100_set_power_mode(priv, priv->power_mode);
	if (err) {
		IPW_DEBUG_WX("failed setting power mode.\n");
		return err;
	}

	IPW_DEBUG_WX("SET Power Management Mode -> %d \n",
			 priv->power_mode);
	
	return 0;
}

static int ipw2100_wx_get_power(struct net_device *dev, 
			        struct iw_request_info *info, 
			        union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int mode = IPW_POWER_LEVEL(priv->power_mode);

	if (!(priv->power_mode & IPW_POWER_ENABLED)) {
		/* We are in IPW_POWER_MODE_CAM - Always On - No PSP */
		wrqu->power.disabled = 1;
	} else {
		wrqu->power.disabled = 0;
	
		if ((wrqu->power.flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT)
			wrqu->power.flags = IW_POWER_TIMEOUT;
		else
			wrqu->power.flags = IW_POWER_PERIOD;

		if (mode == IPW_POWER_AUTO) /* Auto */
			wrqu->power.flags = IW_POWER_ON;
		else {
			if (wrqu->power.flags == IW_POWER_TIMEOUT) 
				wrqu->power.value = timeout_duration[mode - 1];
			else 
				wrqu->power.value = period_duration[mode - 1];
		}
	}

	IPW_DEBUG_WX("GET Power Management Mode -> %d \n",
			 priv->power_mode);
	
	return 0;
}


/*
 *
 * IWPRIV handlers 
 *
 */
#ifdef CONFIG_IPW2100_PROMISC
static int ipw2100_wx_set_promisc(struct net_device *dev, 
				  struct iw_request_info *info, 
				  union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int *parms = (int *)extra;
	int enable = (parms[0] > 0);
	
	if (enable) {
	  if (priv->ieee->iw_mode == IW_MODE_MONITOR) {
	    ipw2100_set_channel(priv, parms[1], 0);
	    return 0;
	  }
		ipw2100_enable_monitor(priv, parms[1]);
	} else {
		if (priv->ieee->iw_mode != IW_MODE_MONITOR)
			return 0;
		ipw2100_disable_monitor(priv);
	}

	return 0;
}

static int ipw2100_wx_reset(struct net_device *dev, 
			    struct iw_request_info *info, 
			    union iwreq_data *wrqu, char *extra)
{
	ipw2100_reset_adapter(ipw2100_netdev(dev));
	return 0;
}

#endif


static int ipw2100_wx_set_powermode(struct net_device *dev, 
				    struct iw_request_info *info, 
				    union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err, mode = *(int *)extra;
	
	if ((mode < 1) || (mode > POWER_MODES)) 
		mode = IPW_POWER_AUTO;
	
	if (priv->power_mode != mode) {
		err = ipw2100_set_power_mode(priv, mode);
		if (err) {
			IPW_DEBUG_WX("failed setting power mode.\n");
			return err;
		}
	}
	
	return 0;
}

#define MAX_POWER_STRING 80
static int ipw2100_wx_get_powermode(struct net_device *dev, 
				    struct iw_request_info *info, 
				    union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int level = IPW_POWER_LEVEL(priv->power_mode);
	s32 timeout, period;
	
	if (!(priv->power_mode & IPW_POWER_ENABLED)) {
		snprintf(extra, MAX_POWER_STRING, 
			 "Power save level: %d (Off)", level);
	} else {
		switch (level) {
		case IPW_POWER_MODE_CAM:
			snprintf(extra, MAX_POWER_STRING, 
				 "Power save level: %d (None)", level);
			break;
		case IPW_POWER_AUTO:
		snprintf(extra, MAX_POWER_STRING, 
			 "Power save level: %d (Auto)", level);
			break;
		default:
			timeout = timeout_duration[level - 1] / 1000;
			period = period_duration[level - 1] / 1000;
			snprintf(extra, MAX_POWER_STRING, 
				 "Power save level: %d "
				 "(Timeout %dms, Period %dms)", 
				 level, timeout, period);
		}
	}

	wrqu->data.length = strlen(extra) + 1;

	return 0;
}


static int ipw2100_wx_set_preamble(struct net_device *dev, 
				   struct iw_request_info *info, 
				   union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	int err, mode = *(int *)extra;
	int preamble = priv->preamble;
	
	if (mode == 1)
		preamble = IPW_PREAMBLE_LONG;
	else if (mode == 0)
		preamble = IPW_PREAMBLE_AUTO;
	else 
		return -EINVAL;

	if (preamble != priv->preamble) {
		priv->preamble = preamble;
		err = ipw2100_system_config(priv, 0);
		if (err) {
			IPW_DEBUG_WX("failed setting preamble.\n");
			return err;
		}
	}

	return 0;
}

static int ipw2100_wx_get_preamble(struct net_device *dev, 
				    struct iw_request_info *info, 
				    union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = ipw2100_netdev(dev);

	if (priv->preamble == IPW_PREAMBLE_LONG) 
		snprintf(wrqu->name, IFNAMSIZ, "long (1)");
	else 
		snprintf(wrqu->name, IFNAMSIZ, "auto (0)");

	return 0;
}

static iw_handler ipw2100_wx_handlers[] =
{
        NULL,                     /* SIOCSIWCOMMIT */
        ipw2100_wx_get_name,      /* SIOCGIWNAME */
        NULL,                     /* SIOCSIWNWID */
        NULL,                     /* SIOCGIWNWID */
        ipw2100_wx_set_freq,      /* SIOCSIWFREQ */
        ipw2100_wx_get_freq,      /* SIOCGIWFREQ */
        ipw2100_wx_set_mode,      /* SIOCSIWMODE */
        ipw2100_wx_get_mode,      /* SIOCGIWMODE */
        NULL,                     /* SIOCSIWSENS */
        NULL,                     /* SIOCGIWSENS */
        NULL,                     /* SIOCSIWRANGE */
        ipw2100_wx_get_range,     /* SIOCGIWRANGE */
        NULL,                     /* SIOCSIWPRIV */
        NULL,                     /* SIOCGIWPRIV */
        NULL,                     /* SIOCSIWSTATS */
        NULL,                     /* SIOCGIWSTATS */
        NULL,                     /* SIOCSIWSPY */
        NULL,                     /* SIOCGIWSPY */
        NULL,                     /* SIOCGIWTHRSPY */
        NULL,                     /* SIOCWIWTHRSPY */
        ipw2100_wx_set_wap,       /* SIOCSIWAP */
        ipw2100_wx_get_wap,       /* SIOCGIWAP */
        NULL,                     /* -- hole -- */
        NULL,                     /* SIOCGIWAPLIST -- depricated */
        ipw2100_wx_set_scan,      /* SIOCSIWSCAN */
        ipw2100_wx_get_scan,      /* SIOCGIWSCAN */
        ipw2100_wx_set_essid,     /* SIOCSIWESSID */
        ipw2100_wx_get_essid,     /* SIOCGIWESSID */
        ipw2100_wx_set_nick,      /* SIOCSIWNICKN */
        ipw2100_wx_get_nick,      /* SIOCGIWNICKN */
        NULL,                     /* -- hole -- */
        NULL,                     /* -- hole -- */
        ipw2100_wx_set_rate,      /* SIOCSIWRATE */
        ipw2100_wx_get_rate,      /* SIOCGIWRATE */
        ipw2100_wx_set_rts,       /* SIOCSIWRTS */
        ipw2100_wx_get_rts,       /* SIOCGIWRTS */
        ipw2100_wx_set_frag,      /* SIOCSIWFRAG */
        ipw2100_wx_get_frag,      /* SIOCGIWFRAG */
        ipw2100_wx_set_txpow,     /* SIOCSIWTXPOW */
        ipw2100_wx_get_txpow,     /* SIOCGIWTXPOW */
        ipw2100_wx_set_retry,     /* SIOCSIWRETRY */
        ipw2100_wx_get_retry,     /* SIOCGIWRETRY */
        ipw2100_wx_set_encode,    /* SIOCSIWENCODE */
        ipw2100_wx_get_encode,    /* SIOCGIWENCODE */
        ipw2100_wx_set_power,     /* SIOCSIWPOWER */
        ipw2100_wx_get_power,     /* SIOCGIWPOWER */
};

#define IPW2100_PRIV_SET_PROMISC	SIOCIWFIRSTPRIV 
#define IPW2100_PRIV_RESET		SIOCIWFIRSTPRIV+1
#define IPW2100_PRIV_SET_POWER		SIOCIWFIRSTPRIV+2
#define IPW2100_PRIV_GET_POWER		SIOCIWFIRSTPRIV+3
#define IPW2100_PRIV_SET_LONGPREAMBLE	SIOCIWFIRSTPRIV+4
#define IPW2100_PRIV_GET_LONGPREAMBLE	SIOCIWFIRSTPRIV+6

static const struct iw_priv_args ipw2100_private_args[] = { 

#ifdef CONFIG_IPW2100_PROMISC
	{
		IPW2100_PRIV_SET_PROMISC, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "monitor" 
	}, 
	{
		IPW2100_PRIV_RESET, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 0, 0, "reset" 
	},
#endif /* CONFIG_IPW2100_PROMISC */

	{
		IPW2100_PRIV_SET_POWER, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_power"
	},
	{
		IPW2100_PRIV_GET_POWER, 
		0, IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | MAX_POWER_STRING, "get_power" 
	},
	{
		IPW2100_PRIV_SET_LONGPREAMBLE, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_preamble" 
	},
	{
		IPW2100_PRIV_GET_LONGPREAMBLE, 
		0, IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | IFNAMSIZ, "get_preamble" 
	},

};

static iw_handler ipw2100_private_handler[] = {
#ifdef CONFIG_IPW2100_PROMISC
	ipw2100_wx_set_promisc,
	ipw2100_wx_reset, 
#else /* CONFIG_IPW2100_PROMISC */
	NULL,
	NULL,
#endif /* CONFIG_IPW2100_PROMISC */
	ipw2100_wx_set_powermode, 
	ipw2100_wx_get_powermode, 
	ipw2100_wx_set_preamble, 
	ipw2100_wx_get_preamble,
};

struct iw_handler_def ipw2100_wx_handler_def = 
{
	.standard = ipw2100_wx_handlers,
	.num_standard = sizeof(ipw2100_wx_handlers) / sizeof(iw_handler),
	.num_private = sizeof(ipw2100_private_handler) / sizeof(iw_handler),
 	.num_private_args = sizeof(ipw2100_private_args) /
	sizeof(struct iw_priv_args),
	.private = (iw_handler *)ipw2100_private_handler,
	.private_args = (struct iw_priv_args *)ipw2100_private_args,	
};

/*
 * Get wireless statistics.
 * Called by /proc/net/wireless
 * Also called by SIOCGIWSTATS
 */
struct iw_statistics *ipw2100_wx_wireless_stats(struct net_device * dev)
{
	enum {
		POOR = 30,
		FAIR = 60,
		GOOD = 80,
		VERY_GOOD = 90,
		EXCELLENT = 95,
		PERFECT = 100
	};
	int rssi_qual;
	int tx_qual;
	int beacon_qual;

	struct ipw2100_priv *priv = ipw2100_netdev(dev);
	struct iw_statistics *wstats;
	u32 rssi, quality, tx_retries, missed_beacons, tx_failures;
	u32 ord_len = sizeof(u32);

	if (!priv)
		return (struct iw_statistics *) NULL;

	/* if hw is disabled, then ipw2100_get_ordinal() can't be called.
	 * ipw2100_wx_wireless_stats seems to be called before fw is 
	 * initialized. */
	if (priv->disabled)
		return (struct iw_statistics *) NULL;

	wstats = &priv->wstats;

	if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_PERCENT_MISSED_BCNS,
				&missed_beacons, &ord_len))
		goto fail_get_ordinal;

        /* If we don't have a connection the quality and level is 0*/
	if (!priv->connected) {
		wstats->qual.qual = 0; 
		wstats->qual.level = 0;
	} else {
		if (ipw2100_get_ordinal(priv, IPW_ORD_RSSI_AVG_CURR, 
					&rssi, &ord_len))
			goto fail_get_ordinal;
		wstats->qual.level = rssi + IPW2100_RSSI_TO_DBM;
		if (rssi < 10) 
			rssi_qual = rssi * POOR / 10;
		else if (rssi < 15)
			rssi_qual = (rssi - 10) * (FAIR - POOR) / 5 + POOR;
		else if (rssi < 20)
			rssi_qual = (rssi - 15) * (GOOD - FAIR) / 5 + FAIR;
		else if (rssi < 30)
			rssi_qual = (rssi - 20) * (VERY_GOOD - GOOD) / 
				10 + GOOD;
		else
			rssi_qual = (rssi - 30) * (PERFECT - VERY_GOOD) / 
				10 + VERY_GOOD;
		
		if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_PERCENT_RETRIES,
					&tx_retries, &ord_len))
			goto fail_get_ordinal;
		
		if (tx_retries > 75) 
			tx_qual = (90 - tx_retries) * POOR / 15;
		else if (tx_retries > 70)
			tx_qual = (75 - tx_retries) * (FAIR - POOR) / 5 + POOR;
		else if (tx_retries > 65)
			tx_qual = (70 - tx_retries) * (GOOD - FAIR) / 5 + FAIR;
		else if (tx_retries > 50)
			tx_qual = (65 - tx_retries) * (VERY_GOOD - GOOD) / 
				15 + GOOD;
		else
			tx_qual = (50 - tx_retries) * 
				(PERFECT - VERY_GOOD) / 50 + VERY_GOOD;
		
		if (missed_beacons > 50) 
			beacon_qual = (60 - missed_beacons) * POOR / 10;
		else if (missed_beacons > 40)
			beacon_qual = (50 - missed_beacons) * (FAIR - POOR) / 
				10 + POOR;
		else if (missed_beacons > 32)
			beacon_qual = (40 - missed_beacons) * (GOOD - FAIR) / 
				18 + FAIR;
		else if (missed_beacons > 20)
			beacon_qual = (32 - missed_beacons) * 
				(VERY_GOOD - GOOD) / 20 + GOOD;
		else
			beacon_qual = (20 - missed_beacons) * 
				(PERFECT - VERY_GOOD) / 20 + VERY_GOOD;
		
		quality = min(beacon_qual, min(tx_qual, rssi_qual));

#ifdef CONFIG_IPW_DEBUG
		if (beacon_qual == quality)
			IPW_DEBUG_WX("Quality clamped by Missed Beacons\n");
		else if (tx_qual == quality)
			IPW_DEBUG_WX("Quality clamped by Tx Retries\n");
		else if (quality != 100)
			IPW_DEBUG_WX("Quality clamped by Signal Strength\n");
		else
			IPW_DEBUG_WX("Quality not clamped.\n");
#endif

		wstats->qual.qual = quality;
		wstats->qual.level = rssi + IPW2100_RSSI_TO_DBM;
	}
	
	wstats->qual.noise = IPW2100_RSSI_TO_DBM;
	wstats->qual.updated = 7;	

        /* FIXME: this is percent and not a # */
	wstats->miss.beacon = missed_beacons; 

	if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_TX_FAILURES, 
				&tx_failures, &ord_len))
		goto fail_get_ordinal;
	wstats->discard.retries = tx_failures;

	return wstats;

 fail_get_ordinal:
	IPW_DEBUG_WX("failed querying ordinals.\n");

	return (struct iw_statistics *) NULL;
}

void ipw2100_wx_event_work(struct ipw2100_priv *priv)
{
      union iwreq_data wrqu;
      int len = ETH_ALEN;

      IPW_DEBUG_WX("enter\n");
      if (priv->wx_ap_event_pending) {
              priv->wx_ap_event_pending = 0;

	      /* This is a temprorary fix to support this within
	       * 2.4; ideally we would just use schedule_delayed_work 
	       * instead of schedule_work and then we wouldn't have to
	       * sleep here */
	      set_current_state(TASK_UNINTERRUPTIBLE);
	      schedule_timeout(2);

              wrqu.ap_addr.sa_family = ARPHRD_ETHER;

              /* Fetch BSSID from the hardware */
              if (priv->disabled || priv->rf_kill ||
                  ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID,
                                      &priv->ctx->bssid,  &len)) {
		    memset(wrqu.ap_addr.sa_data, 0, ETH_ALEN);
              } else {
		    memcpy(wrqu.ap_addr.sa_data, priv->ctx->bssid, ETH_ALEN);
		    memcpy(&priv->ieee->bssid, priv->ctx->bssid, ETH_ALEN);
              }
              wireless_send_event(priv->ndev, SIOCGIWAP, &wrqu, NULL);
      }
}

#define IPW2100_FW_MAJOR_VERSION 1
#define IPW2100_FW_MINOR_VERSION 3

#define IPW2100_FW_MINOR(x) ((x & 0xff) >> 8)
#define IPW2100_FW_MAJOR(x) (x & 0xff)

#define IPW2100_FW_VERSION ((IPW2100_FW_MINOR_VERSION << 8) | \
                             IPW2100_FW_MAJOR_VERSION)

#define IPW2100_FW_PREFIX "ipw2100-" __stringify(IPW2100_FW_MAJOR_VERSION) \
"." __stringify(IPW2100_FW_MINOR_VERSION)

#define IPW2100_FW_NAME(x) IPW2100_FW_PREFIX "" x ".fw"

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD

static char *firmware = NULL;

/* Module paramter for path to the firmware*/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

MODULE_PARM(firmware, "s");

#else /* LINUX_VERSION_CODE < 2.6.0 */

#include <linux/moduleparam.h>
module_param(firmware, charp, 0);

#endif /* LINUX_VERSION_CODE < 2.6.0 */

MODULE_PARM_DESC(firmware, "complete path to firmware file");

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */


/*

BINARY FIRMWARE HEADER FORMAT

offset      length   desc
0           2        version
2           2        mode == 0:BSS,1:IBSS,2:PROMISC
4           4        fw_len
8           4        uc_len
C           fw_len   firmware data
12 + fw_len uc_len   microcode data

*/

struct ipw2100_fw_header {
	short version;
	short mode;
	unsigned int fw_size;
	unsigned int uc_size;
} __attribute__ ((packed));


#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD

/*
 * 
 * The following was originally based on the mod_firmware_load in
 * drivers/sound/sound_firmware.c.  Primary changes revolved around 
 * making it work for firmware images > 128k and to support having
 * both a firmware and microcode image in the file loaded.
 *
 */
static void ipw2100_fw_free(struct ipw2100_fw *fw)
{
	struct ipw2100_fw_chunk *c;
	struct list_head *e;

	while (!list_empty(&fw->fw.chunk_list)) {
		e = fw->fw.chunk_list.next;
		c = list_entry(e, struct ipw2100_fw_chunk, list);
		list_del(e);
		vfree(c->buf);
		vfree(c);
	}

	while (!list_empty(&fw->uc.chunk_list)) {
		e = fw->uc.chunk_list.next;
		c = list_entry(e, struct ipw2100_fw_chunk, list);
		list_del(e);
		vfree(c->buf);
		vfree(c);
	}
}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static int ipw2100_fw_load(int fd, struct ipw2100_fw_chunk_set *cs, long size)
#else
static int ipw2100_fw_load(struct file *filp, struct ipw2100_fw_chunk_set *cs, long size)
#endif
{
	struct ipw2100_fw_chunk *c;
	int i = 0;
	
	/* Break firmware image into chunks of 128k */
	cs->size = size;
	cs->chunks = cs->size >> 17;
	
	if (cs->size & 0x1FFFF)
		cs->chunks++;
	
	IPW_DEBUG_FW("Loading %ld bytes from %u chunks\n",
			 cs->size, cs->chunks);
	
	/* Load the chunks */
	while (size > 0) {
		i++;
		
		c = (struct ipw2100_fw_chunk *)vmalloc(
			sizeof(struct ipw2100_fw_chunk));
		if (c == NULL) {
			printk(KERN_INFO "Out of memory loading firmware "
			       "chunk %d.\n", i);
			goto fail;
		}
		c->pos = 0;
		
		if (size >= 0x20000)
			c->len = 0x20000;
		else
			c->len = size;
		
		c->buf = (unsigned char *)vmalloc(c->len);
		if (c->buf == NULL) {
			printk(KERN_INFO "Out of memory loading firmware "
			       "chunk %d.\n", i);
			goto fail;
			
		}
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		if (read(fd, c->buf, c->len) != c->len) {
#else
		if (vfs_read(filp, c->buf, c->len, &filp->f_pos) != c->len) {
#endif
			printk(KERN_INFO "Failed to read chunk firmware "
			       "chunk %d.\n", i);
			goto fail;
		}
		
		list_add_tail(&c->list, &cs->chunk_list);
		
		IPW_DEBUG_FW("Chunk %d loaded: %lu bytes\n",
				 i, c->len);
		size -= c->len;
	}
	
	return 0;
	
 fail:
	return 1;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

static int errno;
static int ipw2100_do_mod_firmware_load(const char *fn, struct ipw2100_fw *fw)
{
	int fd;
	long l;
	struct ipw2100_fw_header h;

	/* Make sure the lists are init'd so that error paths can safely walk
        * them to free potentially allocated storage */
	INIT_LIST_HEAD(&fw->fw.chunk_list);
	INIT_LIST_HEAD(&fw->uc.chunk_list);
	
	fd = open(fn, 0, 0);
	if (fd == -1) {
		printk(KERN_INFO "Unable to load '%s'.\n", fn);
		return 1;
	}
	l = lseek(fd, 0L, 2);
	lseek(fd, 0L, 0);
	
	IPW_DEBUG_FW("Loading %ld bytes for firmware '%s'\n", l, fn);
	
	if (read(fd, (char *)&h, sizeof(h)) != sizeof(h)) {
		printk(KERN_INFO "Failed to read '%s'.\n", fn);
		goto fail;
	}

	if (IPW2100_FW_MAJOR(h.version) != IPW2100_FW_MAJOR_VERSION) {
		printk(KERN_WARNING "Firmware image not compatible "
		       "(detected version id of %u). "
		       "See Documentation/networking/README.ipw2100\n",
		       h.version);
		goto fail;
	}

	fw->version = h.version;

	if (ipw2100_fw_load(fd, &fw->fw, h.fw_size))
		goto fail;

	if (ipw2100_fw_load(fd, &fw->uc, h.uc_size))
		goto fail;

	close(fd);
	return 0;

 fail:
	ipw2100_fw_free(fw);
	close(fd);
	return 1;
}

#else

static int ipw2100_do_mod_firmware_load(const char *fn, struct ipw2100_fw *fw)
{
	struct file *filp;
	long l;
	struct ipw2100_fw_header h;

	/* Make sure the lists are init'd so that error paths can safely walk
        * them to free potentially allocated storage */
	INIT_LIST_HEAD(&fw->fw.chunk_list);
	INIT_LIST_HEAD(&fw->uc.chunk_list);
	
	filp = filp_open(fn, 0, 0);
	if (IS_ERR(filp)) {
		printk(KERN_INFO "Unable to load '%s'.\n", fn);
		return 1;
	}

	l = i_size_read(filp->f_dentry->d_inode);
	IPW_DEBUG_FW("Loading %ld bytes for firmware '%s'\n", l, fn);
	
	if (vfs_read(filp, (char *)&h, sizeof(h), &filp->f_pos) != sizeof(h)) {
		printk(KERN_INFO "Failed to read '%s'.\n", fn);
		goto fail;
	}

	if (IPW2100_FW_MAJOR(h.version) != IPW2100_FW_MAJOR_VERSION) {
		printk(KERN_WARNING "Firmware image not compatible "
		       "(detected version id of %u). "
		       "See Documentation/networking/README.ipw2100\n",
		       h.version);
		goto fail;
	}

	fw->version = h.version;

	if (ipw2100_fw_load(filp, &fw->fw, h.fw_size))
		goto fail;

	if (ipw2100_fw_load(filp, &fw->uc, h.uc_size))
		goto fail;

	filp_close(filp, current->files);
	return 0;

 fail:
	ipw2100_fw_free(fw);
	filp_close(filp, current->files);
	return 1;
}

#endif /* LINUX_VERSION_CODE < 2.6.0 */


static int ipw2100_mod_firmware_load(const char *fn, struct ipw2100_fw *fw)
{
	int r;
	mm_segment_t fs = get_fs();
	set_fs(get_ds());
	r = ipw2100_do_mod_firmware_load(fn, fw);
	set_fs(fs);
	return r;
}

static inline struct list_head *ipw2100_fw_read(
	struct list_head *e, struct ipw2100_fw_chunk_set *cs,
	unsigned char *data, size_t len)
{
	struct ipw2100_fw_chunk *c = list_entry(e, struct ipw2100_fw_chunk, 
						list);
	unsigned int avail = c->len - c->pos;
	if (avail <= len) {
		struct ipw2100_fw_chunk *tmp;

		memcpy(data, c->buf + c->pos, avail);
		c->pos = 0;

		IPW_DEBUG_INFO("advancing to next chunk...\n");

		e = e->next;
		tmp = list_entry(e, struct ipw2100_fw_chunk, list);

		if (avail != len) {
			memcpy(data + avail, 
			       tmp->buf + tmp->pos, 
			       len - avail);
			tmp->pos += len - avail;
		}

		return e;
	} 

	memcpy(data, c->buf + c->pos, len);
	c->pos += len;

	return e;
}

static inline struct list_head *ipw2100_fw_readw(
	struct list_head *e, struct ipw2100_fw_chunk_set *cs, 
	unsigned short *data)
{
	return ipw2100_fw_read(e, cs, (unsigned char *)data, sizeof(*data));
}

static inline struct list_head *ipw2100_fw_readl(
	struct list_head *e, struct ipw2100_fw_chunk_set *cs,
	unsigned int *data)
{
	return ipw2100_fw_read(e, cs, (unsigned char *)data, sizeof(*data));
}

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */

static int ipw2100_mod_firmware_load(struct ipw2100_fw *fw)
{
	struct ipw2100_fw_header *h = 
		(struct ipw2100_fw_header *)fw->fw_entry->data;

	if (IPW2100_FW_MAJOR(h->version) != IPW2100_FW_MAJOR_VERSION) {
		printk(KERN_WARNING "Firmware image not compatible "
		       "(detected version id of %u). "
		       "See Documentation/networking/README.ipw2100\n",
		       h->version);
		return 1;
	}

	fw->version = h->version;
	fw->fw.data = fw->fw_entry->data + sizeof(struct ipw2100_fw_header);
	fw->fw.size = h->fw_size;
	fw->uc.data = fw->fw.data + h->fw_size;
	fw->uc.size = h->uc_size;

	return 0;
}

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */


int ipw2100_get_firmware(struct ipw2100_priv *priv, struct ipw2100_fw *fw)
{
	char *fw_name;

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD
	int err = 0;

	printk(KERN_DEBUG "%s: Using legacy firmware load.\n",
	       priv->ndev->name);

	if (!firmware) {
		switch (priv->ctx->port_type) {
		case IBSS:
			fw_name = "/etc/firmware/" IPW2100_FW_NAME("-i");
			break;
#ifdef CONFIG_IPW2100_PROMISC
		case MONITOR:
			fw_name = "/etc/firmware/" IPW2100_FW_NAME("-p");
			break;
#endif
		case BSS:
		default:
			fw_name = "/etc/firmware/" IPW2100_FW_NAME("");
			break;
		}
	} else 
		fw_name = firmware;

	err = ipw2100_mod_firmware_load(fw_name, fw);
	if (err) {
		printk(KERN_ERR "%s: Firmware '%s' not available. "
		       "See Documentation/networking/README.ipw2100\n",
		       priv->ndev->name, fw_name);
		return -EIO;
	}

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */
	int rc;

	printk(KERN_DEBUG "%s: Using hotplug firmware load.\n",
	       priv->ndev->name);
	
	switch (priv->ctx->port_type) {
	case IBSS:
		fw_name = IPW2100_FW_NAME("-i");
		break;
#ifdef CONFIG_IPW2100_PROMISC
	case MONITOR:
		fw_name = IPW2100_FW_NAME("-p");
		break;
#endif
	case BSS:
	default:
		fw_name = IPW2100_FW_NAME("");
		break;
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	rc = request_firmware(&fw->fw_entry, fw_name, pci_name(priv->pdev));
#else
	rc = request_firmware(&fw->fw_entry, fw_name, &priv->pdev->dev);
#endif /* LINUX_VERSION_CODE */

	if (rc < 0) {
		printk(KERN_ERR
		       "%s: Firmware '%s' not available or load failed.\n",
		       priv->ndev->name, fw_name);
		return rc;
	}
	IPW_DEBUG_INFO("firmware data %p size %d", fw->fw_entry->data,
			   fw->fw_entry->size);

	ipw2100_mod_firmware_load(fw);

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	return 0;
}

void ipw2100_release_firmware(struct ipw2100_priv *priv,
			      struct ipw2100_fw *fw)
{
	fw->version = 0;
#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD

	ipw2100_fw_free(fw);

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	if (fw->fw_entry)
		release_firmware(fw->fw_entry);
	fw->fw_entry = NULL;

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */
}


int ipw2100_get_fwversion(struct ipw2100_priv *priv, char *buf, size_t max)
{
	char ver[MAX_FW_VERSION_LEN];
	u32 len = MAX_FW_VERSION_LEN;
	u32 tmp;
	int i;
	/* firmware version is an ascii string (max len of 14) */
	if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_FW_VER_NUM,
				ver, &len)) 
		return -EIO;
	tmp = max;
	if (len >= max)
		len = max - 1;
	for (i = 0; i < len; i++)
		buf[i] = ver[i];
	buf[i] = '\0';
	return tmp;
}

int ipw2100_get_ucodeversion(struct ipw2100_priv *priv, char *buf, size_t max)
{
	u32 ver;
	u32 len = sizeof(ver);
	/* microcode version is a 32 bit integer */
	if (ipw2100_get_ordinal(priv, IPW_ORD_UCODE_VERSION,
				&ver, &len))
		return -EIO;
	return snprintf(buf, max, "%08X", ver);
}

/* 
 * On exit, the firmware will have been freed from the fw list 
 */
int ipw2100_fw_download(struct ipw2100_priv *priv, struct ipw2100_fw *fw)
{
	/* firmware is constructed of N contiguous entries, each entry is
	 * structured as:
	 * 
	 * offset    sie         desc
	 * 0         4           address to write to
	 * 4         2           length of data run
         * 6         length      data
	 */
	unsigned int addr;
	unsigned short len;

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD
	unsigned char data[32];
	
	struct ipw2100_fw_chunk_set *cs = &fw->fw;
	struct list_head *e = cs->chunk_list.next;
	unsigned int size = cs->size;

	while (size > 0) {
		e = ipw2100_fw_readl(e, cs, &addr);
		size -= sizeof(addr);

		e = ipw2100_fw_readw(e, cs, &len);
		size -= sizeof(len);

		if (len > 32) {
			printk(KERN_ERR 
			       "Invalid firmware run-length of %d bytes\n",
			       len);
			return -EINVAL;
		}

		e = ipw2100_fw_read(e, cs, data, len);
		size -= len;

		write_nic_memory(priv->ndev, addr, len, data);
	}

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	const unsigned char *firmware_data = fw->fw.data;
	unsigned int firmware_data_left = fw->fw.size;

	while (firmware_data_left > 0) {
	   	addr = *(u32 *)(firmware_data);
		firmware_data      += 4;
		firmware_data_left -= 4;

	   	len = *(u16 *)(firmware_data);
		firmware_data      += 2;
		firmware_data_left -= 2;

		if (len > 32) {
			printk(KERN_ERR
			       "Invalid firmware run-length of %d bytes\n",
			       len);
			return -EINVAL;
		}

		write_nic_memory(priv->ndev, addr, len, firmware_data);
		firmware_data      += len;
		firmware_data_left -= len;
	}
#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	return 0;
}

struct symbol_alive_response {
	u8 cmd_id;
	u8 seq_num;
	u8 ucode_rev;
	u8 eeprom_valid;
	u16 valid_flags;
	u8 IEEE_addr[6];
	u16 flags;
	u16 pcb_rev;
	u16 clock_settle_time;	// 1us LSB
	u16 powerup_settle_time;	// 1us LSB
	u16 hop_settle_time;	// 1us LSB
	u8 date[3];		// month, day, year
	u8 time[2];		// hours, minutes
	u8 ucode_valid;
};

int ipw2100_ucode_download(struct ipw2100_priv *priv, struct ipw2100_fw *fw)
{
	struct net_device *dev = priv->ndev;

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD

	struct ipw2100_fw_chunk_set *cs = &fw->uc;
	struct list_head *e = cs->chunk_list.next;
	unsigned int size = cs->size;
	unsigned short uc_data;

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	const unsigned char *microcode_data = fw->uc.data;
	unsigned int microcode_data_left = fw->uc.size;

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	struct symbol_alive_response response;
	int i, j;
	u8 data;

	/* Symbol control */
	write_nic_word(dev, IPW2100_CONTROL_REG, 0x703);
	readl((void *)(dev->base_addr));
	write_nic_word(dev, IPW2100_CONTROL_REG, 0x707);
	readl((void *)(dev->base_addr));

	/* HW config */
	write_nic_byte(dev, 0x210014, 0x72);	/* fifo width =16 */
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210014, 0x72);	/* fifo width =16 */
	readl((void *)(dev->base_addr));

	/* EN_CS_ACCESS bit to reset control store pointer */
	write_nic_byte(dev, 0x210000, 0x40);
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210000, 0x0);
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210000, 0x40);
	readl((void *)(dev->base_addr));

	/* copy microcode from buffer into Symbol */

#ifdef CONFIG_IPW2100_LEGACY_FW_LOAD

	while (size > 0) {
		e = ipw2100_fw_readw(e, cs, &uc_data);
		size -= sizeof(uc_data);
		write_nic_byte(dev, 0x210010, uc_data & 0xFF);
		write_nic_byte(dev, 0x210010, (uc_data >> 8) & 0xFF);
	}

#else /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	while (microcode_data_left > 0) {
		write_nic_byte(dev, 0x210010, *microcode_data++);
		write_nic_byte(dev, 0x210010, *microcode_data++);
		microcode_data_left -= 2;
	}

#endif /* CONFIG_IPW2100_LEGACY_FW_LOAD */

	/* EN_CS_ACCESS bit to reset the control store pointer */
	write_nic_byte(dev, 0x210000, 0x0);
	readl((void *)(dev->base_addr));

	/* Enable System (Reg 0)
	 * first enable causes garbage in RX FIFO */
	write_nic_byte(dev, 0x210000, 0x0);
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210000, 0x80);
	readl((void *)(dev->base_addr));

	/* Reset External Baseband Reg */
	write_nic_word(dev, IPW2100_CONTROL_REG, 0x703);
	readl((void *)(dev->base_addr));
	write_nic_word(dev, IPW2100_CONTROL_REG, 0x707);
	readl((void *)(dev->base_addr));

	/* HW Config (Reg 5) */
	write_nic_byte(dev, 0x210014, 0x72);	// fifo width =16
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210014, 0x72);	// fifo width =16
	readl((void *)(dev->base_addr));

	/* Enable System (Reg 0)
	 * second enable should be OK */
	write_nic_byte(dev, 0x210000, 0x00);	// clear enable system
	readl((void *)(dev->base_addr));
	write_nic_byte(dev, 0x210000, 0x80);	// set enable system

	/* check Symbol is enabled - upped this from 5 as it wasn't always
	 * catching the update */
	for (i = 0; i < 10; i++) {
		udelay(10);

		/* check Dino is enabled bit */
		read_nic_byte(dev, 0x210000, &data);
		if (data & 0x1)
			break;
	}

	if (i == 10) {
		printk(KERN_ERR "%s: Error initializing Symbol\n",
		       dev->name);
		return -EIO;
	}

	/* Get Symbol alive response */
	for (i = 0; i < 30; i++) {
		/* Read alive response structure */
		for (j = 0; 
		     j < (sizeof(struct symbol_alive_response) >> 1); 
		     j++)
			read_nic_word(dev, 0x210004,
				      ((u16 *)&response) + j);

		if ((response.cmd_id == 1) &&
		    (response.ucode_valid == 0x1))
			break;
		udelay(10);
	}

	if (i == 30) {
		printk(KERN_ERR "%s: No response from Symbol - hw not alive\n",
		       dev->name);
		printk_buf((u8*)&response, sizeof(response));
		return -EIO;
	}

	return 0;
}
